#!/usr/bin/python

# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import json
import base64

from shinken.log import logger

### Will be populated by the UI with it's own value
app = None


def user_login_redirect():
    app.bottle.redirect("/")
    return {}


def user_logout():
    # To delete it, send the same, with different date
    user_id = app.request.get_cookie("user", secret=app.auth_secret)
    if user_id:
        app.response.set_cookie('user', '', secret=app.auth_secret, path='/', httponly=True)
    else:
        app.response.set_cookie('user', '', secret=app.auth_secret, path='/', httponly=True)
    app.bottle.redirect("/")
    return {}


def user_logout_api():
    user_id = app.request.get_cookie("user", secret=app.auth_secret)
    if user_id:
        app.response.set_cookie('user', '', secret=app.auth_secret, path='/', httponly=True)
    else:
        app.response.set_cookie('user', '', secret=app.auth_secret, path='/', httponly=True)
    return {}


def user_auth():
    login64 = app.request.forms.get('login', '')
    password64 = app.request.forms.get('password', '')
    try:
        password = base64.b64decode(password64).decode('utf8', 'ignore')
        login = base64.b64decode(login64).decode('utf8', 'ignore')
    except Exception, exp:  # bad encoding? get out!
        return app.abort(400, str(exp))
    is_mobile = app.request.forms.get('is_mobile', '0')
    contact = app.check_auth(login, password)
    is_valid = contact
    
    if is_valid:
        if isinstance(contact, dict):
            contact_id = contact['_id']
        else:
            contact_id = contact.uuid
        app.response.set_cookie('user', contact_id, secret=app.auth_secret, path='/', expires='Fri, 01 Jan 2100 00:00:00 GMT', httponly=True)
        token = app.get_token(login)
        app.response.headers['X-Shinken-Token'] = token
        if is_mobile == '1':
            app.bottle.redirect("/mobile/main")
        else:
            app.bottle.redirect("/problems")
    else:
        app.bottle.redirect("/user/login?error=Invalid user or Password")
    
    return {'app': app, 'is_auth': is_valid}


def user_auth_api():
    login64 = app.request.forms.get('login', '')
    password64 = app.request.forms.get('password', '')
    
    try:
        password = base64.b64decode(password64).decode('utf8', 'ignore')
        login = base64.b64decode(login64).decode('utf8', 'ignore')
    except Exception, exp:  # bad encoding? get out!
        return app.abort(400, str(exp))
    hidden_password = '*' * len(password)
    
    logger.debug("user_auth_api %s and %s" % (login, hidden_password))
    contact = app.check_auth(login, password)
    
    if contact:
        if isinstance(contact, dict):
            contact_id = contact.get('_id', '')
        else:
            contact_id = contact.uuid
        app.response.set_cookie('user', contact_id, secret=app.auth_secret, path='/', expires='Fri, 01 Jan 2100 00:00:00 GMT', httponly=True)
        token = app.get_token(contact_id)
        app.response.headers['X-Shinken-Token'] = token
        return json.dumps(token)
    else:
        return app.abort(401, '')


def user_auth_nagvis_api():
    restrict_to_shinken_admin64 = app.request.forms.get('restrict_to_shinken_admin', '0')
    try:
        restrict_to_shinken_admin = base64.b64decode(restrict_to_shinken_admin64).decode('utf8', 'ignore')
        if restrict_to_shinken_admin == '1':
            restrict_to_shinken_admin = True
        else:
            restrict_to_shinken_admin = False
    except Exception, exp:  # bad encoding? get out!
        restrict_to_shinken_admin = False
    
    # If a http header is provided for SSO and it corresponds to a valid user, return username
    if app.remote_user_enable == '1':
        if app.remote_user_variable in app.request.headers:
            logger.debug("NagVis auth via HTTP header requested")
            username_from_header = app.request.headers[app.remote_user_variable]
            
            case_sensitive = app.remote_user_case_sensitive
            if case_sensitive:
                user_name = app.datamgr.get_contact(username_from_header)
            else:
                user_name = app.datamgr.get_contact_case_insensitive(username_from_header)
                
            if user_name and (user_name.is_admin or not restrict_to_shinken_admin):
                logger.debug("NagVis auth with HTTP header successful for user %s" % user_name.contact_name)
                return user_name.contact_name
            elif user_name and not user_name.is_admin and restrict_to_shinken_admin:
                logger.debug("NagVis auth with HTTP header rejected for user %s (user is not a Shinken administrator)" % user_name.contact_name)
                return app.abort(401, '')
            else:
                logger.debug("NagVis auth with HTTP header rejected for user %s" % user_name.contact_name)
                return app.abort(401, '')
            
    # If a cookie has been provided and it corresponds to a valid user session, return username
    user_id = app.request.get_cookie("user", secret=app.auth_secret)
    if user_id:
        logger.debug("NagVis auth via cookie requested")
        user = app.datamgr.get_contact(user_id, by_id=True)
        if user and (user.is_admin or not restrict_to_shinken_admin):
            logger.debug("NagVis auth via cookie successful for user %s" % user.contact_name)
            return user.contact_name
        elif user and not user.is_admin and restrict_to_shinken_admin:
            logger.debug("NagVis auth via cookie rejected for user %s (user is not a Shinken administrator)" % user.contact_name)
            return app.abort(401, '')
        else:
            logger.debug("NagVis auth via cookie rejected for user %s" % user_id)
            return app.abort(401, '')
    
    # If no cookie provided in request, proceed to check auth credentials
    login64 = app.request.forms.get('login', '')
    password64 = app.request.forms.get('password', '')
    
    try:
        password = base64.b64decode(password64).decode('utf8', 'ignore')
        login = base64.b64decode(login64).decode('utf8', 'ignore')
    except Exception, exp:  # bad encoding? get out!
        return app.abort(400, str(exp))
    hidden_password = '*' * len(password)
    
    logger.debug("user_auth_api %s and %s" % (login, hidden_password))
    contact = app.check_auth(login, password)
    
    if contact:
        if isinstance(contact, dict):
            contact_id = contact['_id']
        else:
            contact_id = contact.uuid
            
        if contact.is_admin or not restrict_to_shinken_admin:
            app.response.set_cookie('user', contact_id, secret=app.auth_secret, path='/', expires='Fri, 01 Jan 2100 00:00:00 GMT', httponly=True)
            token = app.get_token(login)
            app.response.headers['X-Shinken-Token'] = token
            return json.dumps(token)
        else:
            return app.abort(401, '')
    else:
        return app.abort(401, '')


# manage the /. If the user is known, go to problems page.
# Should be /dashboard in the future. If not, go login :)
def get_root():
    # user = app.request.get_cookie("user", secret=app.auth_secret)
    return app.bottle.redirect("/static/ui/index.html")


def login_mobile():
    user = app.get_user_auth()
    if user:
        app.bottle.redirect("/mobile/main")
    
    err = app.request.GET.get('error', None)
    login_text = app.login_text
    
    return {'error': err, 'login_text': login_text}


def get_placeholder():
    for mod in app.modules_manager.get_internal_instances():
        if getattr(mod, 'login_placeholder', ''):
            return {'placeholder': mod.login_placeholder}
    return {}


def get_token():
    user = app.get_user_auth()
    token = app.get_token(user.uuid)
    app.response.headers['X-Shinken-Token'] = token
    return json.dumps(token)


pages = {
    user_login_redirect : {'routes': ['/login'], 'static': True, 'wrappers': []},
    get_placeholder     : {'routes': ['/login/placeholder'], 'wrappers': []},
    user_auth           : {'routes': ['/user/auth'], 'view': 'auth', 'method': 'POST', 'static': True, 'wrappers': []},
    user_auth_api       : {'routes': ['/auth'], 'method': 'POST', 'wrappers': ['json']},
    user_auth_nagvis_api: {'routes': ['/auth_nagvis'], 'method': 'POST', 'wrappers': ['json']},
    user_logout_api     : {'routes': ['/apilogout'], 'method': 'POST', 'wrappers': ['json']},
    user_logout         : {'routes': ['/user/logout', '/logout'], 'static': True, 'wrappers': []},
    get_root            : {'routes': ['/'], 'static': True, 'wrappers': []},
    get_token           : {'routes': ['/get_token'], 'static': True, 'wrappers': ['auth', 'json']},
}
