#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import random
import time

from shinken.satellitelink import SatelliteLink, SatelliteLinks
from shinken.property import BoolProp, IntegerProp, StringProp
from shinken.log import logger
from shinken.http_client import HTTPExceptions
from shinken.util import to_mb_size


class SchedulerLink(SatelliteLink):
    """Please Add a Docstring to describe the class here"""
    
    id = 0
    
    # Ok we lie a little here because we are a mere link in fact
    my_type = 'scheduler'
    
    properties = SatelliteLink.properties.copy()
    properties.update({
        'scheduler_name'    : StringProp(fill_brok=['full_status']),
        'port'              : IntegerProp(default='7768', fill_brok=['full_status']),
        'weight'            : IntegerProp(default='1', fill_brok=['full_status']),
        'skip_initial_broks': BoolProp(default='0', fill_brok=['full_status']),
    })
    
    running_properties = SatelliteLink.running_properties.copy()
    running_properties.update({
        'conf'              : StringProp(default=None),
        'need_conf'         : StringProp(default=True),
        'external_commands' : StringProp(default=[]),
        'push_flavor'       : IntegerProp(default=0),
        'current_satellites': StringProp(default={'broker': [], 'reactionner': [], 'poller': [], 'receiver': []}),
    })
    
    
    def get_name(self):
        return self.scheduler_name
    
    
    def run_external_commands(self, commands):
        if self.con is None:
            self.create_connection()
        if not self.alive:
            return None
        logger.debug("[SchedulerLink] Sending %d commands" % len(commands))
        try:
            self.con.post('run_external_commands', {'cmds': commands})
        except HTTPExceptions, exp:
            self.con = None
            logger.debug(exp)
            return False
    
    
    def register_to_my_realm(self):
        self.realm.schedulers.append(self)
    
    
    def give_satellite_cfg(self):
        return {'port'               : self.port, 'address': self.address,
                'name'               : self.scheduler_name,
                'instance_id'        : self.id,
                'active'             : self.conf is not None,
                'push_flavor'        : self.push_flavor,
                'use_ssl'            : self.use_ssl,
                'hard_ssl_name_check': self.hard_ssl_name_check,
                'timeout'            : self.timeout,
                'data_timeout'       : self.data_timeout,
                }
    
    
    # Some parameters can give as 'overridden parameters' like use_timezone
    # so they will be mixed (in the scheduler) with the standard conf sent by the arbiter
    def get_override_configuration(self):
        r = {}
        properties = self.__class__.properties
        for prop, entry in properties.iteritems():
            if entry.override:
                r[prop] = getattr(self, prop)
        return r
    
    
    def create_and_put_active_scheduler_configuration(self, shard, arbiter_trace, configuration_incarnation):
        realm_name = self.realm.get_name()
        logger.info('[DISPATCH][%s] SENDING SHARD   Trying to send shard %d to scheduler %s' % (realm_name, shard.id, self.get_name()))
        if not self.need_conf:
            logger.info('[DISPATCH][%s] SCHEDULER ALREADY SET   The scheduler %s do not need a shard.' % (realm_name, self.get_name()))
            return False
        
        # We give this configuration a new 'flavor'
        shard.push_flavor = random.randint(1, 1000000)
        # REF: doc/shinken-conf-dispatching.png (3)
        # REF: doc/shinken-scheduler-lost.png (2)
        override_conf = self.get_override_configuration()
        satellites_for_sched = self.realm.get_satellites_links_for_scheduler()
        serialized_shard = shard.get_serialized_configuration()
        shard_size = len(serialized_shard)
        arbiter_trace['arbiter_time'] = time.time()  # update arbiter time with the current time
        
        # Prepare the conf before sending it
        conf_package = {
            'conf'                             : serialized_shard,
            'override_conf'                    : override_conf,
            'modules'                          : self.modules,
            'satellites'                       : satellites_for_sched,
            'instance_name'                    : self.scheduler_name,
            'push_flavor'                      : shard.push_flavor,
            'skip_initial_broks'               : self.skip_initial_broks,
            'realm'                            : self.realm.get_name(),
            'activated'                        : True,
            'spare'                            : self.spare,
            'arbiter_trace'                    : arbiter_trace,
            'configuration_incarnation'        : configuration_incarnation,
            'vmware__statistics_compute_enable': self.vmware__statistics_compute_enable,
        }
        
        t1 = time.time()
        is_sent = self.put_conf(conf_package)
        sent_time = time.time() - t1
        sent_speed = shard_size / sent_time
        
        if not is_sent:
            logger.error('[DISPATCH][%s] SHARD NOT SENT   shard [%d] - flavor [%s] dispatching error for scheduler %s' % (realm_name, shard.id, shard.push_flavor, self.get_name()))
            return False
        logger.info("[DISPATCH][%s] SENT TIME   Shard [%s] sent time is %.2fs (size=%.3fMB speed=%.3fMB/s)" % (realm_name, shard.id, sent_time, to_mb_size(shard_size), to_mb_size(sent_speed)))
        logger.info('[DISPATCH][%s] SHARD SENT TO SCHEDULER   Dispatch OK of shard [%d] - flavor [%s] in scheduler %s' % (realm_name, shard.id, shard.push_flavor, self.get_name()))
        
        self.conf = shard
        self.push_flavor = shard.push_flavor
        self.need_conf = False
        shard.is_assigned = True
        shard.assigned_to = self
        
        # We update all data for this scheduler
        self.managed_confs = {shard.id: shard.push_flavor}
        return True
    
    
    def create_and_put_inactive_scheduler_configuration(self, arbiter_trace, configuration_incarnation):
        if not self.alive or not self.reachable or self.conf is not None:
            return
        realm_name = self.realm.get_name()
        # We give this configuration a new 'flavor'
        push_flavor = random.randint(1, 1000000)
        extended_conf = {
            'conf'                             : None,
            'override_conf'                    : {},
            'modules'                          : (),
            'satellites'                       : {},
            'instance_name'                    : self.get_name(),
            'push_flavor'                      : push_flavor,
            'skip_initial_broks'               : None,
            'realm'                            : realm_name,
            'activated'                        : False,
            'spare'                            : self.spare,
            'arbiter_trace'                    : arbiter_trace,
            'vmware__statistics_compute_enable': self.vmware__statistics_compute_enable,
            'configuration_incarnation'        : configuration_incarnation,
        }
        
        t1 = time.time()
        
        is_sent = self.put_conf(extended_conf)
        
        # It can be:
        # * spare = spare daemon that just go as spare
        # * idle  = NOT spare daemon but need to sleep a bit until the dispatch can come
        conf_type_sent = 'spare' if self.spare else 'idle'
        
        if not is_sent:
            logger.error('[DISPATCH][%s][%s] SHARD NOT SENT   %s configuration - flavor [%s] dispatching error for scheduler %s' % (realm_name, conf_type_sent.upper(), conf_type_sent, push_flavor, self.get_name()))
            return
        logger.info("[DISPATCH][%s][%s] SENT TIME   Shard sent time is %.2fs" % (realm_name, conf_type_sent.upper(), (time.time() - t1)))
        logger.info('[DISPATCH][%s][%s] SHARD SENT TO SCHEDULER   Dispatch OK of %s configuration - flavor [%s] in scheduler %s' % (realm_name, conf_type_sent.upper(), conf_type_sent, push_flavor, self.get_name()))
        self.push_flavor = push_flavor
        self.need_conf = True
    
    
    def update_managed_list(self):
        super(SchedulerLink, self).update_managed_list()
        self._refresh_current_satellites()
    
    
    def _refresh_current_satellites(self):
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return
            
            try:
                current_satellites = self.con.get('get_current_satellites')
                logger.debug("[DISPATCH] [%s] have satellites [%s]" % (self.get_name(), current_satellites))
                
                if current_satellites is None:  # HTTP fail? will be detected by ping
                    return
                self.current_satellites = current_satellites
            except HTTPExceptions, exp:
                logger.warning("[%s][%s] Call to know what the daemon is currently managing did fail: %s" % (self._get_my_realm_name(), self.get_name(), exp))
    
    
    def assert_only_allowed_brokers(self, allowed_broker_names):
        to_remove = []
        current_brokers = self.current_satellites['broker']
        for current_broker in current_brokers:
            if current_broker not in allowed_broker_names:
                to_remove.append(current_broker)
        # Maybe there is no problem
        if len(to_remove) == 0:
            return
        
        # Maybe the scheduler did gone, will be detected in the ping
        if self.con is None:
            return
        
        logger.info('[DISPATCH] [%s] [%s] We are removing old brokers %s from the scheduler as they are no more need currently.' % (self._get_my_realm_name(), self.get_name(), ','.join(to_remove)))
        
        try:
            self.con.post('satellites_to_remove', {'to_remove': {'broker': to_remove}})
            # Now we did remove some satellites, refresh the list so we are up to date
            # and we are OK with what the daemon REALLY did ^^
            self._refresh_current_satellites()
        except HTTPExceptions as exp:
            logger.warning("[DISPATCH] [%s] [%s] Cannot remove old brokers %s in the scheduler. We will retry in the next seconds: %s" % (self._get_my_realm_name(), self.get_name(), ','.join(to_remove), exp))


class SchedulerLinks(SatelliteLinks):
    """Please Add a Docstring to describe the class here"""
    
    name_property = "scheduler_name"
    inner_class = SchedulerLink
