#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.fast_copy import fast_deepcopy
from .massive_change_returns import MASSIVE_CHANGE_RETURNS
from .massive_change_status import MASSIVE_CHANGE_STATUS
from ....dao.items.baseitem import BaseItem
from ....front_end.helper_object import COUNTER
from ....front_end.mass_change_helper import MassChangeCounterSelect


class MassiveChangeElement(object):
    
    def __init__(self, item):
        self.item = item
        if isinstance(item, BaseItem):
            self._id = item.get_uuid()
            self.name = item.get_name()
            self.item_type = item.get_type()
            
            self.raw_item = fast_deepcopy(item.get_raw_item(flatten_links=False))
        else:
            self._id = item.get('_id', '')
            self.name = item.get('name', '')
            self.item_type = item.get('item_type', '')
            
            self.raw_item = fast_deepcopy(item)
        
        self.properties = []
        self.modifications = 0
        self.overrides = []
        
        # Validation
        self.has_validation_messages = False
        self.error_counter = 0
        self.warning_counter = 0
        self._validation_errors_html = ""
        self._validation_warnings_html = ""
    
    
    def add_property(self, prop, modified):
        self.properties.append(prop)
        if modified:
            self.modifications += 1
    
    
    def add_overrides(self, override, modified):
        self.overrides.append(override)
        if modified:
            self.modifications += 1
    
    
    def get_overrides(self):
        return self.overrides
    
    
    def get_properties(self):
        return self.properties
    
    
    def get_property(self, name):
        for prop in self.get_properties():
            if prop.name == name:
                return prop
        return None
    
    
    def get_uuid(self):
        return self._id
    
    
    def get_name(self):
        return self.name
    
    
    def get_name_to_html(self, app):
        return app.helper.helper_object.build_link_dom_element_depend_if_admin(self.item_type, self._id, self.get_name())
    
    
    def set_validation(self, validation, app):
        self.has_validation_messages = validation.get('has_messages', False)
        if self.has_validation_messages:
            self.error_counter = len(validation.get('error_messages', 0))
            self._validation_errors_html = app.helper.helper_component.get_list_not_ordered_html(
                validation.get('error_messages'),
                title=app._('common.errors'),
                class_to_add_container="shinken-error-message")
            
            self.warning_counter = len(validation.get('warning_messages', 0))
            self._validation_warnings_html = app.helper.helper_component.get_list_not_ordered_html(
                validation.get('warning_messages'),
                title=app._('common.warnings'),
                class_to_add_container="shinken-warning-message")
            if self.error_counter:
                for _property in self.get_properties():
                    _property._modified = MASSIVE_CHANGE_STATUS.UNMODIFIED
                    _property.message = MASSIVE_CHANGE_RETURNS.ITEM_NOT_VALIDATE
    
    
    def get_counter_cell_content(self, app):
        if not self.has_validation_messages:
            return ""
        counters_values = {COUNTER.ERROR: self.error_counter or "", COUNTER.WARNING: self.warning_counter or ""}
        _to_return = '<div class="shinken-counters-container shinken-user-select-none shinken-toggle-button-active">%s</div>'
        return _to_return % app.helper.helper_object.get_counters_html(counters=[COUNTER.ERROR, COUNTER.WARNING], counters_value=counters_values)
    
    
    def get_counter_cell_filter_value(self):
        _to_return = MassChangeCounterSelect.EMPTY
        if self.error_counter and self.warning_counter:
            _to_return = "ERROR,WARNING"
        elif self.error_counter:
            _to_return = "ERROR"
        elif self.warning_counter:
            _to_return = "WARNING"
        return _to_return
