#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

from collections import namedtuple
from ....dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE, METADATA, NOT_TO_LOOK, STOP_INHERITANCE_VALUES, VALUE_FORCE_DEFAULT, ADVANCED_PERIODS, SERVICE_OVERRIDE
from ....front_end.component_select_options import ComponentOption
from ....dao.helpers import get_default_label
from ....front_end.mass_change_helper import MassChangeHelperHtml

mock_property = namedtuple('MockProp', ['name', 'handle_additive_inheritance', 'merging'])

FORBIDDEN_PROPERTIES = {
    ITEM_TYPE.HOSTS                    : [SERVICE_OVERRIDE],
    ITEM_TYPE.HOSTTPLS                 : [SERVICE_OVERRIDE],
    ITEM_TYPE.HOSTGROUPS               : [],
    ITEM_TYPE.CLUSTERS                 : [SERVICE_OVERRIDE, 'service_excludes_by_id'],
    ITEM_TYPE.CLUSTERTPLS              : [SERVICE_OVERRIDE, 'service_excludes_by_id'],
    ITEM_TYPE.SERVICESHOSTS            : [],
    ITEM_TYPE.SERVICESHOSTTPLS         : [],
    ITEM_TYPE.SERVICESCLUSTERS         : [],
    ITEM_TYPE.SERVICESCLUSTERTPLS      : [],
    ITEM_TYPE.SERVICETPLS              : [],
    ITEM_TYPE.CONTACTS                 : ['host_notification_period', 'service_notification_period', 'host_notifications_enabled', 'service_notifications_enabled', 'host_notification_commands', 'service_notification_commands',
                                          'host_notification_options', 'service_notification_options', 'min_business_impact'],
    ITEM_TYPE.CONTACTTPLS              : ['host_notification_period', 'service_notification_period', 'host_notifications_enabled', 'service_notifications_enabled', 'host_notification_commands', 'service_notification_commands',
                                          'host_notification_options', 'service_notification_options', 'min_business_impact'],
    ITEM_TYPE.CONTACTGROUPS            : [],
    ITEM_TYPE.ESCALATIONS              : ['first_notification', 'last_notification'],
    ITEM_TYPE.NOTIFICATIONWAYS         : [],
    ITEM_TYPE.COMMANDS                 : [],
    ITEM_TYPE.BUSINESSIMPACTMODULATIONS: [],
    ITEM_TYPE.MACROMODULATIONS         : [],
    ITEM_TYPE.RESULTMODULATIONS        : [],
    ITEM_TYPE.TIMEPERIODS              : [],
}

ALLOWED_PROPERTIES = {
    ITEM_TYPE.HOSTS                    : ['pack'],
    ITEM_TYPE.HOSTTPLS                 : ['pack', 'for_all_users'],
    ITEM_TYPE.HOSTGROUPS               : ['pack'],
    ITEM_TYPE.CLUSTERS                 : ['pack'],
    ITEM_TYPE.CLUSTERTPLS              : ['pack'],
    ITEM_TYPE.SERVICESHOSTS            : ['pack', 'check_command_args'],
    ITEM_TYPE.SERVICESHOSTTPLS         : ['pack', 'check_command_args'],
    ITEM_TYPE.SERVICESCLUSTERS         : ['pack'],
    ITEM_TYPE.SERVICESCLUSTERTPLS      : ['pack'],
    ITEM_TYPE.SERVICETPLS              : ['pack'],
    ITEM_TYPE.CONTACTS                 : ['pack'],
    ITEM_TYPE.CONTACTTPLS              : ['pack', 'for_all_users'],
    ITEM_TYPE.CONTACTGROUPS            : ['pack'],
    ITEM_TYPE.ESCALATIONS              : ['pack'],
    ITEM_TYPE.NOTIFICATIONWAYS         : ['pack'],
    ITEM_TYPE.COMMANDS                 : ['pack'],
    ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['pack'],
    ITEM_TYPE.MACROMODULATIONS         : ['pack'],
    ITEM_TYPE.RESULTMODULATIONS        : ['pack'],
    ITEM_TYPE.TIMEPERIODS              : ['pack', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday', ADVANCED_PERIODS],
}

LINKED_PROPERTIES_FLAT_WHEN_CHANGED = {
    ITEM_TYPE.HOSTS                    : [],
    ITEM_TYPE.HOSTTPLS                 : [],
    ITEM_TYPE.HOSTGROUPS               : [],
    ITEM_TYPE.CLUSTERS                 : ['bp_rule'],
    ITEM_TYPE.CLUSTERTPLS              : ['bp_rule'],
    ITEM_TYPE.SERVICESHOSTS            : [],
    ITEM_TYPE.SERVICESHOSTTPLS         : ['host_name'],
    ITEM_TYPE.SERVICESCLUSTERS         : [],
    ITEM_TYPE.SERVICESCLUSTERTPLS      : ['host_name'],
    ITEM_TYPE.SERVICETPLS              : [],
    ITEM_TYPE.CONTACTS                 : [],
    ITEM_TYPE.CONTACTTPLS              : [],
    ITEM_TYPE.CONTACTGROUPS            : [],
    ITEM_TYPE.ESCALATIONS              : [],
    ITEM_TYPE.NOTIFICATIONWAYS         : [],
    ITEM_TYPE.COMMANDS                 : [],
    ITEM_TYPE.BUSINESSIMPACTMODULATIONS: [],
    ITEM_TYPE.MACROMODULATIONS         : [],
    ITEM_TYPE.RESULTMODULATIONS        : [],
    ITEM_TYPE.TIMEPERIODS              : [],
    
}


def init_forbidden_properties():
    # Add property enabled and name for each item_type
    for item_type, item_def in DEF_ITEMS.iteritems():
        FORBIDDEN_PROPERTIES[item_type].extend([item_def['key_name'], 'enabled'])
        FORBIDDEN_PROPERTIES[item_type].extend(NOT_TO_LOOK)


init_forbidden_properties()


def add_item_in_list(all_items, _item_id, _item_type, _item_state, _item_name, enabled):
    mock_item = {
        DEF_ITEMS[_item_type]['key_name']: _item_name,
        '_id'                            : _item_id,
        'enabled'                        : enabled
    }
    METADATA.update_metadata(mock_item, METADATA.ITEM_TYPE, _item_type)
    METADATA.update_metadata(mock_item, METADATA.STATE, _item_state)
    all_items[_item_id] = mock_item


def parse_form_links_to_html(links, prop_name, item_type, app, datamanagerV2):
    _to_return = []
    if len(links) == 1 and links[0].get('name', '') in STOP_INHERITANCE_VALUES:
        if links[0]['name'] == VALUE_FORCE_DEFAULT:
            return get_default_label(item_type, prop_name, links[0]['name'], app)
        else:
            return '%s [%s]' % (app._('element.forced_default'), get_default_label(item_type, prop_name, links[0]['name'], app))
    for link in links:
        _to_return.append(parse_form_link_to_html(link, prop_name, item_type, app, datamanagerV2))
        continue
    return ''.join(_to_return)


def parse_form_link_to_html(link, prop_name, item_type, app, datamanagerV2):
    linked_item = {}
    _name = ''
    link_item_type = None
    
    if link['exists']:
        _link_item_type = link['item_type']
        _where = {'_id': link['_id']}
        _states = (ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING) if ITEM_TYPE.has_work_area(_link_item_type) else (ITEM_STATE.NEW, ITEM_STATE.STAGGING)
        items = datamanagerV2.find_merge_state_items(_link_item_type, _states, where=_where)
        if items:
            linked_item = items[0]
            link_item_type = linked_item.get_type()
            _name = linked_item.get_name()
    else:
        possible_link_item_type = DEF_ITEMS[item_type]['props_links'][prop_name]
        for _link_item_type in possible_link_item_type:
            _key = DEF_ITEMS[_link_item_type]['key_name']
            _states = (ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING) if ITEM_TYPE.has_work_area(_link_item_type) else (ITEM_STATE.NEW, ITEM_STATE.STAGGING)
            _where = {_key: link['name']}
            items = datamanagerV2.find_merge_state_items(_link_item_type, _states, where=_where)
            if items:
                linked_item = items[0]
                link_item_type = linked_item.get_type()
                _name = linked_item.get_name()
                break
        
        if not linked_item:
            _name = link['name']
    
    _current_entry = ComponentOption(
        METADATA.get_metadata(linked_item, METADATA.STATE),
        linked_item.get('enabled', '1'),
        _name,
        False,
        0,
        state_element=METADATA.get_metadata(linked_item, METADATA.STATE),
        _id=linked_item.get('_id', None),
        item_type=link_item_type
    )
    
    return app.helper.helper_object.get_tag_container_html(prop_name, _current_entry, add_close=False, add_dragging=False)


def parse_command_link_to_html(value, prop_name, item_type, app, datamanagerV2):
    _name = value['node']['link'].get('name', '')
    if _name in STOP_INHERITANCE_VALUES:
        if _name == VALUE_FORCE_DEFAULT:
            return get_default_label(item_type, prop_name, _name, app)
        else:
            return '%s [%s]' % (app._('element.forced_default'), get_default_label(item_type, prop_name, _name, app))
    else:
        _value = parse_form_link_to_html(value['node']['link'], prop_name, item_type, app, datamanagerV2)
        _value += MassChangeHelperHtml.get_command_arg_html(value['node']['args'])
    return _value
