#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import json
import time

from shinken.log import LoggerFactory

# Use only for type hint in PyCharm
try:
    # noinspection PyUnresolvedReferences
    from event_container.module import EventContainerWebUIModule
    from event_container.ec_database_connection import ECDatabaseError
    # noinspection PyUnresolvedReferences
    from event_container.ec_event import Event
except ImportError:
    pass

logger = LoggerFactory.get_logger('event_container')
app = None


def _send_stat_to_module(stat_name, stat_endpoint, filters, stat_time):
    _module = get_event_container_module()
    _module.send_stat_to_reader_stats(stat_name, stat_endpoint, filters, stat_time)


def get_event_container_module():
    # type: () -> EventContainerWebUIModule
    event_container_module = next((mod for mod in app.modules_manager.get_all_instances() if mod.properties['type'] == 'event_container'), None)
    if not event_container_module:
        logger.error("The event_container module wasn't found")
        return app.abort(500, "The event_container module wasn't found")
    return event_container_module


def _get_pagination_info():
    # type: () -> (str, int, int)
    first_ordering_uuid = ''
    page_size = 100
    page_index = 0
    
    get_range = 'range not get from request'
    try:
        get_range = app.request.GET.get('range', '').strip()
        if get_range:
            for param in get_range.split('~'):
                p_name, p_value = param.split(':')
                
                if p_name == 'first_ordering_uuid':
                    first_ordering_uuid = p_value
                elif p_name == 'page_size':
                    page_size = int(p_value)
                elif p_name == 'page':
                    page_index = int(p_value)
                else:
                    raise Exception('unknown range param : [%s]' % get_range)
    except Exception as e:
        logger.warning('fail to parse range param : [%s] with error : [%s]' % (get_range, e))
    
    return first_ordering_uuid, page_size, page_index


def _get_filters():
    date_filter = ''
    get_filters = ''
    filters_json = 'filters not get from request'
    try:
        get_filters = []
        cnt = 0
        filters_json = app.request.body.getvalue()
        filters_dict = json.loads(filters_json)
        get_filter = filters_dict.get('filter%s' % cnt, '')
        while get_filter:
            get_filters.append(get_filter.strip())
            cnt += 1
            get_filter = filters_dict.get('filter%s' % cnt, '')
        date_filter = filters_dict.get('filter_days', '')
    except Exception as e:
        logger.warning('fail to parse filter : [%s] with error : [%s]' % (filters_json, e))
    
    return date_filter, get_filters, filters_json


def _list_all_values():
    all_values = {
        u'realm': list(app.datamgr.get_realms()),
    }
    return all_values


def event_container():
    event_container_start = int(time.time())
    
    user = app.get_user_auth()
    
    # Get pagination information
    first_ordering_uuid, page_size, page_index = _get_pagination_info()
    
    # Get all filtering
    date_filter, get_filters, filters_json = _get_filters()
    
    # Get all realm
    all_values = _list_all_values()
    
    event_container_module = get_event_container_module()
    try:
        final_events, has_next = event_container_module.get_events(user, get_filters, date_filter, first_ordering_uuid=first_ordering_uuid, page_index=page_index, page_size=page_size)
    except ECDatabaseError as e:
        return app.abort(514, 'Can\'t get events from module [%s]. Connection to database [%s] is down.' % (event_container_module.get_name(), e._uri))
    except Exception:
        return app.abort(500, 'Can\'t get events from module [%s]' % (event_container_module.get_name()))
    
    configuration_id = app.datamgr.get_configuration_id()
    ret = {
        u'elements'                          : final_events,
        u'nb_element_filter'                 : len(final_events),
        u'has_next'                          : has_next,
        u'page_index'                        : page_index,
        u'configuration_id'                  : configuration_id,
        u'timestamp_backend_processing_start': event_container_start,
        u'timestamp_backend_processing_end'  : int(time.time()),
        u'all_values'                        : all_values,
    }
    
    _send_stat_to_module('Event manager', app.request.path + app.request.query_string, filters_json, time.time() - event_container_start)
    return ret


def debug_event_container():
    user = app.get_user_auth()
    event_container_module = get_event_container_module()
    
    # Get pagination information
    first_ordering_uuid, page_size, page_index = _get_pagination_info()
    
    filters = []
    # if user.is_admin:
    #     filters = [u'host_name:nautilus Host-01', u'host_name:nautilus Host-02']
    
    date_filter = ''
    
    final_events, have_next = event_container_module.get_events(user, filters, date_filter, first_ordering_uuid=first_ordering_uuid, page_index=page_index, page_size=page_size)
    if not first_ordering_uuid:
        first_ordering_uuid = final_events[0]['ordering_uuid']
    return {'data': final_events, 'have_next': have_next, 'first_ordering_uuid': first_ordering_uuid, 'page_index': page_index, 'page_size': page_size}


pages = {
    event_container      : {'routes': ['/event-container'], 'wrappers': ['json', 'auth'], 'method': 'POST'},
    debug_event_container: {'routes': ['/event-container-debug'], 'view': 'event_container_debug', 'wrappers': ['auth']}
}
