#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2016:
#    Shinken solutions
import os
import json
import time

from shinken.log import logger

CONTEXT_PATH = '/var/lib/shinken/context.json'
CURRENT_VERSION = '02.06.00'

if os.path.exists(CONTEXT_PATH):
    context = json.loads(open(CONTEXT_PATH, 'r').read())
    CURRENT_VERSION = context.get('current_version', CURRENT_VERSION)

# Global value that will be changed by the main app
app = None


def get_user():
    logger.debug("call get_user")
    user = app.get_user_auth()
    
    # Second we look for bdd access.
    collection = app.get_user_collection()
    if not collection:
        logger.info("get_user No more backend for saving the data")
        app.abort(500, 'No more backend for saving the data')
        return json.dumps(False)
    
    # logger.debug("user %s" % user)
    user_id = user.uuid
    # Our main key is the uuid
    user_info = collection.find_one({'uuid': user_id}, {'_id': 0})
    
    # if no previous entries, None is good
    if user_info:
        user_group = app.get_contact_groups(user)
        if user_group:
            user_info['groups'] = []
            for group in user_group:
                user_info['groups'].append({"uuid": group.uuid, "name": group.get_name()})
        
        user_info['acl'] = {}
        for key, value in user.__dict__.iteritems():
            if key.startswith('acl_'):
                user_info['acl'][key[4:]] = value
                
        is_dst = time.daylight and time.localtime().tm_isdst > 0
        
        user_info['tiles_background'] = app.tiles_background
        user_info['apply_filter_method'] = app.apply_filter_method
        user_info['userName'] = user.contact_name
        user_info['is_admin'] = user.is_admin
        user_info['backend_localtime_epoch'] = int(time.time())
        user_info['time_zone_backend'] = int(time.altzone if is_dst else time.timezone)
        user_info['known_modules'] = list(set([mod.properties['type'] for mod in app.modules_manager.get_all_instances()]))
        
        if 'prefs' not in user_info:
            user_info['prefs'] = {}
        
        if 'historicTab' not in user_info['prefs']:
            user_info.get('prefs')['historicTab'] = {}
        
        if 'show_historic_outputs' not in user_info['prefs']['historicTab']:
            user_info.get('prefs').get('historicTab')['show_historic_outputs'] = app.history__display_outputs
        
        if 'collapse_all_historic_outputs' not in user_info['prefs']['historicTab']:
            user_info.get('prefs').get('historicTab')['collapse_all_historic_outputs'] = app.history__collapse_outputs
        
        if 'size_sla_column' not in user_info['prefs']['historicTab']:
            user_info.get('prefs').get('historicTab')['size_sla_column'] = app.history__size_sla_pane
        
        user_info['shinken_version'] = CURRENT_VERSION
        user_info['configuration_id'] = app.datamgr.get_configuration_id()
        
        # The UI need the check plugin styles css hash: if does change, need to reload the UI
        user_info['check_plugin_styles_css_hash'] = app.get_check_plugin_styles_css_hash()
        
        # for admin only we will add the architecture map
        if user.is_admin:
            user_info['architecture_map'] = app.arch_export_handler.get_api_architecture_export()
        
        return json.dumps(user_info)
    else:
        logger.info("get_user user not found")
        app.abort(404, 'User not found')
        return json.dumps(False)


def save_user():
    user = app.get_user_auth()
    
    # Second we look for bdd access.
    collection = app.get_user_collection()
    if not collection:
        logger.info("save_user No more backend for saving the data")
        app.abort(500, 'No more backend for saving the data')
        return json.dumps(False)
    
    # Third load the data to save.
    if hasattr(app.request.body, 'getvalue'):
        data = json.loads(app.request.body.getvalue())
    else:  # file?
        data = json.loads(app.request.body.read())
    
    user_id = user.uuid
    # Eventually we save the data.
    logger.debug("hive data to save %s", data)
    data['uuid'] = user_id
    data['groups'] = []  # We clean group we don't want save this.
    collection.update({'_id': user_id}, data, upsert=True)
    return json.dumps({'uuid': user_id})


# Remove all user screens & shares. Also delete user entry.
# Allow to recreate from old userdata.
def clean_user(uuid):
    shares = app.get_share_collection()
    hives = app.get_hive_collection()
    lists = app.get_list_collection()
    dashboards = app.get_dashboard_collection()
    removed_hives = hives.remove({'owner.uuid': uuid})
    removed_lists = lists.remove({'owner.uuid': uuid})
    removed_dashboards = dashboards.remove({'owner.uuid': uuid})
    removed_shares = shares.remove({'owner.uuid': uuid})
    removed = {
        "screen": {
            "hives"     : removed_hives.get('n', 0),
            "lists"     : removed_lists.get('n', 0),
            "dashboards": removed_dashboards.get('n', 0)
        },
        "shares": removed_shares.get('n', 0)
    }
    return json.dumps(removed)


pages = {
    get_user  : {'routes': ['/user'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    save_user : {'routes': ['/user'], 'method': 'POST', 'wrappers': ['auth', 'json']},
    clean_user: {'routes': ['/user/:uuid/clean'], 'method': 'DELETE', 'wrappers': ['auth', 'json']}
}
