#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import traceback
import cPickle
import errno
import inspect
import select
import socket
import threading
import time
import zlib
import os
from log import logger
from shinken.webui import bottlecore as bottle
import shinkensolutions.shinkenjson as json
from .thread_helper import get_thread_id

ALLOW_HTTP_CALLS_DEBUG = os.path.exists('/tmp/SHINKEN_ALLOW_HTTP_CALLS_DEBUG_FLAG')

try:
    import ssl
except ImportError:
    ssl = None

try:
    from cherrypy import wsgiserver as cheery_wsgiserver
except ImportError:
    cheery_wsgiserver = None

bottle.debug(True)


class InvalidWorkDir(Exception):
    pass


class PortNotFree(Exception):
    pass


# CherryPy is allowing us to have a HTTP 1.1 server, and so have a KeepAlive
class CherryPyServer(bottle.ServerAdapter):
    def run(self, handler):
        daemon_thread_pool_size = self.options['daemon_thread_pool_size']
        server = cheery_wsgiserver.CherryPyWSGIServer((self.host, self.port), handler, numthreads=daemon_thread_pool_size, shutdown_timeout=1)
        logger.info('Initializing a CherryPy backend with %d threads' % daemon_thread_pool_size)
        use_ssl = self.options['use_ssl']
        ca_cert = self.options['ca_cert']
        ssl_cert = self.options['ssl_cert']
        ssl_key = self.options['ssl_key']
        
        if use_ssl:
            server.ssl_certificate = ssl_cert
            server.ssl_private_key = ssl_key
        return server


class CherryPyBackend(object):
    def __init__(self, host, port, use_ssl, ca_cert, ssl_key, ssl_cert, hard_ssl_name_check, daemon_thread_pool_size):
        self.port = port
        self.use_ssl = use_ssl
        try:
            
            @bottle.error(500)
            def print_traceback(error):
                if error.traceback:
                    formatted_lines = error.traceback.split('\n')
                    first_line = formatted_lines[0]
                    to_print = []
                    if first_line == 'None':
                        for line in traceback.format_stack():
                            to_print.append("%s" % line)
                    else:
                        to_print.append("ERROR stack : %s" % first_line)
                        for line in formatted_lines[1:]:
                            to_print.append("%s" % line)
                elif isinstance(error, bottle.HTTPError):
                    # app.abord make a HTTPError without traceback see SEF-6067
                    to_print = ['HTTPError : code[%s] message:[%s]' % (error.status, error.output)]
                else:
                    to_print = [str(error)]
                return "<br/>".join(to_print)
            
            
            self.srv = bottle.run(host=host, port=port, server=CherryPyServer, use_ssl=use_ssl, ca_cert=ca_cert, ssl_key=ssl_key, ssl_cert=ssl_cert, daemon_thread_pool_size=daemon_thread_pool_size, quiet=True)
        except socket.error, exp:
            msg = "Error: Sorry, the port %d is not free: %s" % (self.port, str(exp))
            raise PortNotFree(msg)
        except Exception, e:
            # must be a problem with http workdir:
            raise InvalidWorkDir(e)
    
    
    # When call, it do not have a socket
    def get_sockets(self):
        return []
    
    
    # We stop our processing, but also try to hard close our socket as cherrypy is not doing it...
    def stop(self):
        # TODO: find why, but in ssl mode the stop() is locking, so bailout before
        if self.use_ssl:
            return
        try:
            self.srv.stop()
        except Exception, exp:
            logger.warning('Cannot stop the CherryPy backend : %s' % exp)
    
    
    # Will run and LOCK
    def run(self):
        try:
            self.srv.start()
        except socket.error, exp:
            msg = "Error: Sorry, the port %s is not free: %s" % (self.port, str(exp))
            raise PortNotFree(msg)
        finally:
            self.srv.stop()


class HTTPDaemon(object):
    def __init__(self, host, port, use_ssl, ca_cert, ssl_key, ssl_cert, hard_ssl_name_check, daemon_thread_pool_size):
        self.port = port
        self.host = host
        self.bottle = bottle
        self.abort = bottle.abort
        # Port = 0 means "I don't want HTTP server"
        if self.port == 0:
            return
        
        self.use_ssl = use_ssl
        
        self.registered_fun = {}
        self.registered_fun_names = []
        self.registered_fun_defaults = {}
        
        protocol = 'http'
        if use_ssl:
            protocol = 'https'
        self.uri = '%s://%s:%s' % (protocol, self.host, self.port)
        logger.info("Opening HTTP socket at %s" % self.uri)
        
        # Hack the BaseHTTPServer so only IP will be looked by wsgiref, and not names
        __import__('BaseHTTPServer').BaseHTTPRequestHandler.address_string = lambda x: x.client_address[0]
        
        self.srv = CherryPyBackend(host, port, use_ssl, ca_cert, ssl_key, ssl_cert, hard_ssl_name_check, daemon_thread_pool_size)
        self.lock = threading.RLock()
        self.is_stopping = False
    
    
    # Get the server socket but not if disabled or closed
    def get_sockets(self):
        if self.port == 0 or self.srv is None:
            return []
        return self.srv.get_sockets()
    
    
    def run(self):
        self.srv.run()
    
    
    def register(self, obj):
        methods = inspect.getmembers(obj, predicate=inspect.ismethod)
        merge = [fname for (fname, f) in methods if fname in self.registered_fun_names]
        if merge:
            methods_in = [m.__name__ for m in obj.__class__.__dict__.values() if inspect.isfunction(m)]
            methods = [m for m in methods if m[0] in methods_in]
        
        for (fname, f) in methods:
            if fname.startswith('_'):
                continue
            # Get the args of the function to catch them in the queries
            argspec = inspect.getargspec(f)
            args = argspec.args
            varargs = argspec.varargs
            keywords = argspec.keywords
            defaults = argspec.defaults
            # If we got some defauts, save arg=value so we can lookup
            # for them after
            if defaults:
                default_args = zip(argspec.args[-len(argspec.defaults):], argspec.defaults)
                _d = {}
                for (argname, defavalue) in default_args:
                    _d[argname] = defavalue
                self.registered_fun_defaults[fname] = _d
            # remove useless self in args, because we alredy got a bonded method f
            if 'self' in args:
                args.remove('self')
            self.registered_fun_names.append(fname)
            self.registered_fun[fname] = (f)
            
            
            # WARNING : we MUST do a 2 levels function here, or the f_wrapper
            # will be uniq and so will link to the last function again and again
            def register_callback(fname, args, f, obj, lock):
                def f_wrapper():
                    if ALLOW_HTTP_CALLS_DEBUG:
                        caller = bottle.request.environ.get('HTTP_X_CALLER_NAME', '(unknown)')
                        logger.debug('[HTTP] incoming http request: thread=%s caller=%-15s  functioncalled=%s' % (get_thread_id(), caller, fname))
                    hours_since_epoch = int(time.time()) / 3600
                    try:
                        t0 = time.time()
                        args_time = aqu_lock_time = calling_time = json_time = 0
                        need_lock = getattr(f, 'need_lock', obj.default_lock)
                        
                        # Warning : put the bottle.response set inside the wrapper
                        # because outside it will break bottle
                        d = {}
                        method = getattr(f, 'method', 'get').lower()
                        for aname in args:
                            v = None
                            if method == 'post':
                                v = bottle.request.forms.get(aname, None)
                                # Post args are zlibed and cPickled
                                if v is not None:
                                    v = zlib.decompress(v)
                                    v = cPickle.loads(v)
                            
                            elif method == 'get':
                                v = bottle.request.GET.get(aname, None)
                            if v is None:
                                # Maybe we got a default value?
                                default_args = self.registered_fun_defaults.get(fname, {})
                                if aname not in default_args:
                                    raise bottle.HTTPError(500, 'Missing argument %s' % aname)
                                v = default_args[aname]
                            d[aname] = v
                        
                        # Clean args from bottle+cherrypy in memory
                        # Always clean bottle environ data as bottle it won't do it itself (or cherrypy?)
                        if method == 'post':
                            bottle.request.environ['wsgi.input'].truncate(0)  # StringIO
                            bottle.request.environ['bottle.request.body'].truncate(0)  # StringIO too
                            # forms parameters are kept too, in Multidict (with a .dict)
                            bottle.request.environ['bottle.request.post'].dict.clear()
                            bottle.request.environ['bottle.request.forms'].dict.clear()
                        
                        args_time = time.time() - t0
                        
                        t1 = time.time()
                        if need_lock:
                            # logger.debug("HTTP: calling lock for %s" % fname)
                            lock.acquire()
                        aqu_lock_time = time.time() - t1
                        
                        t2 = time.time()
                        
                        try:
                            ret = f(**d)
                        # Always call the lock release if need
                        finally:
                            # Ok now we can release the lock
                            if need_lock:
                                lock.release()
                        calling_time = time.time() - t2
                        t3 = time.time()
                        encode = getattr(f, 'encode', 'json').lower()
                        j = json.dumps(ret)
                        json_time = time.time() - t3
                        
                        if calling_time > 0.05 or aqu_lock_time > 0.05 or json_time > 0.05 or args_time > 0.05:
                            logger.debug("Debug perf: %s [args:%.4f] [aqu_lock:%.4f] [calling:%.4f] [json:%.4f]" %
                                         (fname, args_time, aqu_lock_time, calling_time, json_time))
                        
                        # all app daemons will have this but for schedulers app can be Scheduler and not a daemon
                        if hasattr(obj.app, 'http_errors_count'):
                            # Clean up older error counts
                            for key in obj.app.http_errors_count.keys():
                                if (int(key) < (hours_since_epoch - 24)) or int(key) > hours_since_epoch:
                                    del obj.app.http_errors_count[key]
                        
                        return j
                    except Exception as e:
                        is_public_api = getattr(f, 'public_api', obj.public_api)
                        if is_public_api and isinstance(e, bottle.HTTPError):
                            logger.error('error in %s : %s' % (fname, e))
                        else:
                            # all app daemons will have this but for schedulers app can be Scheduler and not a daemon
                            if hasattr(obj.app, 'http_errors_count'):
                                # Store HTTP errors for the last 24 hours
                                if hours_since_epoch in obj.app.http_errors_count:
                                    obj.app.http_errors_count[int(hours_since_epoch)] += 1
                                else:
                                    obj.app.http_errors_count[int(hours_since_epoch)] = 1
                                logger.error('An error occurred, add http_errors_count %s' % obj.app.http_errors_count)
                            logger.print_stack()
                        
                        raise
                
                
                # here we build the route with prefix and api version. To add an api_version, you need a route prefix.
                # In default and many case (no prefix and no api version), the route will be : /my_method
                # In the case of the interface have only a route prefix, the route will be : /my-prefix/my_method
                # In the case of the interface have route prefix and api version, the route will be: /my-prefix/v1/my_method
                version_route_prefix = ''
                if obj.API_VERSION != 0:
                    version_route_prefix = 'v%s/' % obj.API_VERSION
                final_route = '%s/%s%s' % (obj.route_prefix, version_route_prefix, fname)
                # and the name with - instead of _ if need
                final_route_dash_replaced = '%s/%s%s' % (obj.route_prefix, version_route_prefix, fname.replace('_', '-'))
                # Ok now really put the route in place
                bottle.route(final_route, callback=f_wrapper, method=getattr(f, 'method', 'get').upper())
                if final_route != final_route_dash_replaced:
                    bottle.route(final_route_dash_replaced, callback=f_wrapper, method=getattr(f, 'method', 'get').upper())
            
            
            register_callback(fname, args, f, obj, self.lock)
        
        bottle.route('/', callback=HTTPDaemon._slash)
    
    
    # Add a simple / page
    @staticmethod
    def _slash():
        return "OK"
    
    
    # TODO to remove ( legacy form pyro )
    def unregister(self, obj):
        return
    
    
    def handleRequests(self, s):
        self.srv.handle_request()
    
    
    def create_uri(address, port, obj_name, use_ssl=False):
        return "PYRO:%s@%s:%d" % (obj_name, address, port)
    
    
    def set_timeout(con, timeout):
        con._pyroTimeout = timeout
    
    
    def _do_shutdown(self):
        if self.is_stopping:
            return
        else:
            self.is_stopping = True
        self.srv.stop()
        self.srv = None
    
    
    # Close all sockets and delete the server object to be sure
    # no one is still alive
    def shutdown(self, quiet=False):
        if self.srv is None:
            return
        if not quiet:
            logger.debug('Closing the http socket on the process %s' % os.getpid())
        # First we will not need any more lock because we are going down, so try to release it if we hold it
        try:
            self.lock.release()
        except:
            pass
        thread = threading.Thread(target=self._do_shutdown, name='Stopping cherrypy http server')
        thread.daemon = True
        thread.start()
        # This thread will never be joined, but it should be called only once a process
    
    
    def get_socks_activity(self, timeout):
        try:
            ins, _, _ = select.select(self.get_sockets(), [], [], timeout)
        except select.error, e:
            errnum, _ = e
            if errnum == errno.EINTR:
                return []
            raise
        return ins


daemon_inst = None
