#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import copy
from ..item_controller.work_area_helper import get_item_state, WORK_AREA_INFO_KEY
from ...dao.def_items import ITEM_STATE, ITEM_TYPE, HISTORY_ACTION, DEF_ITEMS
from collections import namedtuple

State = namedtuple('State', 'state wa_bypass')


class ACTIONS:
    DISABLE_OBJECT = 'DISABLE_OBJECT'
    ENABLE_OBJECT = 'ENABLE_OBJECT'
    CLONE_OBJECT = 'CLONE_OBJECT'
    CREATE_FROM_OBJECT = 'CREATE_FROM_OBJECT'
    IMPORT_FROM_SOURCE = 'IMPORT_FROM_SOURCE'
    DELETE_UI_ENTRY = 'DELETE_UI_ENTRY'
    DELETE_ITEM_WORKING_AREA = 'DELETE_ITEM_WORKING_AREA'
    LIST_ELEMENTS = 'LIST_ELEMENTS'
    SAVE_OBJECT = 'SAVE_OBJECT'
    ACCEPT_SUBMIT = 'ACCEPT_SUBMIT'
    REJECT_SUBMIT = 'REJECT_SUBMIT'
    VALIDATE_CHANGES = 'VALIDATE_CHANGES'
    SUBMIT_TO_STAGING = 'SUBMIT_TO_STAGING'
    SAVE_IN_WORK_AREA = 'SAVE_IN_WORK_AREA'
    UNLOCK_WORK_AREA = 'UNLOCK_WORK_AREA'


DEF_ACTIONS = {
    ACTIONS.DISABLE_OBJECT          : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.ENABLE_OBJECT           : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.CLONE_OBJECT            : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.CREATE_FROM_OBJECT      : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.IMPORT_FROM_SOURCE      : {
        'transitions': {State(ITEM_STATE.NEW, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.NEW, True) : ITEM_STATE.STAGGING
                        }
    },
    ACTIONS.DELETE_UI_ENTRY         : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.DELETE_ITEM_WORKING_AREA: {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.LIST_ELEMENTS           : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.SAVE_OBJECT             : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.ACCEPT_SUBMIT           : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.STAGGING,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.STAGGING,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.REJECT_SUBMIT           : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.WORKING_AREA,
                        }
    },
    ACTIONS.VALIDATE_CHANGES        : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
                        }
    },
    ACTIONS.SUBMIT_TO_STAGING       : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.WORKING_AREA,
                        }
    },
    ACTIONS.SAVE_IN_WORK_AREA       : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.WORKING_AREA,
                        }
    },
    ACTIONS.UNLOCK_WORK_AREA        : {
        'transitions': {State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
                        State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.WORKING_AREA,
                        }
    },
}


class ItemContext(object):
    """
    Marshall all elements of item state in the front end and back end
    """
    
    
    def __init__(self, app, item_id, item_type, action=None, item_state=None, user=None, bypass_work_area=False):
        """
        Marshall all elements of item state in the front end and back end
        :param item_id:  item_id
        :type item_id: str
        :param item_type:
        :type item_type: ITEM_TYPE
        :param item_state:
        :type item_state: ITEM_STATE
        :param app: Synchronizer daemon instance
        :type app: Synchronizer
        """
        
        user = user or app.get_user_auth()
        
        self.item_type = item_type
        self.item_id = item_id
        self.action = action
        self.user = user
        self.is_admin = user.is_admin()
        self.is_expert = user.is_expert()
        self.is_new = (app.request.GET.get('pending', '0') == '1') or (app.request.GET.get('item_state') == 'new')
        self.in_creation = (app.request.GET.get('in_creation', '0') == '1')
        self.bypass_work_area = bypass_work_area or (app.request.GET.get('bypass_work_area', '0') == '1')
        self.must_submit_to_staging = (app.request.GET.get('submit_to_stagging', '0') == '1')
        self.prev_elt = None  # it will be compute on _compute_current_state_for_item
        self.from_state = self._compute_current_state_for_item(app)
        self.with_change = (app.request.GET.get('with_change', '0') == '1')
        self.to_state = self._compute_next_state_for_item()
        self.item_state = item_state if item_state else self.from_state
    
    
    def do_action(self, action):
        pass
    
    
    # we need to update the work_area_info of the staging object
    # compute a new context with a new action and to state
    def update_work_area(self):
        """ Return a copy of the context with the destination state set to staging"""
        staging_context = copy.copy(self)
        staging_context.prev_elt = self.prev_elt
        staging_context.to_state = ITEM_STATE.STAGGING
        staging_context.action = HISTORY_ACTION.UPDATE_WORK_AREA_INFO
        staging_context.with_change = False
        return staging_context
    
    
    def _compute_current_state_for_item(self, app):
        has_work_area = ITEM_TYPE.has_work_area(self.item_type)
        if self.is_new:
            states_to_lookup = [ITEM_STATE.NEW, ITEM_STATE.STAGGING]
            if has_work_area:
                states_to_lookup.append(ITEM_STATE.WORKING_AREA)
            items = app.datamanagerV2.find_merge_state_items(where={'_id': self.item_id}, item_type=self.item_type, item_states=states_to_lookup)
            if items:
                self.prev_elt = items[0]
            
            return app.request.GET.get('item_state', ITEM_STATE.NEW)
    
        if not has_work_area:
            self.prev_elt = app.datamanagerV2.find_item_by_id(self.item_id, self.item_type, ITEM_STATE.STAGGING)
            return ITEM_STATE.STAGGING
        
        self.prev_elt = app.datamanagerV2.find_item_by_id(self.item_id, self.item_type, ITEM_STATE.WORKING_AREA)
        if self.prev_elt:
            work_area_info = self.prev_elt.get(WORK_AREA_INFO_KEY, None)
            return get_item_state(work_area_info)
        
        self.prev_elt = app.datamanagerV2.find_item_by_id(self.item_id, self.item_type, ITEM_STATE.STAGGING)
        if self.prev_elt:
            work_area_info = self.prev_elt.get(WORK_AREA_INFO_KEY, None)
            return get_item_state(work_area_info)
        else:
            return ITEM_STATE.WORKING_AREA
    
    
    def _compute_next_state_for_item(self):
        """Computes the state in which the item will be after the action"""
        if self.is_admin and self.bypass_work_area:
            return ITEM_STATE.STAGGING

        if ITEM_TYPE.has_work_area(self.item_type):
            return ITEM_STATE.WORKING_AREA
        
        return ITEM_STATE.STAGGING
