#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import logger
from . import PostDeleter, PostSaver, PreSaver, PostMover, PreDeleter, PreMover
from ..trash_manager.trash_manager import trash_manager
from ..item_saving_formatter import build_link
from ..item_comparator import compute_item_diff_existing_vs_from_source
from ...business.item_controller.item_running_state import ACTIONS
from ..def_items import ITEM_STATE, HISTORY_ACTION, DEF_ITEMS, METADATA
from ..helpers import get_name_from_type, compute_diff


def _add_source_change(datamanagerV2, item, item_type, action, old_item=None, item_property=None):
    if not old_item and not item_property:
        return
    merge_from_source_item = datamanagerV2.find_item_by_id(item['_id'], item_type, ITEM_STATE.MERGE_SOURCES)
    if not merge_from_source_item:
        return
    
    effective = False
    source_info = METADATA.get_metadata(merge_from_source_item, METADATA.SOURCE_INFO)['_info']
    if item_property:
        if source_info.get(item_property, None):
            old_action = source_info[item_property].get('property_modif_auto', None)
            if not old_action or old_action == HISTORY_ACTION.AUTO_MODIFICATION:
                source_info[item_property]['property_modif_auto'] = action
                effective = True
    else:
        old_item = datamanagerV2.get_raw_item(old_item, item_type=item_type, flatten_links=False)
        item = datamanagerV2.get_raw_item(item, item_type=item_type, flatten_links=False)
        diffs_properties = compute_diff(item, old_item, item_type)
        for item_property in diffs_properties:
            if source_info.get(item_property, None):
                old_action = source_info[item_property].get('property_modif_auto', None)
                if not old_action or old_action == HISTORY_ACTION.AUTO_MODIFICATION:
                    source_info[item_property]['property_modif_auto'] = action
                    effective = True
    
    if effective:
        datamanagerV2.save_item(merge_from_source_item, None, item_type, item_state=ITEM_STATE.MERGE_SOURCES)


class _CallbackComputeDiffStagingProduction(PostSaver, PostDeleter, PostMover):
    def called_after_save(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        from ...business.sync_ui_common import syncuicommon
        if item_state != ITEM_STATE.STAGGING or action == HISTORY_ACTION.UPDATE_WORK_AREA_INFO:
            return
        
        self.log(item_type, item_state, item_id, action, 'compute diff staging <-> production')
        item_staging = datamanagerV2.find_item_by_id(item_id, item_type, ITEM_STATE.STAGGING)
        item_production = datamanagerV2.find_item_by_id(item_id, item_type, ITEM_STATE.PRODUCTION)
        syncuicommon.compute_one_diff_staging_production(item_id, item_type, item_staging, item_production)
    
    
    called_after_delete = called_after_save


callback_compute_diff_staging_production = _CallbackComputeDiffStagingProduction()


class _CallbackComputeDiffStagingSource(PreSaver, PreMover):
    def called_before_save(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        if item_state not in (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA) or action in (HISTORY_ACTION.UPDATE_WORK_AREA_INFO, ACTIONS.IMPORT_FROM_SOURCE):
            return
        
        if action == HISTORY_ACTION.APPLY_CHANGES:
            datamanagerV2.delete_item(item, item_type=item_type, item_state=ITEM_STATE.CHANGES)
            return
        
        synchronizer = datamanagerV2.synchronizer
        item_source = datamanagerV2.find_item_by_id(item_id, item_type, ITEM_STATE.MERGE_SOURCES)
        item_from_ui = item
        have_enable_source = len([s for s in synchronizer.sources if s.enabled and s.source_name != 'syncui']) > 0
        
        # logger.debug('[callback][%s:%s:%s] - item_source[%s] have_enable_source[%s] item_from_ui[%s]' % (item_type, item_state, item_id, item_source is not None, have_enable_source, item_from_ui is not None))
        if item_source and item_from_ui and have_enable_source:
            self.log(item_type, item_state, item_id, action, 'compute diff staging <-> merge item')
            _add_source_change(datamanagerV2, item, item_type, action, old_item=old_item)
            changes = compute_item_diff_existing_vs_from_source(item_type, item_from_ui, item_source, datamanagerV2)
            diff = {'_id': item_from_ui['_id'], 'changes': changes}
            
            # logger.debug('[callback][%s:%s:%s] - diff [%s]' % (item_type, item_state, item_id, diff))
            
            # If the compared items have no common sync keys, they can't be merged together and are considered as separate items
            # So their differences have no meaning and are removed
            if len(changes) == 0 or set(item_source['_SYNC_KEYS']).intersection(set(item_from_ui['_SYNC_KEYS'])) == set([]):
                datamanagerV2.delete_item(diff, item_type=item_type, item_state=ITEM_STATE.CHANGES)
            else:
                datamanagerV2.save_item(diff, item_type=item_type, item_state=ITEM_STATE.CHANGES)


callback_compute_diff_staging_source = _CallbackComputeDiffStagingSource()


class _CallbackOnRename(PostSaver, PostMover):
    def called_after_save(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        if not item_state in (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING):
            return
        if old_item and get_name_from_type(item_type, old_item) != get_name_from_type(item_type, item):
            self.log(item_type, item_state, item_id, action, 'callback post rename')
            
            reverse_links = item.get_reverse_links()
            for reverse_link in reverse_links:
                item_link = datamanagerV2.find_item_by_id(reverse_link.item_id, item_type=reverse_link.item_type, item_state=reverse_link.item_state)
                if item_link:
                    item_link_change = item_link.get_changes()
                    must_recompute_change = False
                    for link_property in DEF_ITEMS[item_link.get_type()]['item_links'][item_type]:
                        if [l for l in item_link.get_links(link_property, item_type) if l.get('_id', None) == item_id]:
                            _add_source_change(datamanagerV2, item_link, item_link.get_type(), HISTORY_ACTION.AUTO_MODIFICATION, item_property=link_property)
                            if item_link_change and item_link_change.get(link_property, []):
                                must_recompute_change = True
                    
                    if must_recompute_change:
                        callback_compute_diff_staging_source.called_before_save(item_link['_id'], item_link.get_type(), item_link.get_state(), item_link, None, user, HISTORY_ACTION.AUTO_MODIFICATION, datamanagerV2)
                        datamanagerV2.save_item(item_link, user, item_link.get_type(), item_link.get_state(), HISTORY_ACTION.AUTO_MODIFICATION)


callback_on_rename = _CallbackOnRename()


class _CallbackDeleteChangeInfo(PostDeleter):
    def called_after_delete(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        if item_state not in (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA):
            return
        item_exist = datamanagerV2.find_item_by_id(item_id, item_type, ITEM_STATE.WORKING_AREA if item_state == ITEM_STATE.STAGGING else ITEM_STATE.STAGGING)
        if not item_exist:
            self.log(item_type, item_state, item_id, action, 'delete change info')
            datamanagerV2.delete_item(item, item_type=item_type, item_state=ITEM_STATE.CHANGES)


callback_delete_change_info = _CallbackDeleteChangeInfo()


class _CallbackRemoveFromSourceCache(PostDeleter):
    def called_after_delete(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        if item_state not in (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA):
            return
        item_exist = datamanagerV2.find_item_by_id(item_id, item_type, ITEM_STATE.WORKING_AREA if item_state == ITEM_STATE.STAGGING else ITEM_STATE.STAGGING)
        if item_exist:
            return
        source_item = datamanagerV2.find_item_by_id(item_id, item_type, ITEM_STATE.MERGE_SOURCES)
        if not source_item:
            return
        
        self.log(item_type, item_state, item_id, action, 'remove item from source cache')
        self._remove_item_from_import_cache(datamanagerV2.synchronizer, source_item)
    
    
    def _remove_item_from_import_cache(self, synchronizer, source_item):
        conn = synchronizer.get_synchronizer_deamon_connection()
        conn.request("GET", "/remove_item_from_import_cache?merged_item_hash=%s" % METADATA.get_metadata(source_item, METADATA.SOURCE_HASH))
        r1 = conn.getresponse()
        r1.read()
        conn.close()


callback_remove_from_source_cache = _CallbackRemoveFromSourceCache()


class _callbackConfigurationStatsCacheReset(PostSaver, PostDeleter, PostMover):
    
    def reset_cache_callback(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        from ...business.sync_ui_common import syncuicommon
        self.log(item_type, item_state, item_id, action, 'reset configuration stats cache')
        syncuicommon.reset_configuration_stats_cache()
    
    
    called_after_save = reset_cache_callback
    called_after_delete = reset_cache_callback


callback_configuration_stats_cache_reset = _callbackConfigurationStatsCacheReset()


class _callbackComputeDoubleLinks(PostSaver, PostMover):
    def compute_double_links_callback(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        if not datamanagerV2.compute_double_links or item_state in (ITEM_STATE.CHANGES, ITEM_STATE.WORKING_AREA, ITEM_STATE.MERGE_SOURCES) or action == HISTORY_ACTION.UPDATE_WORK_AREA_INFO:
            return
        
        self.log(item_type, item_state, item_id, action, 'compute double links')
        datamanagerV2._make_double_link(item, item_type, user, old_item)
    
    
    called_after_save = compute_double_links_callback


callback_compute_double_links = _callbackComputeDoubleLinks()


class _CallbackBuildLink(PreSaver, PreMover):
    def called_before_save(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        if item_state not in (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION) or action == HISTORY_ACTION.UPDATE_WORK_AREA_INFO:
            return
        self.log(item_type, item_state, item_id, action, 'build links')
        build_link(item, item_type, item_state, datamanagerV2)


callback_build_link = _CallbackBuildLink()


class _CallbackSaveDeletedItem(PreDeleter):
    def save_deleted_item(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        if item_state not in (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA):
            return
        
        item_to_delete = datamanagerV2.find_item_by_id(item_id, item_type, item_state)
        if item_to_delete:
            trash_manager.save_deleted_item(item_to_delete, item_type, user)
    
    
    called_before_delete = save_deleted_item


callback_save_deleted_item = _CallbackSaveDeletedItem()
