#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013:
#    Gabes Jean, naparuba@gmail.com
#
# This file is part of Shinken Enterprise, all rights reserved.

import base64
import httplib
import json
import math
import os
import shutil
import ssl
import threading
import traceback
import uuid
from _ssl import SSLError
from datetime import datetime
from threading import RLock

try:
    import pwd
    import grp
except ImportError:
    # don't expect to have this on windows :)
    pwd = grp = None
from PIL import Image

import shinken.objects.module
from shinken.log import logger
from shinkensolutions.lib_checks.graphite import GraphiteConfReader
from shinkensolutions.lib_checks.common import EXIT_STATUS
from shinken.misc.type_hint import List, Dict, Callable

SATELLITE_NAMES = ('scheduler', 'reactionner', 'broker', 'receiver', 'poller')
UPPER_REALM_SATELLITE_NAMES = ('arbiter', 'synchronizer')

# This dict allow to add check for module present on a daemon, the key is the type of the module.
# We need to separate Module check and daemon check, cause module check are optional and based on module installed on the shinken machine
MODULE_CHECK_DICT = {
    'webui'                 : {
        'checks'    : [
            {'service_description': 'Broker - $KEY$ - Module Visualisation UI - Metrology Reader', 'label': 'WebUI Metrology Reader'}
        ],
        'submodules': {
            'sla': {
                'checks'    : [
                    {'service_description': 'Broker - $KEY$ - Module Visualisation UI - SLA Reader', 'label': 'WebUI SLA Reader'}
                ],
                'submodules': {}
            }
        }
    },
    'broker_module_livedata': {
        'checks'    : [
            {'service_description': 'Broker - $KEY$ - Module Livedata', 'label': ''}
        ],
        'submodules': {}
    },
    'graphite_perfdata'     : {
        'checks'    : [
            {'service_description': 'Broker - $KEY$ - Module Metrology Writer', 'label': 'Graphite-Perfdata Metrology Writer'}
        ],
        'submodules': {}
    },
    'sla'                   : {
        'checks'    : [
            {'service_description': 'Broker - $KEY$ - Module SLA Writer', 'label': 'SLA Writer'}
        ],
        'submodules': {}
    }
}

# In this dict, all daemons checks are defined. Some of these are in package (like Database), create a package will make square of little icons
CHECK_DICT = {
    'synchronizer': {'checks': [
        {"service_description": "Alive", "label": "Alive"},
        {"service_description": "Performance API Connection", "label": "Performance"}
    ],
        'checks_packages'    : [
            {'name'  : 'Database',
             'checks': [
                 {"service_description": "Synchronizer - DB - Connection", "label": ""},
                 {"service_description": "Synchronizer - DB - Last Flush Time", "label": ""},
                 {"service_description": "Synchronizer - DB - Open Connections", "label": ""},
             ],
             },
        ]
    },
    
    'arbiter'     : {'checks': [
        {"service_description": "Alive", "label": "Alive"},
        {"service_description": "Performance", "label": "Performance"}
    ],
        'checks_packages'    : []
    },
    
    'broker'      : {'checks': [
        {"service_description": "Alive", "label": "Alive"},
        {"service_description": "Performance API Connection", "label": "Performance"},
        {"service_description": "Performance Modules Queues", "label": "Queues"}
    ],
        'checks_packages'    : [
            {'name'  : 'Database',
             'checks': [
                 {"service_description": "Broker - DB - Connection", "label": ""},
                 {"service_description": "Broker - DB - Last Flush Time", "label": ""},
                 {"service_description": "Broker - DB - Open Connections", "label": ""}
             ],
             },
        ],
    },
    
    'poller'      : {'checks': [
        {"service_description": "Running Well", "label": "Alive"},
        {"service_description": "Performance", "label": "Performance"},
    ],
        'checks_packages'    : []
    },
    
    'reactionner' : {'checks': [
        {"service_description": "Running Well", "label": "Alive"},
        {"service_description": "Performance", "label": "Performance"},
    ],
        'checks_packages'    : []
    },
    
    'receiver'    : {'checks': [
        {"service_description": "Alive", "label": "Alive"},
        {"service_description": "Performance API Connection", "label": "Performance"}
    ],
        'checks_packages'    : []
    },
    
    'scheduler'   : {'checks': [
        {"service_description": "Running Well", "label": "Alive"},
        {"service_description": "Performance", "label": "Performance"},
    ],
        'checks_packages'    : []
    }
}

TRADS = {
    "fr": {
        "do_not_edit_map"               : u"Carte générée par Shinken Entreprise. Ne pas modifier.",
        "do_not_edit_host"              : u"Hôte généré par un export de l’architecture [%s], ne pas modifier.",
        "shinken_architecture"          : u"Architecture Shinken",
        "shinken_architecture_link_name": u"Vue détaillée",
        "realms_tree"                   : u"Arbre des royaumes",
        "realms_tree_link_name"         : u"Arbre des royaumes",
        "realm"                         : u"Royaume",
        "realm_status"                  : u"Statut du royaume",
        "no_daemon"                     : u"Aucun démon",
        "one_daemon"                    : u"1 démon",
        "nb_daemons"                    : u"%s démons",
    },
    "en": {
        "do_not_edit_map"               : u"Map generated by Shinken Enterprise. Do not edit.",
        "do_not_edit_host"              : u"Shinken host generated by an export of [%s] architecture, do not edit.",
        "shinken_architecture"          : u"Shinken architecture",
        "shinken_architecture_link_name": u"Detailed view",
        "realms_tree"                   : u"Realms tree",
        "realms_tree_link_name"         : u"Realms tree",
        "realm"                         : u"Realm",
        "realm_status"                  : u"Realm Status",
        "no_daemon"                     : u"No daemon",
        "one_daemon"                    : u"1 daemon",
        "nb_daemons"                    : u"%s daemons",
    }
}


def translate(key, lang="en"):
    return TRADS[lang].get(key, '')


TREE_LEFT_MARGIN = 2
TREE_TOP_MARGIN = 1
MINIMAL_REALM_SIZE = 6

_PATH_LISTENER_HOSTS_MAPPING = '/var/lib/shinken/architecture_export_hosts_mapping.json'

unicode_ascii_table = {
    ord(u'é'): u'e',
    ord(u'è'): u'e',
    ord(u'ê'): u'e',
    ord(u'ë'): u'e',
    ord(u'à'): u'a',
    ord(u'â'): u'a',
    ord(u'ä'): u'a',
    ord(u'ç'): u'c',
    ord(u'ü'): u'ue',
    ord(u'ù'): u'u',
    ord(u'û'): u'u',
    ord(u'î'): u'i',
    ord(u'ï'): u'i',
    ord(u'ß'): u'ss',
}


# noinspection PyMethodMayBeStatic
class ArchitectureExportMapper(object):
    SAVE_RETENTION_LOCK = RLock()
    
    
    def __init__(self, path, architecture_name, server_uuid, synchronizer, lang, listener_use_ssl, listener_login, listener_password, architecture_broks, ssh_port, ssh_user, ssh_key_file, ssh_timeout, map_realm_layout):
        self.lang = lang
        self.path = path
        self.map_path = path + "/etc/maps/"
        self.global_map_file_path = path + "/etc/maps/shinken_global-%s.cfg" % server_uuid
        self.detailed_map_file_path = path + "/etc/maps/shinken_architecture-%s.cfg" % server_uuid
        self.background_path = path + "/share/userfiles/images/maps/"
        self.increment = 0
        self.server_uuid = server_uuid
        dir_name = os.path.dirname(path)
        if not os.path.exists(dir_name):
            os.makedirs(dir_name)
        self.tmp_path = "/tmp/arch_shinken-%s.tmp" % server_uuid
        self._already_seen_realm_name = set()
        # transform the unicode char from and then transforms the unicode as ascii and drop unknown chars
        if isinstance(architecture_name, str):
            architecture_name = architecture_name.decode('utf-8')
        if isinstance(architecture_name, unicode):
            architecture_name = architecture_name.translate(unicode_ascii_table).encode('ascii', 'ignore')
        self.architecture_name = architecture_name
        self.architecture_broks = architecture_broks
        
        self.shinken_architecture_map_name = self._translate("shinken_architecture")
        self.global_map_name = self._translate("realms_tree")
        
        self.synchronizer_address = "localhost"
        self.listener_port = 7777
        self.listener_base_path = "/shinken/listener-shinken/v1/hosts/"
        self.get_synchronizer_address(synchronizer)
        
        self.ssh_port = ssh_port
        self.ssh_user = ssh_user
        self.ssh_key_file = ssh_key_file
        self.ssh_timeout = ssh_timeout
        
        self.http_clients = {}
        self._listener_hosts_mapping = {}
        self._load_listener_hosts_mapping()
        self.listener_use_ssl = (listener_use_ssl == 1)
        self.listener_login = listener_login
        self.listener_password = listener_password
        if map_realm_layout in MAP_REALM_LAYOUTS.keys():
            self.map_realm_layout = MAP_REALM_LAYOUTS[map_realm_layout]
        else:
            logger.error("[architecture-export] The 'map_realm_layout' parameter has an illegal value : %s" % map_realm_layout)
    
    
    def _translate(self, key):
        return translate(key, self.lang)
    
    
    def _load_listener_hosts_mapping(self):
        try:
            file_path = _PATH_LISTENER_HOSTS_MAPPING
            if not os.path.exists(os.path.dirname(file_path)):
                os.makedirs(os.path.dirname(file_path))
            if os.path.isfile(file_path):
                self._listener_hosts_mapping = json.load(open(file_path, 'r'))
            return self._listener_hosts_mapping
        except Exception as e:
            logger.warning("[architecture-export] Previous listener hosts mapping retention cannot be load from file [%s]. \n%s" % (_PATH_LISTENER_HOSTS_MAPPING, e))
    
    
    def save_listener_hosts_mapping(self):
        if self._listener_hosts_mapping:
            try:
                with ArchitectureExportMapper.SAVE_RETENTION_LOCK:
                    file_path = _PATH_LISTENER_HOSTS_MAPPING
                    try:
                        if not os.path.exists(os.path.dirname(file_path)):
                            os.makedirs(os.path.dirname(file_path))
                        with open(file_path, 'w') as fd:
                            json.dump(self._listener_hosts_mapping, fd)
                        logger.debug('[architecture-export] [%s] listener hosts mapping saved in %s' % (self.architecture_name, file_path))
                    except Exception as e:
                        logger.error("[architecture-export] [%s] Permission denied on %s : %s" % (self.architecture_name, _PATH_LISTENER_HOSTS_MAPPING, e))
            except:
                logger.warning("[architecture-export] [%s] listener hosts mapping saved in [%s] failed : [%s]" % (self.architecture_name, _PATH_LISTENER_HOSTS_MAPPING, traceback.format_exc()))
    
    
    def get_increment(self):
        self.increment += 1
        return self.increment
    
    
    def get_listener_headers(self):
        _listener_headers = getattr(self, '_listener_headers', None)
        if _listener_headers is None:
            # base64 encode the username and password
            auth = base64.encodestring('%s:%s' % (self.listener_login, self.listener_password)).replace('\n', '')
            _listener_headers = {
                'Authorization': "Basic %s" % auth,
                "Content-type" : "application/json",
            }
        return _listener_headers
    
    
    def get_listener_conn(self):
        if not self.http_clients.get(self.synchronizer_address, None):
            if self.listener_use_ssl:
                if hasattr(ssl, 'SSLContext'):
                    ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1)
                    ssl_context.check_hostname = False
                    ssl_context.verify_mode = ssl.CERT_NONE
                    http_conn = httplib.HTTPSConnection(self.synchronizer_address, self.listener_port, timeout=3, context=ssl_context)
                else:
                    http_conn = httplib.HTTPSConnection(self.synchronizer_address, self.listener_port, timeout=3)
            else:
                http_conn = httplib.HTTPConnection(self.synchronizer_address, self.listener_port, timeout=3)
            self.http_clients[self.synchronizer_address] = http_conn
        return self.http_clients[self.synchronizer_address]
    
    
    def _get_satellite_data(self, sat_type, sat):
        if not sat.enabled:
            return None
        sat_data = {
            'type'           : sat_type,
            'name'           : sat.get_name(),
            'address'        : sat.address,
            'port'           : sat.port,
            'checks'         : list(CHECK_DICT[sat_type]['checks']),
            'checks_packages': list(CHECK_DICT[sat_type]['checks_packages']),
            'nb_slot'        : 0,
            'spare'          : sat.spare,
            'modules'        : [],
            'modules_checks' : []
        }
        sat_data['nb_slot'] += len(sat_data['checks'])
        for packages in sat_data['checks_packages']:
            if len(packages['checks']) % 4 == 0:
                sat_data['nb_slot'] += len(packages['checks']) / 4
            else:
                sat_data['nb_slot'] += len(packages['checks']) / 4 + 1
        if sat.modules:
            for module in sat.modules:
                if isinstance(module, shinken.objects.module.Module):
                    tmp_module = {'type': module.get_type(), 'name': module.get_name(), 'submodules': self.get_submodules(module.get_modules(), [])}
                    modules_checks = {'checks': [], 'submodules': []}
                    
                    if tmp_module['type'] in MODULE_CHECK_DICT.keys():
                        for check in MODULE_CHECK_DICT[tmp_module['type']]['checks']:
                            modules_checks['checks'].append(check)
                    
                    if tmp_module['type'] == "graphite_perfdata":
                        tmp_module["host"] = getattr(module, "host", None)
                        tmp_module["port"] = getattr(module, "port", None)
                    
                    submodule_checks = self.get_submodule_checks(tmp_module['type'], tmp_module['submodules'])
                    for submodule_check in submodule_checks:
                        modules_checks['submodules'].append(submodule_check)
                    
                    sat_data['modules'].append(tmp_module)
                    sat_data['modules_checks'].append(modules_checks)
        return sat_data
    
    
    def get_submodules(self, submodules, tmp_submodules):
        for submodule in submodules:
            tmp_submodule = {'type': submodule.get_type(), 'name': submodule.get_name(), 'submodules': self.get_submodules(submodule.get_modules(), [])}
            tmp_submodules.append(tmp_submodule)
        
        return tmp_submodules
    
    
    def get_submodule_checks(self, module_type, submodules):
        submodule_checks = []
        
        for submodule in submodules:
            tmp_submodule_checks = {'checks': [], 'submodules': []}
            if module_type in MODULE_CHECK_DICT and submodule['type'] in MODULE_CHECK_DICT[module_type]['submodules'].keys():
                for check in MODULE_CHECK_DICT[module_type]['submodules'][submodule['type']]['checks']:
                    tmp_submodule_checks['checks'].append(check)
            
            tmp_submodule_checks['submodules'] = self.get_submodule_checks(submodule['type'], submodule['submodules'])
            submodule_checks.append(tmp_submodule_checks)
        
        return submodule_checks
    
    
    # This will check is the host is already in the list
    def check_if_host_known(self, address, hosts):
        for host_data in hosts:
            if address == host_data['address']:
                return True
        return False
    
    
    def _get_realm_data(self, realm, conf=""):
        realm_name = realm.get_name()
        # if we already add this realm before, exit now
        if realm_name in self._already_seen_realm_name:
            return None
        self._already_seen_realm_name.add(realm_name)
        realm_data = {
            'type' : 'realm',
            'name' : realm_name,
            'hosts': []
        }
        if conf:
            for sat_type in UPPER_REALM_SATELLITE_NAMES:
                for sat in getattr(conf, sat_type + 's', []):
                    sat_data = self._get_satellite_data(sat_type, sat)
                    if sat_data:
                        if self.check_if_host_known(sat_data['address'], realm_data['hosts']) is True:
                            for host_data in realm_data['hosts']:
                                if host_data['address'] == sat_data['address']:
                                    host_data['sats'].append(sat_data)
                                    break
                        else:
                            host_data = {
                                'type'   : 'host',
                                'address': sat_data['address'],
                                'sats'   : [],
                                'realm'  : realm_name
                            }
                            host_data['sats'].append(sat_data)
                            realm_data['hosts'].append(host_data)
        for sat_type in SATELLITE_NAMES:
            # get all satellites of this type in the realm
            for sat in realm.get_satellites_by_type(sat_type):
                sat_data = self._get_satellite_data(sat_type, sat)
                if sat_data:
                    if self.check_if_host_known(sat_data['address'], realm_data['hosts']) is True:
                        for host_data in realm_data['hosts']:
                            if host_data['address'] == sat_data['address']:
                                host_data['sats'].append(sat_data)
                                break
                    else:
                        host_data = {
                            'type'   : 'host',
                            'address': sat_data['address'],
                            'sats'   : [],
                            'realm'  : realm_name
                        }
                        host_data['sats'].append(sat_data)
                        realm_data['hosts'].append(host_data)
        # manage sub realms, if found call us recursively otherwise remove the key
        realm_data['sub_realms'] = []
        sub_realms_data = realm_data['sub_realms']
        for sub_realm in realm.get_realms():
            temp = self._get_realm_data(sub_realm)
            if temp is not None:
                sub_realms_data.append(temp)
        
        return realm_data
    
    
    def _get_architecture_realm(self):
        arch_dict = {
            'type'  : 'architecture',
            'name'  : "[%s] %s" % (self.architecture_name, self.shinken_architecture_map_name),
            'length': 3,
            'height': 3,
        }
        return arch_dict
    
    
    def generate_submodule_block(self, submodule, daemon_name, x, y, x_offset, y_offset, z, config_file, hostname, icon_size, icon_size_px):
        submodules_written = 0
        
        submodule_icon_size = (icon_size / 2)
        submodule_icon_size_px = submodule_icon_size - 3
        x_offset += icon_size_px + 2
        y_offset += 2
        start_y_offset = y_offset
        for check in submodule['checks']:
            label_x = "center"
            service_name = check['service_description'].replace('$KEY$', daemon_name)
            
            self.write_service(x=x + x_offset,
                               y=y + y_offset,
                               config_file=config_file,
                               hostname=hostname,
                               service_description=service_name,
                               z=z + 1,
                               icon_size=submodule_icon_size,
                               label="Modules",
                               display_label=False,
                               label_x=label_x,
                               is_package=True)
            
            submodules_written += 1
            if y_offset == start_y_offset:
                y_offset = start_y_offset + submodule_icon_size_px
            else:
                y_offset = start_y_offset
                x_offset += submodule_icon_size_px
        
        return submodules_written
    
    
    # This will write a service on nagvis
    def write_service(self, x, y, config_file, hostname, service_description, z, label, icon_size=40, display_label=True, label_x="center", is_package=False):
        if display_label:
            label_show = 1
        else:
            label_show = 0
        
        label_offset = -9
        if is_package:
            label_offset = -11  # In case of packages (block of little icons), we need to place it higher
        
        service = '''
define service {
    host_name=%(host_name)s
    service_description=%(service_description)s
    x=%(x)d
    y=%(y)d
    z=%(z)d
    label_show=%(label_show)d
    icon_size=%(icon_size)d
    label_x=%(label_x)s
    label_y=%(label_y)d
    label_text=%(label_text)s
    object_id=%(object_id)s
}

''' % {
            "host_name"          : hostname,
            "service_description": service_description,
            "x"                  : x,
            "y"                  : y,
            "z"                  : z + 1,
            "label_show"         : label_show,
            "icon_size"          : icon_size,
            "label_y"            : label_offset,  # Center label text vertically
            "label_x"            : label_x,
            "label_text"         : label,
            "object_id"          : "shinkenservice%d" % self.get_increment()
        }
        config_file.write(service.encode('utf-8'))
    
    
    def generate_host_name(self, realm, address):
        return "%s - %s - %s" % (self.architecture_name, realm.upper(), address)
    
    
    # This will write a host on nagvis
    def write_host(self, x, y, config_file, name, z, length, tile_size):
        # We now all hosts will be written on top left corner, so we add few pixels on x and y to shift the icon from the corner
        host = '''
define host {
    host_name=%(host_name)s
    x=%(x)d
    y=%(y)d
    z=%(z)d
    label_x=%(label_x)s
    label_y=%(label_y)d
    label_text=%(label_text)s
    icon_size=%(icon_size)d
    object_id=%(object_id)s
    label_width=%(label_width)d
}

''' % {
            "host_name"  : name,
            "x"          : x + 5,  # Shift the host a few pixels away from the bloc corner
            "y"          : y + 7,  # Shift the host a few pixels away from the bloc corner
            "z"          : z + 1,
            "label_x"    : "+30",  # Put the label on the right of the icon
            "label_y"    : 0,
            "label_text" : "<div title=\"%s\">%s</div>" % (name, name),
            "icon_size"  : 30,
            "object_id"  : "shinkenhost%d" % self.get_increment(),
            "label_width": (length - 1) * tile_size - 15  # As the text is next to the icon we remove 1 block, which is used to the icon and corners
        }
        config_file.write(host.encode('utf-8'))
    
    
    def write_architecture_name(self, x, y, config_file, text):
        to_write = '''
define textbox {
    object_id=shinkenarchitecturename%(increment)d
    text=%(text)s
    x=%(x)d
    y=%(y)d
    border_color=transparent
}

''' % {
            "text"     : text,
            "x"        : x,
            "y"        : y,
            "increment": self.get_increment()
        }
        config_file.write(to_write.encode('utf-8'))
    
    
    # This will write a daemon text on nagvis
    def write_daemon(self, x, y, config_file, text, z, tile_size, width):
        to_write = '''
define textbox {
    text = %(text)s
    x=%(x)d
    y=%(y)d
    z=%(z)d
    h=%(height)d
    w=%(width)d
    border_color=transparent
    object_id=shinkendaemonname%(increment)d
}

''' % {
            "text"     : text,
            "x"        : x,
            "y"        : y,
            "z"        : z,
            "height"   : tile_size,
            "width"    : width,
            "increment": self.get_increment()
        }
        config_file.write(to_write.encode('utf-8'))
    
    
    # This will write a realm text on nagvis
    def write_realm_text(self, x, y, config_file, text, z, width):
        to_write = '''
define textbox {
    text = %(text)s
    x=%(x)d
    y=%(y)d
    z=%(z)d
    h=%(height)d
    w=%(width)d
    border_color=transparent
    object_id=shinkenrealm%(increment)d
}

''' % {
            "text"     : text,
            "x"        : x,
            "y"        : y,
            "z"        : z,
            "height"   : 45,
            "width"    : width,
            "increment": self.get_increment()
        }
        config_file.write(to_write.encode('utf-8'))
    
    
    # This will write a left text on nagvis
    def write_left_text(self, x, y, config_file, text, z, tile_size, width, font_size):
        to_write = '''
define textbox {
    object_id=%(object_id)s
    text = %(text)s
    x=%(x)d
    y=%(y)d
    z=%(z)d
    h=%(height)d
    w=%(width)d
    border_color=transparent
}

''' % {
            "object_id": "shinkenlogocaption%d" % self.get_increment(),
            "text"     : text,
            "x"        : x,
            "y"        : y + tile_size,
            "z"        : z,
            "font_size": font_size,
            "height"   : tile_size,
            "width"    : width
        }
        config_file.write(to_write.encode('utf-8'))
    
    
    # This will write the shinken logo on nagvis
    def write_logo(self, x, y, config_file, small=False):
        if small is True:
            logo = "enterprise-white-small.png"
        else:
            logo = "enterprise-white.png"
        block = '''
define shape {
    object_id=%(object_id)s
    icon=%(logo)s
    x=%(x)d
    y=%(y)d
    z=3
}

''' % {
            "object_id": "shinkenlogoimg%d" % self.get_increment(),
            "logo"     : logo,
            "x"        : x,
            "y"        : y
        }
        config_file.write(block.encode('utf-8'))
    
    
    # This will write realms on nagvis
    def write_realm(self, x, y, config_file, realm, z, icon_size):
        # As we want to center the icon, we need to divide x and y by 2
        to_write = '''
define map {
    map_name=%(map_name)s-%(architecture_name)s
    x=%(x)d
    y=%(y)d
    z=%(z)d
    icon_size=%(size)s
    label_x=+30
    label_y=0
    object_id=shinkenstatusrealm%(increment)s
    label_text=%(realm_status)s
    label_width=200
}

''' % {
            "map_name"         : realm.replace(' ', '_'),
            "architecture_name": self.server_uuid,
            "x"                : x,
            "y"                : y,
            "z"                : z + 1,
            "size"             : icon_size,
            "realm_status"     : self._translate("realm_status"),
            "increment"        : self.get_increment()
        }
        config_file.write(to_write.encode('utf-8'))
    
    
    # This will create a square in nagvis, from the start to the end
    def square_generator(self, start, end, z, block_tab, host=False):
        # As we only have 3 different tile, if we are in 4 depth, we need to take back the first tile, 5 depth = tile n2, etc
        if z > 5:
            while z > 5:
                z -= 2
        if host is True:
            z = "_host" + str(z)
        
        # It's a simple square generator algorithm
        for y in xrange(int(start['y']), int(end['y'])):
            for x in xrange(int(start['x']), int(end['x'])):
                if x == start['x']:
                    if y == start['y']:
                        block_tab[x][y] = "top_left_corner" + str(z) + ".png"
                    elif y == end['y'] - 1:
                        block_tab[x][y] = "bottom_left_corner" + str(z) + ".png"
                    else:
                        block_tab[x][y] = "left_border" + str(z) + ".png"
                elif x == end['x'] - 1:
                    if y == start['y']:
                        block_tab[x][y] = "top_right_corner" + str(z) + ".png"
                    elif y == end['y'] - 1:
                        block_tab[x][y] = "bottom_right_corner" + str(z) + ".png"
                    else:
                        block_tab[x][y] = "right_border" + str(z) + ".png"
                elif y == start['y']:
                    block_tab[x][y] = "top_border" + str(z) + ".png"
                elif y == end['y'] - 1:
                    block_tab[x][y] = "bottom_border" + str(z) + ".png"
                else:
                    if host is True:
                        block_tab[x][y] = "full_host.png"
                    else:
                        block_tab[x][y] = "full" + str(z) + ".png"
        return block_tab
    
    
    # Computes size of architecture object
    # Returns a copy of the arch dict describing the architecture with additional "length" and "width" keys
    def compute_size_of_arch_objects(self, arch):
        for objects in arch:
            if objects['type'] == 'architecture':
                # Default architecture map is 5 blocks high (title + top and bottom logos) and 2 blocks wide (margins)
                objects['height'] = 5
                objects['length'] = 2
                
                if objects.get('sub_realms'):
                    objects['sub_realms'] = self.compute_size_of_arch_objects(objects['sub_realms'])
                    for sub_realm in objects['sub_realms']:
                        objects['height'] += sub_realm['height']
                        objects['length'] += sub_realm['length']
            
            # Compute size of realm
            if objects['type'] == 'realm':
                # An empty block has height of 3 blocks (realm title + borders)
                objects['height'] = 3
                objects['length'] = 2
                realm_width = 0
                
                # Compute size of hosts
                if objects['hosts']:
                    # Two hosts are displayed on each line.
                    host_counter = 0
                    hosts_width = 0
                    hosts_height = 0
                    hosts_width_buffer = 0
                    hosts_height_buffer = 0
                    check_size = 2
                    daemon_length = 7
                    
                    for host in objects['hosts']:
                        host_counter += 1
                        # We compute host width depending on its number of daemons (satellites)
                        # 7 is the width of a daemon display (label + 2 status icons)
                        # As daemons are displayed on the lines, we divide by 2
                        if len(host['sats']) % 2 == 0:
                            host['length'] = (len(host['sats']) / 2) * daemon_length
                        else:
                            host['length'] = ((len(host['sats']) + 1) / 2) * daemon_length
                        
                        for daemon in host['sats']:
                            host['length'] += (daemon['nb_slot'] - 2) * 2
                            if daemon['modules_checks']:
                                checks_len = 0
                                for module in daemon['modules_checks']:
                                    checks_len += len(module['checks'])
                                    for submodule in module['submodules']:
                                        columns = math.ceil(float(len(submodule['checks'])) / 4)
                                        checks_len += (columns * 2)
                                # We will display modules_checks as little icon, in this case, length of 1 tile = length of 3 little checks
                                # And height of 1 check = width of 2 little checks, so each 6 checks (3 up, 3 down), we have one more check_size
                                if checks_len <= 6:
                                    host['length'] += check_size
                                # If we have more than 6 modules_checks, we need to create more tiles (one per block, one block is 2 checks up, 2 checks down, + the margin of 1 check_size)
                                elif checks_len % 4:
                                    host['length'] += int(checks_len / 6) + check_size
                                # In case we do not finish a block (2 checks left, 0 right), we don't need to add a complete check size tile)
                                else:
                                    host['length'] += int(checks_len / 6) + check_size / 2
                        # A host has a height of 3 tiles
                        host['height'] = 3
                        
                        # If we have more than one daemon, we display all daemons on 2 lines, so the host takes one more block
                        if len(host['sats']) > 1:
                            host['height'] += 1
                        
                        if host_counter % 2 == 1:
                            hosts_height_buffer = hosts_height
                            hosts_height_buffer += host['height']
                            
                            hosts_width_buffer = host['length']
                        else:
                            hosts_height_buffer = max(hosts_height, hosts_height_buffer)
                            hosts_height = max(hosts_height, hosts_height_buffer)
                            
                            hosts_width_buffer += host['length']
                            hosts_width = max(hosts_width, hosts_width_buffer)
                    
                    # If last item is first of line, we must add his height to the block height
                    if host_counter % 2 == 1:
                        hosts_width = max(hosts_width, hosts_width_buffer)
                        hosts_height = max(hosts_height, hosts_height_buffer)
                    
                    objects['height'] += hosts_height
                    realm_width = hosts_width
                
                # Compute size of subrealm
                if objects['sub_realms']:
                    sub_realm_counter = 0
                    sub_realms_width = 0
                    sub_realms_height = 0
                    sub_realms_width_buffer = 0
                    sub_realms_height_buffer = 0
                    
                    # If this realm has sub_realms, we need to compute the size of its sub realms first
                    objects['sub_realms'] = self.compute_size_of_arch_objects(objects['sub_realms'])
                    for sub_realms in objects['sub_realms']:
                        sub_realm_counter += 1
                        
                        if sub_realm_counter % 2 == 1:
                            sub_realms_height_buffer = sub_realms_height
                            sub_realms_height_buffer += sub_realms['height']
                            
                            sub_realms_width_buffer = sub_realms['length']
                        else:
                            sub_realms_height_buffer = max(sub_realms['height'], sub_realms_height_buffer)
                            sub_realms_height = max(sub_realms_height, sub_realms_height_buffer)
                            
                            sub_realms_width_buffer += sub_realms['length']
                            sub_realms_width = max(sub_realms_width, sub_realms_width_buffer)
                    
                    # If last item is first of line, we must add his height to the block height
                    if sub_realm_counter % 2 == 1:
                        sub_realms_width = max(sub_realms_width, sub_realms_width_buffer)
                        sub_realms_height = max(sub_realms_height, sub_realms_height_buffer)
                    
                    objects['height'] += sub_realms_height
                    realm_width = max(realm_width, sub_realms_width)
                
                if (not objects['hosts']) and (not objects['sub_realms']):
                    # If the realm is completely empty, give it a minimum size to avoid a broken NagVis map display (SEF-3984)
                    realm_width = MINIMAL_REALM_SIZE
                
                # Hosts and sub realms width has been computed.
                objects['length'] += realm_width
        
        return arch
    
    
    # This will generate the concatenated map (recursive function)
    def block_in_block_generator(self, arch, tile_size, config_file, block_x, block_y, z, block_tab, position=1):
        first_height = 0
        first = 1
        save_x = block_x
        icon_y_margin = 15
        
        for objects in arch:
            if position == 1:
                first_height = objects['height']
            if first == 1:
                first = 0
            
            # Coordinates of top left corner of block
            start = {'x': block_x, 'y': block_y}
            # Coordinates of bottom right corner of block
            end = {'x': objects['length'] + block_x, 'y': objects['height'] + block_y}
            
            # If we are in a realm, we need to generate all sub_realms and hosts
            if objects['type'] == 'realm':
                block_tab = self.square_generator(start, end, z, block_tab)
                inside_y = block_y
                if objects['hosts']:
                    # We add an offset of 1 block to avoid it being directly drawed on the edge of the realm block on x
                    # We add an offset of 2 blocks corresponding to the realm title on y
                    block_tab, inside_y, last_height = self.block_in_block_generator(objects['hosts'], tile_size, config_file, block_x + 1, block_y + 2, z + 1, block_tab)
                else:
                    # If the realm doesn't contain any hosts, we leave a 2 blocks vertical space to avoid having the realms being draw on top of each other
                    inside_y += 2
                if objects['sub_realms']:
                    # Offset sub_realm by 1 block to the left to avoid it being drawn of the edge of the current realm
                    inside_x = block_x + 1
                    block_tab, inside_y, last_height = self.block_in_block_generator(objects['sub_realms'], tile_size, config_file, inside_x, inside_y, z + 1, block_tab)
                
                x = start['x'] * tile_size
                # Write realm name 1 block from the top of the block to give it a bit of margin
                y = (start['y'] + 1) * tile_size
                realm_text = "%s : %s" % (self._translate("realm"), objects['name'])
                self.write_realm_text(x, y, config_file, realm_text, z, objects['length'] * tile_size)
                
                # Center realm icon minus some pixels to leave blank space for the "Realm status" text
                x = ((start['x'] * tile_size + end['x'] * tile_size) / 2) - (len(self._translate("realm_status")) * 3)
                # Write realm status icon 1 block from the top of the block to give it a bit of margin
                y = (start['y'] + 1) * tile_size + icon_y_margin
                self.write_realm(x, y, config_file, objects['name'], z, 30)
            
            elif objects['type'] == 'host':
                
                daemon_length = 7
                icon_size = 40
                little_icon_size = icon_size / 2
                check_size = 2
                daemon_width = 120
                little_icon_size_px = 15
                title_padding = 6
                
                block_tab = self.square_generator(start, end, z, block_tab, True)
                # We put the host title 6 pixels on right and bottom of the top-left corner of the box
                x = (start['x']) * tile_size + title_padding
                y = (start['y']) * tile_size + title_padding
                hostname = self.generate_host_name(objects['realm'], objects['address'])
                self.write_host(x, y, config_file, hostname, z, objects['length'], tile_size)
                
                daemon_position_horizontal_offset = 0
                daemon_position_vertical_offset = 1
                write_daemon_on_first_line = True
                offset_to_next_item = 0
                
                for daemon in objects['sats']:
                    # First daemon in on top, second is on bottom, after the second we do the same next to them
                    spare_label = ""
                    if daemon["spare"]:
                        spare_label = "<span class=\"shinken-daemon-spare\">SPARE</span>"
                    daemon_label = "<span class=\"shinken-daemon-type\">" + daemon['type'] + "</span>" + spare_label + "<br/>" + daemon['name']
                    
                    if write_daemon_on_first_line:
                        x = (start['x'] + daemon_position_horizontal_offset) * tile_size + tile_size / 3
                        y = (start['y'] + daemon_position_vertical_offset) * tile_size + tile_size / 2
                        self.write_daemon(x, y, config_file, daemon_label, z + 1, tile_size, daemon_width)
                    else:
                        x = (start['x'] + daemon_position_horizontal_offset) * tile_size + tile_size / 3
                        y = (start['y'] + daemon_position_vertical_offset) * tile_size - tile_size / 4
                        self.write_daemon(x, y, config_file, daemon_label, z + 1, tile_size - 7, daemon_width)
                    
                    # Daemon name is 3 blocks wide
                    service_position_x_offset = 3
                    nb_service = 0
                    check_name_prefix = "%s - %s - " % (daemon['type'][0].upper() + daemon['type'][1:], daemon['name'])
                    
                    # If we have more than 2 checks, we need to create some padding
                    offset_to_next_item += (daemon['nb_slot'] - 2) * 2
                    # We save this for later
                    checks_len = 0
                    if daemon['modules_checks']:
                        for module in daemon['modules_checks']:
                            checks_len += len(module['checks'])
                            for submodule in module['submodules']:
                                columns = math.ceil(float(len(submodule['checks'])) / 4)
                                checks_len += (columns * 2)
                        if checks_len <= 6:
                            offset_to_next_item += check_size
                        elif checks_len % 4:
                            offset_to_next_item += int(checks_len / 6) + check_size
                        else:
                            offset_to_next_item += int(checks_len / 6) + check_size / 2
                    
                    checks_to_display = daemon['checks']
                    checks_packages = daemon['checks_packages']
                    
                    for check in checks_to_display:
                        service_description = check_name_prefix + check["service_description"]
                        
                        x = (start['x'] * tile_size) + tile_size * (daemon_position_horizontal_offset + service_position_x_offset + nb_service) + tile_size / 3
                        # Service can be on top or on bottom of the box, so we need to change the position
                        if write_daemon_on_first_line:
                            y = (start['y'] + daemon_position_vertical_offset) * tile_size + tile_size / 2
                        else:
                            y = (start['y'] + daemon_position_vertical_offset) * tile_size - 10
                        
                        self.write_service(x=x,
                                           y=y,
                                           config_file=config_file,
                                           hostname=hostname,
                                           icon_size=icon_size,
                                           service_description=service_description,
                                           z=z + 1,
                                           label=check['label'])
                        nb_service += 1
                        service_position_x_offset += 1
                    
                    little_icon_first_line_padding = 2
                    little_icon_second_line_padding = -7
                    
                    modules_block_offset = 0
                    modules_block_right_padding = (little_icon_size_px + little_icon_first_line_padding) * 5
                    
                    # Module checks are displayed differently, so they are handled separately (the X modules checks form 1 icon)
                    if daemon['modules_checks']:
                        
                        x = (start['x'] + daemon_position_horizontal_offset + service_position_x_offset + nb_service) * tile_size + tile_size / 2
                        
                        if write_daemon_on_first_line:
                            # Since little icons don't have the same margin is bigger ones, we add 3px of offset to compensate
                            y = (start['y'] + daemon_position_vertical_offset) * tile_size + tile_size / 2 + little_icon_first_line_padding
                        else:
                            # Since little icons don't have the same margin is bigger ones, we add 3px of offset to compensate
                            y = (start['y'] + daemon_position_vertical_offset) * tile_size + little_icon_second_line_padding
                        
                        x_offset = 0
                        y_offset = 0
                        modules_checks_written = 0
                        label_written = True
                        label_written_one_time = False
                        
                        last_squares_written = 0
                        
                        # Module check are differents, we need to put them in a variant format
                        for module in daemon['modules_checks']:
                            for check in module['checks']:
                                label_x = "center"
                                if check.get('service_description') is None:
                                    logger.error("[architecture-export] [%s] the version of Shinken that sends the map is not compatible" % self.architecture_name)
                                    break
                                service_name = check['service_description'].replace('$KEY$', daemon['name'])
                                
                                # Simple algorithm, label will be formatted if we have a odd number of modules checks
                                if checks_len > 2:
                                    column_number = int(round(checks_len / 2))
                                    if column_number == 2:
                                        if label_written_one_time is False:
                                            label_written = False
                                            label_written_one_time = True
                                            label_x = "-5"
                                    elif column_number % 2 != 0:
                                        if label_written_one_time is False:
                                            label_written = False
                                            label_written_one_time = True
                                            label_x = "+" + str(4 + (little_icon_size_px * ((column_number / 2) - 1)))
                                    else:
                                        if label_written_one_time is False:
                                            label_written = False
                                            label_written_one_time = True
                                            label_x = "+" + str((little_icon_size_px * ((column_number / 2) - 1)) - 4)
                                else:
                                    if label_written_one_time is False:
                                        label_written = False
                                        label_written_one_time = True
                                        label_x = "-12"
                                
                                self.write_service(x=x + x_offset,
                                                   y=y + y_offset,
                                                   config_file=config_file,
                                                   hostname=hostname,
                                                   service_description=service_name,
                                                   z=z + 1,
                                                   icon_size=little_icon_size,
                                                   label="Modules",
                                                   display_label=not label_written,
                                                   label_x=label_x,
                                                   is_package=True)
                                
                                squares_written = 0
                                if (len(module['checks']) - 1) == modules_checks_written:
                                    submodules_written = 0
                                    for submodule in module['submodules']:
                                        submodules_written += self.generate_submodule_block(submodule, daemon['name'], x, y, x_offset, y_offset, z, config_file, hostname, little_icon_size, little_icon_size_px)
                                    squares_written = math.ceil(float(submodules_written) / 4)
                                
                                modules_checks_written += 1
                                label_written = True
                                
                                modules_block_offset = x + x_offset
                                
                                if y_offset == 0:
                                    y_offset = little_icon_size_px
                                    last_squares_written = squares_written
                                else:
                                    if squares_written > 0:
                                        y_offset = 0
                                        if last_squares_written > squares_written:
                                            x_offset += last_squares_written * little_icon_size_px
                                        else:
                                            x_offset += little_icon_size_px + squares_written * little_icon_size_px
                                    else:
                                        if last_squares_written > 0:
                                            y_offset = 0
                                            x_offset += little_icon_size_px + last_squares_written * little_icon_size_px
                                            last_squares_written = 0
                                        else:
                                            y_offset = 0
                                            x_offset += little_icon_size_px
                        
                        service_position_x_offset += 1
                    
                    # Packages checks are displayed differently, so they are handled separately (4 checks form 1 icon)
                    for packages in checks_packages:
                        label_written = True
                        label_written_one_time = False
                        x_offset = 0
                        y_offset = 0
                        index = 0
                        x = (start['x'] + daemon_position_horizontal_offset + service_position_x_offset + nb_service) * tile_size + tile_size / 3
                        if modules_block_offset > 0:
                            x = modules_block_offset + modules_block_right_padding
                        
                        if write_daemon_on_first_line:
                            # Since little icons don't have the same margin is bigger ones, we add 3px of offset to compensate
                            y = (start['y'] + daemon_position_vertical_offset) * tile_size + tile_size / 2 + little_icon_first_line_padding
                        else:
                            # Since little icons don't have the same margin is bigger ones, we add 3px of offset to compensate
                            y = (start['y'] + daemon_position_vertical_offset) * tile_size + little_icon_second_line_padding
                        
                        package_padding = 2
                        for check in packages['checks']:
                            label_x = "center"
                            if len(packages['checks']) % 2 != 0:
                                if index == round(float(len(packages['checks']) + 1) / 4) - 1 and label_written_one_time is False:
                                    label_written = False
                                    label_written_one_time = True
                                    # This hard code is here for make label centered in case of odd number of column
                                    if (len(packages['checks']) + 1) / 2 % 2 == 0:
                                        # In case we do not have a odd number of columns, we need to center correctly, by move the text to the left
                                        label_x = "-" + str(little_icon_size / 4 + package_padding)
                            else:
                                if index == round(float(len(packages['checks'])) / 4) - 1 and label_written_one_time is False:
                                    label_written = False
                                    label_written_one_time = True
                                    # This is here for make label centered
                                    if (len(packages['checks'])) / 2 % 2 == 0:
                                        # In case we do not have a odd number of columns, we need to center correctly, by move the text to the left
                                        label_x = "-" + str(little_icon_size / 4 + package_padding)
                            self.write_service(x=x + x_offset,
                                               y=y + y_offset,
                                               config_file=config_file,
                                               hostname=hostname,
                                               service_description=check["service_description"],
                                               z=z + 1,
                                               icon_size=little_icon_size,
                                               label_x=label_x,
                                               label=packages['name'],
                                               display_label=not label_written,
                                               is_package=True)
                            label_written = True
                            if y_offset == 0:
                                y_offset = little_icon_size_px
                            else:
                                y_offset = 0
                                x_offset += little_icon_size_px
                                index += 1
                        
                        if len(packages['checks']) % 4 != 0:
                            nb_service += len(packages['checks']) / 4 + 2
                        else:
                            nb_service += len(packages['checks']) / 4 + 1
                    
                    # If we are on top, next will be above it
                    if write_daemon_on_first_line:
                        daemon_position_vertical_offset = 3
                        write_daemon_on_first_line = False
                    else:
                        write_daemon_on_first_line = True
                        daemon_position_vertical_offset = 1
                        daemon_position_horizontal_offset += daemon_length + offset_to_next_item
                        offset_to_next_item = 0
            
            if position == 1:
                block_x += objects['length']
                position = 2
            else:
                block_x = save_x
                block_y += max(first_height, objects['height'])
                position = 1
        
        if arch and position == 2:
            block_y += first_height
        
        return block_tab, block_y, first_height
    
    
    # We sort all hosts by size
    def sort_hosts(self, hosts, limit):
        i = 0
        while i < limit - 1:
            if hosts[i + 1]['height'] > hosts[i]['height']:
                hosts[i], hosts[i + 1] = hosts[i + 1], hosts[i]
                i = 0
            i += 1
        return hosts
    
    
    @staticmethod
    def by_size_sorter(realm1, realm2):
        # type: (Dict, Dict) -> bool
        return (realm1['length'] * realm1['height']) > (realm2['length'] * realm2['height'])
    
    
    @staticmethod
    def by_name_sorter(realm1, realm2):
        # type: (Dict, Dict) -> bool
        return realm2['name'] > realm1['name']
    
    
    # We sort all realms using the comparator functions above
    def sort_all_realms(self, realms, sorter_fn, limit):
        # type: (List[Dict], Callable[[Dict,Dict],bool], int) -> List[Dict]
        
        i = 0
        while i < limit:
            if i < limit - 1 and sorter_fn(realms[i + 1], realms[i]):
                realms[i], realms[i + 1] = realms[i + 1], realms[i]
                i = 0
                continue
            if realms[i]['sub_realms']:
                realms[i]['sub_realms'] = self.sort_all_realms(realms[i]['sub_realms'], sorter_fn, len(realms[i]['sub_realms']))
            if realms[i]['hosts']:
                realms[i]['hosts'] = self.sort_hosts(realms[i]['hosts'], len(realms[i]['hosts']))
            i += 1
        return realms
    
    
    # This generate a background png
    def create_background(self, block_tab, tile_size, length, height, background_file, tree=False):
        # creates a new empty image and fill it with block/tile image
        new_im = Image.new('RGB', (length * tile_size, tile_size * height))
        for x in xrange(0, length):
            for y in xrange(0, height):
                if x == 0 or y == 0:
                    position = (x * tile_size, y * tile_size)
                    if tree is True:
                        new_im.paste(Image.open(self.path + "/share/userfiles/images/shapes/tree_blank.png"), position)
                    else:
                        new_im.paste(Image.open(self.path + "/share/userfiles/images/shapes/blank.png"), position)
                position = (x * tile_size, y * tile_size)
                if block_tab[x][y]:
                    if tree is True:
                        new_im.paste(Image.open(self.path + "/share/userfiles/images/shapes/tree_block.png"), position)
                    else:
                        new_im.paste(Image.open(self.path + "/share/userfiles/images/shapes/" + block_tab[x][y]), position)
                else:
                    if tree is True:
                        new_im.paste(Image.open(self.path + "/share/userfiles/images/shapes/tree_blank.png"), position)
                    else:
                        new_im.paste(Image.open(self.path + "/share/userfiles/images/shapes/blank.png"), position)
        
        new_im.save(background_file, "PNG")
    
    
    # This is the main function of the nagvis generator
    def nagvis_map_generator(self, all_arch, path, name, parent_map=""):
        MAP_LEFT_MARGIN = 1
        MAP_TOP_MARGIN = 2
        tile_size = 45
        date = datetime.now()
        background_name = "background_%s%s%s%s%s%s-%s.png" % (name.replace(' ', '_'), date.year, date.month, date.day, date.hour, date.minute, self.server_uuid)
        tmp_path = '%s.tmp' % path
        config_file = open(tmp_path, "w")
        config_to_write = '''
define global {
    alias=%(alias)s
    label_show=1
    label_border=transparent
    grid_steps=%(grid_steps)s
    iconset=shinken
    map_image=%(map_image)s
    stylesheet=%(stylesheet)s
''' % {
            "alias"     : name,
            "grid_steps": tile_size,
            "map_image" : background_name,
            "stylesheet": "shinken.css"
        }
        if parent_map:
            config_to_write += "parent_map=%s\n" % parent_map.replace(' ', '_')
        config_to_write += "}\n\n"
        config_file.write(config_to_write.encode('utf-8'))
        
        architecture_info = self.compute_size_of_arch_objects([all_arch])[0]
        block_tab = [[0] * architecture_info['height'] for _ in range(architecture_info['length'])]
        
        # Block top left coordinates
        start = {'x': 0, 'y': 0}
        # Block bottom right coordinates
        end = {'x': architecture_info['length'], 'y': architecture_info['height']}
        
        # Draw logo one tile from the left border of the screen
        x = tile_size
        # Write small logo one tile from the top of the screen (minus a little margin for better display)
        y = tile_size - 5
        self.write_logo(x, y, config_file, True)
        
        # Write architecture name on the right of the logo
        x = 5 * tile_size
        y = tile_size - 5
        self.write_architecture_name(x, y, config_file, "Architecture: %s" % self.architecture_name)
        
        block_tab = self.square_generator(start, end, 1, block_tab)
        # Write realm data on the map, with a left margin and top margin (for the logo)
        block_tab, y_fact, last_height = self.block_in_block_generator(architecture_info['sub_realms'], tile_size, config_file, MAP_LEFT_MARGIN, MAP_TOP_MARGIN, 2, block_tab, 0)
        
        # Draw logo one tile from the left border of the screen
        x = tile_size
        # Write logo after the last block of the map (excluding map top margin for logo and architecture name)
        y = (end['y'] - MAP_TOP_MARGIN - 1) * tile_size
        self.write_logo(x, y, config_file)
        
        # Draw logo one tile from the left border of the screen
        x = tile_size
        y = (end['y'] - MAP_TOP_MARGIN) * tile_size
        self.write_left_text(x, y, config_file, self._translate("do_not_edit_map"), 2, tile_size, 6 * tile_size, 12)
        
        config_file.close()
        # Move temporary working file into definitive map file
        shutil.move(tmp_path, path)
        
        background_file = self.background_path + background_name
        self.create_background(block_tab, tile_size, architecture_info['length'], architecture_info['height'], background_file)
        
        uid = pwd.getpwnam("shinken").pw_uid
        gid = grp.getgrnam("apache").gr_gid
        os.chown(path, uid, gid)
        os.chmod(path, 0664)
        os.chown(background_file, uid, gid)
    
    
    # We look of many leaf and plug we need
    def take_tree_size_of_arch(self, arch, total, stage):
        for realm in arch:
            if realm['sub_realms']:
                total, stage = self.take_tree_size_of_arch(realm['sub_realms'], total, stage)
            total += 2
        stage += 1
        return total, stage
    
    
    # This generate plugs
    def write_parent_line(self, child_x, child_y, config_file, parent_x, parent_y):
        if child_x == parent_x:
            config_to_write = '''
define line {
    x=%(parent_x)s,%(child_x)s
    y=%(parent_y)s,%(child_y)s
    line_type=%(line_type)d
    z=%(z_index)d
    line_width=%(line_width)d
    line_color_border=%(line_color_border)s
    line_color=%(line_color)s
}

''' % {
                "parent_x"         : parent_x,
                "child_x"          : child_x,
                "parent_y"         : parent_y,
                "child_y"          : child_y,
                "line_type"        : 12,
                "z_index"          : 50,
                "line_width"       : 2,
                "line_color_border": "transparent",
                "line_color"       : "#EFECE5"
            }
        else:
            config_to_write = '''
define line {
    x=%(parent_x)s,%(child_x)s
    y=%(parent_y)s,%(child_y)s
    line_type=%(line_type)d
    line_width=%(line_width)d
    line_color_border=%(line_color_border)s
    line_color=%(line_color)s
}

''' % {
                "parent_x"         : parent_x,
                "child_x"          : child_x,
                "parent_y"         : parent_y,
                "child_y"          : child_y,
                "line_type"        : 12,
                "line_width"       : 2,
                "line_color_border": "transparent",
                "line_color"       : "#EFECE5"
            }
        config_file.write(config_to_write.encode('utf-8'))
    
    
    # This generate a leaf on the tree map
    def write_leaf(self, realm, config_file, x, y, block_size, childs):
        margin = 38
        self.write_realm_text(x + margin,
                              y + block_size / 4 + margin - 15,  # Give 15px more space to the text to leave room for a second text line if realm name is too long
                              config_file, realm['name'],
                              1,
                              block_size - 75)  # Remove 75px from text width to account for margin on both sides
        for child in childs:
            self.write_parent_line(child['x'] + block_size / 2,
                                   child['y'] + (margin - 1),
                                   config_file,
                                   x + block_size / 2,
                                   y + block_size - margin)
        
        nb_daemon = 0
        for host in realm['hosts']:
            nb_daemon += len(host['sats'])
        
        if nb_daemon == 1:
            nb_daemon = self._translate("one_daemon")
        elif nb_daemon == 0:
            nb_daemon = self._translate("no_daemon")
        else:
            nb_daemon = self._translate("nb_daemons") % nb_daemon
        to_write = '''
define map {
    map_name=%(map_name)s
    x=%(x)d
    y=%(y)d
    z=%(z_index)d
    icon_size=%(icon_size)s
    label_text=%(label_text)s
    label_x=+%(label_x)s
    label_y=%(label_y)d
    object_id=%(object_id)s
    label_width=%(label_width)s
}

''' % {
            "map_name"   : "%s-%s" % (realm['name'].replace(' ', '_'), self.server_uuid),
            "x"          : x + margin + 15,  # Shift the icon from 15px to the left
            "y"          : y + block_size / 2 + 12,  # Move the icon 12px to the bottom for alignment
            "z_index"    : 1,
            "icon_size"  : 50,
            "label_text" : nb_daemon,
            "label_x"    : 50,
            "label_y"    : 0,
            "label_width": block_size / 2 - margin,
            "object_id"  : "shinkentreerealm%d" % self.get_increment()
        }
        config_file.write(to_write.encode('utf-8'))
    
    
    # Recursive function which will create a tree
    def create_tree_from_list(self, arch, tab, config_file, block_size, x=0, first=True, y=0):
        childs = []
        for realm in arch:
            actual_childs = []
            actual_x = x
            if realm['sub_realms']:
                tab, x, actual_childs, first = self.create_tree_from_list(realm['sub_realms'], tab, config_file, block_size, x, first, y + 1)
                actual_x = (actual_x + x) / 2
                if actual_x != x:
                    x -= 1
            self.write_leaf(realm, config_file, actual_x * block_size, y * block_size, block_size, actual_childs)
            child = {'x': actual_x * block_size, 'y': y * block_size}
            childs.append(child)
            if first is True:
                first = False
            tab[actual_x][y] = realm['name']
            x += 1
        return tab, x, childs, first
    
    
    # This generate the tree map
    def global_tree_generator(self, arch, map_path, background_path):
        tmp_map_path = '%s.tmp' % map_path
        
        block_size = 225
        logo_width = 287
        logo_height = 100
        
        date = datetime.now()
        background_name = "background_shinken_global%(year)s%(month)s%(day)s%(hour)s%(minute)s.png" % {
            "year"  : date.year,
            "month" : date.month,
            "day"   : date.day,
            "hour"  : date.hour,
            "minute": date.minute
        }
        config_file = open(tmp_map_path, "w")
        config_to_write = '''
define global {
    alias=%(alias)s
    label_show=1
    label_border=%(label_border)s
    grid_steps=%(block_size)d
    iconset=%(iconset)s
    map_image=%(background_name)s
    stylesheet=%(stylesheet)s
}

''' % {
            "alias"          : "[%s] %s" % (self.architecture_name, self.global_map_name),
            "label_border"   : "transparent",
            "block_size"     : block_size,
            "iconset"        : "shinken",
            "background_name": background_name,
            "stylesheet"     : "shinken.css"
        }
        config_file.write(config_to_write.encode('utf-8'))
        to_write = '''
define textbox {
    object_id=%(object_id)s
    text=%(text)s
    x=%(x)d
    y=%(y)d
    z=%(z_index)d
    h=%(height)d
    w=%(width)d
    border_color=%(border_color)s
}

''' % {
            "object_id"   : "shinkenlogocaption%d" % self.get_increment(),
            "text"        : self._translate("do_not_edit_map"),
            "x"           : block_size / 4,
            "y"           : block_size / 4 + logo_height - 10,  # Shift text 100px from the bottom (100px is the logo size)
            "z_index"     : 2,
            "height"      : 20,
            "width"       : logo_width,
            "border_color": "transparent"
        }
        config_file.write(to_write.encode('utf-8'))
        self.write_logo(block_size / 4, block_size / 4, config_file)
        
        # Write architecture name on the right of the logo (margin + logo_size)
        x = block_size / 4 + logo_width + 50
        # Write architecture name at the same level as the logo
        y = block_size / 4 + 40
        self.write_architecture_name(x, y, config_file, "Architecture: %s" % self.architecture_name)
        
        total, stage = self.take_tree_size_of_arch(arch['sub_realms'], 1, 0)
        stage = stage + 2
        total = total * 2 + 1
        block_tab = [[0] * stage for _ in range(total)]
        alone = 0
        if arch['sub_realms']:
            if arch['sub_realms'][0]:
                # If the realm is alone without sub or it is alone with only one sub, we need to start 1 block on the right
                if not arch['sub_realms'][0]['sub_realms'] or len(arch['sub_realms'][0]['sub_realms']) == 1:
                    block_tab, x, childs, first = self.create_tree_from_list(arch['sub_realms'], block_tab, config_file, block_size, x=2, y=1)
                    alone = 1
        if alone == 0:
            block_tab, x, childs, first = self.create_tree_from_list(arch['sub_realms'], block_tab, config_file, block_size, x=0, y=1)
        self.create_background(block_tab, block_size, x + 4, stage, background_path + background_name, True)
        
        config_file.close()
        # Move temporary working file into definitive map file
        shutil.move(tmp_map_path, map_path)
        
        uid = pwd.getpwnam("shinken").pw_uid
        gid = grp.getgrnam("apache").gr_gid
        os.chown(map_path, uid, gid)
        os.chmod(map_path, 0664)
        os.chown(background_path + background_name, uid, gid)
    
    
    def send_host_to_listener(self, host, arch_listener_hosts):
        actual_id = ""
        name = host['shinken_formatted_config']['host_name']
        
        if name in self._listener_hosts_mapping:
            id_from_retention = self._listener_hosts_mapping[name]
            # now check if the ifd from retention is present inside the listener source
            if id_from_retention in arch_listener_hosts and host.get("id_manually_generated", False):
                actual_id = id_from_retention
        
        url_path = self.listener_base_path
        if actual_id:
            url_path = url_path + actual_id
        
        data = json.dumps(host['shinken_formatted_config'])
        try:
            conn = self.get_listener_conn()
            conn.request("PUT", url_path, data, headers=self.get_listener_headers())
            response = conn.getresponse()
            if not actual_id:
                actual_id = response.read()
            else:
                # we need to fully read the response to be able to reuse the connection
                response.read()
        except SSLError as e:
            logger.error("[architecture-export] [%s] Try to request the listener with HTTPS, but it failed. Verify if your listener-shinken is on HTTPS" % self.architecture_name)
            return {'error': e}
        except Exception as e:
            return {'error': e}
        # keep the name -> id mapping update
        self._listener_hosts_mapping[name] = actual_id
        
        if response.status not in (httplib.CREATED, httplib.OK):
            logger.error(
                "[architecture-export] [%s] An error occured when calling the Architecture Listener at %s:%s%s http status %s %s" % (self.architecture_name, self.synchronizer_address, self.listener_port, url_path, response.status, response.read()))
        return arch_listener_hosts
    
    
    def remove_host_shinken_by_listener(self, address, arch_listener_hosts, realm_name):
        name = self.generate_host_name(realm_name, address)
        actual_id = ""
        if name in self._listener_hosts_mapping:
            actual_id = self._listener_hosts_mapping[name]
            if actual_id in arch_listener_hosts:
                arch_listener_hosts.pop(actual_id, None)
        if actual_id:
            url_path = self.listener_base_path + actual_id
            try:
                conn = self.get_listener_conn()
                conn.request("DELETE", url_path, headers=self.get_listener_headers())
                response = conn.getresponse()
                response.read()
            except SSLError as e:
                logger.error("[architecture-export] [%s] Try to request the listener with HTTPS, but it failed. Verify if your listener-shinken is on HTTPS" % self.architecture_name)
                return {'error': e}
            except Exception as e:
                return {'error': e}
            # keep the name -> id mapping update
            self._listener_hosts_mapping.pop(name, None)
            if response.status not in (httplib.CREATED, httplib.OK):
                logger.error("[architecture-export] [%s] An error occured when calling the Architecture Listener at %s:%s%s http status %s %s" % (
                    self.architecture_name, self.synchronizer_address, self.listener_port, url_path, response.status, response.read()))
            
            self.save_listener_hosts_mapping()
            return arch_listener_hosts
    
    
    def add_unique_value(self, _list, _value):
        if _value not in _list:
            _list.append(_value)
    
    
    def generate_hosts_from_shinken_architecture_info(self, arch_all, arch_listener_hosts=None):
        # recursivly go deeper in the realm and send hosts to the listener
        
        if arch_listener_hosts is None:
            arch_listener_hosts = {}
        
        for objects in arch_all:
            if objects['type'] == 'realm':
                if objects['sub_realms']:
                    arch_listener_hosts = self.generate_hosts_from_shinken_architecture_info(objects['sub_realms'], arch_listener_hosts)
                if objects['hosts']:
                    for host in objects['hosts']:
                        host_to_send = self.add_satellites_templates(host)
                        host_to_send = self.add_config_keys_for_listener(host_to_send)
                        host_to_send = self.add_id_from_listener(host_to_send)
                        arch_listener_hosts[host["_id"]] = host_to_send
        return arch_listener_hosts
    
    
    def add_satellites_templates(self, host):
        templates = set()
        duplicate_foreach = {
            'arbiter'                                : [],
            'broker'                                 : [],
            'poller'                                 : [],
            'reactionner'                            : [],
            'receiver'                               : [],
            'scheduler'                              : [],
            'synchronizer'                           : [],
            'module-metrology-writer'                : [],
            'module-visualisation-ui'                : [],
            'module-visualisation-ui-sla'            : [],
            'module-visualisation-ui-event-container': [],
            'module-sla'                             : [],
            'module-event_container'                 : [],
            'module-livedata'                        : [],
        }
        
        if host.get('sats') is not None:
            for sats in host['sats']:
                foreach = '%s$(%d)$' % (sats['name'], sats['port'])
                if sats['modules']:
                    for module in sats['modules']:
                        if module['type'] == 'broker_module_livedata':
                            templates.add('shinken-broker-module-livedata')
                            self.add_unique_value(duplicate_foreach['module-livedata'], foreach)
                        if module['type'] == 'sla':
                            self.add_unique_value(duplicate_foreach['module-sla'], foreach)
                            templates.add('shinken-broker-module-sla-writer')
                        if module['type'] == 'event_container':
                            self.add_unique_value(duplicate_foreach['module-event_container'], foreach)
                            templates.add('shinken-broker-module-event-manager-writer')
                        if module['type'] == 'graphite_perfdata':
                            self.add_unique_value(duplicate_foreach['module-metrology-writer'], foreach)
                            templates.add('shinken-broker-module-metrology-writer')
                        if module['type'] == 'webui':
                            foreach_webui = '%s-%s$(%d)$$(%s)$' % (sats['name'], module['name'], sats['port'], module['name'])
                            self.add_unique_value(duplicate_foreach['module-visualisation-ui'], foreach_webui)
                            templates.add('shinken-broker-module-visualisation-ui')
                            for submodule in module['submodules']:
                                if submodule['type'] == 'sla':
                                    self.add_unique_value(duplicate_foreach['module-visualisation-ui-sla'], foreach_webui)
                                    templates.add('shinken-broker-module-visualisation-ui-sla-reader')
                                if submodule['type'] == 'event_container':
                                    self.add_unique_value(duplicate_foreach['module-visualisation-ui-event-container'], foreach_webui)
                                    templates.add('shinken-broker-module-visualisation-ui-event-manager-reader')
                
                self.add_unique_value(duplicate_foreach[sats['type']], foreach)
                temp = 'shinken-' + sats['type']
                templates.add(temp)
        
        host['templates'] = templates
        host['duplicate_foreach'] = duplicate_foreach
        
        return host
    
    
    def add_config_keys_for_listener(self, host):
        # Add Shinken configuration to prepare hosts for being sent to the listener
        
        if not host.get('is_architecture_host', True):
            name = "%s - %s" % (self.architecture_name, host['address'])
        else:
            name = self.generate_host_name(host['realm'].upper(), host['address'])
        template = list(host['templates'])
        template.sort()
        
        config = {
            'host_name'                    : name,
            'address'                      : host['address'],
            'display_name'                 : self._translate('do_not_edit_host') % self.architecture_name,
            'use[FORCE]'                   : ','.join(template),
            'enabled'                      : '1',
            '_ARBITER_LIST'                : (','.join(host['duplicate_foreach']['arbiter'])) or '',
            '_BROKER_LIST'                 : (','.join(host['duplicate_foreach']['broker'])) or '',
            '_POLLER_LIST'                 : (','.join(host['duplicate_foreach']['poller'])) or '',
            '_REACTIONNER_LIST'            : (','.join(host['duplicate_foreach']['reactionner'])) or '',
            '_RECEIVER_LIST'               : (','.join(host['duplicate_foreach']['receiver'])) or '',
            '_SCHEDULER_LIST'              : (','.join(host['duplicate_foreach']['scheduler'])) or '',
            '_SYNCHRONIZER_LIST'           : (','.join(host['duplicate_foreach']['synchronizer'])) or '',
            '_MODULE_METROLOGY_LIST'       : (','.join(host['duplicate_foreach']['module-metrology-writer'])) or '',
            '_MODULE_SLA_LIST'             : (','.join(host['duplicate_foreach']['module-sla'])) or '',
            '_MODULE_EVENT_MANAGER_LIST'   : (','.join(host['duplicate_foreach']['module-event_container'])) or '',
            '_MODULE_UI_LIST'              : (','.join(host['duplicate_foreach']['module-visualisation-ui'])) or '',
            '_MODULE_UI_SLA_LIST'          : (','.join(host['duplicate_foreach']['module-visualisation-ui-sla'])) or '',
            '_MODULE_UI_EVENT_MANAGER_LIST': (','.join(host['duplicate_foreach']['module-visualisation-ui-event-container'])) or '',
            '_MODULE_LIVEDATA_LIST'        : (','.join(host['duplicate_foreach']['module-livedata'])) or "",
        }
        
        if host.get("custom_data"):
            for data in host["custom_data"]:
                config[data] = host["custom_data"][data]
        
        url_list = []
        # Architecture details external URL
        url_list.append("%(label)s~=%(url)s~=%(icon)s~=%(open_method)s" % {
            "label"      : self._translate("shinken_architecture_link_name"),
            "url"        : self.architecture_broks['arch_map']['url'],
            "icon"       : "blocks",
            "open_method": "POPUP(80)"
        })
        # Realms tree external URL
        url_list.append("%(label)s~=%(url)s~=%(icon)s~=%(open_method)s" % {
            "label"      : self._translate("realms_tree_link_name"),
            "url"        : self.architecture_broks['tree_map']['url'],
            "icon"       : "sitemap",
            "open_method": "POPUP(80)"
        })
        config['notes_multi_url'] = "~#~".join(url_list)
        
        host['shinken_formatted_config'] = config
        
        return host
    
    
    def add_id_from_listener(self, host):
        name = host['shinken_formatted_config']['host_name']
        
        if name in self._listener_hosts_mapping:
            id_from_retention = self._listener_hosts_mapping[name]
            host["_id"] = id_from_retention
            host["id_manually_generated"] = False
        else:
            # No existing ID, generate one
            host["_id"] = 'core-host-' + uuid.uuid1().get_hex()
            host["id_manually_generated"] = True
        
        return host
    
    
    def _get_graphite_configuration_tree(self, node):
        # Perform breadth first tree traversal to get all carbon-cache and carbon-relay daemons
        
        graphite_nodes_list = []
        nodes_to_visit = []
        nodes_to_visit.append(node)
        
        while len(nodes_to_visit) > 0:
            current_node = nodes_to_visit.pop()
            graphite_nodes_list.append(current_node)
            
            status = current_node.connect(quit_on_fail=False, timeout=self.ssh_timeout)
            if status != EXIT_STATUS.OK:
                logger.error(
                    "[architecture-export] The module could not perform SSH connexion on host \"%s\" to determine Graphite configuration. This Graphite node and its children nodes (carbon-relay or carbon-cache nodes) will be missing from the host generated by this module" % current_node.graphite_hostname)
                continue
            
            current_node.read_carbon_conf()
            current_node.close()
            if current_node.is_relay:
                for n in current_node.nodes:
                    nodes_to_visit.append(
                        GraphiteConfReader(graphite_hostname=n[0],
                                           graphite_port=n[1],
                                           ssh_port=self.ssh_port,
                                           user=self.ssh_user,
                                           ssh_key_file=self.ssh_key_file,
                                           passphrase="",
                                           graphite_conf_file="/opt/graphite/conf/carbon.conf")
                    )
        
        return graphite_nodes_list
    
    
    def _scan_host_for_graphite_tpl(self, host, graphite_hosts):
        # Scans host for graphite installations (carbon-cache and carbon-relay) and adds found daemons into graphite_hosts list passed as parameter
        for sats in host['sats']:
            if sats['modules']:
                for module in sats['modules']:
                    if module['type'] == "graphite_perfdata":
                        if module.get("host") is None or module.get("port") is None:
                            logger.error(
                                "[architecture-export] The module could not find Graphite-Perfdata configuration on the host. "
                                "This Graphite node and its children nodes (carbon-relay or carbon-cache nodes) will be missing from the host generated by this module"
                            )
                            continue
                        
                        conf_reader = GraphiteConfReader(
                            graphite_hostname=module.get("host"),
                            graphite_port=module.get("port"),
                            ssh_port=self.ssh_port,
                            user=self.ssh_user,
                            ssh_key_file=self.ssh_key_file,
                            passphrase="",
                            graphite_conf_file="/opt/graphite/conf/carbon.conf",
                        )
                        graphite_tree_configuration = self._get_graphite_configuration_tree(conf_reader)
                        graphite_hosts += graphite_tree_configuration
    
    
    def _group_graphite_hosts_by_address(self, host_list):
        grouped_by_addr = {}
        for host in host_list:
            if grouped_by_addr.get(host.graphite_hostname):
                grouped_by_addr[host.graphite_hostname].append(host)
            else:
                grouped_by_addr[host.graphite_hostname] = [host]
        
        return grouped_by_addr
    
    
    def add_graphite_templates_and_hosts(self, host_list):
        # Read Graphite configuration on each host to add Graphite templates (to monitor carbon-cache and carbon-relay daemons)
        
        graphite_hosts = []
        shinken_graphite_hosts = []
        
        for id in host_list:
            # The host retrieved from listener is not part of the shinken architecture (but may be a graphite/mongo host only)
            if isinstance(host_list[id], basestring):
                continue
            
            self._scan_host_for_graphite_tpl(host_list[id], graphite_hosts)
        
        for cfg in graphite_hosts:
            for host in host_list:
                if isinstance(host_list[host], basestring):
                    continue
                
                if host_list[host]["address"] != cfg.graphite_hostname:
                    continue
                
                for sats in host_list[host]['sats']:
                    foreach = "%s$(%d)$" % (sats['name'], sats['port'])
                    if sats['modules']:
                        for module in sats['modules']:
                            if module['type'] == "graphite_perfdata":
                                shinken_graphite_hosts.append(cfg)
                                self.add_unique_value(host_list[host]["duplicate_foreach"]["module-metrology-writer"], foreach)
                                if "shinken-broker-module-metrology-writer" not in host_list[host]["templates"]:
                                    host_list[host]["templates"].add("shinken-broker-module-metrology-writer")
        
        standalone_graphite_hosts = self._group_graphite_hosts_by_address([h for h in graphite_hosts if h not in shinken_graphite_hosts])
        for addr in standalone_graphite_hosts:
            host_to_send = self.add_satellites_templates({
                "name"                : "graphite_%s" % cfg.graphite_hostname,
                "address"             : addr,
                "is_architecture_host": False,
            })
            host_to_send = self.add_satellites_templates(host_to_send)
            host_to_send["templates"].add("graphite")
            
            for cfg in standalone_graphite_hosts[addr]:
                if host_to_send.get("custom_data"):
                    # If graphite_port is already configured, override if host is a relay
                    if host_to_send["custom_data"].get("_GRAPHITE_PORT"):
                        if cfg.is_relay:
                            host_to_send["custom_data"] = {
                                "_GRAPHITE_PORT": cfg.graphite_port,
                            }
                else:
                    host_to_send["custom_data"] = {
                        "_GRAPHITE_PORT": cfg.graphite_port,
                    }
            
            host_to_send = self.add_config_keys_for_listener(host_to_send)
            host_to_send = self.add_id_from_listener(host_to_send)
            host_list[host_to_send["_id"]] = host_to_send
        
        return host_list
    
    
    def find_hosts_to_disable_on_listener(self, arch_all, arch_listener_hosts=None):
        # recursivly go deeper in the realm and disable hosts to the listener
        
        if arch_listener_hosts is None:
            arch_listener_hosts = {}
        for objects in arch_all:
            if objects['type'] == 'realm':
                realm_name = objects['name'].upper()
                if objects['sub_realms']:
                    arch_listener_hosts = self.find_hosts_to_disable_on_listener(objects['sub_realms'], arch_listener_hosts)
                if objects['hosts']:
                    for host in objects['hosts']:
                        arch_listener_hosts = self.remove_host_shinken_by_listener(host['address'], arch_listener_hosts, realm_name)
        return arch_listener_hosts
    
    
    def get_hosts_in_listener(self):
        conn = self.get_listener_conn()
        try:
            conn.request("GET", self.listener_base_path, headers=self.get_listener_headers())
            response = conn.getresponse()
        except SSLError as e:
            logger.error("[architecture-export] [%s] Try to request the listener with HTTPS, but it failed. Verify if your listener-shinken is on HTTPS" % self.architecture_name)
            return {'error': e}
        except Exception as e:
            return {'error': e}
        if response.status != httplib.OK:
            # we need to fully read the response to be able to reuse the connection
            response.read()
            if response.status == httplib.FORBIDDEN:
                logger.error("[architecture-export] [%s] Bad login and password for the listener" % self.architecture_name)
            return {'error': httplib.HTTPException}
        ret = json.loads(response.read())
        arch_listener_hosts = {}
        for host in ret:
            seuuid = host['_SE_UUID']
            if host['host_name'].startswith(self.architecture_name):
                self._listener_hosts_mapping[host['host_name']] = seuuid
                arch_listener_hosts[seuuid] = host['host_name']
        return arch_listener_hosts
    
    
    def get_synchronizer_address(self, synchronizer):
        if synchronizer is None:
            return
        
        self.synchronizer_address = getattr(synchronizer, 'address', 'localhost')
    
    
    # This will generate one map by realm
    def generate_all_maps(self, arch, map_path, parent_map, rotation_file):
        for realms in arch:
            realm_map_path_name = "%s-%s" % (realms['name'].replace(' ', '_'), self.server_uuid)
            to_rotation = ",%s %s:%s" % (self._translate("realm"), realms['name'], realm_map_path_name)
            rotation_file.write(to_rotation.encode('utf-8'))
            if realms['sub_realms']:
                self.generate_all_maps(realms['sub_realms'], map_path, realm_map_path_name, rotation_file)
            tmp = {'type': 'architecture', 'name': 'tmp', 'length': 3, 'height': 3, 'sub_realms': []}
            tmp['sub_realms'].append(realms)
            map_name = "[%s] %s: %s" % (self.architecture_name, self._translate("realm"), realms['name'])
            
            logger.debug("[architecture-export] [%s] Generating NagVis map for realm %s" % (self.architecture_name, realms['name']))
            self.nagvis_map_generator(tmp, map_path + realm_map_path_name + ".cfg", map_name, parent_map)
    
    
    # We made a realm list
    def take_all_realms_in_list(self, arch, list_of_realms):
        for realms in arch:
            cfg_name = "%s-%s.cfg" % (realms['name'], self.server_uuid)
            list_of_realms.append(cfg_name)
            if realms['sub_realms']:
                list_of_realms = self.take_all_realms_in_list(realms['sub_realms'], list_of_realms)
        return list_of_realms
    
    
    def get_topmost_realm(self, realms_list):
        topmost_realm = realms_list.get_default()
        has_parent = True
        iteration = 0
        max_iterations = 15
        while has_parent and iteration < max_iterations:
            iteration += 1
            for r in realms_list:
                if getattr(r, 'realm_members'):
                    if topmost_realm in r.realm_members:
                        topmost_realm = r
                        break
                    else:
                        has_parent = False
        return topmost_realm
    
    
    def get_architecture_realm(self, conf):
        realms = conf.realms
        map_data = []
        # Create a super frame upper than all realm that contains Arbiters
        architecture_realm = self._get_architecture_realm()
        
        # Default realm is not necessarily the topmost realm. We retrieve a toplevel realm to begin generating architecture description
        map_data.append(self._get_realm_data(self.get_topmost_realm(realms), conf))
        for realm in realms:
            realm_data = self._get_realm_data(realm)
            if realm_data:
                map_data.append(realm_data)
        
        architecture_realm['sub_realms'] = map_data
        return architecture_realm
    
    
    def _disable_hosts_on_listener(self, hosts):
        for _id in hosts.iterkeys():
            # don't reuse the connection and avoid keep alive errors
            conn = httplib.HTTPConnection(self.synchronizer_address, self.listener_port, timeout=3)
            data = json.dumps(
                {
                    "enabled": "0",
                }
            )
            conn.request("PUT", self.listener_base_path + _id, data, headers=self.get_listener_headers())
            response = conn.getresponse()
            # we need to fully read the response to be able to reuse the connection
            response.read()
            logger.info('[architecture-export] [%s] Old host [%s] disabled on synchronizer.' % (self.architecture_name, _id))
    
    
    def make_map(self, thread_id, threads_status, lock, architecture_realm):
        t = threading.Thread(None, target=self.do_make_map, name='do_make_map', args=(thread_id, threads_status, lock, architecture_realm,))
        t.daemon = True
        threads_status[thread_id]['thread'] = t
        t.start()
    
    
    def remove_map(self, architecture_realm):
        t = threading.Thread(None, target=self.do_remove_map, name='do_remove_map', args=(architecture_realm,))
        t.daemon = True
        t.start()
    
    
    def do_make_map(self, thread_id, threads_status, lock, architecture_realm):
        # Now we have the shinken architecture, we check if it changed from last time
        try:
            # Here we look at the "higher broker realm", it will be used for the config
            logger.debug("[architecture-export] [%s] Retrieving hosts correspondig to this architecture from the Synchronizer (listener-shinken)" % (self.architecture_name))
            architecture_hosts_from_listener = self.get_hosts_in_listener()
            if 'error' in architecture_hosts_from_listener.keys():
                raise architecture_hosts_from_listener['error']
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
                logger.error("[architecture-export] [%s] Could not retrieve existing hosts from listener: %s" % (self.architecture_name, e))
            return
        
        try:
            logger.debug("[architecture-export] [%s] Building host list from architecture description received" % (self.architecture_name))
            arch_listener_hosts = self.generate_hosts_from_shinken_architecture_info(architecture_realm['sub_realms'], architecture_hosts_from_listener)
            arch_listener_hosts = self.add_graphite_templates_and_hosts(arch_listener_hosts)
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
                logger.error("[architecture-export] [%s] Could not build host list from architecture description %s" % (self.architecture_name, e))
            return
        
        try:
            logger.debug("[architecture-export] [%s] Sending hosts in Synchronizer (listener-shinken)" % (self.architecture_name))
            for host_id in arch_listener_hosts:
                if isinstance(arch_listener_hosts[host_id], basestring):
                    continue
                
                self.send_host_to_listener(arch_listener_hosts[host_id], arch_listener_hosts)
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
                logger.error("[architecture-export] [%s] Could not send hosts to listener: %s" % (self.architecture_name, e))
            return
        
        try:
            logger.debug("[architecture-export] [%s] Sending hosts in Synchronizer (listener-shinken)" % (self.architecture_name))
            # Hosts retrieved from listener that are still in a string format have not been processed by the architecture generation, which means they are not part of the architecture anymore. Disable them
            hosts_to_disable = dict((host, arch_listener_hosts[host]) for host in arch_listener_hosts if isinstance(arch_listener_hosts[host], basestring))
            if hosts_to_disable:
                self._disable_hosts_on_listener(hosts_to_disable)
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
                logger.error("[architecture-export] [%s] Old hosts were not removed in listener: %s" % (self.architecture_name, e))
            return
        
        try:
            logger.debug("[architecture-export] [%s] Removing obsolete NagVis maps backgrounds" % (self.architecture_name))
            # First we remove all old backgrounds and maps
            filelist = [f for f in os.listdir(self.background_path) if f.startswith("background")]
            for f in filelist:
                if self.server_uuid in f:
                    os.remove(os.path.join(self.background_path, f))
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
                logger.error("[architecture-export] [%s] Cannot delete background files in %s : %s" % (self.architecture_name, self.background_path, e))
            return
        
        try:
            logger.debug("[architecture-export] [%s] Removing obsolete NagVis maps" % (self.architecture_name))
            # should remove file containing the architecture_name
            filelist = [f for f in os.listdir(self.map_path)]
            for f in filelist:
                
                if self.server_uuid in f:
                    os.remove(os.path.join(self.map_path, f))
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
            logger.error("[architecture-export] [%s] Cannot remove file in %s : %s" % (self.architecture_name, self.map_path, e))
            return
        
        # Now we can make maps
        rotation_file_path = self.path + "/etc/conf.d/rotation_shinken_%s.ini.php" % self.server_uuid
        tmp_rotation_file_path = '%s.tmp' % rotation_file_path
        try:
            rotation_file = open(tmp_rotation_file_path, "w")
            rotation_file.write(";<?php return 1; ?>\n;-----------------------------\n; Rotation pool definitions\n;-----------------------------\n")
            rotation_file.write('''[rotation_%(rotation_name)s]
                                   maps="%(global_map_display_name)s:%(global_map_name)s,%(architecture_map_display_name)s:%(architecture_map_name)s''' %
                                {
                                    "rotation_name"                : self.architecture_name.replace(' ', '_'),
                                    "global_map_display_name"      : self.global_map_name,
                                    "global_map_name"              : "shinken_global-%s" % self.server_uuid,
                                    "architecture_map_display_name": self.shinken_architecture_map_name,
                                    "architecture_map_name"        : "shinken_architecture-%s" % self.server_uuid
                                })
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
            logger.error("[architecture-export] [%s] Shinken cannot write in file %s : %s" % (self.architecture_name, rotation_file_path, e))
            return
        
        try:
            architecture_realm['sub_realms'] = self.compute_size_of_arch_objects(architecture_realm['sub_realms'])
            architecture_realm['sub_realms'] = self.sort_all_realms(architecture_realm['sub_realms'], self.map_realm_layout, len(architecture_realm['sub_realms']))
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
            logger.error("[architecture-export] [%s] Cannot load architecture_realm : %s" % (self.architecture_name, e))
            rotation_file.close()
            os.unkink(tmp_rotation_file_path)
            return
        
        try:
            logger.debug("[architecture-export] [%s] Generating realms tree NagVis map" % (self.architecture_name))
            self.global_tree_generator(architecture_realm, self.global_map_file_path, self.background_path)
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
            logger.error("[architecture-export] [%s] Cannot generating realms tree %s" % (self.architecture_name, e))
            rotation_file.close()
            os.unkink(tmp_rotation_file_path)
            return
        
        try:
            logger.debug("[architecture-export] [%s] Generating detailed architecture NagVis map" % (self.architecture_name))
            self.nagvis_map_generator(architecture_realm, self.detailed_map_file_path, "[%s] %s" % (self.architecture_name, self.shinken_architecture_map_name))
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
            logger.error("[architecture-export] [%s] Cannot generating detailed architecture %s" % (self.architecture_name, e))
            rotation_file.close()
            os.unkink(tmp_rotation_file_path)
            return
        
        try:
            shinken_architecture_map_name = os.path.splitext(os.path.split(self.detailed_map_file_path)[1])[0]
            self.generate_all_maps(architecture_realm['sub_realms'], self.map_path, shinken_architecture_map_name, rotation_file)
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
            logger.error("[architecture-export] [%s] Unable to create file %s : %s" % (self.architecture_name, self.detailed_map_file_path, e))
            rotation_file.close()
            os.unkink(tmp_rotation_file_path)
            return
        
        try:
            logger.debug("[architecture-export] [%s] Generating rotation configuration for NagVis maps" % (self.architecture_name))
            rotation_file.write("\"\ninterval=30\n")
            rotation_file.close()
            shutil.move(tmp_rotation_file_path, rotation_file_path)
            
            uid = pwd.getpwnam("shinken").pw_uid
            gid = grp.getgrnam("apache").gr_gid
            os.chown(rotation_file_path, uid, gid)
            os.chmod(rotation_file_path, 0664)
            
            self.save_listener_hosts_mapping()
        except Exception as e:
            with lock:
                threads_status[thread_id]['status'] = "ERROR"
            logger.error("[architecture-export] [%s] Error sending request to the listener-shinken, please verify your configuration %s" % (self.architecture_name, e))
            return
        
        with lock:
            threads_status[thread_id]['status'] = "DONE"
    
    
    def do_remove_map(self, architecture_realm):
        arch_listener_hosts = self.get_hosts_in_listener()
        if 'error' in arch_listener_hosts.keys():
            raise arch_listener_hosts['error']
        self.find_hosts_to_disable_on_listener(architecture_realm['sub_realms'], arch_listener_hosts)
        
        # we remove all old backgrounds and maps
        filelist = [f for f in os.listdir(self.background_path) if f.startswith("background")]
        for f in filelist:
            if self.server_uuid in f:
                os.remove(os.path.join(self.background_path, f))
        
        # we remove file containing the architecture_name
        filelist = [f for f in os.listdir(self.map_path)]
        for f in filelist:
            if self.server_uuid in f:
                os.remove(os.path.join(self.map_path, f))
        # and finally remove the rotation file
        rotation_file_path = self.path + "/etc/conf.d/rotation_shinken_%s.ini.php" % self.server_uuid
        if os.path.exists(rotation_file_path):
            os.remove(rotation_file_path)


MAP_REALM_LAYOUTS = {
    'sort_by_name': ArchitectureExportMapper.by_name_sorter,
    'sort_by_size': ArchitectureExportMapper.by_size_sorter
}
