#!/usr/bin/python

# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import types
import time
from shinken.misc.regenerator import Regenerator
from shinken.util import safe_print, get_obj_full_name
from shinken.log import logger
from livestatus_query_metainfo import HINT_HOST, HINT_HOSTS, HINT_SERVICES_BY_HOST, HINT_SERVICE, HINT_SERVICES_BY_HOSTS, HINT_SERVICES, HINT_HOSTS_BY_GROUP, HINT_SERVICES_BY_GROUP, HINT_SERVICES_BY_HOSTGROUP

ALL_CONTACTS = {}
ALL_HOSTS = {}
ALL_HOSTGROUPS = {}
ALL_SERVICES = {}

# Should we look at case for filter AuthUser and so the hint computation
REMOVE_USER_CASE_SENSITIVE = True


def recompute_specific_contact(cname):
    global ALL_HOSTS
    global ALL_SERVICES
    global ALL_HOSTGROUPS
    
    if ALL_HOSTS:
        recompute_specific_contact_heap(cname)


def recompute_specific_contact_heap(cname):
    global ALL_HOSTS
    global ALL_SERVICES
    global ALL_HOSTGROUPS
    global ALL_CONTACTS
    global REMOVE_USER_CASE_SENSITIVE
    
    hosts = ALL_HOSTS
    services = ALL_SERVICES
    hostgroups = ALL_HOSTGROUPS
    contacts = ALL_CONTACTS
    
    ## Will compute contact visibility for services, but only when the contact is asking for the first time
    
    if contacts == {}:  # not ready, skip it
        return
    # The contact we are looking for, but maybe we should not look at case
    if REMOVE_USER_CASE_SENSITIVE:
        cont = contacts.find_by_name(cname)
    else:  # do not look at case, all in LOWER!
        # Force a lower name for saving, so will save only once for all possible requests
        cname = cname.lower()
        cont = contacts.get_contact_case_insensitive(cname)
    if cont is None:
        logger.error("[livestatus] asking for a computation of user index with an unknown user: %s" % cname)
        return
    logger.debug('[livestatus] computation of host/checks index for the contact: %s' % (cont.get_name()))
    
    # Compute host visibility
    lst = set()
    for (k, host) in hosts.items.iteritems():
        if cont in host.contacts:
            lst.add(k)
    
    hosts._id_contact_heap[cname] = list(lst)
    hosts._id_contact_heap[cname].sort(key=lambda x: get_obj_full_name(hosts.items[x]))
    
    logger.debug('[livestatus] Final computation for user:%s hosts count:%d' % (cname, len(hosts._id_contact_heap[cname])))
    
    # Now Services
    lst = set()
    # strict: one must be an explicitly contact of a service in order to see it.
    if services.service_authorization_strict:
        for (k, v) in services.items.iteritems():
            if cont in v.contacts:
                lst.add(k)
    else:
        # Only look at host level
        for (hid, host) in hosts.items.iteritems():
            # still host then service checks
            if cont in host.contacts:
                for v in host.services:
                    lst.add(v.id)
    services._id_contact_heap[cname] = list(lst)
    # and finally sort it
    services._id_contact_heap[cname].sort(key=lambda x: get_obj_full_name(services.items[x]))
    
    logger.debug('[livestatus] Final computation for user:%s checks view:%d' % (cname, len(services._id_contact_heap[cname])))
    
    hostgroups._id_contact_heap[cname] = set()
    if hostgroups.group_authorization_strict:
        # only host contacts can be hostgroup-contacts at all
        # now, which hosts does the contact know?
        contact_host_ids = set(hosts._id_contact_heap[cname])
        for (k, hg) in hostgroups.items.iteritems():
            # now look if c is contact of all v.members
            # we already know the hosts for which c is a contact
            hostgroup_host_ids = set([h.id for h in hg.members])
            # if all of the hostgroup_host_ids are in contact_host_ids
            # then the hostgroup belongs to the contact
            if hostgroup_host_ids <= contact_host_ids:
                hostgroups._id_contact_heap[cname].add(hg.id)
    else:
        # loose: a contact of a member becomes contact of the whole group
        for (k, hg) in hostgroups.items.iteritems():
            for h in hg.members:
                if cont in h.contacts:
                    hostgroups._id_contact_heap[cname].add(k)
    
    # Sort hostgroup things
    hostgroups._id_contact_heap[cname] = list(hostgroups._id_contact_heap[cname])
    hostgroups._id_contact_heap[cname].sort(key=lambda x: get_obj_full_name(hostgroups.items[x]))


def itersorted(self, hints=None):
    global REMOVE_USER_CASE_SENSITIVE
    preselected_ids = []
    preselection = False
    if hints is not None:
        logger.debug("[Livestatus Regenerator] Hint is %s" % hints["target"])
    if hints is None:
        # return all items
        hints = {}
    elif hints['target'] == HINT_HOST:
        try:
            preselected_ids = [self._id_by_host_name_heap[hints['host_name']]]
            preselection = True
        except Exception, exp:
            # This host is unknown
            pass
    elif hints['target'] == HINT_HOSTS:
        try:
            preselected_ids = [self._id_by_host_name_heap[h] for h in hints['host_name'] if h in self._id_by_host_name_heap]
            preselection = True
        except Exception, exp:
            # This host is unknown
            pass
    elif hints['target'] == HINT_HOSTS_BY_GROUP:
        try:
            preselected_ids = self._id_by_hostgroup_name_heap[hints['hostgroup_name']]
            preselection = True
        except Exception, exp:
            # This service is unknown
            pass
    elif hints['target'] == HINT_SERVICES_BY_HOST:
        try:
            preselected_ids = self._id_by_host_name_heap[hints['host_name']]
            preselection = True
        except Exception, exp:
            # This service is unknown
            pass
    elif hints['target'] == HINT_SERVICE:
        try:
            preselected_ids = [self._id_by_service_name_heap[hints['host_name'] + '/' + hints['service_description']]]
            preselection = True
        except Exception:
            pass
    elif hints['target'] == HINT_SERVICES:
        try:
            preselected_ids = [self._id_by_service_name_heap[host_name + '/' + service_description] for host_name, service_description in hints['host_names_service_descriptions'] if
                               host_name + '/' + service_description in self._id_by_service_name_heap]
            preselection = True
        except Exception, exp:
            logger.error("[Livestatus Regenerator] Hint_services exception: %s" % exp)
            pass
    elif hints['target'] == HINT_SERVICES_BY_HOSTS:
        try:
            preselected_ids = [id for h in hints['host_name'] if h in self._id_by_host_name_heap for id in self._id_by_host_name_heap[h]]
            preselection = True
        except Exception:
            pass
    elif hints['target'] == HINT_SERVICES_BY_GROUP:
        try:
            preselected_ids = self._id_by_servicegroup_name_heap[hints['servicegroup_name']]
            preselection = True
        except Exception, exp:
            # This service is unknown
            pass
    elif hints['target'] == HINT_SERVICES_BY_HOSTGROUP:
        try:
            preselected_ids = self._id_by_hostgroup_name_heap[hints['hostgroup_name']]
            preselection = True
        except Exception, exp:
            # This service is unknown
            pass
    if 'authuser' in hints:
        _authuser = hints['authuser']
        # if we are not case sensitive, switch to lower more, as we did/will compute will all lower
        if not REMOVE_USER_CASE_SENSITIVE:
            _authuser = _authuser.lower()
        logger.debug('[livestatus] Hint Targets? %s' % hints)
        logger.debug('[livestatus] Hint Preselection? %s' % preselection)
        logger.debug('[livestatus] Hint preselecteds ids? %s' % len(preselected_ids))
        
        __hints = getattr(self, '_id_contact_heap', {}).get(_authuser, None)
        try:
            logger.debug('[livestatus] Hint restrict? %s' % len(__hints))
        except:
            pass
        can_lazy_heap = getattr(self, 'can_lazy_heap', False)
        logger.debug('[livestatus] Can lazy heap: %s' % (can_lazy_heap))
        if __hints is None and can_lazy_heap:
            logger.debug('[livestatus] Asking for a lasy compute of the service=>contact hints for the contact %s' % _authuser)
            t0 = time.time()
            recompute_specific_contact(_authuser)
            logger.debug('[livestatus] Lasy compute in %.2f' % (time.time() - t0))
        if preselection:
            try:
                for _id in [pid for pid in preselected_ids if pid in self._id_contact_heap[_authuser]]:
                    yield self.items[_id]
            except Exception:
                # this hints['authuser'] was not in self._id_contact_heap
                # we do nothing, so the caller gets an empty list
                pass
        else:
            try:
                for _id in self._id_contact_heap[_authuser]:
                    yield self.items[_id]
            except Exception:
                pass
    else:
        if preselection:
            for _id in preselected_ids:
                yield self.items[_id]
        else:
            for _id in self._id_heap:
                yield self.items[_id]


class LiveStatusRegenerator(Regenerator):
    def __init__(self, module_conf, service_authorization_strict=False, group_authorization_strict=True, use_hints=True, remote_user_case_sensitive=True):
        global REMOVE_USER_CASE_SENSITIVE
        super(self.__class__, self).__init__(module_conf)
        self.service_authorization_strict = service_authorization_strict
        self.group_authorization_strict = group_authorization_strict
        self.use_hints = use_hints
        self.remote_user_case_sensitive = remote_user_case_sensitive
        REMOVE_USER_CASE_SENSITIVE = remote_user_case_sensitive
    
    
    def all_done_linking(self, part_configuration_incarnation):
        global ALL_HOSTS
        global ALL_SERVICES
        global ALL_HOSTGROUPS
        global ALL_CONTACTS
        """In addition to the original all_done_linking our items will get sorted"""
        
        inst_id = part_configuration_incarnation.get_part_id()
        __start = time.time()
        logger.info('[regenerator] Relinking configuration: %d' % inst_id)
        # We will relink all objects if need. If we are in a scheduler, this function will just bailout because it's not need :)
        super(self.__class__, self).all_done_linking(part_configuration_incarnation)
        
        # save contacts
        ALL_CONTACTS = self.contacts
        
        if not self.use_hints:
            logger.debug('Not using hints, skipping the creation')
            return
        # now sort the item collections by name
        
        t0 = time.time()
        # First install a new attribute _id_heap, which holds the
        # item ids in sorted order
        setattr(self.services, '_id_heap', self.services.items.keys())
        self.services._id_heap.sort(key=lambda x: get_obj_full_name(self.services.items[x]))
        setattr(self.hosts, '_id_heap', self.hosts.items.keys())
        self.hosts._id_heap.sort(key=lambda x: get_obj_full_name(self.hosts.items[x]))
        setattr(self.contacts, '_id_heap', self.contacts.items.keys())
        self.contacts._id_heap.sort(key=lambda x: get_obj_full_name(self.contacts.items[x]))
        setattr(self.servicegroups, '_id_heap', self.servicegroups.items.keys())
        self.servicegroups._id_heap.sort(key=lambda x: get_obj_full_name(self.servicegroups.items[x]))
        setattr(self.hostgroups, '_id_heap', self.hostgroups.items.keys())
        self.hostgroups._id_heap.sort(key=lambda x: get_obj_full_name(self.hostgroups.items[x]))
        setattr(self.contactgroups, '_id_heap', self.contactgroups.items.keys())
        self.contactgroups._id_heap.sort(key=lambda x: get_obj_full_name(self.contactgroups.items[x]))
        setattr(self.commands, '_id_heap', self.commands.items.keys())
        self.commands._id_heap.sort(key=lambda x: get_obj_full_name(self.commands.items[x]))
        setattr(self.timeperiods, '_id_heap', self.timeperiods.items.keys())
        self.timeperiods._id_heap.sort(key=lambda x: get_obj_full_name(self.timeperiods.items[x]))
        # Then install a method for accessing the lists' elements in sorted order
        setattr(self.services, '__itersorted__', types.MethodType(itersorted, self.services))
        setattr(self.hosts, '__itersorted__', types.MethodType(itersorted, self.hosts))
        setattr(self.contacts, '__itersorted__', types.MethodType(itersorted, self.contacts))
        setattr(self.servicegroups, '__itersorted__', types.MethodType(itersorted, self.servicegroups))
        setattr(self.hostgroups, '__itersorted__', types.MethodType(itersorted, self.hostgroups))
        setattr(self.contactgroups, '__itersorted__', types.MethodType(itersorted, self.contactgroups))
        setattr(self.commands, '__itersorted__', types.MethodType(itersorted, self.commands))
        setattr(self.timeperiods, '__itersorted__', types.MethodType(itersorted, self.timeperiods))
        
        # Speedup authUser requests by populating _id_contact_heap with contact-names as key and
        # an array with the associated host and service ids
        setattr(self.hosts, '_id_contact_heap', dict())
        setattr(self.services, '_id_contact_heap', dict())
        setattr(self.hostgroups, '_id_contact_heap', dict())
        setattr(self.servicegroups, '_id_contact_heap', dict())
        
        # Also set the self.service_authorization_strict in service heap as they will need it when compute contact
        # visibility
        self.services.service_authorization_strict = self.service_authorization_strict
        self.hostgroups.group_authorization_strict = self.group_authorization_strict
        # setattr(self.services, '__recompute_specific_contact_heap', types.MethodType(recompute_specific_contact_heap, self.services))
        # setattr(self.services, '__can_lazy_heap', True)
        self.services.can_lazy_heap = True
        self.hosts.can_lazy_heap = True
        self.hostgroups.can_lazy_heap = True
        
        # Keep a link for theses elements
        ALL_HOSTS = self.hosts
        ALL_CONTACTS = self.contacts
        ALL_HOSTGROUPS = self.hostgroups
        ALL_SERVICES = self.services
        
        t1 = time.time()
        t2 = time.time()
        t3 = time.time()
        
        logger.debug('Service Heaps:')
        
        # services without contacts inherit the host's contacts (no matter of strict or loose)
        t4 = time.time()
        t5 = time.time()
        t6 = time.time()
        t7 = time.time()
        # Add another helper structure which allows direct lookup by name
        # For hosts: _id_by_host_name_heap = {'name1':id1, 'name2': id2,...}
        # For services: _id_by_host_name_heap = {'name1':[id1, id2,...], 'name2': [id6, id7,...],...} = hostname maps to list of service_ids
        # For services: _id_by_service_name_heap = {'name1':id1, 'name2': id6,...} = full_service_description maps to service_id
        setattr(self.hosts, '_id_by_host_name_heap', dict([(get_obj_full_name(v), k) for (k, v) in self.hosts.items.iteritems()]))
        setattr(self.services, '_id_by_service_name_heap', dict([(get_obj_full_name(v), k) for (k, v) in self.services.items.iteritems()]))
        setattr(self.services, '_id_by_host_name_heap', dict())
        [self.services._id_by_host_name_heap.setdefault(get_obj_full_name(v.host), []).append(k) for (k, v) in self.services.items.iteritems()]
        
        for hn in self.services._id_by_host_name_heap.keys():
            self.services._id_by_host_name_heap[hn].sort(key=lambda x: get_obj_full_name(self.services[x]))
        
        t8 = time.time()
        
        # Add another helper structure which allows direct lookup by name
        # For hostgroups: _id_by_hostgroup_name_heap = {'name1':id1, 'name2': id2,...}
        # For servicegroups: _id_by_servicegroup_name_heap = {'name1':id1, 'name2': id2,...}
        setattr(self.hostgroups, '_id_by_hostgroup_name_heap', dict([(get_obj_full_name(v), k) for (k, v) in self.hostgroups.items.iteritems()]))
        setattr(self.servicegroups, '_id_by_servicegroup_name_heap', dict([(get_obj_full_name(v), k) for (k, v) in self.servicegroups.items.iteritems()]))
        
        t9 = time.time()
        
        # For hosts: key is a hostgroup_name, value is an array with all host_ids of the hosts in this group
        setattr(self.hosts, '_id_by_hostgroup_name_heap', dict())
        [self.hosts._id_by_hostgroup_name_heap.setdefault(get_obj_full_name(hg), []).append(k) for (k, v) in self.hosts.items.iteritems() for hg in v.hostgroups]
        # For services: key is a servicegroup_name, value is an array with all service_ids of the services in this group
        setattr(self.services, '_id_by_servicegroup_name_heap', dict())
        [self.services._id_by_servicegroup_name_heap.setdefault(get_obj_full_name(sg), []).append(k) for (k, v) in self.services.items.iteritems() for sg in v.servicegroups]
        # For services: key is a hostgroup_name, value is an array with all service_ids of the hosts in this group
        setattr(self.services, '_id_by_hostgroup_name_heap', dict())
        [self.services._id_by_hostgroup_name_heap.setdefault(get_obj_full_name(hg), []).append(k) for (k, v) in self.services.items.iteritems() for hg in v.host.hostgroups]
        t10 = time.time()
        
        # print self.services._id_by_host_name_heap
        for hn in self.services._id_by_host_name_heap.keys():
            self.services._id_by_host_name_heap[hn].sort(key=lambda x: get_obj_full_name(self.services[x]))
        t11 = time.time()
        
        # Everything is new now. We should clean the cache
        self.cache.wipeout()
        
        t12 = time.time()
        
        tfinal = t12
        for i in range(13):
            j = i + 1
            ti = locals()['t%d' % i]
            try:
                tj = locals()['t%d' % j]
            except KeyError:
                continue
            logger.debug('[regenerator] REGENERATOR TIME: T%d->T%d: %.2f' % (i, j, (tj - ti)))
        logger.debug('[regenerator] OVERALLTIME: %.2f' % (tfinal - t0))
        logger.info('[regenerator] Relinking configuration: %d is done in %.2fseconds for hosts:%d and services:%d' % (inst_id, (time.time() - __start), len([s for s in self.hosts]), len([s for s in self.services])))
    
    
    def register_cache(self, cache):
        self.cache = cache
    
    
    def before_after_hook(self, brok, obj):
        super(LiveStatusRegenerator, self).before_after_hook(brok, obj)
        self.cache.impact_assessment(brok, obj)
