#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinken.misc.type_hint import Tuple
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.date_helper import date_now, get_now, Date, from_to_date_generator
from sla_abstract_component import AbstractComponent
from sla_archive import SLAArchive
from sla_common import shared_data
from sla_component_manager import ComponentManager
from sla_compute_percent_sla import ComputePercentSla
from sla_database import SLADatabase
from sla_info import SLAInfo
from sla_migration import SLAMigration


class SLAReader(AbstractComponent):
    
    def __init__(self, conf, component_manager, compute_percent_sla, sla_archive, sla_migration, sla_database, sla_info):
        # type: (ShinkenModuleDefinition, ComponentManager,  ComputePercentSla, SLAArchive, SLAMigration, SLADatabase, SLAInfo) -> None
        super(SLAReader, self).__init__(conf, component_manager)
        self.compute_percent_sla = compute_percent_sla
        self.sla_archive = sla_archive
        self.sla_migration = sla_migration
        self.sla_database = sla_database
        self.sla_info = sla_info
        
        self.archive_sla_read = (0, 0)  # type: Tuple[int, float]
        self.raw_sla_read = (0, 0)  # type: Tuple[int, float]
    
    
    def init(self):
        pass
    
    
    def tick(self):
        pass
    
    
    def read_archive(self, item_uuid, date=None, date_ranges=None, limit_ranges=None, where=None, lookup=None, only_existing_day=False):
        if date and date_ranges:
            raise ValueError('you can\'t specify a date and a date_ranges')
        
        start_time = time.time()
        archives = self._find_archive(item_uuid, date=date, date_ranges=date_ranges, limit_ranges=limit_ranges, where=where, lookup=lookup)
        finished_time = time.time() - start_time
        
        if date:
            if archives:
                archive = archives
                self.archive_sla_read = (self.archive_sla_read[0] + 1, self.archive_sla_read[1] + finished_time)
            else:
                archive = self._build_not_in_db_archive(item_uuid, date, limit_ranges, only_existing_day=only_existing_day)
            return archive
        elif date_ranges:
            if len(archives) > 0:
                self.archive_sla_read = (self.archive_sla_read[0] + len(archives), self.archive_sla_read[1] + finished_time)
            in_archive_dates = set([Date(e['yday'], e['year']) for e in archives])
            for date in from_to_date_generator(date_ranges[0], date_ranges[1]):
                if date not in in_archive_dates:
                    archive = self._build_not_in_db_archive(item_uuid, date, limit_ranges, only_existing_day=only_existing_day)
                    if archive:
                        archives.append(archive)
            archives.sort(SLAReader._sort_history)
            
            return archives
    
    
    def _build_not_in_db_archive(self, item_uuid, date, limit_ranges, only_existing_day=False):
        current_date = date_now()
        current_time = get_now()
        
        entry = None
        if current_date == date:
            start_time = time.time()
            entry = self.sla_archive.build_archive(date, item_uuid, end_of_day=current_time)
            finished_time = time.time() - start_time
            self.raw_sla_read = (self.raw_sla_read[0] + 1, self.raw_sla_read[1] + finished_time)
        
        if not entry and self.sla_database.collection_exist(date):
            start_time = time.time()
            entry = self.sla_archive.build_archive(date, item_uuid)
            finished_time = time.time() - start_time
            self.raw_sla_read = (self.raw_sla_read[0] + 1, self.raw_sla_read[1] + finished_time)
        
        if not entry:
            entry = self.sla_archive.build_archive_for_missing_day(date, item_uuid, only_existing_day=only_existing_day)
        
        if entry and limit_ranges:
            current_range_display = limit_ranges[0]
            max_range_display = limit_ranges[1]
            if current_range_display or max_range_display:
                entry['ranges'] = entry['ranges'][current_range_display:current_range_display + max_range_display]
        
        return entry
    
    
    def _find_archive(self, item_uuid, date=None, date_ranges=None, limit_ranges=None, where=None, lookup=None):
        if limit_ranges:
            current_range_display = limit_ranges[0]
            max_range_display = limit_ranges[1]
            lookup = {'ranges': {'$slice': [current_range_display, max_range_display]}}
        
        full_archive = not bool(lookup)
        if date:
            archive = self.sla_database.find_archive(item_uuid, date, where=where, lookup=lookup)
            self._update_read_archive(archive, full_archive=full_archive)
            return archive
        else:
            archives = self.sla_database.find_archives(item_uuid, date_ranges, where=where, lookup=lookup)
            self._update_read_archives(archives, full_archive=full_archive)
            return archives
    
    
    def _update_one_archive(self, archive, full_archive=True):
        if not archive:
            return False
        
        if not (self.sla_migration.must_migrate(archive) or self.compute_percent_sla.must_update_archive(archive)):
            return False
        
        to_save = archive
        if not full_archive:
            to_save = self.sla_database.find_archive_by_id(archive['_id'])
            self.sla_migration.migrate_archive(archive)
            self.compute_percent_sla.update_sla(archive)
        
        self.sla_migration.migrate_archive(to_save)
        self.compute_percent_sla.update_sla(to_save)
        return to_save
    
    
    def _update_read_archives(self, archives, full_archive=True):
        to_update_archive = []
        for archive in archives:
            to_save = self._update_one_archive(archive, full_archive=full_archive)
            if to_save:
                to_update_archive.append(to_save)
        
        if to_update_archive:
            self.sla_database.save_archives(to_update_archive, update=True)
    
    
    def _update_read_archive(self, archive, full_archive=True):
        to_save = self._update_one_archive(archive, full_archive=full_archive)
        if to_save:
            self.sla_database.save_archive(to_save)
        return bool(to_save)
    
    
    def _build_where(self, where, item_uuid, date, date_ranges):
        if where:
            _where = where.copy()
        else:
            _where = {}
        
        if item_uuid:
            _where['uuid'] = item_uuid
        
        if date:
            _where['year'] = date.year
            _where['yday'] = date.yday
        
        if date_ranges:
            start_date = date_ranges[0]
            end_date = date_ranges[1]
            
            if start_date and not end_date:
                _where['$or'] = [{'$and': [{'year': start_date.year}, {'yday': {'$gte': start_date.yday}}]}, {'year': {'$gt': start_date.year}}]
            
            if not start_date and end_date:
                _where['$or'] = [{'$and': [{'year': end_date.year}, {'yday': {'$lte': end_date.yday}}]}, {'year': {'$lt': end_date.year}}]
            
            if start_date and end_date:
                start_date_bound = {'$or': [{'$and': [{'year': start_date.year}, {'yday': {'$gte': start_date.yday}}]}, {'year': {'$gt': start_date.year}}]}
                end_date_bound = {'$or': [{'$and': [{'year': end_date.year}, {'yday': {'$lte': end_date.yday}}]}, {'year': {'$lt': end_date.year}}]}
                _where['$and'] = [start_date_bound, end_date_bound]
        
        _where_with_name = None
        if not shared_data.get_migration_archive_done():
            _where_with_name = _where.copy()
            del _where_with_name['uuid']
            host_name, service_description = self.sla_info.get_name(item_uuid)
            if host_name:
                _where_with_name['hname'] = host_name
                if service_description:
                    _where_with_name['sdesc'] = service_description
                    _where_with_name['type'] = 'service'
                else:
                    _where_with_name['type'] = 'host'
        
        return _where, _where_with_name
    
    
    # An history entry got a year and a yday.
    @staticmethod
    def _sort_history(e1, e2):
        y1 = e1['year']
        y2 = e2['year']
        d1 = e1['yday']
        d2 = e2['yday']
        if y1 != y2:
            return -1 * cmp(y1, y2)
        return -1 * cmp(d1, d2)
