#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import os
import time
import uuid
from datetime import datetime, timedelta

from shinken.basesubprocess import BaseSubProcess, EventHandler
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.log import LoggerFactory, PART_INITIALISATION

from component.sla_common import LOG_PART
from component.sla_common import shared_data
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration, read_float_in_configuration
from component.sla_component_manager import ComponentManager
from component.sla_database import SLADatabase
from component.sla_database_connection import SLADatabaseConnection
from component.sla_error_handler import error_handler, ERROR_LEVEL
from component.sla_info import SLAInfo
from component.sla_migration import SLAMigration
from shinkensolutions.date_helper import get_now, Date

MIGRATING_BATCH_SIZE = 1000
MIGRATING_PAUSE_TIME = 0.1
DAILY_CLEAN_BATCH_SIZE = 10000
DAILY_CLEAN_PAUSE_TIME = 2
INFINITE_NB_STORED_DAYS = -1
LOOP_SPEED = 10


class HandleInitialBroksDone(EventHandler):
    migrator = None  # type: Migrator
    
    
    def __init__(self, migrator):
        super(HandleInitialBroksDone, self).__init__(event_name='handle-initial-broks', error_handler=error_handler, time_wait_for_started=10)
        self.migrator = migrator
        self.logger = migrator.logger
    
    
    def callback(self):
        self.migrator.configuration_uuid = uuid.uuid4().hex
        shared_data.set_migration_archive_done(False)
        self.logger.info('New configuration receive. Ask for a recheck of archive version.')


class Migrator(BaseSubProcess):
    component_manager = None  # type: ComponentManager
    sla_info = None  # type: SLAInfo
    sla_database_connection = None  # type: SLADatabaseConnection
    sla_database = None  # type: SLADatabase
    sla_migration = None  # type: SLAMigration
    
    
    def __init__(self, daemon_name, module_name, conf):
        # type: (str, str, ShinkenModuleDefinition) -> None
        super(Migrator, self).__init__('migration', father_name=daemon_name, loop_speed=LOOP_SPEED, only_one_process_by_class=True, error_handler=error_handler)
        self.conf = conf
        self.module_name = module_name
        self.logger = LoggerFactory.get_logger(conf.get_name())
        
        self.nb_stored_days = 0
        self.configuration_uuid = None
        self.start_archive_migration_time = None
        self.nb_archive_migrate = 0
        
        self._handle_initial_broks_done = HandleInitialBroksDone(self)
        
        self.logger.info(PART_INITIALISATION, 'Creating new migrator process name : %s' % self.get_process_name())
    
    
    def get_process_name(self):
        return '%s [ - Module: %s - %s]' % (self.father_name, self.module_name, self.name)
    
    
    def get_logger_name(self):
        return [self.father_name.replace('shinken-', '').strip(), self.module_name, '%s (pid:%s)' % (self.name, os.getpid())]
    
    
    def on_init(self):
        global DAILY_CLEAN_BATCH_SIZE, DAILY_CLEAN_PAUSE_TIME, MIGRATING_BATCH_SIZE, MIGRATING_PAUSE_TIME
        self.component_manager = ComponentManager(self.logger)
        self.sla_database_connection = SLADatabaseConnection(self.conf, self.component_manager)
        self.sla_info = SLAInfo(self.conf, self.component_manager, self.sla_database_connection)
        self.sla_database = SLADatabase(self.conf, self.component_manager, self.sla_database_connection, self.sla_info)
        self.sla_migration = SLAMigration(self.conf, self.component_manager, self.sla_info)
        
        DAILY_CLEAN_BATCH_SIZE = read_int_in_configuration(self.conf, 'daily_clean_batch_size', DAILY_CLEAN_BATCH_SIZE)
        DAILY_CLEAN_PAUSE_TIME = read_float_in_configuration(self.conf, 'daily_clean_pause_time', DAILY_CLEAN_PAUSE_TIME)
        MIGRATING_BATCH_SIZE = read_int_in_configuration(self.conf, 'broker_module_sla_migration_batch_size', MIGRATING_BATCH_SIZE)
        MIGRATING_PAUSE_TIME = read_float_in_configuration(self.conf, 'broker_module_sla_migration_pause_time', MIGRATING_PAUSE_TIME)
        
        self.nb_stored_days = read_int_in_configuration(self.conf, 'nb_stored_days', INFINITE_NB_STORED_DAYS)
        if self.nb_stored_days != INFINITE_NB_STORED_DAYS and self.nb_stored_days < 7:
            self.logger.warning(LOG_PART.INITIALISATION, 'Parameter nb_stored_days set by user at [%s] is too low we set it a minimal value:7' % self.nb_stored_days)
            self.nb_stored_days = 7
        
        self.logger.info(LOG_PART.INITIALISATION, 'Parameter load for migrating')
        self.logger.info(LOG_PART.INITIALISATION, '   - daily_clean_batch_size ----------------- :[%s]' % DAILY_CLEAN_BATCH_SIZE)
        self.logger.info(LOG_PART.INITIALISATION, '   - daily_clean_pause_time ----------------- :[%s]' % DAILY_CLEAN_PAUSE_TIME)
        self.logger.info(LOG_PART.INITIALISATION, '   - broker_module_sla_migration_batch_size - :[%s]' % MIGRATING_BATCH_SIZE)
        self.logger.info(LOG_PART.INITIALISATION, '   - broker_module_sla_migration_pause_time - :[%s]' % MIGRATING_PAUSE_TIME)
        self.logger.info(LOG_PART.INITIALISATION, '   - nb_stored_days ------------------------- :[%s]' % 'INFINITE' if self.nb_stored_days == INFINITE_NB_STORED_DAYS else self.nb_stored_days)
        
        self.configuration_uuid = None
        self.start_archive_migration_time = None
        self.nb_archive_migrate = 0
        
        self.component_manager.init()
        self._handle_initial_broks_done.start_thread()
    
    
    def on_close(self):
        pass
    
    
    def loop_turn(self):
        self.component_manager.tick()
        if self.nb_stored_days != INFINITE_NB_STORED_DAYS:
            self._do_daily_clean()
        
        self.sla_database.check_daily_format()
        self._migrate_acknowledge()
        self._migrate_archives()
    
    
    def handle_initial_broks_done(self):
        self._handle_initial_broks_done.send_event()
    
    
    def _do_daily_clean(self):
        try:
            t0 = time.time()
            last_stored_timetuple = (datetime.today() + timedelta(days=-self.nb_stored_days)).timetuple()
            last_stored_date = Date(last_stored_timetuple.tm_yday, last_stored_timetuple.tm_year)
            _where = {'$or': [{'$and': [{'year': last_stored_date.year}, {'yday': {'$lt': last_stored_date.yday}}]}, {'year': {'$lt': last_stored_date.year}}]}
            
            nb_items_to_del = self.sla_database.count_archives_before_date(last_stored_date)
            must_clean = bool(nb_items_to_del)
            if must_clean:
                self._stats_save_start_daily_clean(nb_items_to_del)
                self.logger.info(LOG_PART.DAILY_CLEAN, 'Archive anterior to %s has been found ( %s records )' % (self.logger.format_sla_date(last_stored_date), nb_items_to_del))
            
            batch_id = 0
            while nb_items_to_del > 0:
                group_ids = self.sla_database.find_archives_before_date(last_stored_date, DAILY_CLEAN_BATCH_SIZE)
                batch = [i['_id'] for i in group_ids if i is not None]
                self.sla_database.remove_archives({'_id': {'$in': batch}})
                batch_id += 1
                time.sleep(DAILY_CLEAN_PAUSE_TIME)
                nb_items_to_del -= len(group_ids)
                shared_data.set_nb_sla_left_to_remove(nb_items_to_del)
            
            if must_clean:
                time_taken = time.time() - t0
                self._stats_save_finished_daily_clean(time_taken)
                self.logger.info(LOG_PART.DAILY_CLEAN, 'Archive anterior to %s has been remove ( %s records ) in %s' % (self.logger.format_sla_date(last_stored_date), nb_items_to_del, self.logger.format_duration(time_taken)))
        except Exception as e:
            error_handler.handle_exception('Removing %s days old SLA archive fail with error : %s. This will be retry in %ssec.' % (self.nb_stored_days, LOOP_SPEED, e), e, self.logger, ERROR_LEVEL.WARNING)
    
    
    def _migrate_acknowledge(self):
        time_start = time.time()
        to_migrate_acknowledge = self.sla_database.find_acknowledge_not_migrate()
        
        new_data = []
        for to_migrate in to_migrate_acknowledge:
            item_uuid = self.sla_info.get_uuid(to_migrate['hname'], to_migrate.get('sdesc', ''))
            if item_uuid:
                to_migrate['item_uuid'] = item_uuid
                del to_migrate['type']
                del to_migrate['hname']
                to_migrate.pop('sdesc', None)
                new_data.append(to_migrate)
        
        if new_data:
            self.logger.info('Must migrate %s acknowledge entry' % len(new_data))
            self.sla_database.update_acknowledges(new_data)
            self.logger.info('Migrate %s acknowledge entry done in %s' % (len(new_data), self.logger.format_chrono(time_start)))
    
    
    def _migrate_archives(self):
        if self.configuration_uuid is None or shared_data.get_migration_archive_done():
            return False
        
        self.start_archive_migration_time = time.time()
        shared_data.set_migration_in_progress(True)
        self.nb_archive_migrate = 0
        
        nb_to_migrate_archive = self.sla_database.count_archives_to_migrate(self.configuration_uuid)
        to_migrate_archives = self.sla_database.find_archives_to_migrate(self.configuration_uuid, MIGRATING_BATCH_SIZE)
        
        if nb_to_migrate_archive != 0:
            self._stats_set_total_sla_to_migrate(nb_to_migrate_archive)
        
        self.logger.info('Need to migrate %s archives' % nb_to_migrate_archive)
        
        while to_migrate_archives:
            for archive in to_migrate_archives:
                self.nb_archive_migrate += 1
                shared_data.set_nb_sla_left_to_migrate(nb_to_migrate_archive - self.nb_archive_migrate)
                self.sla_migration.migrate_archive(archive)
                archive['configuration_uuid'] = self.configuration_uuid
            
            self.sla_database.save_archives(to_migrate_archives, update=True)
            self.logger.info('Migrating in progress %s/%s archives' % (self.nb_archive_migrate, nb_to_migrate_archive))
            to_migrate_archives = self.sla_database.find_archives_to_migrate(self.configuration_uuid, MIGRATING_BATCH_SIZE)
            self.interruptable_sleep(0.1)
        
        time_taken = time.time() - self.start_archive_migration_time
        self._stats_save_finished_migration(time_taken)
        if self.nb_archive_migrate:
            self.logger.info('Migrate %s archive done in:%s.' % (self.nb_archive_migrate, self.logger.format_duration(time_taken)))
        else:
            self.logger.info('No archive need migration. Check done in:%s.' % self.logger.format_duration(time_taken))
        
        self.start_archive_migration_time = None
        self.nb_archive_migrate = 0
    
    
    @staticmethod
    def _stats_save_start_daily_clean(nb_items_to_del):
        shared_data.set_daily_clean_in_progress(True)
        shared_data.set_total_sla_to_remove(nb_items_to_del)
        shared_data.set_nb_sla_left_to_remove(nb_items_to_del)
    
    
    @staticmethod
    def _stats_save_finished_daily_clean(time_taken):
        shared_data.set_daily_clean_in_progress(False)
        shared_data.set_execution_time_last_daily_clean(time_taken)
    
    
    @staticmethod
    def _stats_save_finished_migration(time_taken):
        shared_data.set_migration_archive_done(True)
        shared_data.set_migration_in_progress(False)
        shared_data.set_execution_time_last_migration(time_taken)
    
    
    @staticmethod
    def _stats_set_total_sla_to_migrate(value):
        shared_data.set_total_sla_to_migrate(value)
        shared_data.set_nb_sla_left_to_migrate(value)
