#!/usr/bin/env python
# -*- coding: utf-8 -*-

#
# "THE BEER-WARE LICENSE" (Revision 43~maze)
#
# <maze@pyth0n.org> wrote these files. As long as you retain this notice you
# can do whatever you want with this stuff. If we meet some day, and you think
# this stuff is worth it, you can buy me a beer in return.

from __future__ import print_function

import atexit
import math
import os
import random
import re
import sys
import time

# Hook stdout to print in UTF8 every time
import codecs

stdout_utf8 = codecs.getwriter("utf-8")(sys.stdout)
sys.stdout = stdout_utf8


class Characters(object):
    # Box drawing
    # NOTE: if you have more heavy chars, I'm interested, because all I did found is ┏ but the vertical sign is not continue (space in putty at least)
    vbar = u'│'
    hbar = u'─'
    hbar_light = u'─'
    corner_top_left = u'┌'
    corner_top_right = u'┐'
    corner_bottom_left = u'└'
    corner_bottom_right = u'┘'
    
    # Others
    hbar_dotted = u'᠁'
    vbar_dotted = u'⁞'
    
    # Ok or not
    check = u'√'
    cross = u'Х'
    double_exclamation = u'‼'
    
    # arrows
    arrow_left = u'→'
    arrow_double = u'↔'
    arrow_bottom = u'↓'
    arrow_top = u'↑'
    
    # Small numbers
    small_slash = u'̷'
    small_open = u'₍'
    small_0 = u'₀'
    small_1 = u'₁'
    small_2 = u'₂'
    small_3 = u'₃'
    small_4 = u'₄'
    small_5 = u'₅'
    small_6 = u'₆'
    small_7 = u'₇'
    small_8 = u'₈'
    small_9 = u'₉'
    small_close = u'₎'
    
    # Topic display prefix
    topic_display_prefix = u'¦'
    topic_small_picto = u'█'
    
    # Gun
    higer_gun = u'߹'
    middle_gun = u'█'
    lower_gun = u'߸'
    
    # Spinners
    spinners = u"⣷⣯⣟⡿⢿⣻⣽⣾"


CHARACTERS = Characters()

PY3 = sys.version_info >= (3,)

STRIP_ANSI = re.compile(r'\x1b\[(\d+)(;\d+)?(;\d+)?[m|K]')
COLOR_ANSI = (
    (0x00, 0x00, 0x00), (0xcd, 0x00, 0x00),
    (0x00, 0xcd, 0x00), (0xcd, 0xcd, 0x00),
    (0x00, 0x00, 0xee), (0xcd, 0x00, 0xcd),
    (0x00, 0xcd, 0xcd), (0xe5, 0xe5, 0xe5),
    (0x7f, 0x7f, 0x7f), (0xff, 0x00, 0x00),
    (0x00, 0xff, 0x00), (0xff, 0xff, 0x00),
    (0x5c, 0x5c, 0xff), (0xff, 0x00, 0xff),
    (0x00, 0xff, 0xff), (0xff, 0xff, 0xff),
)


class LolCat(object):
    def __init__(self, mode=256, output=sys.stdout):
        self.mode = mode
        self.output = output
    
    
    def _distance(self, rgb1, rgb2):
        return sum(map(lambda c: (c[0] - c[1]) ** 2,
                       zip(rgb1, rgb2)))
    
    
    def ansi(self, rgb):
        r, g, b = rgb
        
        if self.mode in (8, 16):
            colors = COLOR_ANSI[:self.mode]
            matches = [(self._distance(c, map(int, rgb)), i) for i, c in enumerate(colors)]
            matches.sort()
            color = matches[0][1]
            
            return '3%d' % (color,)
        else:
            gray_possible = True
            sep = 2.5
            
            while gray_possible:
                if r < sep or g < sep or b < sep:
                    gray = r < sep and g < sep and b < sep
                    gray_possible = False
                
                sep += 42.5
            
            if gray:
                color = 232 + int(float(sum(rgb) / 33.0))
            else:
                color = sum([16] + [int(6 * float(val) / 256) * mod
                                    for val, mod in zip(rgb, [36, 6, 1])])
            
            return '38;5;%d' % (color,)
    
    
    def wrap(self, *codes):
        return '\x1b[%sm' % (''.join(codes),)
    
    
    def rainbow(self, freq, i):
        r = math.sin(freq * i) * 127 + 128
        g = math.sin(freq * i + 2 * math.pi / 3) * 127 + 128
        b = math.sin(freq * i + 4 * math.pi / 3) * 127 + 128
        return [r, g, b]
    
    
    def cat(self, fd, options):
        if options.animate:
            self.output.write('\x1b[?25l')
        
        for line in fd:
            options.os += 1
            self.println(line, options)
        
        if options.animate:
            self.output.write('\x1b[?25h')
    
    
    def get_line(self, s, offset, spread=3.0):
        if spread is None:
            spread = 99999.0
        
        # if options.force or self.output.isatty():
        s = STRIP_ANSI.sub('', s)
        r = ''
        if isinstance(s, str):
            s = s.decode('utf-8', 'replace')
        for i, c in enumerate(s):
            rgb = self.rainbow(0.1, offset + i / spread)
            if isinstance(c, str):
                c = c.encode('utf-8', 'replace')
            r += u''.join([self.wrap(self.ansi(rgb)), c if PY3 else c, ])
        r += '\x1b[0m'
        return r
    
    
    def println(self, s, options):
        s = s.rstrip()
        if options.force or self.output.isatty():
            s = STRIP_ANSI.sub('', s)
        
        if options.animate:
            self.println_ani(s, options)
        else:
            self.println_plain(s, options)
        
        self.output.write('\n')
        self.output.flush()
    
    
    def println_ani(self, s, options):
        if not s:
            return
        
        for i in range(1, options.duration):
            self.output.write('\x1b[%dD' % (len(s),))
            self.output.flush()
            options.os += options.spread
            self.println_plain(s, options)
            time.sleep(1.0 / options.speed)
    
    
    def println_plain(self, s, options):
        for i, c in enumerate(s if PY3 else s.decode(options.charset_py2, 'replace')):
            rgb = self.rainbow(options.freq, options.os + i / 3.0)
            self.output.write(''.join([
                self.wrap(self.ansi(rgb)),
                c if PY3 else c.encode(options.charset_py2, 'replace'),
            ]))


def detect_mode(term_hint='xterm-256color'):
    '''
    Poor-mans color mode detection.
    '''
    if 'ANSICON' in os.environ:
        return 16
    elif os.environ.get('ConEmuANSI', 'OFF') == 'ON':
        return 256
    else:
        term = os.environ.get('TERM', term_hint)
        if term.endswith('-256color') or term in ('xterm', 'screen'):
            return 256
        elif term.endswith('-color') or term in ('rxvt',):
            return 16
        else:
            return 256  # optimistic default


def run():
    """Main entry point."""
    import optparse
    
    # Reset terminal colors at exit
    def reset():
        sys.stdout.write('\x1b[0m')
        sys.stdout.flush()
    
    
    atexit.register(reset)
    
    parser = optparse.OptionParser(usage=r'%prog [<options>] [file ...]')
    parser.add_option('-p', '--spread', type='float', default=3.0,
                      help='Rainbow spread')
    parser.add_option('-F', '--freq', type='float', default=0.1,
                      help='Rainbow frequency')
    parser.add_option('-S', '--seed', type='int', default=0,
                      help='Rainbow seed')
    parser.add_option('-a', '--animate', action='store_true', default=False,
                      help='Enable psychedelics')
    parser.add_option('-d', '--duration', type='int', default=12,
                      help='Animation duration')
    parser.add_option('-s', '--speed', type='float', default=20.0,
                      help='Animation speed')
    parser.add_option('-f', '--force', action='store_true', default=False,
                      help='Force colour even when stdout is not a tty')
    
    parser.add_option('-3', action='store_const', dest='mode', const=8,
                      help='Force 3 bit colour mode')
    parser.add_option('-4', action='store_const', dest='mode', const=16,
                      help='Force 4 bit colour mode')
    parser.add_option('-8', action='store_const', dest='mode', const=256,
                      help='Force 8 bit colour mode')
    
    parser.add_option('-c', '--charset-py2', default='utf-8',
                      help='Manually set a charset to convert from, for python 2.7')
    
    options, args = parser.parse_args()
    options.os = random.randint(0, 256) if options.seed == 0 else options.seed
    options.mode = options.mode or detect_mode()
    
    lolcat = LolCat(mode=options.mode)
    
    if not args:
        args = ['-']
    
    for filename in args:
        if filename == '-':
            lolcat.cat(sys.stdin, options)
        else:
            try:
                with open(filename, 'r') as handle:
                    lolcat.cat(handle, options)
            except IOError as error:
                sys.stderr.write(str(error) + '\n')


lolcat = LolCat(mode=detect_mode())

# coding: utf-8
# Copyright (c) 2008-2011 Volvox Development Team
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# Author: Konstantin Lepa <konstantin.lepa@gmail.com>

"""ANSII Color formatting for output in terminal."""

import os
import codecs
import sys

# We want to print as UTF8
# TODO: what about windows? will it works even on old XP?
stdout_utf8 = codecs.getwriter("utf-8")(sys.stdout)

__ALL__ = ['colored', 'cprint']

VERSION = (1, 1, 0)

ATTRIBUTES = dict(
    list(zip([
        'bold',
        'dark',
        '',
        'underline',
        'blink',
        '',
        'reverse',
        'concealed'
    ],
        list(range(1, 9))
    ))
)
del ATTRIBUTES['']

HIGHLIGHTS = dict(
    list(zip([
        'on_grey',
        'on_red',
        'on_green',
        'on_yellow',
        'on_blue',
        'on_magenta',
        'on_cyan',
        'on_white'
    ],
        list(range(40, 48))
    ))
)

COLORS = dict(
    list(zip([
        'grey',
        'red',
        'green',
        'yellow',
        'blue',
        'magenta',
        'cyan',
        'white',
    ],
        list(range(90, 98))
    ))
)

RESET = '\033[0m'


def colored(text, color=None, on_color=None, attrs=None):
    """Colorize text.

    Available text colors:
        red, green, yellow, blue, magenta, cyan, white.

    Available text highlights:
        on_red, on_green, on_yellow, on_blue, on_magenta, on_cyan, on_white.

    Available attributes:
        bold, dark, underline, blink, reverse, concealed.

    Example:
        colored('Hello, World!', 'red', 'on_grey', ['blue', 'blink'])
        colored('Hello, World!', 'green')
    """
    if os.getenv('ANSI_COLORS_DISABLED') is None:
        fmt_str = '\033[%dm%s'
        if color is not None:
            text = fmt_str % (COLORS[color], text)
        
        if on_color is not None:
            text = fmt_str % (HIGHLIGHTS[on_color], text)
        
        if attrs is not None:
            for attr in attrs:
                text = fmt_str % (ATTRIBUTES[attr], text)
        # Shinken mod
        if color is not None:
            text += RESET
    return text


def cprint(text, color=None, on_color=None, attrs=None, **kwargs):
    """Print colorize text.

    It accepts arguments of print function.
    """
    # Force writing to a utf8 encoded stdout
    print((colored(text, color, on_color, attrs)), **kwargs)


def sprintf(text, color=None, on_color=None, attrs=None, **kwargs):
    return colored(text, color, on_color, attrs)


def is_tty():
    # Look if we are in a tty or not
    if hasattr(sys.stdout, 'isatty'):
        return sys.stdout.isatty()
    return False


if not is_tty():
    # Protect sys.stdout write for utf8 outputs
    import codecs
    
    stdout_utf8 = codecs.getwriter("utf-8")(sys.stdout)
    
    
    def cprint(s, color='', end='\n'):
        if not isinstance(s, basestring):
            s = str(s)
        if end == '':
            stdout_utf8.write(s)
        else:
            stdout_utf8.write(s)
            stdout_utf8.write('\n')
    
    
    def sprintf(s, color='', end=''):
        return s


def get_unicode_string(s):
    if isinstance(s, str):
        return unicode(s, 'utf8', errors='ignore')
    return str(s)


# raw_title means do not format it, use it's own color
def print_h1(title, raw_title=False, only_first_part=False, line_color='cyan', title_color='yellow'):
    p1 = 12
    # 4.5= enter add 4, exit 5, so if enter == exit, will be 4.5 in avg.
    l_title = len(title) - int(title.count(u'\x1b[') * 4.5)
    p2 = l_title + 2  # +2 for spaces around the title
    p3 = 80 - p1 - p2
    
    cprint(CHARACTERS.hbar * p1, color=line_color, end='')
    if not raw_title:
        cprint(' %s ' % title, color=title_color, end='')
    else:
        cprint(' ' + title + ' ', end='')
    if not only_first_part:
        cprint(CHARACTERS.hbar * p3, color=line_color)
    else:
        cprint('')


if __name__ == '__main__':
    mode = sys.argv[1]
    try:
        text = sys.argv[2]
    except:
        text = ''
    
    if mode == 'header':
        print_h1(text)
        sys.exit(0)
    
    if mode == 'red-header':
        print_h1(text, title_color='red')
        sys.exit(0)
    
    elif mode == "std-line":
        cprint(' %s %s' % (CHARACTERS.hbar, text))
        sys.exit(0)
    
    elif mode == "std-line-no-end":
        cprint(' %s %-60s' % (CHARACTERS.hbar, text), end='')
        sys.stdout.flush()
        sys.exit(0)
    
    elif mode == "ok-result-line":
        cprint(u'   %s%s ' % (CHARACTERS.corner_bottom_left, CHARACTERS.arrow_left), end='')
        cprint(u' %s OK' % CHARACTERS.check, color='green')
        sys.stdout.flush()
        sys.exit(0)
    
    elif mode == 'note-line-grey':
        lines = text.splitlines()
        cprint('\n'.join(['   | %s' % s for s in lines]), color='grey')
        sys.exit(0)
    
    elif mode == 'OK':
        cprint(u' %s OK' % CHARACTERS.check, color='green')
    
    elif mode == 'WARNING':
        cprint(u' %s WARNING' % CHARACTERS.double_exclamation, color='yellow')

    elif mode == "ERROR":
        cprint(u' %s ERROR' % CHARACTERS.double_exclamation, color='red')
    
    elif mode == 'magenta-no-end':
        cprint(text, color='magenta', end='')

    else:
        cprint("UNKNOWN MODE %s" % mode)
