#!/usr/bin/python

# This script is designed to take a cfg file in entry, and output a json file with an object & key/values matched from the source file
# * only keys that are selected will be migrated
# * migrated keys will be set in a json object in the distant file
# * migrated values will be commented in the original file
# * a comment can be set about migrated lines




import json
import shutil
import optparse
import os
import sys
import string
import re



if not os.getuid() == 0:
    print "ERROR: this script must be run as root"
    sys.exit(2)


def cut_line(line):
    tmp = re.split("[" + string.whitespace + "]+", line, 1)
    r = [elt for elt in tmp if elt != '']
    return r


if __name__ == '__main__':
    parser = optparse.OptionParser("%prog -i INTPUT_FILE -o OUTPUT_FILE -k KEYS -p NEW_PARAMETER", version="%prog: ", description='This tool is used for migrate data from a cfg file into a json file\nWARNING: this can migrate only ONE object.')
    parser.add_option('-i', '--input', dest='input_file',  help="Input file to migrate (as .cfg file)")
    parser.add_option('-o', '--ouput', dest='output_file', help="Output file (as json data)")
    parser.add_option('-k', '--keys', dest='migrate_keys', help="Comma separated keys to migrate.")
    parser.add_option('-p', '--parameter', dest='file_parameter', help="Parameter to set in the input_file to call the new output json file.")
    
    
    opts, args = parser.parse_args()
    
    # Look if the user ask for local or global, and if not, guess
    input_file = opts.input_file
    output_file = opts.output_file
    migrate_keys = opts.migrate_keys
    file_parameter = opts.file_parameter
    
    if not input_file or not output_file or not migrate_keys or not file_parameter:
        parser.print_help()
        sys.exit(1)

    input_file = os.path.abspath(input_file)
    output_file = os.path.abspath(output_file)

    keys = [k.strip() for k in migrate_keys.split(',') if k.strip()]
    
    if not os.path.exists(input_file):
        print "Error: Missing input file: %s" % input_file
        sys.exit(2)
    try:
        f = open(input_file, 'r')
        input_buf = f.read()
        f.close()
    except Exception, exp:
        print "Error: cannot open input file: %s" % exp
        sys.exit(2)
        
    input_lines = input_buf.splitlines()
    nb_lines = len(input_lines)
    
    lines_to_migrate = []
    for (line_nb, l) in enumerate(input_lines):
        l_striped = l.strip()  # be sure to strip space/tab before keys
        migrate = False
        for k in keys:
            if l_striped.startswith(k):
                migrate = True
        if migrate:
            lines_to_migrate.append( (line_nb, l) )
    
    # We will parse the lines to migrate, and key the key/values
    output_data = {}
    for (_, l) in lines_to_migrate:
        elts = cut_line(l.strip())  # cut but the striped version
        output_data[elts[0]] = elts[1]
    
    output_buf = json.dumps(output_data, indent=4)
    
    try:
        f = open(output_file, 'w')
        f.write(output_buf)
        f.close()
    except Exception, exp:
        print "Error: cannot write output file: %s" % exp
        sys.exit(2)
    
    # Now clean/comment lines from the input_file
    # Note: to keep the valid lines number, we do it in **reverse** mode
    lines_to_migrate.reverse()
    new_input_lines = input_lines[:]  # copy the original lines to be sure to always ave it, like if we need to
                                      # display a diff in the future, etc
    for (line_nb, line) in lines_to_migrate:
        new_line = '#%s' % line
        new_input_lines[line_nb] = new_line
    
    first_migrated_line = None
    if len(output_data) >= 1:
        first_migrated_line = lines_to_migrate[-1][0]

    param_line = '    %s     %s\n' % (file_parameter, output_file)
    if first_migrated_line is not None:
        comment_line = '%s##################################################################\n#    NOTE: the parameters [%s] have been migrated into the file "%s". Please update the parameter in this file from now.\n##################################################################' % (param_line, ','.join(keys), output_file)
        new_input_lines.insert(first_migrated_line, comment_line)
    
    # Write input_temporary file
    tmp_file = '%s.tmp' % input_file
    f = open(tmp_file, 'w')
    f.write('\n'.join(new_input_lines))
    f.write('\n')  # put a ending line to be sure there is one
    f.close()
    # move it finally, so we can't corrupt it
    shutil.move(tmp_file, input_file)
    