#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import optparse
import cPickle
import time
import os
import sys
import shutil
import glob

try:
    import pwd
    import grp
except ImportError:
    pwd, grp = None, None

from shinkensolutions.localinstall import get_local_daemons, get_instance_name, get_local_instances_for_type
from shinkensolutions.system_tools import run_command
# The patch_lib is added on build_patch, do not fix theses error
from lib.lib_patch.patch_new_hash_list import PatchNewHashList, PatchNewHashListException
from lib.lib_patch.misc import create_tree, copy_recursively, mkdir_p, COLOR

PATCH_DAT = {}
with open('patch.dat', 'rb') as f:
    PATCH_DAT = cPickle.loads(f.read())

DAEMON_ORDER = {
    'scheduler'   : 0,
    'poller'      : 1,
    'reactionner' : 2,
    'broker'      : 3,
    'receiver'    : 4,
    'synchronizer': 5,
    'arbiter'     : 6,
}

PYTHON_MINOR_VERSION = sys.version_info[1]  # cannot use .minor because python2.6 do not have it
SHINKEN_LIB_PATH = '/usr/lib/python2.%s/site-packages/shinken' % PYTHON_MINOR_VERSION
MODULES_DIR = '/var/lib/shinken/modules'
WEBUI_PATH = '/var/lib/shinken/modules/webui/htdocs/ui'

FILES_DESTS = {
    'shinken_lib'             : {'path': '/usr/lib/python2.%s/site-packages/shinken' % PYTHON_MINOR_VERSION},
    'shinken_solutions_lib'   : {'path': '/usr/lib/python2.%s/site-packages/shinkensolutions' % PYTHON_MINOR_VERSION},
    'modules'                 : {'path': '/var/lib/shinken/modules'},
    'graphite'                : {'path': '/opt/graphite'},
    'shinken_libexec'         : {'path': '/var/lib/shinken/libexec/', 'user': 'shinken', 'rights': 0755},
    'shinken_user_lib'        : {'path': '/var/lib/shinken-user/', 'user': 'shinken'},
    'etc_shinken'             : {'path': '/etc/shinken', 'user': 'shinken'},
    'etc_shinken_user'        : {'path': '/etc/shinken-user', 'user': 'shinken'},
    'etc_shinken_user_example': {'path': '/etc/shinken-user-example', 'user': 'shinken', 'rights': 0555},
    'root_etc'                : {'path': '/etc'},
    'webui'                   : {'path': WEBUI_PATH},
    'shinken_pack'            : {'path': '/etc/shinken/packs', 'user': 'shinken'},
    'init.d'                  : {'path': '/etc/init.d', 'rights': 0755},
    'usr_sbin'                : {'path': '/usr/sbin', 'rights': 0755},
    'site-packages'           : {'path': '/usr/lib/python2.%s/site-packages/' % PYTHON_MINOR_VERSION},
}


#####################################################################
### FILE DISPATCHER
#####################################################################
def _get_files_and_destinations(root_dir=None):
    if root_dir is None:
        my_directory = os.path.abspath(os.path.dirname(__file__))
        root_dir = os.path.join(my_directory, 'files')
    
    to_dispatch_files = []
    for in_patch_dir, destination_entry in FILES_DESTS.iteritems():
        if in_patch_dir == 'webui':
            continue
        if in_patch_dir == 'shinken_pack':
            continue
        
        destination_dir = destination_entry['path']
        destination_user = destination_entry.get('user', None)
        destination_rights = destination_entry.get('rights', None)
        for root, dir_names, file_names in os.walk(os.path.join(root_dir, in_patch_dir)):
            for filename in file_names:
                if filename == '._for_git' or filename.endswith('~'):
                    continue
                
                local_file_path = os.path.join(root, filename)
                
                from_root_path = local_file_path[len(os.path.join(root_dir, in_patch_dir)) + 1:]
                #                destination_file = os.path.join(destination_dir, in_patch_dir, filename)
                
                destination_file = os.path.join(destination_dir, from_root_path)
                inside_files_path = local_file_path[len(root_dir) + 1:]
                entry = {
                    'local_file_path'  : local_file_path,  # '/root/XXX...XXX-PATCH01/files/modules/simple-log/module.py'
                    'destination_file' : destination_file,  # '/var/lib/shinken/modules/simple-log/module.py'
                    'inside_files_path': inside_files_path,  # 'modules/simple-log/module.py'
                    'from_files_path'  : in_patch_dir,  # 'modules'
                    'user'             : destination_user,  # shinken or None
                    'rights'           : destination_rights,  # int 0755 or None
                }
                to_dispatch_files.append(entry)
    return to_dispatch_files


def manage_file_patchnew(hash_list, file_path, local_file_path):
    managed_path = os.path.join(os.path.sep, 'etc', 'shinken')  # include also /etc/shinken-user
    exclude_paths = (
        os.path.join(managed_path, 'pack', 'shinken'),
        os.path.join(managed_path, 'pack', 'shinken-core'),
        os.path.join(managed_path, '_default'),
        os.path.join(managed_path, 'external'),
        os.path.join(os.path.sep, 'etc', 'shinken-user-example'),  # The user-example must be overwrite but not the shinken-user
        os.path.join(os.path.sep, 'etc', 'shinken-skeletons'),  # The skeletons must be overwrite
    )
    
    if not file_path.startswith(managed_path):
        return file_path
    else:
        for exclude_path in exclude_paths:
            if file_path.startswith(exclude_path):
                return file_path
    
    hash_list = hash_list.get_hash_list_for_file(local_file_path)
    
    if not hash_list or not os.path.exists(file_path):
        return file_path
    
    with open(file_path, 'r') as _file:
        user_file_hash = PatchNewHashList.get_hash_for_file(_file.read())
    
    if user_file_hash in hash_list:
        return os.path.join(file_path)
    
    return '%s%s' % (file_path, '.patchnew')


def backup_before_patch(backup_dir):
    _print_info(' - Backup the files before applying the patch', color=COLOR.YELLOW)
    to_dispatch_files = _get_files_and_destinations()
    for entry in to_dispatch_files:
        destination_file = entry['destination_file']
        if not os.path.exists(destination_file):
            _print_info("   * No need to backup the file %s as it does not exists" % (destination_file))
            continue
        inside_files_path = entry['inside_files_path']
        backup_path = os.path.join(backup_dir, inside_files_path)
        # If the backup is already existing, it's an error, we should NOT lost it
        if os.path.exists(backup_path):
            do_exit_error('The backup file %s is already existing, cannot erase it' % backup_path)
        _print_info("   * Backup %s to %s" % (destination_file, backup_path))
        backup_path_directory = os.path.dirname(backup_path)
        try:
            mkdir_p(backup_path_directory)
            shutil.copy2(destination_file, backup_path)
        except Exception as exp:
            do_exit_error('Cannot save the file %s to %s: %s' % (destination_file, backup_path, exp))
    # maybe we will have no file to backup, so force one file to be set
    pth = os.path.join(backup_dir, 'installed')
    with open(pth, 'w') as f:
        f.write('Installed at: %s' % int(time.time()))
    
    _patch_directory = os.path.abspath(os.path.dirname(__file__))
    # For the webui, we backup all the folder because, all the folder will be replaced
    if os.path.exists(os.path.join(_patch_directory, 'files', 'webui')):
        _print_info("   * Backup webui")
        copy_webui(WEBUI_PATH, os.path.join(backup_dir, 'webui'), delete_before=False)
    
    # For the shinken pack, we backup all the folder
    if os.path.exists(os.path.join(_patch_directory, 'files', 'shinken_pack')):
        _print_info("   * Backup shinken packs")
        copy_recursively('/etc/shinken/packs/shinken', os.path.join(backup_dir, 'shinken_pack', 'shinken'))
        copy_recursively('/etc/shinken/packs/shinken-core', os.path.join(backup_dir, 'shinken_pack', 'shinken-core'))
    do_exit()


def restore_after_patch(patch_id):
    _print_info(' - Restoring the files for the patch %s' % patch_id, color=COLOR.YELLOW)
    backup_dir = get_backup_dir(patch_id)
    if not os.path.exists(os.path.join(backup_dir, 'installed')):
        _print_info('The patch %s is not installed.' % patch_id)
        do_exit()
    
    to_dispatch_files = _get_files_and_destinations(backup_dir)
    for entry in to_dispatch_files:
        destination_file = entry['destination_file']
        user = entry['user']
        rights = entry['rights']
        inside_files_path = entry['inside_files_path']
        backup_path = os.path.join(backup_dir, inside_files_path)
        # If the backup is already existing, it's an error, we should NOT lost it
        if not os.path.exists(backup_path):
            do_exit_error('Cannot find the backup file %s, cannot restore it' % backup_path)
        _print_info("   * Restoring %s to %s" % (backup_path, destination_file))
        try:
            mkdir_p(os.path.dirname(destination_file))
            shutil.move(backup_path, destination_file)
            if user is not None:
                uid = pwd.getpwnam(user).pw_uid
                gid = grp.getgrnam(user).gr_gid
                os.chown(destination_file, uid, gid)
            if rights is not None:
                os.chmod(destination_file, rights)
        except Exception as exp:
            do_exit_error('Cannot restore the file %s to %s: %s' % (backup_path, destination_file, exp))
    web_ui_dir = os.path.join(backup_dir, 'webui')
    if os.path.exists(web_ui_dir):
        _print_info("   * Restoring webui")
        copy_webui(web_ui_dir, WEBUI_PATH)
    # At the end, should clean what we put in the backup dir from the files (do not erase all as maybe the patch set something else into it)
    for files_dest_root in FILES_DESTS.keys():
        files_dest_backup_dir = os.path.join(backup_dir, files_dest_root)
        if os.path.exists(files_dest_backup_dir):
            _print_info('   * Cleaning backup dir %s' % files_dest_backup_dir)
            try:
                shutil.rmtree(files_dest_backup_dir)
            except Exception as exp:
                do_exit_error('Cannot clean backup directory %s: %s' % (files_dest_backup_dir, exp))
    # Also remove the installed file, as it it no more need
    pth = os.path.join(backup_dir, 'installed')
    
    if os.path.exists(pth):
        os.unlink(pth)
    do_exit('Patch %s was removed' % patch_id)


def apply_patch_files():
    _print_info(' - Applying patch files', color=COLOR.YELLOW)
    try:
        _hash_list = PatchNewHashList('patch_new_hash_list.dat')
    except PatchNewHashListException as err:
        do_exit_error('Cannot load patch_new_hash_list.dat. Exiting')
    to_dispatch_files = _get_files_and_destinations()
    for entry in to_dispatch_files:
        inside_files_path = entry['inside_files_path']
        local_file_path = entry['local_file_path']
        destination_file = manage_file_patchnew(_hash_list, entry['destination_file'], entry['inside_files_path'])
        user = entry['user']
        rights = entry['rights']
        _print_info("   * Copying %s to %s" % (inside_files_path, destination_file))
        dest_dir = os.path.dirname(destination_file)
        try:
            mkdir_p(dest_dir)
            shutil.copy2(local_file_path, destination_file)
            if user is not None:
                uid = pwd.getpwnam(user).pw_uid
                gid = grp.getgrnam(user).gr_gid
                os.chown(destination_file, uid, gid)
                os.chown(dest_dir, uid, gid)
                _print_info("     - setting as owner %s/%s" % (user, user))
            if rights is not None:
                os.chmod(destination_file, rights)
                _print_info("     - setting rights as %o" % (rights))
        except Exception as exp:
            do_exit_error('Cannot save the file %s to %s: %s' % (local_file_path, destination_file, exp))
    _patch_directory = os.path.abspath(os.path.dirname(__file__))
    if os.path.exists(os.path.join(_patch_directory, 'files', 'webui')):
        _print_info("   * Copying webui")
        copy_webui(os.path.join(_patch_directory, 'files', 'webui'), WEBUI_PATH)
    
    if os.path.exists(os.path.join(_patch_directory, 'files', 'shinken_pack')):
        _print_info("   * Copying Shinken packs")
        try:
            shutil.rmtree('/etc/shinken/packs/shinken')
            shutil.rmtree('/etc/shinken/packs/shinken-core')
        except OSError:
            # If the pack doesn't exists don't worry, it's already delete
            pass
        copy_recursively(os.path.join(_patch_directory, 'files', 'shinken_pack'), '/etc/shinken/packs')
    
    do_exit()


#####################################################################
### Miscellaneous
#####################################################################
def do_exit_error(msg):
    _print_error(msg)
    exit(2)


def do_exit(msg=None, color=COLOR.BLUE):
    if msg:
        print '\n%s%s%s\n' % (color, msg, COLOR.RESET)
    exit(0)


def _print_info(msg, color=COLOR.BLUE):
    print '%s%s%s' % (color, msg, COLOR.RESET)


def _print_error(msg):
    print '\n%sError : %s%s\n' % (COLOR.RED, msg, COLOR.RESET)


def get_backup_dir(patch_id):
    backup_dir = "/var/lib/shinken/backup/shinken-patch-%s" % patch_id
    return backup_dir


def copy_file(_source, _dest):
    file_path = os.path.dirname(_dest)
    create_tree(file_path)
    try:
        shutil.copyfile(_source, _dest)
    except IOError as err:
        do_exit_error(err.message)


#####################################################################
### GET DAEMONS INFOS
#####################################################################
def _get_daemons_instances_by_type():
    daemons = {}
    for daemon_type in get_local_daemons():
        for daemon_id, daemon_enabled in get_local_instances_for_type(daemon_type):
            daemons[daemon_type] = {
                'id'     : daemon_id,
                'name'   : get_instance_name(daemon_type, daemon_id),
                'enabled': daemon_enabled
            }
    return daemons


def _get_enabled_instances(daemon_type):
    instances = []
    for daemon_id, daemon_enabled in get_local_instances_for_type(daemon_type):
        if daemon_enabled:
            instances.append((daemon_type, daemon_id, get_instance_name(daemon_type, daemon_id)))
    return instances


#####################################################################
### MANAGE DAEMONS START/STOP
#####################################################################
def request_start_all_daemons(force):
    daemons_needed = PATCH_DAT.get('RESTART_DAEMONS', [])
    if daemons_needed == 'X' or not daemons_needed:
        do_exit('No need to restart any daemon')
    
    daemons_needed = sorted([dm.strip() for dm in daemons_needed if dm.strip()], key=lambda d: DAEMON_ORDER.get(d, 99))
    daemons_to_start = [daemon_type for daemon_type in daemons_needed if _get_enabled_instances(daemon_type)]
    
    if not daemons_to_start:
        do_exit('No need to restart any daemon')
    
    if force:
        _start_all_daemons(daemons_to_start)
    
    ask_msg = 'This patch need to start such daemons : %s.\nDo you want to start all theses daemons now ?' % ', '.join(daemons_to_start)
    user_choice = _ask_user(ask_msg)
    
    if user_choice in ('o', 'y'):
        _start_all_daemons(daemons_to_start)
    elif user_choice == 'c':
        for daemon_type in daemons_to_start:
            instances = _get_enabled_instances(daemon_type)
            selected_instances = _ask_user_to_select_daemon(instances)
            
            for (_daemon_type, daemon_id, _) in selected_instances:
                _start_daemon_instance(_daemon_type, daemon_id)
    else:
        do_exit_error('No daemon has been started. In order for the patch to be effective, you must restart the following daemons : %s' % ', '.join(daemons_to_start))


def request_restart_services(service, force):
    if not service:
        do_exit('No need to restart any service')
    
    if force:
        _restart_service(service)
        do_exit()
    
    ask_msg = 'This patch need to restart the %s service.\nDo you want to restart this service now ?' % service
    user_choice = _ask_user(ask_msg)
    
    if user_choice in ('o', 'y'):
        _restart_service(service)
    else:
        do_exit('The service %s was not restarted.')


def request_stop_all_daemons(force):
    daemons_needed = PATCH_DAT.get('RESTART_DAEMONS', [])
    if daemons_needed == 'X' or not daemons_needed:
        do_exit('No need to restart any daemon')
    
    daemons_needed = sorted(daemons_needed, key=lambda d: DAEMON_ORDER.get(d, 99))
    daemons_to_stop = [daemon_type for daemon_type in daemons_needed if _get_enabled_instances(daemon_type)]
    
    if not daemons_to_stop:
        do_exit('No need to restart any daemon')
    
    if force:
        _stop_all_daemons(daemons_to_stop)
    
    ask_msg = 'This patch need to stop such daemons : %s.\nDo you want to stop all theses daemons now ?' % ', '.join(daemons_to_stop)
    user_choice = _ask_user(ask_msg)
    
    if user_choice in ('o', 'y'):
        _stop_all_daemons(daemons_to_stop)
    elif user_choice == 'c':
        for daemon_type in daemons_to_stop:
            instances = _get_enabled_instances(daemon_type)
            selected_instances = _ask_user_to_select_daemon(instances)
            
            for (daemon_type, daemon_id, _) in selected_instances:
                _stop_daemon_instance(daemon_type, daemon_id)
    else:
        do_exit_error('No daemon was stopped. You must stop the daemon to apply the patch')


def _start_all_daemons(daemon_lists):
    for daemon_type in daemon_lists:
        _start_daemon_type(daemon_type)
    do_exit()


def _start_daemon_type(daemon_type):
    print run_command('service shinken-%s start' % daemon_type)


def _start_daemon_instance(daemon_type, daemon_id):
    print run_command('service shinken-%s --id %s start' % (daemon_type, daemon_id))


def _stop_all_daemons(daemon_lists):
    for daemon_type in daemon_lists:
        _stop_daemon_type(daemon_type)
    do_exit()


def _stop_daemon_type(daemon_type):
    print run_command('service shinken-%s stop' % daemon_type)


def _stop_daemon_instance(daemon_type, daemon_id):
    print run_command('service shinken-%s --id %s stop' % (daemon_type, daemon_id))


def _restart_service(service):
    print run_command('service %s restart' % service)


#####################################################################
### SANATIZE
#####################################################################
def run_sanatize():
    sanatize_list = PATCH_DAT.get('SANATIZE_LIST', [])
    if not sanatize_list:
        do_exit('There is no sanatize to run into this patch', color=COLOR.YELLOW)
    for sanatize_name in sanatize_list:
        
        print run_command('/var/lib/shinken/libexec/tools/sanatize-data.py -r "%s"' % sanatize_name)
    do_exit()


#####################################################################
### USER CONSIDERATION
#####################################################################
def _ask_user(question, possible_choice=['y', 'o', 'n'], printable_choice='Yes/No', default_choice='Yes'):
    valid_choice = False
    user_choice = default_choice
    while not valid_choice:
        _print_info(question)
        user_choice = raw_input('(%s) by default \'%s\' : ' % (printable_choice, default_choice.capitalize())).strip().lower()
        if not user_choice:
            user_choice = default_choice[0].lower()
            valid_choice = True
        else:
            user_choice = user_choice[0]
            
            if user_choice in possible_choice:
                valid_choice = True
            
            else:
                _print_error('Your answer is not valid, please choose one this answer %s' % printable_choice)
    
    print '\n'
    return user_choice


# Not use for the moment See SEF-5780
def _ask_user_to_select_daemon(instances):
    valid_choice = False
    while not valid_choice:
        _instances = []
        _print_info('Available instances are :')
        _print_info('Number | daemon type  | daemon name')
        for index, instance in enumerate(instances):
            _print_info('- %4d | %12s | %s' % (index + 1, instance[0], instance[2]))
        _print_info('Wich daemons do you want to stop ?')
        user_choice = raw_input(' (you can choose several by separating them by a comma) : ').lower()
        user_choice = user_choice.split(',')
        
        choices = []
        for selected_index in user_choice:
            try:
                selected_index = int(selected_index.strip())
            except ValueError:
                _print_error('this value is not available : %s' % selected_index)
                choices.append(False)
                continue
            
            if not (1 <= selected_index <= len(instances)):
                _print_error('this value is not available : %s' % selected_index)
                choices.append(False)
            
            else:
                _instances.append(instances[selected_index - 1])
        if not False in choices:
            valid_choice = True
    
    return _instances


#####################################################################
### PATCHS INSTALLATION
#####################################################################
def patch_is_installed():
    _backup_dir = "/var/lib/shinken/backup/shinken-patch-%s" % PATCH_DAT['PATCH_ID']
    # If there is no a file in the backup directory, means that it was already installed
    backup_files = os.listdir(_backup_dir) if os.path.exists(_backup_dir) else []
    if len(backup_files) == 0:
        do_exit_error('The patch %s is not installed. The backup directory %s is void' % (PATCH_DAT['PATCH_ID'], _backup_dir))
    do_exit(' - Checking if the patch is always installed: OK')


def patch_is_not_installed():
    _backup_dir = "/var/lib/shinken/backup/shinken-patch-%s" % PATCH_DAT['PATCH_ID']
    # If there is no a file in the backup directory, means that it was already installed
    backup_files = os.listdir(_backup_dir) if os.path.exists(_backup_dir) else []
    if len(backup_files) != 0:
        do_exit_error('The patch %s is already installed. The backup directory %s is not void' % (PATCH_DAT['PATCH_ID'], _backup_dir))
    do_exit(' - Checking if the patch is not already installed: OK')


def check_version():
    try:
        from shinkensolutions.localinstall import VERSION as shinken_version_installed
    except ImportError:
        do_exit_error('This patch must be installer on a existing shinken installation')
    
    shinken_version_in_patch = PATCH_DAT.get('SHINKEN_VERSION', '')
    
    if shinken_version_installed != shinken_version_in_patch:
        do_exit_error('This patch is designed to be install only on version %s' % shinken_version_in_patch)


#####################################################################
### WEBUI
#####################################################################
def copy_webui(source, dest, delete_before=True):
    if delete_before:
        try:
            shutil.rmtree(dest)
        except OSError:
            pass
    copy_recursively(source, dest)


#####################################################################
### Selinux
#####################################################################
def apply_selinux_rules(path):
    _backup_dir = os.path.join(get_backup_dir(PATCH_DAT['PATCH_ID']), 'selinux_rules')
    os.mkdir(_backup_dir)
    
    selinux_pattern = os.path.join(path, '*.pp')
    selinux_files = glob.glob(selinux_pattern)
    if len(selinux_files) == 0:
        do_exit('There is no selinux rules to update into this patch', color=COLOR.YELLOW)
    
    for root, dir_names, file_names in os.walk(path):
        for rule in file_names:
            if rule.endswith('.pp'):
                rule_path = os.path.join(os.path.abspath(root), rule)
                rule_name = rule.strip('.pp')
                src_rule_path = rule_path.replace('.pp', '.te')
                # checkmodule -M -m -o /tmp/$rule_name.mod dependencies/selinux-rules/$rule_name.te 2>/dev/null >/dev/null
                run_command('''checkmodule -M -m -o /tmp/%s.mod %s''' % (rule_name, src_rule_path))
                # semodule_package -o  /tmp/$rule_name.pp -m /tmp/$rule_name.mod
                run_command('''semodule_package -o  /tmp/%s.pp -m /tmp/%s.mod''' % (rule_name, rule_name))
                # semodule -i /tmp/$rule_name.pp
                run_command('''semodule -i /tmp/%s.pp''' % rule_name)
                _print_info('Rule %s applied' % rule_name, color=COLOR.YELLOW)
                # We need to backup the files for revert it if necessary. The .pp is sufficient because for revert we need only its name
                shutil.copy2(rule_path, _backup_dir)


def remove_selinux_rules(path):
    is_in_revert = False
    if not path:
        # No specific folder ? ok take the backup dir
        is_in_revert = True
        path = os.path.join(get_backup_dir(PATCH_DAT['PATCH_ID']), 'selinux_rules')
    
    selinux_pattern = os.path.join(path, '*.pp')
    selinux_files = glob.glob(selinux_pattern)
    if len(selinux_files) == 0:
        if is_in_revert:
            shutil.rmtree(path)
        do_exit('There is no selinux rules to remove into this patch', color=COLOR.YELLOW)
    
    for root, dir_names, file_names in os.walk(path):
        for rule in file_names:
            if rule.endswith('.pp'):
                rule_name = rule.strip('.pp')
                run_command('''semodule -r %s''' % rule_name)
                _print_info('Rule %s removed' % rule_name)
    
    # If we are in a revert-patch.sh/py (hope
    if is_in_revert:
        shutil.rmtree(path)


#####################################################################
### RPMs & System
#####################################################################
def update_rpms_and_system(path):
    # Try to detect 6 or 7:
    minor = PYTHON_MINOR_VERSION
    rpm_pattern = os.path.join(path, '%s.X' % minor, '*.rpm')
    rpm_files = glob.glob(rpm_pattern)
    if len(rpm_files) != 0:
        _print_info('Updating rpm files', color=COLOR.YELLOW)
        for rpm_file in rpm_files:
            _print_info(' - %s' % rpm_file, color=COLOR.BLUE)
        run_command('''source ./lib/common.sh;dorpminstall "%s" 'Patch %s' ''' % (rpm_pattern, PATCH_DAT['PATCH_ID']))
    else:
        _print_info('There is no RPMs to update into this patch', color=COLOR.YELLOW)
    
    _print_info('Checking system parameters', color=COLOR.BLUE)
    run_command('''source ./lib/common.sh;assert_jemalloc_memory ''')


#####################################################################
### Run !!!!
#####################################################################
def parse_opts():
    parser = optparse.OptionParser('Common lib for patch')
    # Common
    parser.add_option('--id', dest='patch_id', action='store', default='', help='patch id')
    parser.add_option('--force', dest='force', action='store_true', default=False, help='bypass all answers')
    # Restart daemon
    parser.add_option('--start', dest='start_all_daemons', action='store_true', default=False, help='request for start all daemons')
    parser.add_option('--stop', dest='stop_all_daemons', action='store_true', default=False, help='request for stop all daemons')
    # Restart external service
    parser.add_option('--restart-services', dest='restart_services', action='store_true', default=False, help='request for start all daemons')
    parser.add_option('--service-list', dest='service_list', action='store', default='', help='list of daemons required')
    # Patch installation
    parser.add_option('--patch-is-installed', dest='patch_is_installed', action='store_true', default=False, help='check if patch is already installed')
    parser.add_option('--patch-is-not-installed', dest='patch_is_not_installed', action='store_true', default=False, help='check if patch is not already installed')
    parser.add_option('--check-version', dest='check_version', action='store_true', default=False, help='check if patch will be installed on the good shinken version')
    # File dispatcher
    parser.add_option('--backup-before-patch', dest='backup_before_patch', action='store_true', default=False, help='Backup before install patch')
    parser.add_option('--restore-after-patch', dest='restore_after_patch', action='store_true', default=False, help='Restore after patch')
    parser.add_option('--apply-patch-files', dest='apply_patch_files', action='store_true', default=False, help='Apply the patch')
    # Sanatize
    parser.add_option('--sanatize', dest='sanatize', action='store_true', default=False, help='run sanatize')
    # Selinux
    parser.add_option('--remove-selinux-rules', dest='remove_selinux_rules', action='store_true', default=False, help='Remove selinux rules')
    parser.add_option('--apply-selinux-rules', dest='apply_selinux_rules', action='store_true', default=False, help='Remove selinux rules')
    parser.add_option('--selinux-rules-path', dest='selinux_rules_path', action='store', default='', help='Selinux rules to add/remove')
    # RPMs
    parser.add_option('--update-rpms-and-system', dest='update_rpms_and_system', action='store_true', default=False, help='Update rpms and system')
    parser.add_option('--rpms-path', dest='rpms_path', action='store', default='', help='Top level path for rpms')
    
    return parser.parse_args()


def main():
    opts, args = parse_opts()
    
    # Restart daemon
    if opts.stop_all_daemons:
        request_stop_all_daemons(opts.force)
    elif opts.start_all_daemons:
        request_start_all_daemons(opts.force)
    
    # Restart services
    if opts.restart_services:
        request_restart_services(opts.service_list, opts.force)
    
    # Patch installation
    elif opts.patch_is_installed:
        patch_is_installed()
    elif opts.patch_is_not_installed:
        patch_is_not_installed()
    elif opts.check_version:
        check_version()
    
    # File dispatcher
    elif opts.backup_before_patch:
        backup_dir = get_backup_dir(PATCH_DAT['PATCH_ID'])
        backup_before_patch(backup_dir)
    elif opts.restore_after_patch:
        if not opts.patch_id:
            do_exit_error('need a patch_id with option --id')
        restore_after_patch(opts.patch_id)
    
    elif opts.apply_patch_files:
        apply_patch_files()
    
    elif opts.sanatize:
        run_sanatize()
    
    elif opts.remove_selinux_rules:
        remove_selinux_rules(opts.selinux_rules_path)
    
    elif opts.apply_selinux_rules:
        apply_selinux_rules(opts.selinux_rules_path)
    
    elif opts.update_rpms_and_system:
        update_rpms_and_system(opts.rpms_path)


if __name__ == '__main__':
    try:
        main()
    except KeyboardInterrupt:
        do_exit_error('Patch not installed.')
