#!/bin/bash

############################################
### MISC
############################################
export GREY="\033[90m"
export BLUE="\033[94m"
export MAGENTA="\033[94m"
export YELLOW="\033[93m"
export RED="\033[31m"
export RESET="\033[0m"

export PYTHON_VERSION=$(python -c 'import sys;print sys.version_info[1]')
export SHINKEN_LIB_PATH=/usr/lib/python2.$PYTHON_VERSION/site-packages/shinken
export MODULES_DIR=/var/lib/shinken/modules

function run() {
    $*
    if [[ $? -ne 0 ]]; then
        printf "$RED--------------------------------------------------------------------------------------------------------------------$RESET\n"
        printf "$RED--------------------------------------------------------------------------------------------------------------------$RESET\n"
        printf "$RED   Error while running $*  $RESET\n"
        printf "$RED--------------------------------------------------------------------------------------------------------------------$RESET\n"
        printf "$RED--------------------------------------------------------------------------------------------------------------------$RESET\n"
        exit 2
    fi
}

function run_python() {
    python2 $*
    if [[ $? -ne 0 ]]; then
        exit 2
    fi
}

function print_step() {
    step="$1"
    step_nb="$2"
    nb_steps="$3"
    printf -- "\n\n$BLUE------ [ STEP $YELLOW$step_nb$BLUE / $YELLOW$nb_steps $BLUE]  $step    -----------------------------------------------$RESET\n\n"
}

function _display_readme() {
    if [[ ! -f readme.txt ]]; then
        printf "$RED readme.txt file missing $RESET\n"
    fi

    printf "$GREY$(cat readme.txt | sed 's/^/ | /g' | sed 's/%/%%/g') $RESET"
    echo
    echo
}

### Main banner to display the readme and ask t the user a clear acceptance to apply the patch
function display_acceptance_banner() {
    force="$1"

    _display_readme

    if [[ "$force" == "--force" ]]; then
        return
    fi
    printf "${MAGENTA}==> Apply the patch?${RESET}"
    read -p " (y/N) by default 'N'" -n 1 -r
    echo
    echo
    echo
    if [[ ! $REPLY =~ ^[YyOo]$ ]]; then
        printf "\nPatch not installed.\n"
        exit 1
    fi
}

### Same as just before, but for uninstall
function display_uninstall_acceptance_banner() {
    force="$1"

    _display_readme

    if [[ "$force" == "--force" ]]; then
        return
    fi
    printf "${MAGENTA}==> Uninstall the patch?${RESET}"
    read -p " (y/N) by default 'N'" -n 1 -r
    echo
    echo
    echo
    if [[ ! $REPLY =~ ^[YyOo]$ ]]; then
        printf "\nPatch not un-installed.\n"
        exit 1
    fi
}

############################################
### Rescue from patchlib
############################################
function init_patch_name_and_location() {
    # NO SPACE ALLOWED in patch name
    # Initializes all needed parameters for the script to run
    # - Backup directory
    # - Log file

    # Params
    # - Name of the patch (for user info and backup directory name)
    local patch_name=$1

    if [[ -z "$patch_name" ]]; then
        printf "${RED} Patch name not defined${RESET}\n"
        exit 1
    fi
    PATCH_NAME="$patch_name"

    PATCH_TS=$(date '+%Y-%m-%dT%H-%M')
    BACKUP_DIR="/var/lib/shinken/backup/shinken-patch-${PATCH_NAME}"
    LOG_FILE=$0.$PATCH_TS.log

    mkdir -p $BACKUP_DIR

    printf "Patch: $PATCH_INFO\n\n"
}

function restore_all_patch_valid_rights {
    RESTO_LOG=/tmp/patch_restauration.log
    > $RESTO_LOG
    LANG=C setfacl --restore=patch.file_rights >>$RESTO_LOG 2>>$RESTO_LOG
    if [ $? != 0 ];then
        doexit "Cannot restore all patch file rights. Please look at the log $RESTO_LOG for errors."
    fi
}