#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import logging
import time

from component.sla_common import shared_data
from component.sla_component_manager import ComponentManager
from component.sla_database import SLADatabase
from component.sla_database_connection import SLADatabaseConnection
from component.sla_error_handler import error_handler, ERROR_LEVEL
from component.sla_info import SLAInfo
from component.sla_writer_stats import SLAWriterStats
from shinken.basemodule import ModuleState
from shinken.brokermodule import WorkerBasedBrokerModule
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.thread_helper import async_call
from shinkensolutions.date_helper import get_now, date_now
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration
from sla_archivator import Archivator
from sla_brok_handler import BrokHandlerModuleWorker
from sla_migrator import Migrator

ACCEPTED_BROK_TYPES = ('service_check_result', 'host_check_result', 'update_service_status', 'update_host_status', 'initial_service_status', 'initial_host_status', 'initial_broks_done')
MARGIN_SLA_INACTIVE = 30


class SLAModuleBroker(WorkerBasedBrokerModule):
    MODULE_WORKER_CLASS = BrokHandlerModuleWorker
    
    
    def __init__(self, configuration, display_info=True):
        # type: (ShinkenModuleDefinition, bool) -> None
        global MARGIN_SLA_INACTIVE
        WorkerBasedBrokerModule.__init__(self, configuration)
        if not display_info:
            self.logger.set_level(logging.ERROR)
        
        self.logger.info('Creating new SLA module name : %s for broker' % (configuration.get_name()))
        
        MARGIN_SLA_INACTIVE = read_int_in_configuration(configuration, 'time_before_shinken_inactive', MARGIN_SLA_INACTIVE)
        
        self.component_manager = ComponentManager(self.logger)
        self.sla_database_connection = SLADatabaseConnection(configuration, self.component_manager)
        self.sla_info = SLAInfo(configuration, self.component_manager, self.sla_database_connection)
        self.sla_database = SLADatabase(configuration, self.component_manager, self.sla_database_connection, self.sla_info)
        
        self.migrator = Migrator('broker', self.name, configuration)
        self.archivator = Archivator('broker', self.name, configuration)
        
        self.sla_writer_stats = SLAWriterStats(configuration, self.component_manager, self.sla_database, self.sla_info, self)
        
        self.external_speed_counter = 0
        self.external_done_counter = 0
        
        shared_data.set_default_values(getattr(configuration, 'configuration_default_value', {}))
    
    
    def init(self, daemon_display_name='broker'):
        WorkerBasedBrokerModule.init(self)
        self.component_manager.init()
        time.sleep(0.1)
        self.migrator.start(daemon_display_name)
        time.sleep(0.1)
        self.archivator.start(daemon_display_name)
    
    
    def want_brok(self, brok):
        return brok.type in ACCEPTED_BROK_TYPES
    
    
    def get_internal_state(self):
        return error_handler.get_internal_state()
    
    
    def get_state(self):
        errors = error_handler.get_errors()
        output = []
        for error in errors:
            output.append(error.message)
        return {'status': error_handler.get_internal_state(), 'output': output}
    
    
    # -------------------- Broker part
    
    def manage_initial_service_status_brok(self, new_info):
        self.sla_info.handle_brok('service', new_info)
    
    
    def manage_initial_host_status_brok(self, new_info):
        self.sla_info.handle_brok('host', new_info)
    
    
    def manage_initial_broks_done_brok(self, new_info):
        self.migrator.handle_initial_broks_done()
    
    
    def stop_all(self):
        self.logger.info('Start stopping all process of SLA Module Broker')
        try:
            self.migrator.stop()
        except:
            pass
        
        try:
            self.archivator.stop()
        except:
            pass
        
        try:
            WorkerBasedBrokerModule.stop_all(self)
        except:
            self.logger.error('Fail to stop workers')
            self.logger.print_stack()
        
        self.sla_info.stop()
        self.sla_writer_stats.stop()
        error_handler.stop()
        self.logger.info('Stopping all process of SLA Module Broker done')
    
    
    def hook_tick(self, broker):
        try:
            WorkerBasedBrokerModule.hook_tick(self, broker)
            self.component_manager.tick()
            self._update_active_range()
        except Exception as e:
            error_handler.handle_exception('Fatal error cause by : %s' % e, e, self.logger, level=ERROR_LEVEL.FATAL)
            raise
    
    
    def _update_active_range(self):
        tick_time = get_now()
        date = date_now()
        sla_range_active = self.sla_database.find_raw_sla_status(date)
        if sla_range_active:
            last_tick_time = sla_range_active['active_ranges'][-1].get('end', 0)
            if last_tick_time != 0 and tick_time - last_tick_time > MARGIN_SLA_INACTIVE:
                last_end = int(last_tick_time + MARGIN_SLA_INACTIVE)
                start = int(tick_time)
                sla_range_active['active_ranges'][-1]['end'] = last_end
                sla_range_active['active_ranges'].append({'start': start})
                self.logger.info('Starting a new SLA module activity period because last tick was more than %ss. It will make shinken inactive range in your elements SLA which is %s-%s.' % (
                    MARGIN_SLA_INACTIVE, self.logger.format_time(last_end), self.logger.format_time(start)))
            sla_range_active['active_ranges'][-1]['end'] = get_now()
        else:
            sla_range_active = {'_id': 'SLA_STATUS', 'active_ranges': [{'start': int(tick_time), 'end': int(tick_time) + 1}]}
        
        self.sla_database.save_raw_sla_status(date, sla_range_active)
    
    
    def get_raw_stats(self, param=''):
        try:
            if self.get_internal_state() == ModuleState.FATAL:
                return {}
            
            raw_stats = WorkerBasedBrokerModule.get_raw_stats(self, param)
            self.sla_writer_stats.process_raw_worker_stats(raw_stats)
            
            # self.logger.debug('asking get_raw_stats')
            data = self.sla_writer_stats.get_raw_stats()
            # self.logger.debug('get_raw_stats : [%s]' % data)
            return data
        except Exception as e:
            error_handler.handle_exception('Fail to get module information cause by : %s. This will only impact your check shinken on your sup of sup.' % e, e, self.logger, level=ERROR_LEVEL.WARNING)
            raise
