#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

"""
This Class is a plugin for the Shinken Broker. It is in charge
to get brok and recreate real objects, and propose a Web interface :)
"""

import base64
import hmac
import imp
import json
import os
import re
import socket

import sys
import threading
import time
import traceback

from shinken.basemodule import BaseModule
from shinken.daemon import Daemon
from shinken.log import logger
from shinken.message import Message
from shinken.misc.regenerator import Regenerator
from shinken.modulesctx import modulesctx
from shinken.modulesmanager import ModulesManager
from shinken.util import safe_print, to_bool
from shinken.webui.bottlewebui import static_file, view, route, request, response, template, abort, HTTPError
from shinken.webui.cherrypybackend import CherryPyServerHTTP
from arch_export_handler import ArchitectureExportHandler
from shinkensolutions.external_resources.external_resources import external_resources
from shinkensolutions.localinstall import get_context_hash

# Local import
from shinken.misc.datamanager import datamgr
from helper import helper
from widget_service import widget_service

JS_APP_VERSION_PATTERN = re.compile(r'''shinken-enterprise.([a-zA-Z0-9]*).js''')
SHINKEN_JS_VERSION = 'main file not load'

# Debug
import shinken.webui.bottlewebui as bottle

bottle.debug(True)

# Import bottle lib to make bottle happy
bottle_dir = os.path.abspath(os.path.dirname(bottle.__file__))
sys.path.insert(0, bottle_dir)

# Look at the webui module root dir too
webuimod_dir = os.path.abspath(os.path.dirname(__file__))
htdocs_dir = os.path.join(webuimod_dir, 'htdocs')

properties = {
    'daemons' : ['broker', 'scheduler'],
    'type'    : 'webui',
    'phases'  : ['running'],
    'external': True,
}

CONTEXT_PATH = '/var/lib/shinken/context.json'
CURRENT_VERSION = '02.06.00'

if os.path.exists(CONTEXT_PATH):
    context = json.loads(open(CONTEXT_PATH, 'r').read())
    CURRENT_VERSION = context.get('current_version', CURRENT_VERSION)


# called by the plugin manager to get an instance
def get_instance(plugin):
    # Add template only if we ask for a webui
    bottle.TEMPLATE_PATH.append(os.path.join(webuimod_dir, 'views'))
    bottle.TEMPLATE_PATH.append(webuimod_dir)
    logger.debug("Get a WebUI instance for plugin %s" % plugin.get_name())
    
    instance = Webui_broker(plugin)
    return instance


class EnableCors(object):
    name = 'enable_cors'
    api = 2
    
    
    def apply(self, fn, context):
        def _enable_cors(*args, **kwargs):
            # set CORS headers
            response.headers['Access-Control-Allow-Origin'] = 'http://127.0.0.1:9000'
            response.headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, OPTIONS, DELETE, PATCH'
            response.headers['Access-Control-Allow-Headers'] = 'Origin, Accept, Content-Type, X-Requested-With, X-CSRF-Token, X-Shinken-Token'
            response.headers['Access-Control-Allow-Credentials'] = 'true'
            if bottle.request.method != 'OPTIONS':
                # actual request; reply with the actual response
                return fn(*args, **kwargs)
        
        
        return _enable_cors


MAX_INT32 = int('1' * 31, base=2)


class Webui_broker(BaseModule, Daemon):
    def __init__(self, modconf):
        BaseModule.__init__(self, modconf)
        
        # IMPORTANT: we must have a static value based on the server installation state (install + patch)
        # because numerous webui can be start, and so process start time cannot be used here
        # NOTE: must be a int32 because of the usage in javascript and in the .tpl currently.
        self.http_start_time = int(get_context_hash(), 16) % MAX_INT32
        
        self.plugins = []
        
        self.serveropts = {}
        umask = getattr(modconf, 'umask', None)
        if umask != None:
            self.serveropts['umask'] = int(umask)
        bindAddress = getattr(modconf, 'bindAddress', None)
        if bindAddress:
            self.serveropts['bindAddress'] = str(bindAddress)
        self.serveropts['numthreads'] = 64
        
        self.port = int(getattr(modconf, 'port', '7767'))
        self.http_port = int(getattr(modconf, 'http_port', '7766'))
        self.host = getattr(modconf, 'host', '0.0.0.0')
        self.show_skonf = int(getattr(modconf, 'show_skonf', '1'))
        self.auth_secret = getattr(modconf, 'auth_secret').encode('utf8', 'replace')
        self.play_sound = to_bool(getattr(modconf, 'play_sound', '0'))
        self.http_backend = getattr(modconf, 'http_backend', 'auto')
        self.login_text = getattr(modconf, 'login_text', None)
        self.allow_html_output = to_bool(getattr(modconf, 'allow_html_output', '0'))
        self.max_output_length = int(getattr(modconf, 'max_output_length', '0'))
        self.manage_acl = to_bool(getattr(modconf, 'manage_acl', '1'))
        self.remote_user_enable = getattr(modconf, 'remote_user_enable', '0')
        self.remote_user_variable = getattr(modconf, 'remote_user_variable', 'X-REMOTE-USER')
        self.remote_user_case_sensitive = to_bool(getattr(modconf, 'remote_user_case_sensitive', '1'))
        
        self.use_ssl = getattr(modconf, 'use_ssl', '0') == '1'
        self.ssl_key = getattr(modconf, 'ssl_key', '')
        self.ssl_cert = getattr(modconf, 'ssl_cert', '')
        
        self.share_dir = getattr(modconf, 'share_dir', 'share')
        self.share_dir = os.path.abspath(self.share_dir)
        # Load the photo dir and make it a absolute path
        self.photo_dir = getattr(modconf, 'photo_dir', 'photos')
        self.photo_dir = os.path.abspath(self.photo_dir)
        
        self.show_trending = getattr(modconf, 'show_trending', '0') == '1'
        
        # Look for an additonnal pages dir
        self.additional_plugins_dir = getattr(modconf, 'additional_plugins_dir', '')
        if self.additional_plugins_dir:
            self.additional_plugins_dir = os.path.abspath(self.additional_plugins_dir)
        
        # We will save all widgets
        self.widgets = {}
        self.rg = Regenerator(modconf, self.get_name())
        
        self.arch_export_handler = ArchitectureExportHandler()
        
        self.bottle = bottle
        # get a list of tag images available for display
        self.img_tags = []
        sets_dir = os.path.join(self.share_dir, 'images', 'sets')
        # Look at the share/images/sets/*/tag.png and save the *
        for p in os.listdir(sets_dir):
            fp = os.path.join(sets_dir, p)
            if os.path.isdir(fp) and os.path.exists(os.path.join(fp, 'tag.png')):
                self.img_tags.append(p)
        
        self.lang = getattr(modconf, 'lang', 'en')
        self.langs_path = '/var/lib/shinken/modules/webui/htdocs/js/traductions'
        self.langs = {'en': None, 'fr': None}
        self.tiles_background = getattr(modconf, 'tiles_background', 'context')
        self.colors_graphics = getattr(modconf, 'colors_graphics', '0095DA,E02C2C')
        self.apply_filter_method = getattr(modconf, 'apply_filter_method', 'key_enter')
        
        self.history__nb_changes_displayed = getattr(modconf, 'history__nb_changes_displayed', '30')
        self.history__size_sla_pane = getattr(modconf, 'history__size_sla_pane', None)
        self.history__display_outputs = getattr(modconf, 'history__default_display_outputs', '1') != '0'
        self.history__collapse_outputs = getattr(modconf, 'history__default_collapse_outputs', '0') == '1'
        
        if self.history__size_sla_pane:
            try:
                self.history__size_sla_pane = int(self.history__size_sla_pane)
            except ValueError:
                self.history__size_sla_pane = None
        try:
            self.history__nb_changes_displayed = int(self.history__nb_changes_displayed)
            if self.history__nb_changes_displayed <= 0:
                logger.warning('The key \'history__nb_changes_displayed can\'t be negative or null\'')
                self.history__nb_changes_displayed = 30
        except ValueError:
            self.history__nb_changes_displayed = 30
        
        self.graphs_errors = {}
        
        self.hook_error()
        
        self.current_version = CURRENT_VERSION[0:12]
        self.current_version = self.current_version.replace('-REL', '').replace('-rel', '')
    
    
    def error_handler(self, error):
        return_value = error.output
        if getattr(error, 'warning', False):
            logger.warning("http error catch [%s]" % return_value)
        else:
            logger.error("http error catch [%s]" % return_value)
        if error.traceback:
            first_line = True
            for line_stack in error.traceback.splitlines():
                if first_line:
                    logger.error("ERROR stack : %s" % line_stack)
                    first_line = False
                else:
                    logger.error("%s" % line_stack)
        if response and response.content_type == 'application/json':
            return_value = json.dumps(error.output)
        else:
            with_menu = False
            user_p = None
            try:
                user_p = self.get_user_auth()
            except:
                pass
            if not user_p:
                user_p = {}
            return_value = self.bottle.template("%include error globals()", e=error, app=self, user=user_p, with_menu=with_menu)
        return return_value
    
    
    def hook_error(self):
        @bottle.error(514)
        def custom514(error):
            return self.error_handler(error)
        
        
        @bottle.error(513)
        def custom513(error):
            return self.error_handler(error)
        
        
        @bottle.error(512)
        def custom512(error):
            return self.error_handler(error)
        
        
        @bottle.error(511)
        def custom511(error):
            return self.error_handler(error)
        
        
        @bottle.error(500)
        def custom500(error):
            return self.error_handler(error)
        
        
        @bottle.error(400)
        def custom500(error):
            return self.error_handler(error)
        
        
        @bottle.error(404)
        def custom404(error):
            return self.error_handler(error)
        
        
        @bottle.error(401)
        def custom401(error):
            if response.content_type != 'application/json':
                response.set_header('location', '/static/ui/index.html')
                response.status = 303
                return ''
            else:
                return self.error_handler(error)
        
        
        @bottle.error(403)
        def custom403(error):
            return self.error_handler(error)
        
        
        @bottle.error(409)
        def custom409(error):
            return self.error_handler(error)
        
        
        @bottle.error(410)
        def custom410(error):
            return self.error_handler(error)
    
    
    # We check if the photo directory exists. If not, try to create it
    def check_photo_dir(self):
        if not os.path.exists(self.photo_dir):
            try:
                os.mkdir(self.photo_dir)
            except Exception, exp:
                pass
    
    
    # Called by Broker so we can do init stuff
    # TODO: add conf param to get pass with init
    # Conf from arbiter!
    def init(self):
        logger.debug('Init of the UI %s' % self.name)
        self.rg.load_external_queue(self.from_module_to_main_daemon_queue)
    
    
    # This is called only when we are in a scheduler
    # and just before we are started. So we can gain time, and
    # just load all scheduler objects without fear :) (we
    # will be in another process, so we will be able to hack objects
    # if need)
    def hook_pre_scheduler_mod_start(self, sched):
        print "pre_scheduler_mod_start::", sched.__dict__
        self.rg.load_from_scheduler(sched)
    
    
    # In a scheduler we will have a filter of what we really want as a brok
    def want_brok(self, b):
        return self.rg.want_brok(b) or self.arch_export_handler.want_brok(b)
    
    
    def main(self):
        logger.set_name(self.name)
        
        # Daemon like init
        self.debug_output = []
        self.modules_dir = modulesctx.get_modulesdir()
        self.modules_manager = ModulesManager('webui', self.find_modules_path(), [])
        self.modules_manager.set_modules(self.modules)
        # We can now output some previously silenced debug output
        self.do_load_modules()
        for inst in self.modules_manager.get_all_alive_instances():
            f = getattr(inst, 'load', None)
            if f and callable(f):
                f(self)
        
        for s in self.debug_output:
            logger.debug(s)
        del self.debug_output
        
        self.check_photo_dir()
        self.datamgr = datamgr
        datamgr.load(self.rg)
        self.helper = helper
        self.helper.set_app(self)
        
        self.widget_service = widget_service
        self.widget_service.set_app(self)
        
        self.request = request
        self.abort = self._abort
        self.response = response
        self.template_call = template
        
        # the regenerator is ready in the valid process, we can start it's freshness thread
        self.rg.launch_freshness_thread(lang=self.lang)
        try:
            # import cProfile
            # cProfile.runctx('''self.do_main()''', globals(), locals(),'/tmp/webui.profile')
            self.do_main()
        except Exception, exp:
            msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': exp, 'trace': traceback.format_exc()})
            self.from_module_to_main_daemon_queue.put(msg)
            # wait 2 sec so we know that the broker got our message, and die
            time.sleep(2)
            raise
    
    
    def _abort(self, code=500, text='Unknown Error: Application stopped.', warning=False):
        logger.error('A %s HTTP error occured : %s' % (code, text))
        http_error = HTTPError(code, text)
        http_error.warning = warning
        raise http_error
    
    
    def wrap_shinken_js_version_header(self, f):
        def __wrap(*args, **kwargs):
            self.response.headers['X-shinken-js-version'] = SHINKEN_JS_VERSION
            return f(*args, **kwargs)
        
        
        return __wrap
    
    
    def wrap_auth(self, f):
        def __wrap(*args, **kwargs):
            # First we look for the user sid so we bail out if it's a false one.
            user = self.get_user_auth()
            if not user:
                logger.info("Invalid user")
                return self.abort(401, 'Invalid user')
            return f(*args, **kwargs)
        
        
        return __wrap
    
    
    def wrap_json(self, f):
        def __wrap(*args, **kwargs):
            self.response.content_type = 'application/json'
            return f(*args, **kwargs)
        
        
        return __wrap
    
    
    # A plugin send us en external command. We just put it
    # in the good queue
    def push_external_command(self, e):
        logger.debug("UI: got an external command: %s" % str(e.__dict__))
        self.from_module_to_main_daemon_queue.put(e)
    
    
    # Real main function
    def do_main(self):
        global SHINKEN_JS_VERSION
        # I register my exit function
        self.set_exit_handler()
        
        # We will protect the operations on the non read+write with a lock and 2 int
        self.global_lock = threading.RLock()
        self.nb_readers = 0
        self.nb_writers = 0
        
        self.data_thread = None
        
        # Step - Check if the view dir really exist
        if not os.path.exists(bottle.TEMPLATE_PATH[0]):
            logger.error("The view path do not exist at %s" % bottle.TEMPLATE_PATH)
            sys.exit(2)
        
        # Step - Find js app version
        core_plugin_dir = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'plugins')
        js_app_dir = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'htdocs/ui/scripts/')
        i = 0
        if os.path.isdir(js_app_dir):
            for file_name in os.listdir(js_app_dir):
                main_file = JS_APP_VERSION_PATTERN.match(os.path.basename(file_name))
                if main_file:
                    SHINKEN_JS_VERSION = main_file.group(1)
                    i += 1
                    logger.info('loading js app version [%s]' % SHINKEN_JS_VERSION)
        if i == 0:
            logger.error('no app js found in [%s]' % js_app_dir)
            sys.exit(2)
        elif i > 1:
            logger.error('multi app js found in [%s]' % js_app_dir)
            sys.exit(2)
        
        # Step - Load the additional plugins so they will have the lead on URI routes
        if self.additional_plugins_dir:
            self.load_plugins(self.additional_plugins_dir)
        
        # Step - Modules can also override some views if need
        for inst in self.modules_manager.get_all_alive_instances():
            f = getattr(inst, 'get_webui_plugins_path', None)
            if f and callable(f):
                mod_plugins_path = os.path.abspath(f(self))
                self.load_plugins(mod_plugins_path)
        
        # Step - Then look at the plugins in toe core and load all we can there
        self.load_plugins(core_plugin_dir)
        
        # We must be sure the check plugins styles css are compiled before
        try:
            self.compiled_css_path, self.compiled_css_hash = external_resources.load_and_combine_css()
        except Exception:  # there was a problem in the css compilation, already log
            raise
        # Step - Declare the whole app static files AFTER the plugin ones
        self.declare_common_static()
        
        # Step - Launch the data thread
        self.data_thread = threading.Thread(None, self.manage_brok_thread, 'datathread')
        self.data_thread.start()
        # TODO: look for alive and killing
        
        # Ok, you want to know why we are using a data thread instead of just call for a select with q._reader, the underlying file handle of the Queue()?
        # That's just because under Windows, select only manage winsock (so network) file descriptor! What a shame!
        logger.debug("UI starting application")
        self.srv = bottle.run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=self.use_ssl, ssl_key=self.ssl_key, ssl_cert=self.ssl_cert, **self.serveropts)
        try:
            self.srv.start()
        except socket.error as e:
            if e.message == 'No socket could be created':
                _error = 'The webui named [%s] can not start because the address %s:%s is already in use' % (self.get_name(), self.host, self.port)
                logger.error(_error)
            else:
                raise e
        
        # ^ IMPORTANT ^
        # We are not managing the lock at this level because we got 2 types of requests:
        # static images/css/js: no need for lock
        # pages: need it. So it's managed at a function wrapper at loading pass
    
    
    def manage_brok_thread(self):
        try:
            self._manage_brok_thread()
        except:
            self.logger.print_stack()
    
    
    # It's the thread function that will get broks
    # and update data. Will lock the whole thing
    # while updating
    def _manage_brok_thread(self):
        # DBG: times={}
        # DBG: time_waiting_no_readers = 0
        # DBG: time_preparing = 0
        
        while not self.interrupted:
            # DBG: print "WEBUI :: GET START"
            try:
                l = self.to_q.get()
            except EOFError:
                if self.interrupted:
                    return
                else:
                    raise
            t0 = time.time()
            # DBG: print "WEBUI :: GET FINISH with", len(l), "in ", t1 - t0
            
            for b in l:
                # DBG: t0 = time.time()
                b.prepare()
                # DBG: time_preparing += time.time() - t0
                # DBG: if not b.type in times:
                # DBG:     times[b.type] = 0
                # For updating, we cannot do it while
                # answer queries, so wait for no readers
                # DBG: t0 = time.time()
                self.wait_for_no_readers()
                # DBG: time_waiting_no_readers += time.time() - t0
                try:
                    # print "Got data lock, manage brok"
                    # DBG: t0 = time.time()
                    
                    # to_remove = ['log', 'initial_timeperiod_status', 'initial_command_status', 'initial_hostgroup_status', 'initial_service_status', 'service_check_result', 'service_next_schedule'  ]
                    # if b.type not in to_remove:
                    #     logger.debug('[Webui] [%s-%s] broks' % (b.type,b.id))
                    self.rg.manage_brok(b)
                    self.arch_export_handler.manage_brok(b)
                except Exception, exp:
                    msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': exp, 'trace': traceback.format_exc()})
                    self.from_module_to_main_daemon_queue.put(msg)
                    # wait 2 sec so we know that the broker got our message, and die
                    time.sleep(2)
                    # No need to raise here, we are in a thread, exit!
                    os._exit(2)
                finally:
                    # We can remove us as a writer from now. It's NOT an atomic operation
                    # so we REALLY not need a lock here (yes, I try without and I got
                    # a not so accurate value there....)
                    self.global_lock.acquire()
                    self.nb_writers -= 1
                    self.global_lock.release()
                    
                    # DBG: t2 = time.time()
                    # DBG: print "WEBUI :: MANAGE ALL IN ", t2 - t1
                    # DBG: print '"WEBUI: in Waiting no readers', time_waiting_no_readers
                    # DBG: print 'WEBUI in preparing broks', time_preparing
                    # DBG: print "WEBUI And in times:"
                    # DBG: for (k, v) in times.iteritems():
                    # DBG:     print "WEBUI\t %s: %s" % (k, v)
                    # DBG: print "WEBUI\nWEBUI\n"
            
            logger.log_perf(t0, '[Broks][Webui]', 'Managed %d broks' % len(l))
    
    
    def load_plugin(self, fdir, plugin_dir):
        logger.debug('UI Loading %s from %s' % (fdir, plugin_dir))
        try:
            # Put the full qualified path of the module we want to load
            # for example we will give  webui/plugins/eltdetail/
            mod_path = os.path.join(plugin_dir, fdir)
            # Then we load the eltdetail.py inside this directory
            m = imp.load_module('%s' % (fdir), *imp.find_module(fdir, [mod_path]))
            m_dir = os.path.abspath(os.path.dirname(m.__file__))
            sys.path.append(m_dir)
            
            logger.debug('UI module %s is loaded from %s' % (m_dir, str(m)))
            pages = m.pages
            do_static = False
            for (f, entry) in pages.items():
                routes = entry.get('routes', None)
                v = entry.get('view', None)
                static = entry.get('static', False)
                wrappers = entry.get('wrappers', ['auth'])
                wrappers.append('shinken-js-version')
                widget_lst = entry.get('widget', [])
                widget_desc = entry.get('widget_desc', None)
                widget_display_name = entry.get('widget_display_name', 'Widget')
                widget_name = entry.get('widget_name', None)
                widget_picture = entry.get('widget_picture', None)
                widget_size = entry.get('widget_size', {'width': 1, 'height': 1})
                widget_options = entry.get('widget_options', [])
                for wo in widget_options:  # by default optiosn are required
                    if not 'required' in wo:
                        wo['required'] = True
                widget_favoritable = entry.get('widget_favoritable', False)
                resizable = entry.get('resizable', False)
                
                old_style = entry.get('old_style', True)
                
                f_name = f.__name__
                f.display_name = f_name
                for wrapper in wrappers:
                    wrap = {'auth': self.wrap_auth, 'json': self.wrap_json, 'shinken-js-version': self.wrap_shinken_js_version_header}.get(wrapper, None)
                    if wrap:
                        f_name = f.display_name
                        f = wrap(f)
                        f.display_name = f_name + '-w(' + wrapper + ')'
                
                # IMPORTANT: apply VIEW BEFORE route!
                if v:
                    f_name = f.display_name
                    f = view(v)(f)
                    f.display_name = f_name + '-v(' + v + ')'
                
                # Maybe there is no route to link, so pass
                if routes:
                    for r in routes:
                        method = entry.get('method', 'GET')
                        logger.debug('UI linking function [%s] and route [%s] for method [%s]' % (getattr(f, 'display_name', str(f)), r, method))
                        
                        # Ok, we will just use the lock for all
                        # plugin page, but not for static objects
                        # so we set the lock at the function level.
                        f_name = f.display_name
                        lock_version = self.lockable_function(f)
                        lock_version.display_name = f_name
                        f = route(r, callback=lock_version, method=[method, 'OPTIONS'])
                
                # If the plugin declare a static entry, register it
                # and remember: really static! because there is no lock
                # for them!
                if static:
                    do_static = True
                
                # It's a valid widget entry if it got all data, and at least one route
                # ONLY the first route will be used for Add!
                # print "Should I load a widget?",widget_name, widget_desc, widget_lst!=[], routes
                if widget_name and widget_desc and widget_lst != [] and routes:
                    for place in widget_lst:
                        if place not in self.widgets:
                            self.widgets[place] = []
                        w = {
                            'name'       : widget_name,
                            'displayName': widget_display_name,
                            'description': widget_desc,
                            'baseURI'    : routes[0],
                            'picture'    : widget_picture,
                            'size'       : widget_size,
                            'options'    : widget_options,
                            'favoritable': widget_favoritable,
                            'oldStyle'   : old_style,
                            'resizable'  : resizable,
                        }
                        self.widgets[place].append(w)
            
            if do_static:
                self.add_static(fdir, m_dir)
            
            # And we add the views dir of this plugin in our TEMPLATE
            # PATH
            bottle.TEMPLATE_PATH.append(os.path.join(m_dir, 'views'))
            
            # And finally register me so the pages can get data and other
            # useful stuff
            m.app = self
        
        
        except Exception, exp:
            logger.warning("Loading plugins: %s" % exp)
            logger.print_stack()
    
    
    # Here we will load all plugins (pages) under the webui/plugins directory.
    # Each one can have a page, views and htdocs dir that we must route correctly
    def load_plugins(self, plugin_dir):
        # Load plugin directories
        if not os.path.exists(plugin_dir):
            return
        
        plugin_dirs = [fname for fname in os.listdir(plugin_dir) if os.path.isdir(os.path.join(plugin_dir, fname))]
        
        sys.path.append(plugin_dir)
        # We try to import them, but we keep only the one of our type
        for fdir in plugin_dirs:
            self.load_plugin(fdir, plugin_dir)
    
    
    def add_static(self, fdir, m_dir):
        static_route = '/static/%s/' % self.http_start_time + fdir + '/:path#.+#'
        
        
        def plugin_static(path):
            self._set_allow_cache_headers()
            return static_file(path, root=os.path.join(m_dir, 'htdocs'))
        
        
        logger.debug("Declaring static route: %s=> %s" % (static_route, plugin_static))
        route(static_route, callback=plugin_static)
    
    
    # It will say if we can launch a page rendering or not.
    # We can only if there is no writer running from now
    def wait_for_no_writers(self):
        wait_start = time.time()
        start = time.time()
        wait_time = 0.001
        while True:
            self.global_lock.acquire()
            # We will be able to run
            if self.nb_writers == 0:
                # Ok, we can run, register us as readers
                self.nb_readers += 1
                self.global_lock.release()
                break
            # Oups, a writer is in progress. We must wait a bit
            self.global_lock.release()
            
            # Before checking again, we should wait a bit to not hammer the cpu.
            # Wait will be longer each call limit to 100ms
            time.sleep(wait_time)
            wait_time = max(wait_time * 2, 0.1)
            
            if time.time() - start > 30:
                logger.error("UI: we wait for read lock for more than 30s! %0.3f" % (time.time() - wait_start))
                start = time.time()
    
    
    # It will say if we can launch a brok management or not
    # We can only if there is no readers running from now
    def wait_for_no_readers(self):
        wait_start = time.time()
        start = time.time()
        wait_time = 0.001
        while True:
            self.global_lock.acquire()
            # We will be able to run
            if self.nb_readers == 0:
                # Ok, we can run, register us as writers
                self.nb_writers += 1
                self.global_lock.release()
                break
            # Ok, we cannot run now, wait a bit
            self.global_lock.release()
            
            # Before checking again, we should wait a bit to not hammer the cpu.
            # Wait will be longer each call limit to 100ms
            time.sleep(wait_time)
            wait_time = max(wait_time * 2, 0.1)
            # We should warn if we cannot update broks
            # for more than 30s because it can be not good
            if time.time() - start > 30:
                logger.error("UI: we wait for write lock for more than 30s! %0.3f" % (time.time() - wait_start))
                start = time.time()
    
    
    # We want a lock manager version of the plugin functions
    def lockable_function(self, f):
        def lock_version(**args):
            f = lock_version.f
            self.wait_for_no_writers()
            
            t = time.time()
            try:
                # set CORS headers
                response.headers['Access-Control-Allow-Origin'] = 'http://127.0.0.1:9000'
                response.headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, OPTIONS, DELETE, PATCH'
                response.headers['Access-Control-Allow-Headers'] = 'Origin, Accept, Content-Type, X-Requested-With, X-CSRF-Token, X-Shinken-Token'
                response.headers['Access-Control-Allow-Credentials'] = 'true'
                
                if bottle.request.method != 'OPTIONS':
                    # And disable cache on all backend calls
                    self._set_no_cache_headers()
                    return f(**args)
                return
            finally:
                logger.debug('[ui] Perf: function %s executed in %s' % (getattr(f, 'display_name', str(f)), (time.time() - t)))
                # We can remove us as a reader from now. It's NOT an atomic operation
                # so we REALLY not need a lock here (yes, I try without and I got
                # a not so accurate value there....)
                self.global_lock.acquire()
                self.nb_readers -= 1
                self.global_lock.release()
        
        
        lock_version.f = f
        return lock_version
    
    
    # For this request, ask for no cache at all. for example static that have always the
    # same address across versions
    def _set_no_cache_headers(self):
        # The Cache-Control is per the HTTP 1.1 spec for clients and proxies
        # (and implicitly required by some clients next to Expires).
        # The Pragma is per the HTTP 1.0 spec for prehistoric clients.
        # The Expires is per the HTTP 1.0 and 1.1 spec for clients and proxies.
        # In HTTP 1.1, the Cache-Control takes precedence over Expires, so it's after all for HTTP 1.0 proxies only.
        response.headers['Cache-Control'] = 'no-cache,no-store,must-revalidate'
        response.headers['Pragma'] = 'no-cache'
        response.headers['Expires'] = '0'
    
    
    def _set_allow_cache_headers(self):
        # Ask for no tuning from cache or in the middle proxy
        response.headers['Cache-Control'] = 'no-transform,public,max-age=86400,s-maxage=86400'
        response.headers['Expires'] = '86400'  # old http 1.0 way
    
    
    def declare_common_static(self):
        
        # Give the combined css file
        # IMPORTANT: defined BEFORE the static/*
        # And disallow cache, as the ui will F5 when the hash will change
        @self.bottle.route('/static/css/check_plugin_styles_css.css')
        def give_check_plugin_styles_css():
            self._set_no_cache_headers()
            return self.bottle.static_file(os.path.basename(self.compiled_css_path), root=os.path.dirname(self.compiled_css_path))
        
        
        @route('/static/photos/:path#.+#')
        def give_photo(path):
            # If the file really exist, give it. If not, give a dummy image.
            if os.path.exists(os.path.join(self.photo_dir, path + '.jpg')):
                return static_file(path + '.jpg', root=self.photo_dir)
            else:
                return static_file('images/user.png', root=htdocs_dir)
        
        
        # Route static ui files (they have their own cache preventing system)
        @self.bottle.route('/static/ui/:path#.+#')
        def server_static_app(path):
            # By default give from the root in bottle_dir/htdocs. If the file is missing, search in the share dir
            path = 'ui/' + path
            root = htdocs_dir
            
            # logger.debug('Asking for static: %s' % path)
            
            # Ok now manage cache things:
            # * all the *.js and *.css things are automatically changed paths
            #   when new version, so we can cache them
            # * but all the others things like html and so on don't have changed paths
            #   so we must protect them
            if path.endswith('.js') or path.endswith('.css'):
                self._set_allow_cache_headers()
            else:  # no cache at all
                # Our dear Internet Explorer have a bug about https+nocache+woff files.
                if not path.endswith('.woff'):
                    self._set_no_cache_headers()
            return self.bottle.static_file(path, root=root)
        
        
        # Route static files css files
        @route('/static/%s' % self.http_start_time + '/:path#.+#')
        def server_static(path):
            # By default give from the root in bottle_dir/htdocs. If the file is missing,search in the share dir
            
            root = htdocs_dir
            p = os.path.join(root, path)
            if not os.path.exists(p):
                root = self.share_dir
            # They have path at each boot, so we can cache them without problems
            self._set_allow_cache_headers()
            return static_file(path, root=root)
        
        
        # And add the favicon ico too
        @route('/favicon.ico')
        def give_favicon():
            # response.headers['Cache-Control'] = 'public, max-age=3600'
            return static_file('favicon.ico', root=os.path.join(htdocs_dir, 'images'))
    
    
    def check_auth(self, user, password):
        if getattr(self.myconf, 'demo', '0') == '1':
            return True
        
        c = None
        
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'check_auth', None)
                if f and callable(f):
                    c = f(user, password)
                    try:
                        logger.debug("UI module %s authorization result for the user %s: %s" % (mod.get_name(), user, bool(c)))
                    except Exception, exp:  # TODO: find a way to print it
                        pass
                    if c:
                        if isinstance(c, dict):
                            contact_name = c.get('contact_name', '')
                        else:
                            contact_name = c.contact_name
                        logger.debug("Matched contact %s" % contact_name)
                        # No need for other modules
                        break
            except Exception, exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
        
        # Return the authed contact (or None if auth failed)
        return c
    
    
    def get_check_plugin_styles_css_hash(self):
        return self.compiled_css_hash
    
    
    # Some helpers for string/byte handling
    def tob(self, s, enc='utf8'):
        return s.encode(enc) if isinstance(s, unicode) else bytes(s)
    
    
    def _lscmp(self, a, b):
        ''' Compares two strings in a cryptographically safe way:
        Runtime is not affected by length of common prefix. '''
        return not sum(0 if x == y else 1 for x, y in zip(a, b)) and len(a) == len(b)
    
    
    def token_encode(self, data):
        key = self.auth_secret
        ''' Encode and sign a pickle-able object. Return a (byte) string '''
        msg = base64.b64encode(json.dumps(data))
        sig = base64.b64encode(hmac.new(self.tob(key), msg).digest())
        return self.tob('!') + sig + self.tob('?') + msg
    
    
    def token_decode(self, data):
        key = self.auth_secret
        ''' Verify and decode an encoded string. Return an object or None.'''
        data = self.tob(data)
        if self.token_is_encoded(data):
            sig, msg = data.split(self.tob('?'), 1)
            if self._lscmp(sig[1:], base64.b64encode(hmac.new(self.tob(key), msg).digest())):
                try:
                    return json.loads(base64.b64decode(msg))
                except:  # maybe there was a previous valid token with pickle, so this will fail, invalidate it
                    return None
        return ''
    
    
    def token_is_encoded(self, data):
        ''' Return True if the argument looks like a encoded cookie.'''
        return bool(data.startswith(self.tob('!')) and self.tob('?') in data)
    
    
    def get_token(self, uname):
        return self.token_encode(uname)
    
    
    # Authentification:
    # * only the cookie is a valid authentification method (ui conf + pure backend page)
    # * only the token  is NOT valid as iframe won't be happy
    # * if both: must be ok on the value, if not means that the UI is not aware of a cookie change
    def get_user_auth(self):
        # First we look for the user sid so we bail out if it's a false one
        cookie_user_id = self.request.get_cookie("user", secret=self.auth_secret)
        
        case_sensitive = True
        
        # cookie is mandatory, no cookie, no gain unless we allow remote login
        if not cookie_user_id and self.remote_user_enable != '1':
            return self.abort(401, "Your cookie is not valid. Please re-authentify with a valid user.")
        
        # Now look at the token, if present, must be the same as the cookie
        token = self.request.headers.get('X-Shinken-Token', '')
        # Maybe token was not in header, maybe in the get parameters?
        if token == '':
            token64 = self.request.query.get('_token', '')
            try:
                token = base64.b64decode(token64)
            except Exception:
                logger.error("[ui] bad base64 token: %s" % token64)
        if token.startswith('"'):
            token = token[1:]
        if token.endswith('"'):
            token = token[:-1]
        token_user_id = ''
        if token:
            token_user_id = self.token_decode(token)
        
        new_user = None
        if cookie_user_id:
            new_user = self.datamgr.get_contact(cookie_user_id, by_id=True)
        elif self.remote_user_enable == '1':
            # Maybe the user did want to look at Header auth, so look for this case, but only if there is no cookie here
            logger.debug('[Auth] Searching for auth in header [%s]' % self.remote_user_variable)
            if self.remote_user_variable in self.request.headers:
                header_user_name = self.request.headers[self.remote_user_variable]
                case_sensitive = self.remote_user_case_sensitive
                logger.debug('[Auth] Header auth, did found user: [%s] - case_sensitive [%s]' % (header_user_name, case_sensitive))
                if case_sensitive:
                    new_user = self.datamgr.get_contact(header_user_name)
                else:
                    new_user = self.datamgr.get_contact_case_insensitive(header_user_name)
                if not new_user:
                    self.abort(401, 'UI Authentification by header fail')
                
                cookie_user_id = getattr(new_user, 'uuid')
        
        # Still not user name aven after a header lookup? Bail out.
        if not cookie_user_id:
            return self.abort(401, "Your cookie/auth is not valid. Please re-authentify with a valid user.")
        
        # Ok now look at both token and cookie.
        # If the token is present, it must be the same as cookie.
        # Only the front have token so this is a optional test.
        if token and cookie_user_id != token_user_id:
            if not new_user:
                self.abort(401, "Your cookie is not valid. Please re-authentify with a valid user.")
            
            new_user_name = getattr(new_user, 'contact_name', 'User with no name')
            # token to used for furhter request
            new_token = self.get_token(cookie_user_id)
            logger.debug("[Auth] User change detected [%s-%s]" % (new_token, new_user_name))
            
            response.content_type = 'application/json'
            output = {
                "purpose" : "userSwitched",
                "token"   : new_token,
                "username": new_user.contact_name
            }
            self.abort(409, output)
        
        user_auth = self.datamgr.get_contact(cookie_user_id, by_id=True)
        if user_auth:
            logger.debug('[Auth] UI Authentification found: [%s]' % cookie_user_id)
        else:
            logger.debug('[Auth] UI Authentification fail: [%s]' % cookie_user_id)
        return user_auth
    
    
    # Try to got for an element the graphs uris from modules
    # The source variable describes the source of the calling. Are we displaying 
    # graphs for the element detail page (detail), or a widget in the dashboard (dashboard) ?
    def get_graph_uris(self, elt, graphstart, graphend, source='detail'):
        # safe_print("Checking graph uris ", elt.get_full_name())
        
        uris = []
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'get_graph_uris', None)
                # safe_print("Get graph uris ", f, "from", mod.get_name())
                if f and callable(f):
                    r = f(elt, graphstart, graphend, source)
                    uris.extend(r)
            except Exception, exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
        
        # safe_print("Will return", uris)
        # Ok if we got a real contact, and if a module auth it
        return uris
    
    
    def get_common_preference(self, key, default=None):
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'get_ui_common_preference', None)
                if f and callable(f):
                    r = f(key)
                    return r
            except Exception, exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
        return default
    
    
    # Maybe a page want to warn if there is no module that is able to give user preference?
    def has_user_preference_module(self):
        for mod in self.modules_manager.get_internal_instances():
            f = getattr(mod, 'get_ui_user_preference', None)
            if f and callable(f):
                return True
        return False
    
    
    # Try to got for an element the graphs uris from modules
    def get_user_preference(self, user, key, default=None):
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'get_ui_user_preference', None)
                if f and callable(f):
                    r = f(user, key)
                    return r
            except Exception, exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
        return default
    
    
    # Try to got for an element the graphs uris from modules
    def set_user_preference(self, user, key, value):
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'set_ui_user_preference', None)
                if f and callable(f):
                    f(user, key, value)
            except Exception, exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
    
    
    def set_common_preference(self, key, value):
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'set_ui_common_preference', None)
                if f and callable(f):
                    f(key, value)
            except Exception, exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
                
                # end of all modules
    
    
    # For a specific place like dashboard we return widget lists
    def get_widgets_for(self, place):
        return self.widgets.get(place, [])
    
    
    # Will get all label/uri for external UI like PNP or NagVis
    def get_external_ui_link(self):
        lst = []
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'get_external_ui_link', None)
                if f and callable(f):
                    r = f()
                    lst.append(r)
            except Exception, exp:
                logger.warning("[%s] Warning: The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
        return lst
    
    
    def insert_template(self, tpl_name, d):
        try:
            r = template(tpl_name, d)
        except Exception, exp:
            pass
    
    
    def get_webui_port(self):
        port = self.port
        return port
    
    
    def get_skonf_port(self):
        port = self.http_port
        return port
    
    
    def get_skonf_active_state(self):
        state = self.show_skonf
        return state
    
    
    def get_first_tag(self, tags):
        for t in tags:
            if t in self.img_tags:
                return t
        return None
    
    
    def is_check_uuid(self, uuid):
        return '-' in uuid
    
    
    def get_name_from_uuid(self, uuid):
        r = None
        if '-' in uuid:
            check = datamgr.get_service_by_uuid(uuid)
            if check:
                r = [check.host.host_name, check.service_description]
        else:
            host = datamgr.get_host_by_uuid(uuid)
            if host:
                r = [host.host_name]
        
        if not r:
            return self.abort(404, 'unknow uuid [%s]' % uuid, True)
        return r
    
    
    # Traduction call
    # TODO: too much hard coded paths there, find a better way
    def _(self, s):
        # First find lang of the current call, and by default take the global lang parameter
        lang = self.request.query.get('lang', self.lang)
        d = self.langs.get(lang, None)
        # If missing, load the file
        if d is None:
            pth = os.path.join(self.langs_path, lang + '.js')
            if not os.path.exists(pth):
                logger.error('Cannot load the lang file %s: no such file' % pth)
                return ''
            f = open(pth, 'r')
            buf = f.read()
            f.close()
            lines = buf.splitlines()
            new_lines = []
            for line in lines:
                line = line.strip()
                if line.startswith('//'):
                    continue
                if line.startswith('var lang'):
                    line = '{'
                if line.startswith('};'):
                    line = '}'
                new_lines.append(line)
            buf = '\n'.join(new_lines)
            o = json.loads(buf)
            self.langs[lang] = o
        o = self.langs[lang]
        elts = [e.strip() for e in s.split('.') if e.strip()]
        for e in elts:
            o = o.get(e, None)
            if o is None:
                logger.error('Traduction: cannot find %s in the lang %s' % (s, lang))
                return ''
        return o
    
    
    def get_contact_groups(self, contact):
        # contactgroups = []
        
        # if contact.contactgroups:
        #    for group in contact.contactgroups.split(','):
        #        contactgroup = self.datamgr.get_contactgroup(group)
        #        contactgroups.append(contactgroup)
        
        # for contactgroup in self.datamgr.get_contactgroups():
        #    if (contact in contactgroup.members) and not (contactgroup in contactgroups):
        #        contactgroups.append(contactgroup)
        
        return contact.contactgroups
    
    
    def _get_mongo(self):
        for inst in self.modules_manager.get_all_alive_instances():
            if inst.properties['type'] == 'mongodb':
                return inst
        return None
    
    
    # May raise exceptions, must be try/excepted when used
    def save_versionned_object(self, col_name, uid_key, item_uuid, data, create_object=False):
        logger.debug("Saving %s object of uuid %s" % (col_name, item_uuid))
        prev_version = 0
        inst = self._get_mongo()
        db = inst.db
        col = getattr(db, col_name)
        if col:
            prev_data = col.find_one({uid_key: item_uuid})
            if prev_data:
                prev_version = int(prev_data.get('saveVersion', 0))
            elif not create_object:
                raise LookupError("%s object not found in database : %s" % (col_name, item_uuid))
        
        data_update = data.get('update', False)
        data.pop('update', None)
        if data_update and prev_version < int(data.get('saveVersion', 0)):
            col.update({uid_key: item_uuid}, data, upsert=True)
        elif not data_update and prev_version == int(data.get('saveVersion', 0)):
            col.update({uid_key: item_uuid}, data, upsert=True)
        else:
            logger.info("Save %s object of uuid %s aborted because submitted version is older. (current: %s. submitted : %s)" % (col_name, item_uuid, prev_version, data.get('saveVersion', 0)))
            raise ValueError(self._('error.save_versionned_object'))
    
    
    def get_screen_collection(self, type):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        if type not in ['hive', 'list', 'dashboard']:
            return None
        col = getattr(db, type)
        return col
    
    
    def get_hive_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'hive')
        return col
    
    
    def get_dashboard_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'dashboard')
        return col
    
    
    def get_list_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'list')
        return col
    
    
    def get_share_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'share')
        return col
    
    
    def get_user_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'user')
        return col
    
    
    def get_tiles_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'tiles')
        return col
