#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

""" This python module contains the class BaseModule
that shinken modules will subclass
"""

import inspect
import os
import signal
import threading
import time
import traceback
import uuid
from multiprocessing import Queue, Process
from Queue import Empty

from shinken.misc.type_hint import List, Any
from shinken.objects.module import Module as ShinkenModuleDefinition

from .basesubprocess import BaseSubProcess
from .log import logger as _logger, PART_INITIALISATION, LoggerFactory, get_chapter_string
from .util import mem_wait_for_fork_possible

# The properties dict defines what the module can do and if it's an external module or not.
properties = {
    'type'    : None,  # name of the module type ; to distinguish between them:
    'external': True,  # is the module "external" (external means here a daemon module)?
    'phases'  : ['configuration', 'late_configuration', 'running', 'retention'],  # Possible configuration phases where the module is involved
}

ARG_SEPARATOR = ':::'

MODULE_INFO_CHAPTER = get_chapter_string('MODULE-INFO')


class UnknownCommandException(Exception):
    pass


class ModulePhases(object):
    CONFIGURATION = 1
    LATE_CONFIGURATION = 2
    RUNNING = 4
    RETENTION = 8


class ModuleState(object):
    OK = 'OK'
    WARNING = 'WARNING'
    CRITICAL = 'CRITICAL'
    FATAL = 'FATAL'


class SOURCE_STATE(object):
    PENDING = 'PENDING'
    RUNNING = 'RUNNING'
    OK = 'OK'
    CRITICAL = 'CRITICAL'
    WARNING = 'WARNING'
    UNKNOWN = 'UNKNOWN'
    NOT_CONFIGURED = 'not-configured'
    NOT_CONFIGURED_BEFORE_IMPORT = 'not-configured-before-import'
    READY_FOR_IMPORT = 'ready-for-import'
    NEVER_IMPORT = 'never-import'
    DIFFERENCE_COMPUTING = 'difference-computing'
    
    STATES_ORDER = (
        NOT_CONFIGURED_BEFORE_IMPORT,
        NOT_CONFIGURED,
        CRITICAL,
        WARNING,
        UNKNOWN,
        OK,
        PENDING,
        RUNNING,
        READY_FOR_IMPORT,
        NEVER_IMPORT,
        DIFFERENCE_COMPUTING,
    )
    
    
    @staticmethod
    def get_best_of_two_states(state1, state2):
        for s in SOURCE_STATE.STATES_ORDER:
            if s in (state1, state2):
                return s
        raise Exception('state [%s] [%s] not in STATES_ORDER' % (state1, state2))


# Object that will be send by the daemon to the module with a specific call (and maybe args)
# Will be used to match respond and look ifthe answer have the same uuid than the request
class ToModuleCommandRequest(object):
    def __init__(self, command_name, args):
        str_send_command = command_name
        if args:
            str_send_command = [command_name]
            str_send_command.extend(args)
            str_send_command = ARG_SEPARATOR.join(str_send_command)
        self._command = str_send_command
        self._uuid = uuid.uuid4().hex
    
    
    def get_command(self):
        return self._command
    
    
    def get_uuid(self):
        return self._uuid
    
    
    def create_respond(self, result_payload):
        respond = FromModuleCommandRespond(self._uuid, result_payload)
        return respond
    
    
    def __str__(self):
        return u'ToModuleCommandRequest[%s-%s]' % (self._uuid, self._command)


# This will be create by a request (with the good uuid, etc) to give back
# a result from the module. Will be used by the original request to match if the
# respond match the request uuid (can be problem in queues)
class FromModuleCommandRespond(object):
    def __init__(self, request_uuid, result_payload):
        self._uuid = request_uuid
        self._payload = result_payload
    
    
    def do_match_request(self, request):
        request_uuid = request.get_uuid()
        return self._uuid == request_uuid
    
    
    def get_payload(self):
        return self._payload


class BaseModule(BaseSubProcess):
    """This is the base class for the shinken modules.
    Modules can be used by the different shinken daemons/services
    for different tasks.
    Example of task that a shinken module can do:
     - load additional configuration objects.
     - recurrently save hosts/services status/perfdata
       information in different format.
     - ...
     """
    
    is_worker_based = False
    
    
    def __init__(self, mod_conf):
        # type: (ShinkenModuleDefinition) -> None
        """Instantiate a new module.
        There can be many instance of the same type. 'mod_conf' is module configuration object for this new module instance.
        """
        self.myconf = mod_conf
        super(BaseModule, self).__init__(mod_conf.get_name())
        
        self.logger = LoggerFactory.get_logger(mod_conf.get_name())
        
        # We can have sub modules
        self.modules = getattr(mod_conf, 'modules', [])
        self.props = mod_conf.properties.copy()
        self.module_type = mod_conf.properties['type']
        # TODO: choose between 'props' or 'properties'..
        self.interrupted = False
        self.properties = self.props
        self.is_external = self.props.get('external', False)
        # though a module defined with no phase is quite useless .
        self.phases = self.props.get('phases', [])
        self.phases.append(None)
        # the queue the module will receive data to manage
        self.to_q = None
        # the queue the module will put its result data
        self.from_module_to_main_daemon_queue = None
        self.commands_from_q = None
        self.commands_to_q = None
        self.process = None
        
        self.init_try = 1
        self.last_init_try = 0.0
        
        # We want to know where we are load from? (broker, scheduler, etc)
        self.loaded_into = 'unknown'
        self.daemon_display_name = 'UNSET'
        
        # Will be created when called by create queues
        self._queue_factory = None
        self.manager = None
        
        # For sub process clean, we will have to keep a trace of all our queues
        self._all_queues = []
        
        # Lock for commands from different threads racing for the result
        self.send_command_lock = threading.Lock()
        
        self.fatal_error_has_been_managed = False
    
    
    def get_internal_state(self):
        return ModuleState.OK
    
    
    def init(self):
        """Handle this module "post" init ; just before it'll be started.
        Like just open necessaries file(s), database(s),
        or whatever the module will need.
        """
        pass
    
    
    def stop_all(self):
        try:
            self.logger.info('Starting stop module:[%s]' % self.name)
            try:
                if hasattr(self, 'quit') and callable(self.quit):
                    self.quit()
            except Exception as e:
                self.logger.error('call quit on module:[%s] fail with error:[%s]' % (self.name, e))
                self.logger.print_stack()
            
            self.stop_process()
            
            # If external, clean its queues
            if self.is_external or self.is_worker_based:
                self.clear_queues()
                self.kill_manager()
            
            self.logger.info('Stop module:[%s] DONE' % self.name)
        except:
            pass
    
    
    def get_process_name(self):
        _daemon_display_name = self.father_name
        if not _daemon_display_name or _daemon_display_name == 'NO-DISPLAY-NAME':
            _daemon_display_name = 'shinken-%s' % self.loaded_into
        return "%s [ - Module: %s ]%s" % (_daemon_display_name, self.name, self._get_worker_display_name())
    
    
    def set_loaded_into(self, daemon_name):
        self.loaded_into = daemon_name
    
    
    def kill_manager(self):
        self.logger.info('Asserting the queue manager process is stopped.')
        if self.manager is None:
            return
        process = getattr(self.manager, '_process', None)
        if process:
            pid = process.pid
            try:
                self.logger.debug('kill_manager:: %s -5' % pid)
                process.terminate()
                process.join(0.1)
                
                start_wait_is_dead = time.time()
                while process.is_alive() and (time.time() - start_wait_is_dead) < 3:
                    process.join(0.1)
                
                if process.is_alive():
                    self.logger.debug('kill_manager:: %s -9' % pid)
                    os.kill(process.pid, signal.SIGKILL)
                    self.logger.debug('kill_manager:: %s wait' % pid)
            except Exception as e:
                self.logger.warning('kill_manager:: fail to stop manager with error e:[%s]' % e)
            
            try:
                self.logger.debug('kill_manager:: %s wait 2' % pid)
                process.join()
            except:
                self.logger.warning('kill_manager:: fail to stop manager with error e:[%s]' % e)
        else:
            self.logger.debug('kill_manager:: classic shutdown')
            self.manager.shutdown()
        self.logger.info('The queue manager process is now stopped.')
        self.manager = None
        self.logger.info('The queue manager process is now clean.')
    
    
    def _get_new_queue(self, manager):
        # If no Manager() object, go with classic Queue()
        if not manager:
            return Queue()
        else:
            return manager.Queue()
    
    
    def queue_factory(self, name):
        queue = self.manager.Queue()
        queue.name = name
        self._all_queues.append(queue)
        self.logger.debug(PART_INITIALISATION, 'Creating a exchange queue [%s] with the module. Queue address (for debug only): %s ' % (name, queue))
        return queue
    
    
    def create_queues(self, manager_factory=None):
        """The manager is None on android, but a true Manager() elsewhere
        Create the shared queues that will be used by shinken daemon
        process and this module process.
        But clear queues if they were already set before recreating new one.
        """
        # We always kill the manager, to void it's queue and be sure we did release the memory
        if self.manager:
            self.kill_manager()
        
        if not self.manager:
            self.logger.info(PART_INITIALISATION, 'Creating a queue manager process for the module.')
            self.manager = manager_factory('- Module: %s' % self.get_name())
        self.clear_queues()
        
        self.from_module_to_main_daemon_queue = self.queue_factory('from_module_to_main_daemon_queue')
        self.to_q = self.queue_factory('to_q')
        self.commands_from_q = self.queue_factory('commands_from_q')
        self.commands_to_q = self.queue_factory('commands_to_q')
    
    
    # Release the resources associated to the queues of this instance
    def clear_queues(self):
        from multiprocessing import util as multiprocessing_util
        to_del = []
        for queue in self._all_queues:
            if queue is None:
                return
            # HACK: the manager can be dead, and so the queue sockets will be unavailable, and will
            #       40s timeout if we try to talk to then, so must remove them in the multiprocessing clean pass
            # WARNING: multiprocessing_util._afterfork_registry can have parallel access, so loop until we have keys
            registry_keys = None
            watch_dog = time.time()
            while registry_keys is None:
                if time.time() - watch_dog > 10:
                    registry_keys = []
                    break
                try:
                    registry_keys = multiprocessing_util._afterfork_registry.keys()
                except RuntimeError:  # oups, should retry
                    registry_keys = []
                    pass
            for k in registry_keys:
                obj = multiprocessing_util._afterfork_registry.get(k, None)
                if obj is None:  # did disappears
                    continue
                if obj == queue:
                    self.logger.debug('Disabling old [%s] queue. Queue address: %s (for debug only)' % (queue.name, queue))
                    to_del.append(k)
            
            
            def after_fork_skip(_queue):
                pass
            
            
            queue._incref = after_fork_skip
            queue._token.address = None
            queue._manager = None
        for k in to_del:
            try:
                del multiprocessing_util._afterfork_registry[k]
            except KeyError:  # already removed
                pass
        # And AFTER clean the afterfork_registry we can drop our pointer, and so maybe the registry as weakref will lose it
        self.to_q = self.from_module_to_main_daemon_queue = self.commands_from_q = self.commands_to_q = None
        self._all_queues = []
    
    
    # Basic modules do not handle workers
    def start_workers(self, daemon_display_name='UNSET'):
        return
    
    
    # Start this module process if it's external. if not -> donothing
    def start(self, http_daemon=None, daemon_display_name='UNSET'):
        self.daemon_display_name = daemon_display_name
        self.update_father(daemon_display_name)
        if not self.is_external:
            return
        self.stop_process()
        self.logger.info(PART_INITIALISATION, "Starting external process")
        is_fork_possible = mem_wait_for_fork_possible('module %s' % self.name, fast_error=True)
        if not is_fork_possible:
            raise Exception('[PERFORMANCE][MEMORY] Cannot start the module process as there is not enough memory for this')
        p = Process(target=self._main, args=())
        
        # Under windows we should not call start() on an object that got
        # its process as object, so we remove it and we set it in a earlier
        # start
        try:
            del self.properties['process']
        except:
            pass
        
        p.start()
        # We save the process data AFTER the fork()
        self.process = p
        self.properties['process'] = p  # TODO: temporary
        self.logger.info(PART_INITIALISATION, "module process is started as pid=%d" % p.pid)
    
    
    # Sometime terminate() is not enough, we must "help" external modules to die...
    @staticmethod
    def __kill(process):
        if os.name == 'nt':
            process.terminate()
        else:
            # Ok, let him 1 second before really KILL IT
            os.kill(process.pid, signal.SIGTERM)
            time.sleep(1)
            # You do not let me another choice guy...
            try:
                if process.is_alive():
                    os.kill(process.pid, signal.SIGKILL)
            except AssertionError:
                try:
                    os.kill(process.pid, signal.SIGKILL)
                except OSError:
                    pass
    
    
    def is_manager_alive(self):
        if not self.manager:
            return False
        process = getattr(self.manager, '_process', None)
        if not process:
            return False
        return process.is_alive()
    
    
    def is_alive(self):
        if self.get_internal_state() == ModuleState.FATAL:
            return False
        
        if not self.is_external:
            return True
        
        try:
            return self.process.is_alive() and self.is_manager_alive()
        except AssertionError:  # maybe the process did go daemon and so we are no more its parent. Kill it any way
            self.logger.warning('Seems that the manager of the module is in a bad state')
            return False
    
    
    def _do_stop_process(self, process):
        self.logger.info("Stopping module process pid=%s " % process.pid)
        try:
            process.terminate()
        except OSError:  # Maybe the process was already gone, like kill by the system with a OOM or an angry admin
            pass
        try:
            process.join(timeout=1)
        except OSError:  # Maybe the process was already gone too, but i prefer to be sure to try a join()
            pass
        if process.is_alive():
            self.logger.info("The sub-process [%s] is still alive, I help it to die" % process.pid)
            self.__kill(process)
    
    
    # Request the module process to stop and release it
    def stop_process(self):
        if self.process:
            self._do_stop_process(self.process)
            self.process = None
    
    
    def get_module_info(self):
        pass
    
    
    def get_submodule_states(self):
        manager = getattr(self, 'modules_manager', None)
        if manager:
            return manager.get_modules_states()
    
    
    def _send_command(self, command_name, args=None):
        # type: (str, List[str]) -> Any
        
        if not (self.is_external and self.commands_to_q):
            raise Exception('%s Module %s is not an external module.' % (MODULE_INFO_CHAPTER, self.get_name()))
        
        if self.process is None:
            raise Exception('%s Module %s process is currently starting.' % (MODULE_INFO_CHAPTER, self.get_name()))
        elif not self.process.is_alive():
            raise Exception('%s Module %s process is currently down and is scheduled for a restart.' % (MODULE_INFO_CHAPTER, self.get_name()))
        
        command_exist = bool(getattr(self, command_name, None))
        if not command_exist:
            raise UnknownCommandException('%s Command %s was unknown for the module %s.' % (MODULE_INFO_CHAPTER, command_name, self.get_name()))
        
        with self.send_command_lock:
            try:
                request = ToModuleCommandRequest(command_name, args)
                timeout = 2
                retry_count = 0
                before = time.time()
                self.commands_to_q.put(request)
                while True:  # will timeout at max 2s because the commands request are block and we are the only one asking for it
                    try:
                        response = self.commands_from_q.get(block=True, timeout=timeout)  # type: FromModuleCommandRespond
                    except Empty:
                        # Empty? Means a timeout
                        retry_count += 1
                        if retry_count == 1:
                            self.logger.warning('%s The command call %s for module did timeout %s (%ss). We will retry one time.' % (MODULE_INFO_CHAPTER, command_name, self.get_name(), timeout))
                            continue
                        # Ok still an error? need to give a real error about the timeout
                        raise
                    if not response.do_match_request(request):
                        self.logger.error('%s The command call %s was called but another respond was present. Retrying.' % (MODULE_INFO_CHAPTER, command_name))
                        continue
                    self.logger.debug('%s The command call %s was executed by the module %s in %.3fs' % (MODULE_INFO_CHAPTER, command_name, self.get_name(), time.time() - before))
                    payload = response.get_payload()
                    return payload
            
            # Empty queue means a timeout
            except Empty:
                _message = '%s Fail to send command call %s for module %s because the module did timeout (%ss)' % (MODULE_INFO_CHAPTER, command_name, self.get_name(), timeout)
                self.logger.error(_message)
                raise Exception(_message)
            
            # Another exception? It's not good and must be shown as a stack
            except Exception as e:
                self.logger.print_stack()
                _message = '%s Fail to send command call %s for module %s because of a unknown error %s' % (MODULE_INFO_CHAPTER, command_name, self.get_name(), e.message)
                self.logger.error(_message)
                raise Exception(_message)
    
    
    def get_state(self):
        status = {"status": ModuleState.OK, "output": "OK"}
        
        if self.is_external and self.commands_to_q:
            if self.process is None:
                return {'status': ModuleState.CRITICAL, 'output': 'Module process is currently starting.'}
            elif not self.process.is_alive():
                return {'status': ModuleState.CRITICAL, 'output': 'Module process is currently down and is scheduled for a restart.'}
            
            try:
                module_info = self._send_command('get_module_info')
                if module_info:
                    status.update(module_info)
            except Exception as e:
                return {'status': ModuleState.CRITICAL, 'output': e.message}
            
            if self.modules:
                try:
                    module_info = self._send_command('get_submodule_states')
                    if module_info:
                        status.update(module_info)
                except Exception:
                    status.update({'status': ModuleState.WARNING, 'output': 'Module is running, but does not provide submodule states.'})
        return status
    
    
    def get_name(self):
        return self.name
    
    
    def has(self, prop):
        """The classic has: do we have a prop or not?"""
        return hasattr(self, prop)
    
    
    # For in scheduler modules, we will not send all broks to external
    # modules, only what they really want
    def want_brok(self, b):
        return True
    
    
    def manage_brok(self, brok):
        """Request the module to manage the given brok.
        There a lot of different possible broks to manage.
        """
        manage = getattr(self, 'manage_' + brok.type + '_brok', None)
        if manage:
            # Be sure the brok is prepared before call it
            brok.prepare()
            return manage(brok)
    
    
    def do_stop(self):
        """Called just before the module will exit
        Put in this method all you need to cleanly
        release all open resources used by your module
        """
        pass
    
    
    def do_loop_turn(self):
        """For external modules only:
        implement in this method the body of you main loop
        """
        raise NotImplementedError()
    
    
    # Set by worker based modules to display their worker in the process list
    def _get_worker_display_name(self):
        return ''
    
    
    def _get_and_execute_command_from_master(self):
        while True:
            cmd_and_arg = ''
            try:  # NOTE: this thread is not allowed to die
                
                # Will block so we don't hammer cpu
                try:
                    request = self.commands_to_q.get(block=True, timeout=1)  # type: ToModuleCommandRequest
                except:
                    request = None
                # Nothing in the queue, just loop
                if request is None:
                    continue
                cmd_and_arg = request.get_command()
                _split = cmd_and_arg.split(ARG_SEPARATOR)
                cmd = _split[0]
                arg = _split[1:]
                f = getattr(self, cmd, None)
                if callable(f):
                    self.logger.debug('%s [%s:%s] Executing command %s with param %s' % (MODULE_INFO_CHAPTER, self.name, os.getpid(), cmd, arg))
                    arg_spec = inspect.getargspec(f)
                    if arg and len(arg_spec.args) == len(arg):
                        result = f(*arg)
                    else:
                        result = f()
                    respond = request.create_respond(result)
                    self.commands_from_q.put(respond)
                else:
                    self.logger.warning('%s [%s:%s] Received unknown command %s from father process !' % (MODULE_INFO_CHAPTER, self.name, os.getpid(), cmd))
            except:
                self.logger.error('%s Our father process did send us the command (%s) that did fail: %s' % (MODULE_INFO_CHAPTER, cmd_and_arg, traceback.format_exc()))
                time.sleep(0.01)  # if we crash in loop, do not hammer CPU
    
    
    def _start_sub_process_threads(self):
        super(BaseModule, self)._start_sub_process_threads()
        
        thr = threading.Thread(target=self._get_and_execute_command_from_master, name='command-from-master')
        thr.daemon = True
        thr.start()
    
    
    # module 'main' method. Only used by external modules.
    def _main(self):
        self._sub_process_common_warmup()
        _logger.set_name(self.name)
        
        self.logger.info(PART_INITIALISATION, 'Worker now running..')
        try:
            # Will block here!
            self.logger.debug(PART_INITIALISATION, 'go main')
            self.main()
            self.do_stop()
            self.logger.info('exiting now..')
        except Exception as exp:
            self.logger.error('Exit with error : %s' % exp)
            self.logger.print_stack()
    
    
    # TODO: apparently some modules would uses 'work' as the main method??
    work = _main
    
    
    def _internal_get_raw_stats(self, param=''):
        _internal_get_raw_stats = {}
        modules_manager = getattr(self, 'modules_manager', None)
        if not modules_manager:
            return _internal_get_raw_stats
        
        for module_name, module in modules_manager.get_all_instances_with_name():
            module_get_raw_stats = getattr(module, 'get_raw_stats', None)
            result = {}
            if module_get_raw_stats:
                arg_spec = inspect.getargspec(module_get_raw_stats)
                if len(arg_spec.args) > 1:
                    result = module_get_raw_stats(param)
                else:
                    result = module_get_raw_stats()
            
            module_type = module.myconf.module_type
            t = _internal_get_raw_stats.get(module_type, {})
            _internal_get_raw_stats[module_type] = t
            _internal_get_raw_stats[module_type][module_name] = result
        return _internal_get_raw_stats
    
    
    def get_raw_stats(self, param=''):
        if self.is_external and self.commands_to_q:
            raw_stats = {
                'modules': {}
            }
            try:
                if param:
                    param = [param]
                else:
                    param = None
                
                raw_stats['modules'] = self._send_command('_internal_get_raw_stats', param)
                return raw_stats
            except Exception as e:
                self.logger.error('Fail in get_raw_stats with error : [%s]' % getattr(e, 'message', str(e)))
                self.logger.print_stack()
