#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.
import codecs
import itertools
import os
from ctypes import c_int
from logging.handlers import TimedRotatingFileHandler
from multiprocessing import RLock, Value

try:
    from pwd import getpwnam
    from grp import getgrnam
except ImportError as e:
    # This import is needed only to change owner on linux, not used on windows
    pass

# Windows is special, as usual, because we will have to set filehandler as non inheritable
if os.name == 'nt':
    import msvcrt
    import win32api
    import win32con

####################################
# END MONKEY PATCH
####################################

import os
import time
import logging
import pprint
import sys
import traceback
from difflib import Differ
from io import StringIO
from logging import Formatter, StreamHandler, DEBUG, WARNING, INFO, CRITICAL, ERROR

from brok import Brok
from datetime import datetime

try:
    from shinken.misc.termcolor import cprint
except (SyntaxError, ImportError), exp:
    # Outch can't import a cprint, do a simple print
    def cprint(s, color='', end=''):
        print s

human_timestamp_log = True  # by default, have human format date

human_date_format = '%Y-%m-%d %H:%M:%S'  # [2017-03-02 13:38:15]

defaultFormatter_u = Formatter(u'[%(created)i] %(levelname)-7s: %(message)s')
defaultFormatter_named_u = Formatter(u'[%(created)i] %(levelname)-7s: [%(name)s] %(message)s')
humanFormatter_u = Formatter(u'[%(asctime)s] %(levelname)-7s: %(message)s', human_date_format)
humanFormatter_named_u = Formatter(u'[%(asctime)s] %(levelname)-7s: [%(name)s] %(message)s', human_date_format)

defaultFormatter = Formatter('[%(created)i] %(levelname)-7s: %(message)s')
defaultFormatter_named = Formatter('[%(created)i] %(levelname)-7s: [%(name)s] %(message)s')
humanFormatter = Formatter('[%(asctime)s] %(levelname)-7s: %(message)s', human_date_format)
humanFormatter_named = Formatter('[%(asctime)s] %(levelname)-7s: [%(name)s] %(message)s', human_date_format)

COLORS = {'DEBUG': 'cyan', 'INFO': 'magenta', 'WARNING': 'yellow', 'CRITICAL': 'red', 'ERROR': 'red'}

PERF_LOG_LEVEL = DEBUG
PERF_LOG_MIN_TIME = 0.1
PERF_LOG_WARN_TIME = 1

LOG_CHAPTER_SIZE = 16
LOG_SECTION_SIZE = 22

LONG_FLUSH_LIMIT = 1  # if flush is longer than 1s, warn about it
LONG_FLUSH_KEEP_DURATION = 60  # keep the long flush during a 1min delay
DEFAULT_LONG_FLUSH_STATS = {'is_too_long': False, 'write_duration': 0.0, 'log_path': ''}


def get_chapter_string(chapter):
    return format_part(chapter, LOG_CHAPTER_SIZE)


def get_section_string(chapter):
    return format_part(chapter, LOG_SECTION_SIZE)


def format_part(section_name, section_size):
    s_format = '[%%-%ds]' % section_size
    return s_format % section_name


class FixedTimedRotatingFileHandler(TimedRotatingFileHandler):
    def __init__(self, *args, **kwargs):
        # NOTE: cannot user super() as python 2.6 Handler is not new-style class
        TimedRotatingFileHandler.__init__(self, *args, **kwargs)
        self._current_pid = os.getpid()
        ####################################
        # WARNING !! MONKEY PATCH for TimedRotatingFileHandler
        # TimedRotatingFileHandler doesn't support multiprocessing
        # SEE SEF-3518
        ####################################
        self.roll_over_version = Value(c_int, 0)
        self._rollover_lock = RLock()
        self.delay = 0
        self.my_roll_over_version = 0
        
        self._last_big_flush_date = 0
        self._last_big_flush = 0.0
    
    
    def get_long_flush_stats(self):
        now = int(time.time())
        if self._last_big_flush_date < now - LONG_FLUSH_KEEP_DURATION:
            return DEFAULT_LONG_FLUSH_STATS
        return {'is_too_long': True, 'write_duration': self._last_big_flush, 'log_path': self._get_current_file_name()}
    
    
    def _get_current_file_name(self):
        if os.name == 'nt':
            return "%s.%s" % (self.baseFilename, time.strftime("%Y-%m-%d", time.localtime(time.time())))
        else:
            return self.baseFilename
    
    
    def flush(self):
        before = time.time()
        # NOTE: cannot user super() as python 2.6 Handler is not new-style class
        TimedRotatingFileHandler.flush(self)
        flush_time = time.time() - before
        if flush_time < LONG_FLUSH_LIMIT:
            return
        now = int(time.time())
        
        # Ok a big one, is the current still ok?
        if self._last_big_flush_date < now - LONG_FLUSH_KEEP_DURATION:
            self._last_big_flush_date = 0
            self._last_big_flush = 0.0
        
        if flush_time > self._last_big_flush:
            # older or lower, update it
            self._last_big_flush_date = now
            self._last_big_flush = flush_time
            _prefix = '[%s] WARNING : [ LOGGER ]' % time.strftime(human_date_format)
            
            log_msg = '[ WRITING ] The log write time is very high (%.2fs). Please look at your log disk performance.' % (flush_time)
            for msg in ('', '-' * 100, log_msg, '-' * 100, ''):
                log_line = '%s %s\n' % (_prefix, msg)
                self.stream.write(log_line)
                cprint(log_line)  # also log in debug
    
    
    def _open(self):
        """
        Open the current base file with the (original) mode and encoding.
        Return the resulting stream.
        """
        current_file_name = self._get_current_file_name()
        if getattr(self, 'encoding', None) is None:  # NOTE: if use inside apache, self.encoding is missing
            stream = open(current_file_name, self.mode)
        else:
            stream = codecs.open(current_file_name, self.mode, self.encoding)
        
        if os.name != 'nt':
            # We need to change the current_file owner :
            os.chown(current_file_name, getpwnam('shinken').pw_uid, getgrnam('shinken').gr_gid)
        
        if os.name == 'nt':
            # NOTE: on windows we do not want such file to be inherit if we are fork() because the logger object
            # won't be available
            fd = stream.fileno()  # The log handler file descriptor
            fh = msvcrt.get_osfhandle(fd)  # The actual windows handler
            win32api.SetHandleInformation(fh, win32con.HANDLE_FLAG_INHERIT, 0)  # Disable inheritance
        return stream
    
    
    def doRollover(self):
        with self._rollover_lock:
            if self.stream:
                self.stream.close()
                self.stream = None
            # get the time that this sequence started at and make it a TimeTuple
            currentTime = int(time.time())
            dstNow = time.localtime(currentTime)[-1]
            t = self.rolloverAt - self.interval
            if self.utc:
                timeTuple = time.gmtime(t)
            else:
                timeTuple = time.localtime(t)
                dstThen = timeTuple[-1]
                if dstNow != dstThen:
                    if dstNow:
                        addend = 3600
                    else:
                        addend = -3600
                    timeTuple = time.localtime(t + addend)
            if self.my_roll_over_version >= self.roll_over_version.value:
                if os.name != 'nt':
                    dfn = self.baseFilename + "." + time.strftime(self.suffix, timeTuple)
                    if os.path.exists(dfn):
                        os.remove(dfn)
                    # Issue 18940: A file may not have been created if delay is True.
                    current_file_name = self._get_current_file_name()
                    if os.path.exists(current_file_name):
                        os.rename(current_file_name, dfn)
                if self.backupCount > 0:
                    for s in self.getFilesToDelete():
                        os.remove(s)
                self.roll_over_version.value += 1
            if not self.delay:
                self.stream = self._open()
            newRolloverAt = self.computeRollover(currentTime)
            while newRolloverAt <= currentTime:
                newRolloverAt = newRolloverAt + self.interval
            # If DST changes and midnight or weekly rollover, adjust for this.
            if (self.when == 'MIDNIGHT' or self.when.startswith('W')) and not self.utc:
                dstAtRollover = time.localtime(newRolloverAt)[-1]
                if dstNow != dstAtRollover:
                    if not dstNow:  # DST kicks in before next rollover, so we need to deduct an hour
                        addend = -3600
                    else:  # DST bows out before next rollover, so we need to add an hour
                        addend = 3600
                    newRolloverAt += addend
            self.rolloverAt = newRolloverAt
            self.my_roll_over_version += 1
    
    
    # Important: we do manage the case that the handler is called from another PID after a fork()
    # and then recreate the lock because if lock, won't be ever release from
    # the other pid-thread
    def handle(self, record):
        cur_pid = os.getpid()
        if cur_pid != self._current_pid:
            self.createLock()
            self._current_pid = cur_pid
        # NOTE: cannot user super() as python 2.6 Handler is not new-style class
        TimedRotatingFileHandler.handle(self, record)


class ColorStreamHandler(StreamHandler):
    def __init__(self, *args, **kwargs):
        # NOTE: cannot user super() as python 2.6 Handler is not new-style class
        StreamHandler.__init__(self, *args, **kwargs)
        self._current_pid = os.getpid()
    
    
    def _set_local_formatter(self, encoding='unicode'):
        if encoding == 'unicode':
            if getattr(self, 'name', None) is not None:
                self.setFormatter(human_timestamp_log and humanFormatter_named_u or defaultFormatter_named_u)
            else:
                self.setFormatter(human_timestamp_log and humanFormatter_u or defaultFormatter_u)
        else:
            if getattr(self, 'name', None) is not None:
                self.setFormatter(human_timestamp_log and humanFormatter_named or defaultFormatter_named)
            else:
                self.setFormatter(human_timestamp_log and humanFormatter or defaultFormatter)
    
    
    # Important: we do manage the case that the handler is called from another PID after a fork()
    # and then recreate the lock because if lock, won't be ever release from
    # the other pid-thread
    def handle(self, record):
        cur_pid = os.getpid()
        if cur_pid != self._current_pid:
            self.createLock()
            self._current_pid = cur_pid
        # NOTE: cannot user super() as python 2.6 Handler is not new-style class
        StreamHandler.handle(self, record)
    
    
    def emit(self, record):
        try:
            msg = self.format(record)
        except UnicodeDecodeError:
            self._set_local_formatter(encoding='str')
            msg = self.format(record)
        except UnicodeEncodeError:
            self._set_local_formatter(encoding='unicode')
        
        try:
            if isinstance(msg, unicode):
                encoded_msg = msg.encode('utf8', 'ignore')
            else:
                encoded_msg = msg
        except UnicodeEncodeError:
            encoded_msg = msg
        except:
            self.handleError(record)
        finally:
            cprint(encoded_msg, COLORS.get(record.levelname, 'white'))


class Log(logging.Logger):
    """
    Shinken logger class, wrapping access to Python logging standard library.
    See : https://docs.python.org/2/howto/logging.html#logging-flow for more detail about
    how log are handled
    """
    
    
    def __init__(self, name="Shinken", level=DEBUG, do_log_parameters=False):
        logging.Logger.__init__(self, name, level)
        self.set_name(name)
        self.do_log_parameters = do_log_parameters
        self.set_human_format(on=True)  # activate by default human format
        self.log_file_path = None
        self._my_daemon = None
    
    
    def setLevel(self, level):
        """
        Set level of logger and handlers.
        The logger need the lowest level (see link above)
        """
        if not isinstance(level, int):
            level = getattr(logging, level, None)
            if not level or not isinstance(level, int):
                raise TypeError('log level must be an integer')
        
        self.level = level
        # Only set level to file and/or console handler
        for handler in self.handlers:
            handler.setLevel(level)
    
    
    def set_name(self, name):
        name = '%-15s' % name
        self.name = name
    
    
    def load_obj(self, _object, _name=None):
        """
        We load the object where we will put log broks with the 'add' method
        """
        self._my_daemon = _object
        
        self.set_name(_name)
        if self.name is not None:
            # We need to se the name format to all other handlers
            for handler in self.handlers:
                handler.setFormatter(defaultFormatter_named)
        
        # Be sure to keep human format choice
        logger.set_human_format(on=human_timestamp_log)
    
    
    def register_local_log(self, path, level=None):
        """
        The shinken logging wrapper can write to a local file if needed
        and return the file descriptor so we can avoid to close it.

        Add logging to a local log-file.

        The file will be rotated once a day
        """
        
        # Todo : Create a config var for backup count
        self.log_file_path = path
        handler = FixedTimedRotatingFileHandler(path, 'midnight', backupCount=5)
        
        if level is not None:
            handler.setLevel(level)
        if self.name is not None:
            handler.setFormatter(defaultFormatter_named)
        else:
            handler.setFormatter(defaultFormatter)
        self.addHandler(handler)
        
        # Todo : Do we need this now we use logging?
        return handler.stream.fileno()
    
    
    def get_stats(self):
        for handler in self.handlers:
            if hasattr(handler, 'get_long_flush_stats'):
                return handler.get_long_flush_stats()
        # Maybe no handler was present, if so return default with no errors
        return DEFAULT_LONG_FLUSH_STATS
    
    
    def log_nagios_message(self, level, message):
        if self._my_daemon is None:
            return
        msg = '[%d] %7s: [%s] %s\n' % (time.time(), level.upper(), logger.name, message)
        brok = Brok('log_monitoring', {'log': msg})
        self._my_daemon.add_Brok(brok)
        # Also log into the scheduler log as classic log
        self.info(message)
    
    
    def get_log_file_path(self):
        return self.log_file_path
    
    
    def set_human_format(self, on=True):
        """
        Set the output as human format.

        If the optional parameter `on` is False, the timestamps format
        will be reset to the default format.
        """
        global human_timestamp_log
        human_timestamp_log = bool(on)
        
        # Apply/Remove the human format to all handlers except the monitoring brok one.
        for handler in self.handlers:
            if self.name is not None:
                handler.setFormatter(human_timestamp_log and humanFormatter_named or defaultFormatter_named)
            else:
                handler.setFormatter(human_timestamp_log and humanFormatter or defaultFormatter)
    
    
    def print_stack(self, prefix='', level=logging.ERROR):
        formatted_lines = traceback.format_exc().splitlines()
        first_line = formatted_lines[0]
        if first_line == 'None':
            for line in traceback.format_stack():
                self.log(level, "%s%s" % (prefix, line))
        else:
            self.log(level, "%sERROR stack : %s" % (prefix, first_line))
            for line in formatted_lines[1:]:
                self.log(level, "%s%s" % (prefix, line))
    
    
    def log_perf(self, start_time, tag_name, msg, min_time=PERF_LOG_MIN_TIME, warn_time=PERF_LOG_WARN_TIME, prefix=''):
        time_spend = time.time() - start_time
        if time_spend > min_time:
            from_str = tag_name if isinstance(tag_name, basestring) else tag_name.__class__.__name__
            log_level = DEBUG
            if time_spend > warn_time:
                log_level = WARNING
            self.log(log_level, '%s[%s][perf][%.3f]s %s' % (prefix, from_str, time_spend, msg))
    
    
    def log_parameters(self, func):
        """ Use this decorator on any function or method to log its parameters """
        
        if self.do_log_parameters:
            def wrapper(*args, **kwargs):
                self.debug("Calling %s with the following parameters :" % func.__name__)
                
                if args is not None:
                    self.debug("| args:")
                    for index, arg in enumerate(args):
                        self.debug("|    arg %s => $%s$" % (index, pprint.pformat(arg)))
                
                if kwargs is not None and kwargs != {}:
                    self.debug("| kwargs:")
                    for (k, v) in kwargs:
                        self.debug("|    %s => $%s$" % (k, pprint.pformat(v)))
                
                return func(*args, **kwargs)
            
            
            return wrapper
        else:
            return func
    
    
    def diff(self, item1, item2):
        """ Returns diff between 2 items in the standard UNIX diff format
        :param item1: First item to compare
        :param item2: Second item to compare
        :return:
        """
        
        
        def colorize(str):
            if str.startswith('-'):
                return "\033[31m%s\033[0m" % str
            elif str.startswith('+'):
                return "\033[32m%s\033[0m" % str
            else:
                return str
        
        
        return "".join(map(colorize, list(Differ().compare(
            StringIO(unicode(pprint.pformat(item1) + "\n")).readlines(),
            StringIO(unicode(pprint.pformat(item2) + "\n")).readlines()))))
    
    
    def unregister_all(self):
        hdls = [hdl for hdl in self.handlers]
        for hld in hdls:
            self.removeHandler(hld)


# --- create the main logger ---
logging.setLoggerClass(Log)
logger = logging.getLogger('Shinken')  # type: Log

if hasattr(sys.stdout, 'isatty'):
    csh = ColorStreamHandler(sys.stdout)
    if logger.name is not None:
        csh.setFormatter(defaultFormatter_named)
    else:
        csh.setFormatter(defaultFormatter)
    logger.addHandler(csh)


def log_perf(start_time, tag_name, msg, log_level=PERF_LOG_LEVEL, min_time=PERF_LOG_MIN_TIME):
    time_spend = time.time() - start_time
    from_str = tag_name if isinstance(tag_name, basestring) else '%s-%s' % (tag_name.__class__.__name__, id(tag_name))
    if time_spend > min_time:
        logger.log(log_level, "[%s][perf][%.3f]s %s" % (from_str, time_spend, msg))


# Function use for old Nag compatibility. We need to send a different king of brok for this king of log
# We have another logger and use one for Shinken logs and another for monitoring data
def naglog_result(level, message, *args):
    logger.log_nagios_message(level, message)


class PartLogger(object):
    
    def __init__(self, name, level=0, parts=None):
        self.name = name
        self.level = level
        self.sub_part_logger = {}
        
        self.display_name = get_section_string(self.name)
        
        if parts:
            self.parts = parts[:]
            self.parts.append(self.display_name)
        else:
            self.parts = [self.display_name]
        
        self._parts_string = ' '.join(self.parts)
    
    
    def _update_part_string(self):
        self._parts_string = ' '.join(self.parts)
    
    
    def get_sub_part(self, sub_part):
        if sub_part in self.sub_part_logger:
            return self.sub_part_logger[sub_part]
        
        sub_part_log = PartLogger(sub_part, self.level + 1, self.parts)
        self.sub_part_logger[sub_part] = sub_part_log
        return sub_part_log
    
    
    # DEPRECATED : DO NOT USE ANYMORE
    def set_display_name(self, display_name):
        self.set_default_part(display_name)
    
    
    def set_default_part(self, default_part):
        self.parts[0] = get_section_string(default_part)
        for sub_part_logger in self.sub_part_logger.itervalues():
            sub_part_logger.set_default_part(default_part)
        self._update_part_string()
    
    
    def debug(self, *args):
        self.log(DEBUG, *args)
    
    
    def info(self, *args):
        self.log(INFO, *args)
    
    
    def warning(self, *args):
        self.log(WARNING, *args)
    
    
    def error(self, *args):
        self.log(ERROR, *args)
    
    
    def critical(self, *args):
        self.log(CRITICAL, *args)
    
    
    def log(self, level, *args):
        if not logger.isEnabledFor(level):
            return
        
        message_index = len(args) - 1
        message = args[message_index]
        message_format = ' '.join(itertools.chain([self._parts_string], [get_section_string(sub_part) for sub_part in args[:message_index]], ['%s' % message]))
        logger.log(level, message_format)
    
    
    def print_stack(self, prefix='', level=logging.ERROR):
        logger.print_stack(prefix='%s %s' % (self._parts_string, prefix), level=level)
    
    
    @staticmethod
    def format_sla_date(sla_date):
        try:
            return datetime.strptime('%s %s 00:00:00' % sla_date, '%j %Y %H:%M:%S').strftime('%d-%m-%Y')
        except:
            return '%s' % sla_date
    
    
    @staticmethod
    def format_time(_time):
        try:
            return datetime.fromtimestamp(_time).strftime('%d-%m-%Y %H:%M:%S')
        except:
            return '%s' % _time
    
    
    @staticmethod
    def format_time_with_micro(_time):
        try:
            return datetime.fromtimestamp(_time).strftime('%d-%m-%Y %H:%M:%S:%f')
        except:
            return '%s' % _time
    
    
    @staticmethod
    def format_time_as_sla(_time):
        try:
            return datetime.fromtimestamp(_time).strftime('%j_%Y %H:%M:%S')
        except:
            return '%s' % _time
    
    
    @staticmethod
    def format_duration_in_sec(raw_time):
        return '%.3fs' % raw_time
    
    
    @staticmethod
    def format_duration(time_period, time_format='auto'):
        from shinkensolutions.shinken_time_helper import print_human_readable_period  # lazy loading to avoid circular import
        return print_human_readable_period(time_period, time_format)
    
    
    @staticmethod
    def format_chrono(raw_time):
        raw_time = time.time() - raw_time
        return PartLogger.format_duration(raw_time)
    
    
    @staticmethod
    def format_datetime(_datetime):
        try:
            return _datetime.strftime('%H:%M:%S %d-%m-%Y (%Z)')
        except:
            return '%s' % _datetime
    
    
    @staticmethod
    def get_level():
        return logger.getEffectiveLevel()
    
    
    @staticmethod
    def set_level(level):
        return logger.setLevel(level)
    
    
    @staticmethod
    def set_human_format():
        return logger.set_human_format(True)
    
    
    @staticmethod
    def is_debug():
        return logger.getEffectiveLevel() == DEBUG
    
    
    def log_perf(self, start_time, tag_name, msg, min_time=PERF_LOG_MIN_TIME, warn_time=PERF_LOG_WARN_TIME):
        logger.log_perf(start_time, tag_name, msg, min_time, warn_time, self._parts_string)


PART_INITIALISATION = 'INITIALISATION'
DEFAULT_LOG = ''
loggers = {DEFAULT_LOG: PartLogger(DEFAULT_LOG)}


# Create logger for a specific part if not already exists
class LoggerFactory(object):
    @classmethod
    def get_logger(cls, name=DEFAULT_LOG):
        # type: (str) -> PartLogger
        if not name:
            return loggers[DEFAULT_LOG]
        
        # find by name
        log = loggers.get(name, None)
        
        # dit not exist we make it
        if not log:
            log = PartLogger(name)
            loggers[name] = log
        
        return log
