#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import re
from shinken.log import logger
from shinken.macroresolver import MacroResolver, ARGUMENT_SEPARATOR
from shinken.objects import Host, Service
from .helpers import escape_XSS, unescape_XSS
from .def_items import ITEM_TYPE, STOP_INHERITANCE_VALUES
from ..dao.datamanagerV2 import FALLBACK_USER

# Policy on protected macro:
# Listed macro are macros that match pattern defined by the synchronizer property : protect_fields__substrings_matching_fields
# This property is defined in synchronizer.cfg but can be overload and also manage by the command : shinken-protected-fields-data-manage
# All macro used in definition of protected macro are also protected (hidden)
#   For admin:
#       If frontend_cipher is enabled:
#           * Listed macro are protected
#       Else:
#           * All macro are visible
#   For si admin:
#       * Global marcos are protected
#       If frontend_cipher is enabled:
#           * Listed macro are protected
#       Else:
#           * Listed macros are protected unless they are directly in host

DOUBLEDOLLAR = '__DOUBLEDOLLAR__'
PROTECTED_EXCLAMATION_MARK = '__EXCLAMATION_MARK__'
_ARGN_MACROS = set(['ARG%d' % i for i in xrange(1, 33)])  # 32


class SPECIAL_VALUE(object):
    MISSING_ARGN = '__MISSING_ARGN__'
    MISSING_DUPLICATE_FOR_EACH = '__MISSING_DFE__'
    MISSING_HOST = '__MISSING_HOST__'
    MISSING_CHECK = '__MISSING_CHECK__'
    MISSING_GLOBAL = '__MISSING_GLOBAL__'
    PROTECTED_PASSWORD = '__PROTECTED_PASSWORD__'
    NOT_IMPLEMENTED = '__NOT_IMPLEMENTED__'


class FROM_INFO(object):
    ARGN = 'argn'
    DUPLICATE_FOR_EACH_KEY = 'duplicate foreach (key)'
    DUPLICATE_FOR_EACH_VALUE = 'duplicate foreach (value)'
    DUPLICATE_FOR_EACH = 'duplicate foreach'
    GLOBAL = 'global'
    HOST = 'host'
    HOST_TPL = 'template'
    CHECK = 'check'


def get_macros(raw_string):
    split_string = re.split(ur'(\$)', raw_string)
    macros = []
    in_macro = False
    macro_name = ''
    for string_part in split_string:
        if not in_macro and string_part == '$':
            in_macro = True
        elif in_macro and string_part == '$':
            in_macro = False
            macros.append(macro_name)
        elif in_macro and string_part:
            macro_name = string_part
    
    macros = [m.strip() for m in macros if m.strip()]
    macros = [(m.upper(), m) for m in macros]
    return macros


def resolve_macros(app, host, host_templates, check, check_command_arg, to_expand, dfe, item_type):
    try:
        user = app.get_user_auth()
    except AttributeError:
        # When analyzer ask for a _resolve_macro it not have access to user auth.
        # Other call make user auth check before calling _resolve_macro so
        # We can assume if there are an AttributeError except the user is an admin.
        user = FALLBACK_USER
    expand_macros_infos = {'!!check_command_arg': check_command_arg}
    macros = get_macros(to_expand)
    logger.debug(u'[macro_resolver] start resolve macros : [%s]' % macros)
    for upper_macro_name, original_macro_name in macros:
        is_protected = app.frontend_cipher.match_protected_property(upper_macro_name, item_type, user=user)
        # Show policy on protected macro
        resolution_info = _resolve_macro(app, upper_macro_name, original_macro_name, host, host_templates, check, upper_macro_name, dfe, expand_macros_infos, is_protected, user)
        expand_macros_infos[upper_macro_name] = (upper_macro_name, original_macro_name, resolution_info)
    
    expand_macros_infos.pop('!!check_command_arg', None)
    logger.debug('[macro_resolver] --- macros resolved ---')
    header = '[macro_resolver] %30s | %20s | %30s'
    logger.debug(header % ('macro name', 'from', 'value'))
    for upper_macro_name, macro_info in expand_macros_infos.iteritems():
        logger.debug(header % (upper_macro_name, macro_info[2]['from'] if len(macro_info) > 2 and macro_info[2] else '', macro_info[2]['value'] if len(macro_info) > 2 and macro_info[2] else ''))
    
    return expand_macros_infos


def _build_macro_info(expand_macros_infos, macro_name, original_macro_name, _from, value, raw_value, value_unprotected, founded_prop, is_protected, from_info=None, data_modulation=None):
    if not founded_prop:
        is_protected = False
    if expand_macros_infos.get(macro_name, None):
        _, _, macro_info = expand_macros_infos[macro_name]
    else:
        resolve_order = len(expand_macros_infos)
        if is_protected:
            value_unprotected = value_unprotected.decode('utf8')
        
        macro_info = {
            'founded_prop'     : founded_prop,
            'from'             : _from,
            'value'            : SPECIAL_VALUE.PROTECTED_PASSWORD if is_protected else escape_XSS(value),
            'raw_value'        : SPECIAL_VALUE.PROTECTED_PASSWORD if is_protected else escape_XSS(raw_value),
            'unprotected_value': value_unprotected,
            'from_info'        : from_info,
            'is_protected'     : is_protected,
            'resolve_order'    : resolve_order,
            'data_modulation'  : data_modulation
        }
        expand_macros_infos[macro_name] = (macro_name, original_macro_name, macro_info)
    return macro_info


# Try to look at the macro somewhere, manage recursive lookup, split macros and such things
def _resolve_macro(app, macro_name, original_macro_name, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=32):
    # Ok, it's just too much level here
    if level <= 0:
        value = app._('element.resolve_macro_max_recursive') % initial_macro
        return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, '', value, value, value, initial_macro, is_protected)
    
    logger.debug('[macro_resolver] Resolving macro [%s] level [%s]' % (macro_name, level))
    
    # Can be global or on elements
    if not macro_name.startswith('HOST') and not macro_name.startswith('_HOST') and not macro_name.startswith('SERVICE') and not macro_name.startswith('_SERVICE'):
        logger.debug('[macro_resolver] try to look at global macro [%s]' % macro_name)
        
        # Maybe it's an ARGN that is referenced into a macro.
        if macro_name in _ARGN_MACROS:
            argn_offset = int(macro_name.replace('ARG', ''))
            check_command_arg = expand_macros_infos['!!check_command_arg']
            logger.debug('[macro_resolver] Did detected an ARGN macro: %d => %s' % (argn_offset, macro_name))
            try:
                raw_value = re.split(r'!|\|-\|', check_command_arg)[argn_offset - 1].replace(PROTECTED_EXCLAMATION_MARK, '!')
                value = raw_value
                value_unprotected = raw_value
                if '$' in raw_value:
                    value, value_unprotected = _resolve_macro_in_macro_value(app, raw_value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=level - 1)
                if ARGUMENT_SEPARATOR in value_unprotected:
                    expand_macros_infos['!!check_command_arg'] = check_command_arg.replace(raw_value, value_unprotected.replace('!', PROTECTED_EXCLAMATION_MARK))
                    return _resolve_macro(app, macro_name, original_macro_name, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=level)
            except IndexError:
                value = ''
                value_unprotected = ''
                raw_value = SPECIAL_VALUE.MISSING_ARGN
            
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.ARGN, value, raw_value, value_unprotected, '', is_protected)
        
        # Maybe it's an internal duplicate_foreach value, so show it
        _duplicate_foreach_macros = ['KEY', 'VALUE']
        for i in xrange(1, 17):  # 16
            _duplicate_foreach_macros.append('VALUE%d' % i)
        
        if macro_name in _duplicate_foreach_macros:
            is_protected = is_protected or app.frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.SERVICESHOSTTPLS, user=user)
            logger.debug('[macro_resolver] resolving [%s] type:[duplicate for each]' % macro_name)
            if dfe:
                _dfe_property_name = dfe.get('property_name', 'Unknown property name')
                if macro_name == 'KEY':
                    _dfe_key = dfe.get('key', '')
                    logger.debug('[macro_resolver] found value [%s]:[%s]' % (macro_name, _dfe_key))
                    
                    return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.DUPLICATE_FOR_EACH_KEY, _dfe_key, _dfe_key, _dfe_key, _dfe_property_name, is_protected)
                else:
                    _dfe_values = dfe.get('args', '')
                    for i in range(len(_dfe_values)):
                        if macro_name == 'VALUE':
                            logger.debug('[macro_resolver] found value [%s]:[%s]' % (macro_name, _dfe_values[0]))
                            value = _dfe_values[0]
                            raw_value = value
                            value_unprotected = value
                            if '$' in _dfe_values[0]:  # if there is another macro, look deeper
                                value, value_unprotected = _resolve_macro_in_macro_value(app, _dfe_values[0], host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=level - 1)
                            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.DUPLICATE_FOR_EACH_VALUE, value, raw_value, value_unprotected, _dfe_property_name, is_protected)
                        
                        if macro_name == ('VALUE%s' % (i + 1)):
                            value = _dfe_values[i]
                            raw_value = value
                            value_unprotected = value
                            logger.debug('[macro_resolver] found value [%s]:[%s]' % (macro_name, value))
                            if '$' in value:  # if there is another macro, look deeper
                                value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=level - 1)
                            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.DUPLICATE_FOR_EACH_VALUE, value, raw_value, value_unprotected, _dfe_property_name, is_protected)
            else:
                logger.debug('[macro_resolver] fail to resolve [%s] type:[duplicate for each] dfe:[%s]' % (macro_name, dfe))
                value = SPECIAL_VALUE.MISSING_DUPLICATE_FOR_EACH
                return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.DUPLICATE_FOR_EACH, value, value, value, '', is_protected)
        
        user_is_admin = user.get('is_admin', '0') == '1'
        is_protected = is_protected or (not user_is_admin and not app.frontend_cipher.encryption_enable)
        # Ok so not a duplicate_foreach specific macro, try to look in real globals
        if macro_name in MacroResolver.macros.keys():
            value = SPECIAL_VALUE.NOT_IMPLEMENTED
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.GLOBAL, value, value, value, '', is_protected)
        
        value = getattr(app.conf, ('$%s$' % macro_name).encode('utf8', 'ignore'), None)
        if value is None:
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.GLOBAL, '', SPECIAL_VALUE.MISSING_GLOBAL, '', '', is_protected)
        raw_value = value
        value_unprotected = value
        if '$' in value:  # if there is another macro, look deeper and return what ever we did find
            value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=level - 1)
        
        # return as we did find in global
        return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.GLOBAL, value, raw_value, value_unprotected, macro_name, is_protected)
    
    # Now we can look in the host or the check
    # First host
    if macro_name.startswith('HOST') or macro_name.startswith('_HOST'):
        logger.debug('[macro_resolver] searching [%s] in host and host template' % macro_name)
        is_protected = is_protected or app.frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.HOSTS, user=user)
        modulation = None
        prop = ''
        value = ''
        if macro_name.upper() in Host.macros:
            prop = Host.macros[macro_name.upper()].lower()
            value = host.get(prop, None)
            raw_value = value
        elif macro_name.startswith('_HOST'):
            prop = '_' + macro_name[(len('_HOST')):].upper()
            value = host.get(prop, None)
            raw_value = value
            if prop in host.get('@modulation', {}):
                value = host['@modulation'][prop][0]
                modulation = host['@modulation'][prop][1]
        
        if value:
            value_unprotected = unescape_XSS(value)
            if '$' in value:  # if there is another macro, look deeper
                is_protected = is_protected or app.frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.HOSTS, user=user)
                value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=level - 1)
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.HOST, value, raw_value, value_unprotected, prop, is_protected, data_modulation=modulation)
        else:  # some values have default, like alias => host_name or address=> host_name
            if prop in ('alias', 'address', 'display_name'):
                value = host.get('host_name', '')
                raw_value = value
                value_unprotected = unescape_XSS(value)
                is_protected = is_protected or app.frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.HOSTS, user=user)
                return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.HOST, value, raw_value, value_unprotected, prop, is_protected, data_modulation=modulation)
        
        # Maybe in a template one?
        for tpl in host_templates:
            is_protected = is_protected or app.frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.HOSTTPLS, user=user)
            logger.debug('[macro_resolver] searching in tpl [%s]' % tpl.get('name', 'no name'))
            value = tpl.get(prop, None)
            if value is not None:
                raw_value = value
                value_unprotected = unescape_XSS(value)
                if '$' in value:  # if there is another macro, look deeper
                    value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=level - 1)
                return _build_macro_info(
                    expand_macros_infos,
                    macro_name,
                    original_macro_name,
                    FROM_INFO.HOST_TPL,
                    value,
                    raw_value,
                    value_unprotected,
                    prop,
                    is_protected,
                    from_info={'name': tpl.get('name', ''), 'uuid': tpl['_id'], },
                    data_modulation=modulation
                )
        # Cannot find anything
        return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.HOST, '', SPECIAL_VALUE.MISSING_HOST, '', '', is_protected, data_modulation=modulation)
    
    # Then check
    if macro_name.startswith('SERVICE') or macro_name.startswith('_SERVICE'):
        is_protected = is_protected or app.frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.SERVICETPLS, user=user)
        modulation = None
        prop = ''
        value = ''
        if macro_name.upper() in Service.macros:
            prop = Service.macros[macro_name.upper()].lower()
            value = check.get(prop, None)
            raw_value = value
        elif macro_name.startswith('_SERVICE'):
            prop = '_' + macro_name[(len('_SERVICE')):].upper()
            value = check.get(prop, None)
            
            raw_value = value
            if prop in check.get('@modulation', {}):
                value = check['@modulation'][prop][0]
                modulation = check['@modulation'][prop][1]
        else:
            raw_value = SPECIAL_VALUE.MISSING_CHECK
        
        logger.debug('[macro_resolver] Trying to get the property [%s] from the check [%s]' % (prop, check.get('name', check.get('service_description', 'name not found'))))
        if value is not None:
            value_unprotected = unescape_XSS(value)
            if '$' in value:  # if there is another macro, look deeper
                value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=level - 1)
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.CHECK, value, raw_value, value_unprotected, prop, is_protected, data_modulation=modulation)
        
        # cannot find anything
        return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.CHECK, '', SPECIAL_VALUE.MISSING_CHECK, '', '', is_protected, data_modulation=modulation)


def _resolve_macro_in_macro_value(app, macro_value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=32):
    # type: (Synchronizer, str, Baseitem, list, Baseitem, str,str, dict, Boolean, Baseitem, int) -> (str, str)
    sub_macros = get_macros(macro_value)
    macro_value_resolved = macro_value
    macro_value_resolved_unprotected = unescape_XSS(macro_value_resolved)
    
    for upper_sub_macro, original_sub_macro in sub_macros:
        logger.debug('[macro_resolver] Resolving sub macro [%s] in macro value [%s] ' % (upper_sub_macro, macro_value))
        sub_res = _resolve_macro(app, upper_sub_macro, original_sub_macro, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, level=level - 1)
        if sub_res is None or sub_res['unprotected_value'] is None or sub_res['unprotected_value'] in STOP_INHERITANCE_VALUES:
            logger.debug("[macro_resolver] inner macro [%s] in [%s] was not found" % (upper_sub_macro, macro_value))
            macro_value_resolved = macro_value_resolved.replace('$%s$' % original_sub_macro, '')
            macro_value_resolved_unprotected = macro_value_resolved_unprotected.replace('$%s$' % original_sub_macro, '')
        else:
            macro_value_resolved = macro_value_resolved.replace('$%s$' % original_sub_macro, sub_res['value'])
            macro_value_resolved_unprotected = macro_value_resolved_unprotected.replace('$%s$' % original_sub_macro, sub_res['unprotected_value'])
    
    return macro_value_resolved, macro_value_resolved_unprotected


def expand_value(to_expand, _resolve_macros):
    for _, original_macro_name, macro_infos in _resolve_macros.itervalues():
        if macro_infos is None:
            continue
        if macro_infos['unprotected_value'] in STOP_INHERITANCE_VALUES:
            macro_infos['unprotected_value'] = ''
        
        to_expand = to_expand.replace(u'$%s$' % original_macro_name, macro_infos['unprotected_value'])
    return to_expand
