#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import math
import os
import re
import time
import urllib
from ..bottle import template

from shinken.log import logger
from shinken.macroresolver import MacroResolver
from shinken.objects.host import Host
from shinken.objects.resultmodulation import ModulationSyntaxError
from shinken.objects.service import Service
from shinken.property import none_object
from shinkensolutions.service_override_parser import parse_service_override_property
from .helper_property import HelperProperty
from .helper_component import HelperComponent
from .helper_list import HelperList
from .helper_attachment_property_rules import HelperTemplateApplication_Rules
from .helper_rules_component import HelperRulesComponent
from .helper_object import HelperObject
from .mass_change_helper import MassChangeHelper
from .component_select_options import ComponentOption
from .selector_type import SELECTOR_TYPE
from ..dao.def_items import DEF_ITEMS, ENTER_VALUE_FOR_MASS_CHANGE, ITEM_STATE, ITEM_TYPE, LINKIFY_MANAGE_STATES, METADATA, PROP_DEFAULT_OR_INHERITANCE, PROP_DEFAULT_VALUE, SERVICE_OVERRIDE, STOP_INHERITANCE_VALUES, VALUE_FORCE_DEFAULT, \
    prop_is_linked
from ..dao.helpers import get_default_value, get_name_from_type, get_property_def, get_property_separator, item_for_link
from ..dao.items.mixins import ServiceOverridesMixin


class TIME_PERIOD_TYPE:
    DAY = "period_day_type"
    ADVANCED = "period_advanced_type"


class IconsForUrlSelect:
    
    def __init__(self, icon):
        default_icon = 'tag'
        self.htdocs_path = os.path.join(os.path.join(os.path.abspath(os.path.dirname(__file__))), '../htdocs')
        path_shinkon = os.path.join(self.htdocs_path, 'font/shinken-icons-url.svg')
        
        font_parser = ParseFontSvg(path_shinkon)
        self.list = font_parser.get_table_icons()
        
        if default_icon not in self.list:
            self.list.append(default_icon)
        
        if icon not in self.list:
            icon = default_icon
        
        self.entries = [ComponentOption('icon-url', True, i, icon == i, 0, label="<span class='shinken-icons-url shinken-icons-url-%s'></span>" % i) for i in self.list]


class PopUpUrlSelect:
    LIST = ['NO', 'OPEN_IN_NEW_TAB', 'POPUP(30)', 'POPUP(40)', 'POPUP(50)', 'POPUP(60)', 'POPUP(70)', 'POPUP(80)']
    
    
    def __init__(self, popupping):
        self.entries = [ComponentOption('popupping', True, i, popupping == i, 0, label=app._('multi-line-object.url_%s' % i)) for i in PopUpUrlSelect.LIST]


class PopUpMonth:
    LIST = ['None', 'january', 'february', 'march', 'april', 'may', 'june', 'july', 'august', 'september', 'october', 'november', 'december']
    
    
    def __init__(self, month):
        self.entries = [ComponentOption('month', True, i, month == i, 0, label=app._('timeperiods.%s' % i)) for i in PopUpMonth.LIST]


class PopUpInteger:
    def __init__(self, key, value_selected, min, max, key_label, ignored_key=False):
        self.entries = [ComponentOption(key, True, i, value_selected == i, 0, label=app._(key_label % i)) for i in range(min, max + 1) if not ignored_key or i not in ignored_key]


class BTN_STATE:
    DISABLED = "-1"
    OFF = "0"
    ON = "1"
    DISCREET = "2"


class ParseFontSvg:
    def __init__(self, file_path):
        file = open(file_path, "r")
        lines = file.readlines()
        file.close()
        self.tab = []
        
        for l in lines:
            if "glyph-name" in l:
                pos1 = l.find("glyph-name=\"")
                pos2 = l.find("unicode=\"")
                
                start_name = pos1 + 12
                end_name = pos2 - 2
                
                glyph_name = l[start_name:end_name]
                self.tab.append(glyph_name)
    
    
    def get_table_icons(self):
        return self.tab


app = None


def atoi(text):
    return int(text) if text.isdigit() else text.lower()


def natural_keys(text):
    '''
    alist.sort(key=natural_keys) sorts in human order
    http://nedbatchelder.com/blog/200712/human_sorting.html
    '''
    return [atoi(c) for c in re.split('(\d+)', text)]


def find(value, lst, key):
    # print 'Finding the value', value
    # print 'And in the list', lst, 'and key', key
    
    if value is None:
        return None
    
    for i in lst:
        v = i.get(key, None)
        if v == value:
            return i
    return None


def isfloat(value):
    try:
        float(value)
        return True
    except ValueError:
        return False


def display_service_override_value(value, user, flatten_state=LINKIFY_MANAGE_STATES):
    have_protected = False
    unparsed = []
    separator = get_property_separator(ITEM_TYPE.HOSTS, SERVICE_OVERRIDE)
    if isinstance(value, basestring):
        parsed = parse_service_override_property(value)
        for check, so_value in parsed.iteritems():
            for prop_name, value in so_value.iteritems():
                if app.frontend_cipher.match_protected_property(prop_name, ITEM_TYPE.SERVICESHOSTS, user=user):
                    value = '[%s]' % app._('element.password_protected')
                    have_protected = True
                unparsed.append('%s, %s %s' % (check, prop_name, value))
        value = separator.join(unparsed)
    else:
        for link in value['links']:
            check_name = ServiceOverridesMixin.get_check_name_with_datamanager(link, app.datamanagerV2, flatten_state)
            
            check_type = ITEM_TYPE.SERVICESHOSTS
            if prop_is_linked(check_type, link['key']):
                property_type = DEF_ITEMS[check_type]['props_links'].get(link['key'], '')[0]
                flattened_value = app.datamanagerV2.flatten_value(link['value'], property_type, link['key'], flatten_state)
            else:
                flattened_value = link['value']
                if app.frontend_cipher.match_protected_property(link['key'], ITEM_TYPE.SERVICESHOSTS, user=user):
                    flattened_value = '[%s]' % app._('element.password_protected')
                    have_protected = True
            
            property_override = "%s, %s %s" % (check_name, link['key'], flattened_value)
            unparsed.append(property_override)
        value = separator.join(unparsed)
    return value, have_protected


class Helper(object):
    INPUT_WITH_DROPDOWN_SEPARATORS = {
        'host_name': list('()&|!'),
    }
    
    
    def __init__(self, app):
        self.app = app
        self.mass_change_helper = MassChangeHelper(app, self)
        self.helper_object = HelperObject(app)
        self.helper_component = HelperComponent(app, self)
        self.helper_list = HelperList(app, self)
        self.helper_property = HelperProperty(app, self)
        self.helper_template_application_rules = HelperTemplateApplication_Rules(app, self)
        self.helper_rules_component = HelperRulesComponent(app, self)
        # Define the dot information on the right of each label.
        self.dot_help = '<span class="tag-help">?</span>'
    
    
    @staticmethod
    def template_format(_template, **params):
        return template(_template, params)
    
    
    def _get_service_overrides_from(self, item_prop, shinken_element_original, is_placeholder=False):
        form_value = METADATA.get_metadata(shinken_element_original, METADATA.FROM, {}).get(item_prop, {}) if shinken_element_original else {}
        if form_value:
            if is_placeholder:
                link = "%s" % self.app._('element.from_template_so')
            else:
                link = "<span class='shinken-overrided-in-template'>%s</span>" % self.app._('element.from_template_so')
        else:
            link = self.app._('element.in_check')
        return link
    
    
    def _get_link_from_service_overrides(self, flat_value, item_prop, shinken_element_original):
        link = flat_value
        from_value = METADATA.get_metadata(shinken_element_original, METADATA.FROM, {}).get(item_prop, {}) if shinken_element_original else {}
        
        if from_value:
            link = '<span class="shinken-value-from-template">%s <br>%s <a href="/elements/%s/%s" draggable="false" class="title shinken-link" target="_blank" >%s</a></span><span class="shinken-value-from-check"></span>' % (
                flat_value, self.app._('element.from_template_so'), from_value['item_type'], from_value['item_id'], from_value['item_name'])
        return link
    
    
    def get_last_comment(self, item):
        work_area_info = item.get('work_area_info', {})
        comments = work_area_info.get('comments', {})
        last_comment = {}
        _max_time = -1
        for comment in comments.values():
            if not comment:
                continue
            _at = comment['at']
            if _max_time < _at:
                last_comment = comment
                _max_time = _at
        return last_comment
    
    
    def get_tag_img(self, tag_name, img_size='imgsize1'):
        output = []
        stripped_name = tag_name.strip()
        if self.app.is_tag_exist(stripped_name):
            output.append('<img class="%s" onerror="$(this).hide()" src="/static/%d/images/sets/%s/tag.png" />' % (
                img_size, self.app.http_start_time, stripped_name))
        return "".join(output)
    
    
    # Get
    def get_navi(self, total, pos, step=30):
        step = float(step)
        nb_pages = int(math.ceil(total / step))
        current_page = int(math.ceil(pos / step))
        
        if current_page == 1 and nb_pages == 1:
            nb_pages = 2
        
        step = int(step)
        # print "NaviDBG", nb_pages, current_page, step
        
        _to_return = []
        
        if (nb_pages == 0 or nb_pages == 1) and current_page == 0:
            # print "NaviDBG2, early exit"
            return None
        
        if current_page >= 2:
            # Name, start, end, is_current
            _to_return.append((u'«', 0, step, False))
            _to_return.append(('...', None, None, False))
        
        # print "Navi Range,", current_page - 1, current_page + 2
        for i in xrange(current_page - 1, current_page + 2):
            if i < 0:
                continue
            if i >= nb_pages:
                continue
            # print "Navi Doing PAGE", i
            is_current = (i == current_page)
            start = int(i * step)
            # Maybe we are generating a page too high, bail out
            if start > total:
                continue
            
            end = int((i + 1) * step)
            _to_return.append(('%d' % (i + 1), start, end, is_current))
        
        if current_page < nb_pages - 2:
            start = int((nb_pages - 1) * step)
            end = int(nb_pages * step)
            _to_return.append(('...', None, None, False))
            _to_return.append((u'»', start, end, False))
        
        return _to_return
    
    
    def print_date(self, t):
        if t == 0 or t is None:
            return 'N/A'
        # return time.asctime(time.localtime(t))
        if self.app.lang == 'fr':
            return time.strftime("%d/%m/%Y %H:%M", time.localtime(t))
        return time.strftime("%m/%d/%Y %H:%M", time.localtime(t))
    
    
    # For a time, print something like
    # 10m 37s  (just duration = True)
    # N/A if got bogus number (like 1970 or None)
    # 1h 30m 22s ago (if t < now)
    # Now (if t == now)
    # in 1h 30m 22s
    # Or in 1h 30m (no sec, if we ask only_x_elements=2, 0 means all)
    def print_duration(self, t, just_duration=False, x_elts=0):
        if (t == 0) or (t is None) or (t == -1):
            return 'N/A'
        
        # Get the difference between now and the time of the user
        seconds = int(time.time()) - int(t)
        
        # If it's now, say it :)
        if seconds == 0:
            return 'Now'
        
        in_future = False
        
        # Remember if it's in the future or not
        if seconds < 0:
            in_future = True
        
        # Now manage all case like in the past
        seconds = abs(seconds)
        # print "In future?", in_future
        
        # print "sec", seconds
        seconds = long(round(seconds))
        # print "Sec2", seconds
        minutes, seconds = divmod(seconds, 60)
        hours, minutes = divmod(minutes, 60)
        days, hours = divmod(hours, 24)
        weeks, days = divmod(days, 7)
        months, weeks = divmod(weeks, 4)
        years, months = divmod(months, 12)
        
        minutes = long(minutes)
        hours = long(hours)
        days = long(days)
        weeks = long(weeks)
        months = long(months)
        years = long(years)
        
        duration = []
        if years > 0:
            duration.append('%dy' % years)
        else:
            if months > 0:
                duration.append('%dM' % months)
            if weeks > 0:
                duration.append('%dw' % weeks)
            if days > 0:
                duration.append('%dd' % days)
            if hours > 0:
                duration.append('%dh' % hours)
            if minutes > 0:
                duration.append('%dm' % minutes)
            if seconds > 0:
                duration.append('%ds' % seconds)
        
        # print "Duration", duration
        # Now filter the number of printed elements if ask
        if x_elts >= 1:
            duration = duration[:x_elts]
        
        # Maybe the user just want the duration
        if just_duration:
            return ' '.join(duration)
        
        # Now manage the future or not print
        if in_future:
            return 'in ' + ' '.join(duration)
        else:  # past :)
            return ' '.join(duration) + ' ago'
    
    
    def get_item_property_display_name(self, item_type, item_property):
        if item_property.startswith('_'):
            item_property_display_name = '%s [%s]' % (item_property, self.app._('element.data'))
        else:
            item_property_display_name = self.app._('%s.%s' % (item_type, item_property))
            if 'TO_TRAD(' in item_property_display_name:
                item_property_display_name = self.app._('element.unknown_import_key')
        return item_property_display_name
    
    
    def diff_to_display(self, item_property, item_type, previous_value, new_value):
        user = self.app.get_user_auth()
        is_protected_password = self.app.frontend_cipher.match_protected_property(item_property, item_type, user)
        if is_protected_password:
            new_value = self.app._('element.password_protected') if new_value else self.app._('apply.value_not_set')
            previous_value = self.app._('element.password_protected')
        else:
            
            if previous_value is None:
                previous_value = ''
            if item_property == SERVICE_OVERRIDE:
                previous_value, _ = display_service_override_value(previous_value, user, [ITEM_STATE.PRODUCTION])
            previous_value = self.app.datamanagerV2.flatten_value(previous_value, item_type, item_property, flatten_states=[ITEM_STATE.PRODUCTION])
            if isinstance(previous_value, basestring):
                previous_value = previous_value.replace(",", u",\u200B").replace('__DEFAULT_NO_TEMPLATE__', '-- Default --').replace('null', '-- None --')
            if previous_value == '':
                previous_value = self.app._('apply.value_not_set')
            
            if new_value is None:
                new_value = ''
            if item_property == SERVICE_OVERRIDE:
                new_value, _ = display_service_override_value(new_value, user)
            new_value = self.app.datamanagerV2.flatten_value(new_value, item_type, item_property)
            if isinstance(new_value, basestring):
                new_value = new_value.replace(",", u",\u200B").replace('__DEFAULT_NO_TEMPLATE__', '-- Default --').replace('null', '-- None --')
            if new_value == '':
                new_value = self.app._('apply.value_not_set')
        
        return previous_value, new_value, is_protected_password
    
    
    def get_macros(self):
        
        macros = {
            'macros_host_properties'   : [macro for macro in Host.macros.iterkeys()],
            'macros_service_properties': [macro for macro in Service.macros.iterkeys()],
            'macros_global'            : [macro for macro in self.app.conf.macros.iterkeys()],
            'macros_global_interpreted': [macro for macro in MacroResolver.macros.iterkeys()]
        }
        return macros
    
    
    def get_list(self, item, item_property):
        return_html = ''
        
        item_value = item.get(item_property, None)
        link_items = item.get_link_items(item_property)
        item_type = item.get_type()
        property_handle_plus = item_property in DEF_ITEMS[item.get_type()].get('fields_with_plus', [])
        has_plus = property_handle_plus and link_items and item.get(item_property, {}).get('has_plus', False)
        
        if not item_value:
            return ''
        
        if has_plus:
            return_html = '''<div class='tag template shinken-plus'><span>+</span></div>'''
        
        if item_value in STOP_INHERITANCE_VALUES:
            info = {
                'tag_class'          : '',
                'from_tpl_class_list': '',
                'title'              : '',
                'image'              : '',
                'value'              : get_default_value(item.get_type(), item_property, app=app, same_as_host=True)
            }
            return_html += '''<div class='%(tag_class)s template %(from_tpl_class_list)s ' title="%(title)s">%(image)s%(value)s</div>''' % info
            return return_html
        
        if property_handle_plus:
            for link_item in link_items:
                if link_item.get_name() in STOP_INHERITANCE_VALUES:
                    continue
                link_item_name = link_item.get_name()
                from_myself = [fi for fi in item.get_from(item_property) if fi.origin in item['_id'] and fi.type == item_type and fi.value in (link_item_name, link_item.get('_id', ''))]
                
                # TODO we must not compute the real form_info.origin but a form_info.origin should come from a template of the current item
                info = {
                    'tag_class'          : 'tag',
                    'from_tpl_class_list': '' if from_myself else 'from-template-list',
                    'title'              : '',
                    'image'              : self.get_tag_img(link_item_name, img_size="list-tag-img"),
                    'value'              : link_item_name
                }
                return_html += '''<div class='%(tag_class)s template %(from_tpl_class_list)s ' title="%(title)s">%(image)s%(value)s</div>''' % info
        elif item_property == 'use':
            for link_item in link_items:
                link_item_name = link_item.get_name()
                info = {
                    'tag_class'          : 'tag',
                    'from_tpl_class_list': '',
                    'title'              : '',
                    'image'              : self.get_tag_img(link_item_name, img_size="list-tag-img"),
                    'value'              : link_item_name
                }
                return_html += '''<div class='%(tag_class)s template %(from_tpl_class_list)s ' title="%(title)s">%(image)s%(value)s</div>''' % info
        elif prop_is_linked(item.get_type(), item_property):
            from_myself = [fi for fi in item.get_from(item_property) if fi.origin == item['_id'] and fi.type == item_type and fi.value == item_value]
            flatten_value = item.flatten_prop(item_property)
            info = {
                'tag_class'          : 'tag',
                'from_tpl_class_list': '' if from_myself else 'from-template-list',
                'title'              : '',
                'image'              : self.get_tag_img(flatten_value, img_size="list-tag-img"),
                'value'              : flatten_value
            }
            return_html += '''<div class='%(tag_class)s template %(from_tpl_class_list)s ' title="%(title)s">%(image)s%(value)s</div>''' % info
        else:
            from_myself = [fi for fi in item.get_from(item_property) if fi.origin == item['_id'] and fi.type == item_type and fi.value == item_value]
            info = {
                'tag_class'          : '',
                'from_tpl_class_list': '',
                'title'              : '',
                'image'              : '',
                'value'              : item_value
            }
            return_html = '''<div class='%(tag_class)s template %(from_tpl_class_list)s ' title="%(title)s">%(image)s%(value)s</div>''' % info
        
        return return_html
    
    
    def get_slider_input(self,
                         item,
                         item_prop,
                         item_type,
                         is_disabled=False,
                         label_range='',
                         default='',
                         data_min=0,
                         data_step=1,
                         data_max=100,
                         data_unit='',
                         placeholder='',
                         data_type='range',
                         required=False,
                         item_prop_label=None,
                         shinken_element_original=False,
                         data_digits=0,
                         mass_change=False):
        is_override = True if shinken_element_original else False
        class_to_add = 'shinken-disabled' if is_disabled else ''
        
        item_prop_name_final = self.helper_property.build_final_prop_name(shinken_element_original, item_prop, is_override)
        
        if not item_prop_label:
            item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        
        item_value = "" if mass_change else item.get(item_prop, "")
        value = 0
        label_range = ''
        default_value = ''
        default_value_label = ''
        
        if not item_prop_label:
            item_prop_label = self.app._('%s.%s' % (item_type, item_prop))
        if default == '' or default is None:
            default = get_default_value(item_type, item_prop)
        
        _show_data_unit = True
        if default != '' and default is not None:
            label_range = '%s%s' % (default, data_unit)
            if data_type == 'range_business_impact':
                if isinstance(default, int):
                    value = int(default)
                    default_value = value
                else:
                    value = get_default_value(item_type, item_prop)
                    default_value = value
                
                if item_type == ITEM_TYPE.SERVICESCLUSTERS or item_type == ITEM_TYPE.SERVICESCLUSTERTPLS:
                    default_value_label = self.app._('element.same_as_cluster')
                elif item_type == ITEM_TYPE.SERVICESHOSTS or item_type == ITEM_TYPE.SERVICESHOSTTPLS:
                    default_value_label = self.app._('element.same_as_host')
                elif item_type == ITEM_TYPE.SERVICETPLS:
                    default_value_label = self.app._('element.same_as_host_parent')
                else:
                    default_value_label = self.app._('element.business_impact_%s' % default_value)
                label_range = default_value_label
                placeholder = '%s' % default_value_label
            elif (item_prop == 'sla_warning_threshold' or item_prop == 'sla_critical_threshold') and item_type in ITEM_TYPE.ALL_SERVICES:
                if item_type == ITEM_TYPE.SERVICESCLUSTERS or item_type == ITEM_TYPE.SERVICESCLUSTERTPLS:
                    default_value_label = self.app._('element.same_as_cluster')
                else:
                    default_value_label = self.app._('element.same_as_host')
                placeholder = '%s' % default_value_label
                _show_data_unit = False
                
                if is_override:
                    default = default_value_label
                else:
                    value = ('{0:.' + str(data_digits) + 'f}').format(float(default))
                    default_value = 'same_as_host'
            else:
                value = ('{0:.' + str(data_digits) + 'f}').format(float(default))
                default_value = value
                default_value_label = value
        
        if not placeholder and default_value:
            placeholder = '%s' % default_value
        
        if item_value == PROP_DEFAULT_VALUE:
            value = item_value
        elif item_value != '' and item_value != 'null':
            try:
                value = float(item_value)
            except ValueError:
                value = 0
            if data_type == 'range_business_impact':
                label_range = self.app._('element.business_impact_%s' % int(item_value))
            elif data_type == 'range_with_input':
                label_range = '%g%s' % (value, data_unit)
                label_range = ('{0:.' + str(data_digits) + 'f}').format(float(label_range.rstrip(data_unit)))
            else:
                label_range = '%g%s' % (value, data_unit)
        
        _value_string_original = ''
        _flat_value = ''
        if is_override:
            _value_string_original = shinken_element_original.get(item_prop, '')
            _flat_value = _value_string_original
            if _value_string_original == '' and default:
                _value_string_original = default_value
                _flat_value = self.get_default_string_html(default, 'default')
            elif _value_string_original == 'null' and default:
                if _show_data_unit:
                    _flat_value = self.get_default_string_html(default, 'default') + data_unit
                else:
                    _flat_value = self.get_default_string_html(default, 'default')
            elif _value_string_original and default:
                placeholder = '%s ' % _value_string_original
                if _show_data_unit:
                    _flat_value = '%s %s' % (_value_string_original, data_unit)
                else:
                    _flat_value = _value_string_original
            if data_type == 'range_business_impact':
                _flat_value = self.get_default_string_html(self.app._('element.business_impact_%s' % _value_string_original), 'default') + data_unit
        
        extra_attr = 'data-default-btn-status="0"'
        custom_selected = ''
        if item_value == 'null' or value == PROP_DEFAULT_VALUE:
            custom_selected = PROP_DEFAULT_VALUE
            extra_attr = 'data-default-btn-status="1"'
        elif item_value == '' and is_override:
            custom_selected = '__NO_OVERRIDE__'
            value = _value_string_original
        elif item_value == '':
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
            extra_attr = 'data-default-btn-status="1"'
        
        to_return = []
        to_return.append(
            '''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, data_type, default_value, required, custom_selected, class_to_add, unit=data_unit, extra_attr=extra_attr, original=shinken_element_original,
                                                                        mass_change=mass_change))
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, required=required, mass_change=mass_change))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, default_value_label))
        to_return.append('''    <td class="shinken-prop-cell-action"><div class='shinken-range-bloc'>''')
        to_return.append('''       <table class='shinken-table-range'><tr>''')
        to_return.append('''       <td class="shinken-lock-cell"><span class="shinkon-lock"></span></td>''')
        to_return.append('''           <td class='shinken-cell-range-range'>''')
        to_return.append('''               <div class="shinken-range-container">''')
        to_return.append('''                   <input type='range' min=%(data_min)d max=%(data_max)d value='%(value)s' step='%(data_step)s' data-last-value-from-user='%(value)s'
                                                     onchange="MANAGER.EventManager.onMoveRangeCursor('%(item_prop_name_final)s', true, event)" oninput="MANAGER.EventManager.onMoveRangeCursor('%(item_prop_name_final)s', false)" >'''
                         % {"item_prop_name_final": item_prop_name_final, "data_min": data_min, "data_max": data_max, "value": value, "data_step": data_step})
        to_return.append('''               </div>''')
        to_return.append('''           </td>''')
        to_return.append('''           <td class='shinken-cell-range-label'>''')
        if data_type == 'range_with_input':
            to_return.append('''        %s''' % self.helper_component.get_input_html('input', label_range, data_type=data_type, data_unit=data_unit, placeholder=placeholder, type="text", item_prop=item_prop_name_final))
        else:
            to_return.append('''               <div class="shinken-label-input-range">%(label)s</div>''' % {"label": label_range})
        to_return.append('''           </td>''')
        if _show_data_unit:
            to_return.append('''           %s''' % self.get_range_btn_cells(item_prop_name_final, default_value, default_value_label, data_unit, is_override, shinken_element_original, item_prop, _original_value=_value_string_original))
        else:
            to_return.append('''           %s''' % self.get_range_btn_cells(item_prop_name_final, default_value, default_value_label, "", is_override, shinken_element_original, item_prop, _original_value=_value_string_original))
        to_return.append('''       </tr></table>''')
        to_return.append('''    </div></td>''')
        if not mass_change:
            if is_override:
                if isfloat(_flat_value):
                    to_return.append('''%s''' % self.helper_property.get_override_flat_value_html(_flat_value, item_prop, shinken_element_original, data_unit=data_unit))
                else:
                    to_return.append('''%s''' % self.helper_property.get_override_flat_value_html(_flat_value, item_prop, shinken_element_original))
            else:
                to_return.append('''    %s''' % self.get_template_html())
        to_return.append('''</tr>''')
        return ''.join(to_return)
    
    
    def get_bool_input(self, item, item_prop, item_type, required=False, default=None, item_prop_label=None, extra_class='', shinken_element_original=False, is_disabled=False, mass_change=False):
        
        is_override = True if shinken_element_original else False
        class_to_add = 'shinken-disabled' if is_disabled else ''
        
        item_prop_name_final = self.helper_property.build_final_prop_name(shinken_element_original, item_prop, is_override)
        
        if not item_prop_label:
            item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        
        if default is None:
            default = get_default_value(item_type, item_prop)
        
        value = "" if mass_change else item.get(item_prop, '')
        
        custom_selected = ''
        if value == 'null' or value == PROP_DEFAULT_VALUE:
            custom_selected = PROP_DEFAULT_VALUE
        elif value == '' and is_override:
            custom_selected = '__NO_OVERRIDE__'
        elif value == '':
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
        
        s_true = self.app._('element.true')
        s_false = self.app._('element.false')
        if default == True:
            default = s_true
        elif default == False:
            default = s_false
        
        if is_override:
            _value_string_original = shinken_element_original.get(item_prop, '')
            if _value_string_original == '1':
                _value_string_original = s_true
            elif _value_string_original == '0':
                _value_string_original = s_false
            else:
                _value_string_original = self.get_default_string_html(default, 'default')
        
        on_class = ''
        off_class = ''
        unset_class = ''
        override_class = ''
        
        if value == '1' or value == True:
            on_class = 'shinken-active'
        elif value == '0' or value == False:
            off_class = 'shinken-active'
        elif value == 'null' or value == PROP_DEFAULT_VALUE:
            unset_class = 'shinken-active'
        else:
            unset_class = '' if is_override else 'shinken-active'
            override_class = 'shinken-active'
        
        to_return = []
        to_return.append(
            '''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'boolean', default, required, custom_selected, class_to_add, extra_class=extra_class, original=shinken_element_original, mass_change=mass_change))
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, mass_change=mass_change))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, default))
        to_return.append('''    <td class="shinken-prop-cell-action">''')
        to_return.append('''        <div class="shinken-btn-radio-container" >''')
        to_return.append('''            <table class="shinken-btn-group-table"><tr>''')
        to_return.append('''            <td class="shinken-lock-cell"><span class="shinkon-lock"></span></td>''')
        to_return.append('''                %s ''' % self.get_boolean_btn_html(item_prop_name_final, 1, on_class))
        to_return.append('''                %s ''' % self.get_boolean_btn_html(item_prop_name_final, 0, off_class))
        if is_override:
            to_return.append('''                %s ''' % self.get_property_btn_default_html(item_prop_name_final, default, unset_class))
            to_return.append('''                %s ''' % self.get_boolean_btn_no_override_html(item_prop_name_final, override_class, shinken_element_original, item_prop))
        else:
            to_return.append('''                %s ''' % self.get_property_btn_default_html(item_prop_name_final, default, unset_class))
        to_return.append('''            </tr></table>''')
        to_return.append('''        </div>''')
        to_return.append('''    </td>''')
        if not mass_change:
            if is_override:
                to_return.append('''    %s''' % self.helper_property.get_override_flat_value_html(_value_string_original, item_prop, shinken_element_original))
            else:
                to_return.append('''    %s''' % self.get_template_html())
        
        to_return.append('''</tr>''')
        return "".join(to_return)
    
    
    def get_string_input(self, item, item_prop, item_type, item_prop_label=None, input_type='input', placeholder='', is_disabled=False, password=False, shinken_element_original=None,
                         remove_macro_part='', default='', required=False, none_text='', extra_class='', is_discovery=False, mass_change=False):
        
        if mass_change:
            required = False
        is_override = True if shinken_element_original else False
        class_to_add = 'shinken-disabled' if is_disabled else ''
        
        item_prop_name_final = self.helper_property.build_final_prop_name(shinken_element_original, item_prop, is_override)
        if not item_prop_label:
            item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        
        _autocomplete = ""
        _type = 'text'
        if password:
            _type = 'password'
            _autocomplete = "autocomplete='new-password'"
        
        if not placeholder and default:
            placeholder = '%s [%s]' % (self.app._('element.default'), default)
        
        if not placeholder and mass_change:
            placeholder = '%s [%s]' % (self.app._('element.default'), self.app._('element.none'))
        
        _value = "" if mass_change else item.get(item_prop, '')
        
        if item_prop == 'duplicate_foreach' and _value == '_DEFAULT_NO_TEMPLATE__': # J EN AI MARRE DE VOIR DE LA MERDE PARTOUT...
            _value = PROP_DEFAULT_VALUE
        
        if isinstance(_value, basestring):
            _value = _value.replace("'", "&#39;")
        
        if isinstance(_value, str):
            _value = _value.decode('utf8', 'ignore')
        
        custom_selected = ''
        if _value is None or _value == '':
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
        if _value == PROP_DEFAULT_VALUE or _value == 'null':
            custom_selected = PROP_DEFAULT_VALUE
            _value = 'null'
        
        if is_override:
            _value_string_original = shinken_element_original.get(item_prop, '')
            if default and not _value_string_original:
                _value_string_original = self.get_default_string_html(default, 'default')
            if _value_string_original:
                placeholder = self._get_service_overrides_from(item_prop, shinken_element_original, is_placeholder=True)
        
        is_protected = False
        if not is_discovery:
            user = self.app.get_user_auth()
            is_protected = self.app.frontend_cipher.match_protected_property(item_prop, item_type, user)
        
        to_return = []
        to_return.append('''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'input_string', default, required, custom_selected, class_to_add, extra_class=extra_class, original=shinken_element_original,
                                                                                     mass_change=mass_change))
        to_return.append('''    %s''' % remove_macro_part)
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, required=required, mass_change=mass_change))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, default))
        to_return.append('''    <td class="shinken-prop-cell-action">''')
        to_return.append(
            '''        %s''' % self.helper_component.get_input_html(input_type, _value, item_prop=item_prop_name_final, autocomplete=_autocomplete, placeholder=placeholder, type=_type, add_lock=is_disabled,
                                                                    is_protected=is_protected, extra_class_table="shinken-prop-input-table"))
        to_return.append('''    </td>''')
        if not mass_change:
            if is_override:
                to_return.append('''    %s''' % self.helper_property.get_override_flat_value_html(_value_string_original, item_prop, shinken_element_original))
            else:
                to_return.append('''    %s''' % self.get_template_html())
        to_return.append('''</tr>''')
        return "".join(to_return)
    
    
    def get_string_input_with_dropdown(self, item, item_prop, item_type, item_prop_label=None, input_type='input', placeholder='', is_disabled=False, password=False, shinken_element_original=None,
                                       remove_macro_part='', default='', required=False, none_text='', restrict_characters="", extra_class='', is_discovery=False, mass_change=False):
        
        is_override = True if shinken_element_original else False
        class_to_add = 'shinken-disabled' if is_disabled else ''
        
        separator = Helper.INPUT_WITH_DROPDOWN_SEPARATORS.get(item_prop, [','])
        
        item_prop_name_final = self.helper_property.build_final_prop_name(shinken_element_original, item_prop, is_override)
        if not item_prop_label:
            item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        
        _autocomplete = ""
        _type = 'text'
        if password:
            _type = 'password'
            _autocomplete = "autocomplete='new-password'"
        
        if not placeholder and default:
            placeholder = '%s [%s]' % (self.app._('element.default'), default)
        
        if not placeholder and mass_change:
            placeholder = '%s [%s]' % (self.app._('element.default'), self.app._('element.none'))
        
        value = '' if mass_change else item.get(item_prop, '')
        if isinstance(value, basestring):
            value = value.replace("'", "&#39;")
        
        if isinstance(value, str):
            value = value.decode('utf8', 'ignore')
        
        custom_selected = ''
        if value is None or value == '':
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
        if value == PROP_DEFAULT_VALUE or value == 'null':
            custom_selected = PROP_DEFAULT_VALUE
            value = ''
        
        if is_override:
            _value_string_original = shinken_element_original.get(item_prop, '')
            if default and not _value_string_original:
                _value_string_original = self.get_default_string_html(default, 'default')
            if _value_string_original:
                placeholder = self._get_service_overrides_from(item_prop, shinken_element_original, is_placeholder=True)
        
        is_protected = False
        if not is_discovery:
            user = self.app.get_user_auth()
            is_protected = self.app.frontend_cipher.match_protected_property(item_prop, item_type, user)
        
        entries = []
        values = []
        
        if is_override:
            value_original = shinken_element_original.get(item_prop, None)
            values_original = []
            entries_original = []
        
        if value and value != 'null' and value != PROP_DEFAULT_VALUE:
            split_re = re.compile('[%s]' % "".join(separator))
            if value.startswith('+'):
                option_plus_state = BTN_STATE.ON
                value = value[1:]
            values = [s.strip() for s in re.split(split_re, value) if s.strip()]
        
        if is_override and value_original and value_original not in STOP_INHERITANCE_VALUES:
            if value_original.startswith('+'):
                value_original = value_original[1:]
            values_original = [s.strip() for s in value_original.split(',') if s.strip()]
        
        # if not hasattr(item_type_select, '__iter__'):
        #     item_type_select = (item_type_select,)
        
        items = list()
        
        item_type_select = DEF_ITEMS[item_type].get('apply_on_type', item_type)
        has_work_area = ITEM_TYPE.has_work_area(item_type_select)
        
        lookup = {DEF_ITEMS[item_type_select]['key_name']: 1, 'enabled': 1}
        double_link_prop = None
        double_links = DEF_ITEMS[item_type].get('double_links', {})
        for double_link in double_links:
            if double_link.get('my_attr') == item_prop:
                double_link_prop = double_link.get('is_link_with_attr')
                break
        if double_link_prop:
            lookup[double_link_prop] = 1
        
        state_order = (ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING) if has_work_area else (ITEM_STATE.NEW, ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA)
        _items = self.app.datamanagerV2.find_merge_state_items(item_type_select, state_order, lookup=lookup)
        _values = set(values)
        for _item in _items:
            extra_class = ""
            if _item['_id'] == item['_id'] and METADATA.get_metadata(_item, METADATA.ITEM_TYPE) == METADATA.get_metadata(item, METADATA.ITEM_TYPE):
                continue
            
            if item_prop == "edition_contacts" and _item.get('is_admin', '0') == '0' and _item.get('expert', '0') == '0':
                extra_class = "shinken-tag-disabled"
            
            item_name = get_name_from_type(item_type_select, _item)
            selected = item_name in _values
            double_link_value = _item.get_link_item(double_link_prop) if double_link_prop else None
            force_to_null = double_link_value and double_link_value.get_name() in (PROP_DEFAULT_VALUE, VALUE_FORCE_DEFAULT)
            is_hidden_for_admin_si = False
            entries.append(
                ComponentOption(
                    METADATA.get_metadata(_item, METADATA.STATE),
                    _item.get('enabled', '1'),
                    item_name, selected,
                    values.index(item_name) if selected else -1,
                    force_to_null=force_to_null,
                    _id=_item.get('_id', ''),
                    item_type=item_type_select,
                    is_hidden_for_admin_si=is_hidden_for_admin_si,
                    extra_class=extra_class
                )
            )
        items.extend(_items)
        
        name_entries = set([i.value for i in entries])
        # for _value in values:
        #     if _value not in name_entries:
        #         entries.append(ComponentOption('unknown', '1', _value, True, values.index(_value)))
        
        if is_override and value_original and value_original not in STOP_INHERITANCE_VALUES:
            for _value in values_original:
                if _value not in name_entries:
                    entries.append(ComponentOption('unknown', '1', _value, False, values_original.index(_value)))
        
        entries.sort(key=lambda k: natural_keys(k.value))
        
        input_with_dropdown_component = []
        input_with_dropdown_component.append('''        <div class="shinken-input-with-dropdown" ''')
        input_with_dropdown_component.append('''            data-separator="%s" ''' % '$^$'.join(separator))
        input_with_dropdown_component.append('''            data-type="input-string-with-dropdown" data-name="%s">''' % item_prop)
        input_with_dropdown_component.append('''            %s''' % (self.helper_component.get_input_with_drop_down_component(entries, item_prop, value, has_no_research=False, current_research_not_editable=True, type=_type, placeholder=placeholder)))
        input_with_dropdown_component.append('''        </div>''')
        
        to_return = []
        to_return.append(
            '''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'input-string-with-dropdown', default, required, custom_selected, class_to_add, extra_class=extra_class, original=shinken_element_original,
                                                                        mass_change=mass_change))
        to_return.append('''    %s''' % remove_macro_part)
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, required=required))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, default))
        to_return.append('''    <td class="shinken-prop-cell-action">''')
        to_return.append('''        <table class="shinken-prop-cell-action-content">''')
        to_return.append('''            <tr>''')
        to_return.append('''            <td class="shinken-lock-cell"><span class="shinkon-lock"></span></td>''')
        to_return.append('''                <td>%s</td>''' % "\n".join(input_with_dropdown_component))
        to_return.append('''            </tr>''')
        to_return.append('''        </table>''')
        to_return.append('''    </td>''')
        if not mass_change:
            if is_override:
                to_return.append('''    %s''' % self.helper_property.get_override_flat_value_html(_value_string_original, item_prop, shinken_element_original))
            else:
                to_return.append('''    %s''' % self.get_template_html())
        to_return.append('''</tr>''')
        return "".join(to_return)
    
    
    def get_checkbox_input(self, item, item_prop, item_type, required=False, item_prop_label=None, is_disabled=False, default=None, extra_class='', mass_change=False):
        if not item_prop_label:
            item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        
        class_to_add = 'shinken-disabled' if is_disabled else ''
        if default is None:
            default = get_default_value(item_type, item_prop)
            if default:
                default = '1'
            else:
                default = '0'
        
        default_value_label = ''
        default_value = '0'
        
        s_true = self.app._('element.true')
        s_false = self.app._('element.false')
        if default:
            default_value = int(default)
            if default_value == 1:
                default_value_label = s_true
                _class_default_checkbox = "shinken-checked"
            else:
                default_value_label = s_false
        
        _value = "" if mass_change else item.get(item_prop, '')
        custom_selected = ''
        if _value is None or _value == '':
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
        elif _value == PROP_DEFAULT_VALUE:
            custom_selected = PROP_DEFAULT_VALUE
            _value = default_value
        
        to_return = []
        to_return.append('''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'checkbox-single', default, required, custom_selected, class_to_add, extra_class=extra_class, mass_change=mass_change))
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, mass_change=mass_change))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, default))
        to_return.append('''    <td class="shinken-prop-cell-action">''')
        to_return.append('''        <div class="shinken-prop-cell-action-content">''')
        to_return.append('''            <table class='shinken-prop-cell-action-table shinken-width-100'><tr class="shinken-prop-cell-action-row">''')
        to_return.append('''                %s''' % self.helper_object.get_check_box_cell_html(_value, default_value, ''))
        to_return.append('''                %s''' % self.get_property_btn_default_html(item_prop, default_value_label, ''))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_label())
        to_return.append('''            </tr></table>''')
        to_return.append('''        </div>''')
        to_return.append('''    </td>''')
        if not mass_change:
            to_return.append('''    %s''' % self.get_template_html())
        to_return.append('''</tr>''')
        
        return "".join(to_return)
    
    
    def get_select_input(self, item, item_prop, item_type, item_type_select, required=False, shinken_element_original=False, is_disabled=False, item_prop_label=None,
                         default='', default_text='', special_look='', none_text='', where={}, extra_class='', mass_change=False):
        
        is_override = True if shinken_element_original else False
        class_to_add = 'shinken-disabled' if is_disabled else ''
        
        item_prop_name_final = self.helper_property.build_final_prop_name(shinken_element_original, item_prop, is_override)
        
        if not item_prop_label:
            item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        if not default:
            default = "%s" % self.app._('element.none')
        if not none_text:
            none_text = self.app._('element.none')
        
        is_heritable = not ITEM_TYPE.is_template(item_type)
        
        value = None if mass_change else item.get(item_prop, None)
        if value is not None:
            value = value.strip()
        
        all_entries = []
        selected_link_id = ''
        if special_look == 'acl_in_tab_history':
            options = [
                {'text' : self.app._('element.history_sla'),
                 'value': 'history_sla',
                 'state': ['none']},
                {'text' : self.app._('element.history'),
                 'value': 'history',
                 'state': ['none']},
                {'text' : self.app._('element.sla'),
                 'value': 'sla',
                 'state': ['none']}
            ]
            
            default = get_default_value(item_type, item_prop)
            
            if default in ('history_sla', 'history', 'sla'):
                default_text = '%s [ %s ]' % (self.app._('element.default'), self.app._('element.%s' % default))
                default_text_forced = '%s [ %s ]' % (self.app._('element.forced_default'), self.app._('element.%s' % default))
            else:
                default_text = '%s [ %s ]' % (self.app._('element.default'), self.app._('element.unknown_value') % default)
                default_text_forced = '%s [ %s ]' % (self.app._('element.forced_default'), self.app._('element.unknown_value') % default)
            
            for option in options:
                option_value = option.get('value', None)
                selected = (value == option_value)
                entry = ComponentOption('stagging', True, option_value, selected, 0 if selected else -1, label=option.get('text', False), item_type=item_type_select)
                all_entries.append(entry)
        
        elif special_look == 'poller':
            all_poller_tags = set()
            
            for p in self.app.conf.pollers:
                for t in getattr(p, 'poller_tags', []):
                    if t.strip() != 'None':  # Poller tag "None" is the same as Empty.
                        all_poller_tags.add(t.strip())
            if default:
                default_text = self.app._('element.default') + ' [ ' + default + ' ]'
                default_text_forced = self.app._('element.forced_default') + ' [ ' + default + ' ]'
            else:
                default_text = self.app._('element.default') + ' [ ' + none_text + ' ]'
                default_text_forced = self.app._('element.forced_default') + ' [ ' + none_text + ' ]'
            
            is_heritable = False
            for tag in all_poller_tags:
                selected = (value == tag)
                entry = ComponentOption('stagging', True, tag, selected, 0 if selected else -1, item_type=item_type_select)
                all_entries.append(entry)
        
        elif special_look == 'reactionner':
            all_reactionner_tags = set()
            for p in self.app.conf.reactionners:
                for t in getattr(p, 'reactionner_tags', []):
                    if t.strip() != 'None':
                        all_reactionner_tags.add(t.strip())
            
            if default:
                default_text = self.app._('element.default') + ' [ ' + default + ' ]'
                default_text_forced = self.app._('element.forced_default') + ' [ ' + default + ' ]'
            else:
                default_text = self.app._('element.default') + ' [ ' + none_text + ' ]'
                default_text_forced = self.app._('element.forced_default') + ' [ ' + none_text + ' ]'
            
            is_heritable = False
            for tag in all_reactionner_tags:
                selected = (value == tag)
                entry = ComponentOption('stagging', True, tag, selected, 0 if selected else -1, item_type=item_type_select)
                all_entries.append(entry)
        
        elif special_look == 'realm':
            all_realms = set()
            for r in self.app.conf.realms:
                all_realms.add(r.get_name())
                if r.default:
                    # If there is a default realm, say it :)
                    default = r.get_name()
                    none_text = '%s (%s)' % (self.app._('element.forced_default'), default)
            if default:
                default_text = self.app._('element.default') + ' [ ' + default + ' ]'
                default_text_forced = self.app._('element.forced_default') + ' [ ' + default + ' ]'
            else:
                default_text = self.app._('element.untagged')
                default_text_forced = self.app._('element.untagged')
            
            is_heritable = True
            for realm in all_realms:
                selected = (value == realm)
                entry = ComponentOption('stagging', True, realm, selected, 0 if selected else -1, item_type=item_type_select)
                all_entries.append(entry)
        
        else:
            key = DEF_ITEMS[item_type_select]['key_name']
            items = self.app.datamanagerV2.find_merge_state_items(item_type_select, (ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING), where=where, lookup={key: 1, 'enabled': 1})
            all_entries = []
            default_text = '%s [%s]' % (self.app._('element.default'), default)
            default_text_forced = '%s [%s]' % (self.app._('element.forced_default'), default)
            for _item in items:
                name = _item.get(key)
                state = []
                private_state = METADATA.get_metadata(_item, METADATA.STATE)
                if private_state == ITEM_STATE.NEW:
                    state.append('new')
                elif private_state == ITEM_STATE.WORKING_AREA:
                    state.append('working_area')
                elif private_state == ITEM_STATE.STAGGING:
                    state.append('stagging')
                else:
                    logger.error('items state unknown:[%s]' % private_state)
                
                if _item.get('enabled', '1') == '0':
                    state.append('disable')
                
                selected = (name == value)
                entry = ComponentOption(METADATA.get_metadata(_item, METADATA.STATE), _item.get('enabled', '1'), name, selected, value.index(name) if selected else -1, _id=_item.get('_id', ''), item_type=item_type_select)
                if selected and entry.id_link:
                    selected_link_id = entry.id_link
                all_entries.append(entry)
        
        custom_selected = ''
        if mass_change:
            custom_selected = ENTER_VALUE_FOR_MASS_CHANGE
        elif value is None:
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
            final_value = default_text
        elif value == 'null':
            value = ''
            custom_selected = VALUE_FORCE_DEFAULT
            final_value = default_text_forced
        elif value == PROP_DEFAULT_VALUE:
            final_value = default_text
            custom_selected = PROP_DEFAULT_VALUE
            value = ''
        elif value == '' and is_override:
            custom_selected = '__NO_OVERRIDE__'
        
        else:
            name_entries = set([i.value for i in all_entries])
            if value not in name_entries:
                all_entries.append(ComponentOption('unknown', '1', value, True, 0))
        
        if special_look != 'acl_in_tab_history':
            all_entries.sort(key=lambda k: natural_keys(k.value))
        
        _value_string_original = ''
        if is_override and shinken_element_original:
            _value_string_original = shinken_element_original.get(item_prop, '')
            
            selected_entry = next((entry for entry in all_entries if entry.value == _value_string_original), None)
            
            if _value_string_original == '' or _value_string_original in STOP_INHERITANCE_VALUES:
                _value_string_original = self.get_default_string_html(default, 'default')
            elif self.app.get_user_auth().is_admin() and selected_entry and selected_entry.item_type and selected_entry.id_link:
                _value_string_original = self.helper_object.build_link_dom_element_depend_if_admin(selected_entry.item_type, selected_entry.id_link, selected_entry.value)
        
        selector_html = self.helper_component.get_selector_html(
            item_prop=item_prop_name_final,
            default_text=default_text,
            default_text_forced=default_text_forced,
            entries=all_entries,
            none_text=none_text,
            element_in_check_text=self._get_service_overrides_from(item_prop, shinken_element_original),
            can_have_link=True,
            mass_change=mass_change,
            has_no_default=False,
            has_no_research=False,
        )
        
        to_return = []
        to_return.append('''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'select-single', default, required, custom_selected, class_to_add, default_label=default_text, none_label=none_text,
                                                                                     original=shinken_element_original, item_type_select=item_type_select, link_id=selected_link_id, mass_change=mass_change))
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, mass_change=mass_change))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, default))
        to_return.append('''    <td class="shinken-prop-cell-action">''')
        to_return.append('''        <table class="shinken-prop-cell-action-content">''')
        to_return.append('''            <tr class="shinken-prop-cell-action-row">''')
        to_return.append('''            <td class="shinken-lock-cell"><span class="shinkon-lock"></span></td>''')
        to_return.append('''                %s''' % self.helper_component.get_multi_selector_html("main-selector", selector_html))
        to_return.append('''            </tr>''')
        to_return.append('''        </table>''')
        to_return.append('''    </td>''')
        if not mass_change:
            if is_override:
                to_return.append('''    %s''' % self.helper_property.get_override_flat_value_html(_value_string_original, item_prop, shinken_element_original))
            else:
                to_return.append('''    %s''' % self.get_template_html())
        to_return.append('''</tr>''')
        return ''.join(to_return)
    
    
    def get_command_input(self, item, item_prop, item_type, item_type_select, item_prop_label=None, required=False, shinken_element_original=False,
                          is_disabled=False, default='', default_text='', none_text='', extra_class='', mass_change=False):
        if mass_change:
            required = False
        is_override = True if shinken_element_original else False
        class_to_add = ''
        if is_disabled == 2:
            class_to_add = 'shinken-partial-command-disabled'
        elif is_disabled:
            class_to_add = 'shinken-disabled'
        
        item_prop_name_final = self.helper_property.build_final_prop_name(shinken_element_original, item_prop, is_override)
        
        if not item_prop_label:
            item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        
        key = DEF_ITEMS[item_type_select]['key_name']
        items = self.app.datamanagerV2.find_merge_state_items(item_type_select, (ITEM_STATE.NEW, ITEM_STATE.STAGGING), where={}, lookup={key: 1, 'enabled': 1})
        all_entries = []
        
        input_type = 'input'
        bp_rulling_tooltip = ''
        if not mass_change and item.get('check_command', '').startswith('bp_rule'):
            class_to_add = 'shinken-disabled'
            input_type = 'textarea'
            bp_rulling_tooltip = 'onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" shi-tip-html="%s"' % (self.app._("element.cannot_edit_from_interface"))
        
        value = None if mass_change else item.get(item_prop)
        cmd_name = ''
        cmd_args = ''
        if value is not None:
            value = value.replace("\"", '&quot;')
            value = value.replace("\'", '&apos;')
            elts = value.split('!', 1)
            cmd_name = elts[0]
            if len(elts) > 1:
                cmd_args = elts[1]
        
        if is_override and item_prop == 'check_command':
            cmd_args = item.get('check_command_args', '')
        
        if not default:
            default = "%s" % self.app._('element.none')
        
        if not none_text or none_text == '':
            none_text = self.app._('element.none')
        
        if not default_text or default_text == '':
            default_text = '%s [ %s ]' % (self.app._('element.default'), default)
        default_text_forced = '%s [ %s ]' % (self.app._('element.forced_default'), default)
        
        selected_link_id = ''
        for _item in items:
            name = _item.get(key)
            selected = (name == cmd_name)
            entry = ComponentOption(METADATA.get_metadata(_item, METADATA.STATE), _item.get('enabled', '1'), name, selected, cmd_name.index(name) if selected else -1, _id=_item.get('_id', ''), item_type=item_type_select)
            if selected and entry.id:
                selected_link_id = entry.id_link
            all_entries.append(entry)
        
        custom_selected = ''
        if mass_change:
            custom_selected = ENTER_VALUE_FOR_MASS_CHANGE
        elif value is None:
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
        elif value == 'null':
            custom_selected = VALUE_FORCE_DEFAULT
        elif value == PROP_DEFAULT_VALUE:
            custom_selected = PROP_DEFAULT_VALUE
        else:
            name_entries = set([i.value for i in all_entries])
            if cmd_name not in name_entries:
                all_entries.append(ComponentOption('unknown', '1', cmd_name, True, 0))
        
        all_entries.sort(key=lambda k: natural_keys(k.value))
        
        placeholder = ''
        _autocomplete = ''
        
        if is_override:
            _value_string_original = shinken_element_original.get(item_prop, '')
            if _value_string_original == '' or _value_string_original in STOP_INHERITANCE_VALUES:
                _value_string_original = self.get_default_string_html(default, 'default')
            
            _value_string_original_temp = _value_string_original.replace("\"", '&quot;')
            _value_string_original_temp = _value_string_original_temp.replace("\'", '&apos;')
            _elts_original = _value_string_original_temp.split('!', 1)
            element_in_check_text = _elts_original[0]
            element_in_check_final = ''
            selected_entry = next((entry for entry in all_entries if entry.value == element_in_check_text), None)
            if len(_elts_original) > 1:
                element_in_check_final = '!%s' % _elts_original[1]
                placeholder = _elts_original[1]
            if self.app.get_user_auth().is_admin() and selected_entry and selected_entry.item_type and selected_entry.id_link:
                _value_string_original = '%s%s' % (self.helper_object.build_link_dom_element_depend_if_admin(selected_entry.item_type, selected_entry.id_link, selected_entry.value), element_in_check_final)
            if element_in_check_text.startswith('bp_rule'):
                bp_rulling_tooltip = 'onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" shi-tip-html="%s"' % (self.app._("element.cannot_edit_from_interface"))
                class_to_add += " shinken-has-bp-rules"
            elif item_prop == 'check_command':
                bp_rulling_tooltip = 'onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" shi-tip-html="%s"' % (self.app._("element.cannont_select_command"))
        
        selector_html = self.helper_component.get_selector_html(
            item_prop=item_prop_name_final,
            default_text=default_text,
            default_text_forced=default_text_forced,
            entries=all_entries,
            none_text=none_text,
            element_in_check_text=self._get_service_overrides_from(item_prop, shinken_element_original),
            can_have_link=True,
            mass_change=mass_change,
            has_no_default=False,
            has_no_research=False,
        )
        args_html = self.get_check_command_input_html(input_type, cmd_args, item_prop_name_final, _autocomplete, placeholder)
        
        to_return = []
        to_return.append(
            '''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'select-single-with-input', default, required, custom_selected, class_to_add, default_label=default_text, none_label=none_text,
                                                                        original=shinken_element_original,
                                                                        item_type_select=item_type_select, link_id=selected_link_id, mass_change=mass_change))
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, required=required, mass_change=mass_change))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, default))
        to_return.append('''    <td class="shinken-prop-cell-action">''')
        to_return.append('''        <table class="shinken-prop-cell-action-content">''')
        to_return.append('''            <tr class="shinken-not-allowed-if-disabled shinken-prop-cell-action-row">''')
        to_return.append('''            <td class="shinken-lock-cell"><span class="shinkon-lock"></span></td>''')
        to_return.append('''                %s''' % self.helper_component.get_multi_selector_html("main-selector", selector_html))
        to_return.append('''            </tr>''')
        to_return.append('''            <tr class="shinken-not-allowed-if-disabled shinken-prop-cell-action-row">''')
        to_return.append('''                <td colspan='2' class="shinken-args-container-cell">%s</td>''' % args_html)
        to_return.append('''            </tr>''')
        to_return.append('''        </table>''')
        to_return.append('''    </td>''')
        if not mass_change:
            if is_override:
                to_return.append('''    %s''' % self.helper_property.get_override_flat_value_html(_value_string_original, item_prop, shinken_element_original))
            else:
                to_return.append('''    %s''' % self.get_template_html(extra_tooltip=bp_rulling_tooltip))
        to_return.append('''</tr>''')
        return ''.join(to_return)
    
    
    def get_multiselect_input(self, item, item_prop, item_type, item_type_select, required=False, shinken_element_original=False, is_disabled=False, item_prop_label=None, where=None, default=None, is_working_area=False, mass_change=False):
        
        is_override = True if shinken_element_original else False
        class_to_add = 'shinken-disabled' if is_disabled else ''
        
        item_prop_name_final = self.helper_property.build_final_prop_name(shinken_element_original, item_prop, is_override)
        if item_type in [ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESCLUSTERS] and item_prop == 'host_name':
            is_order_kept = ''
        else:
            is_order_kept = "1" if get_property_def(item_type, item_prop).merging == 'ordered' else ''
        
        option_plus_state = BTN_STATE.DISABLED
        if item_prop in DEF_ITEMS[item_type].get('fields_with_plus', []):
            option_plus_state = BTN_STATE.OFF
            if is_override:
                option_plus_state = BTN_STATE.DISCREET
        
        if not item_prop_label:
            item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        
        if default is None:
            default = get_default_value(item_type, item_prop)
            if default and isinstance(default, (list, tuple)):
                default = default[0]
            if item_prop == 'view_contacts' and default not in ('nobody', 'everyone'):
                default = 'nobody'
        
        if default and default != none_object:
            default_text = '%s [ %s ]' % (self.app._('element.default'), self.app._('element.%s' % default))
            default_text_forced = '%s [ %s ]' % (self.app._('element.forced_default'), self.app._('element.%s' % default))
            none_text = '%s' % (self.app._('element.%s' % default))
        else:
            default = None
            default_text = '%s [ %s ]' % (self.app._('element.default'), self.app._('element.none'))
            default_text_forced = '%s [ %s ]' % (self.app._('element.forced_default'), self.app._('element.none'))
            none_text = '%s' % (self.app._('element.none'))
        
        if where is None:
            where = {}
        
        double_link_prop = None
        double_links = DEF_ITEMS[item_type].get('double_links', {})
        for double_link in double_links:
            if double_link.get('my_attr') == item_prop:
                double_link_prop = double_link.get('is_link_with_attr')
                break
        
        entries = []
        values = []
        value = None if mass_change else item.get(item_prop, None)
        
        if is_override:
            value_original = shinken_element_original.get(item_prop, None)
            values_original = []
            entries_original = []
        
        if value and value != 'null' and value != PROP_DEFAULT_VALUE:
            if value.startswith('+'):
                option_plus_state = BTN_STATE.ON
                value = value[1:]
            values = [s.strip() for s in value.split(',') if s.strip()]
        
        if is_override and value_original and value_original not in STOP_INHERITANCE_VALUES:
            if value_original.startswith('+'):
                value_original = value_original[1:]
            values_original = [s.strip() for s in value_original.split(',') if s.strip()]
        
        if not hasattr(item_type_select, '__iter__'):
            item_type_select = (item_type_select,)
        
        items = list()
        
        for _item_type_select in item_type_select:
            
            lookup = {DEF_ITEMS[_item_type_select]['key_name']: 1, 'enabled': 1}
            if double_link_prop:
                lookup[double_link_prop] = 1
            
            state_order = (ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING) if is_working_area else (ITEM_STATE.NEW, ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA)
            _items = self.app.datamanagerV2.find_merge_state_items(
                _item_type_select,
                state_order,
                where=where,
                lookup=lookup
            )
            _values = set(values)
            for _item in _items:
                extra_class = ""
                if not mass_change and _item['_id'] == item['_id'] and METADATA.get_metadata(_item, METADATA.ITEM_TYPE) == METADATA.get_metadata(item, METADATA.ITEM_TYPE):
                    continue
                
                if item_prop == "edition_contacts" and _item.get('is_admin', '0') == '0' and _item.get('expert', '0') == '0':
                    extra_class = "shinken-tag-disabled"
                
                item_name = get_name_from_type(_item_type_select, _item)
                selected = item_name in _values
                double_link_value = _item.get_link_item(double_link_prop) if double_link_prop else None
                force_to_null = double_link_value and double_link_value.get_name() in (PROP_DEFAULT_VALUE, VALUE_FORCE_DEFAULT)
                is_hidden_for_admin_si = False
                if ITEM_TYPE.is_template(_item_type_select):
                    is_hidden_for_admin_si = _item.get('for_all_users', '1') != '1'
                entries.append(
                    ComponentOption(
                        METADATA.get_metadata(_item, METADATA.STATE),
                        _item.get('enabled', '1'),
                        item_name, selected,
                        values.index(item_name) if selected else -1,
                        force_to_null=force_to_null,
                        _id=_item.get('_id', ''),
                        item_type=_item_type_select,
                        is_hidden_for_admin_si=is_hidden_for_admin_si,
                        extra_class=extra_class,
                        label="--" if is_disabled else ""
                    )
                )
            items.extend(_items)
        
        name_entries = set([i.value for i in entries])
        for _value in values:
            if _value not in name_entries:
                entries.append(ComponentOption('unknown', '1', _value, True, values.index(_value)))
        
        if is_override and value_original and value_original not in STOP_INHERITANCE_VALUES:
            for _value in values_original:
                if _value not in name_entries:
                    entries.append(ComponentOption('unknown', '1', _value, False, values_original.index(_value)))
        
        entries.sort(key=lambda k: natural_keys(k.value))
        
        custom_selected = ''
        if mass_change:
            custom_selected = ENTER_VALUE_FOR_MASS_CHANGE
        elif value is None:
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
        elif value == '':
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
        elif value == 'null':
            custom_selected = VALUE_FORCE_DEFAULT
        elif value == PROP_DEFAULT_VALUE:
            custom_selected = '__DEFAULT_NO_TEMPLATE_FORCED__' if is_override else PROP_DEFAULT_VALUE
        
        # create the HTML : option + and the value column
        if option_plus_state == BTN_STATE.DISABLED:
            html_option_plus = ''
        else:
            html_option_plus = '''<td class="shinken-btn-plus-cell" >''' \
                               '''  <div class="container-btn-plus"  onclick="MANAGER.EventManager.clickOnBtnPlus('%s')" >''' \
                               '''      %s ''' \
                               '''  </div> ''' \
                               '''</td> ''' % (item_prop_name_final, self.helper_object.build_icons_plus())
        
        # item_prop, selector_type, default_text, default_text_forced, entries, none_text, element_in_check_text
        selector_html = self.helper_component.get_selector_html(
            item_prop=item_prop_name_final,
            default_text=default_text,
            default_text_forced=default_text_forced,
            entries=entries,
            none_text=none_text,
            element_in_check_text=self._get_service_overrides_from(item_prop, shinken_element_original),
            has_plus=option_plus_state is not BTN_STATE.DISABLED,
            mass_change=mass_change,
            has_no_default=False,
            has_no_research=False,
            selected_class="shinken-select-no-events"
        )
        
        select_entries = [entry for entry in entries if entry.selected]
        
        is_null_selected = ''
        is_default_selected = ''
        
        if value == 'null':
            is_null_selected = 'shinken-selected'
        elif value == PROP_DEFAULT_VALUE:
            is_default_selected = 'shinken-selected'
        
        if is_default_selected:
            select_entries.append(ComponentOption('hidden', '1', PROP_DEFAULT_VALUE, True, 0, 'hidden'))
        elif is_null_selected:
            select_entries.append(ComponentOption('hidden', '1', 'null', True, 0, 'hidden'))
        
        if is_order_kept:
            select_entries.sort(key=lambda k: k.selected_order)
        
        selected_list = []
        for entry in select_entries:
            if entry.value not in STOP_INHERITANCE_VALUES:
                selected_list.append('%s' % self.helper_object.get_tag_container_parent_html(item_prop_name_final, entry))
        
        if is_override:
            select_entries_original = [entry for entry in entries if entry.value in values_original]
            selected_list_original = []
            for entry_original in select_entries_original:
                selected_list_original.append('%s' % self.helper_object.get_tag_container_parent_html(item_prop_name_final, entry_original))
        
        _value_string_original = ''
        if is_override and shinken_element_original:
            _value_string_original = ''.join(selected_list_original)
            if default:
                if _value_string_original == '':
                    _value_string_original = self.get_default_string_html(self.app._('element.%s' % default), 'default')
                elif _value_string_original == 'null':
                    _value_string_original = default_text
                elif _value_string_original == '__DEFAULT_NO_TEMPLATE__':
                    _value_string_original = default_text_forced
        
        # Close the list and the value column
        if is_override:
            override_list_html = '''<div class="shinken-list-override" >%s</div>''' % ''.join(selected_list)
            selected_list_html = '''<div class="list-selected-in-select" ></div>'''
        else:
            override_list_html = '''<div class="shinken-list-override" ></div>'''
            selected_list_html = '''<div class="list-selected-in-select" >%s</div>''' % ''.join(selected_list)
        inherited_list_html = '''<div class="list-inherited" ></div>'''
        
        multiselect_component = self.helper_component.get_multi_selector_html("main-selector", selector_html, selected_list_html, override_list_html, inherited_list_html)
        
        # Construct the line
        to_return = []
        to_return.append(
            '''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'select-multi', default, required, custom_selected, class_to_add, default_label=default_text, none_label=none_text, state_plus=option_plus_state,
                                                                        original=shinken_element_original, mass_change=mass_change, is_order_kept=is_order_kept))
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, mass_change=mass_change))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, default))
        to_return.append('''    <td class="shinken-prop-cell-action">''')
        to_return.append('''        <table class="shinken-prop-cell-action-content">''')
        to_return.append('''            <tr class="shinken-prop-cell-action-row">''')
        to_return.append('''            <td class="shinken-lock-cell"><span class="shinkon-lock"></span></td>''')
        if not mass_change:
            to_return.append('''                %s''' % html_option_plus)
        to_return.append('''                %s''' % multiselect_component)
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_single_for_multiselect_html("mass_change_element_single", item_type, item_prop, entries, default_text))
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_extra_multiselect_html(item_type, item_prop))
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_label(self.app._('mass-change.label.by')))
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_single_for_multiselect_html("mass_change_element_extra", item_type, item_prop, entries, default_text))
        to_return.append('''            </tr>''')
        to_return.append('''        </table>''')
        to_return.append('''    </td>''')
        if not mass_change:
            if is_override:
                to_return.append('''    %s''' % self.helper_property.get_override_flat_value_html(_value_string_original, item_prop, shinken_element_original))
            else:
                to_return.append('''    %s''' % self.get_template_html())
        to_return.append('''</tr>''')
        return ''.join(to_return)
    
    
    def get_data_property_html(self, item, item_type, item_prop, required=False, shinken_element_original=False, is_disabled=False, extra_class='', mass_change=False):
        is_override = True if shinken_element_original else False
        class_to_add = 'shinken-disabled' if is_disabled else ''
        item_prop_value = '' if mass_change else item.get(item_prop, '')
        item_prop_label = item_prop[1:]
        
        custom_selected = ''
        if item_prop_value is None or item_prop_value == '':
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
        
        _value_string_original = ''
        placeholder = ''
        is_protected = False
        if is_override:
            user = self.app.get_user_auth()
            placeholder = self._get_service_overrides_from(item_prop, shinken_element_original, is_placeholder=True)
            if self.app.frontend_cipher.match_protected_property(item_prop, item_type, user):
                _value_string_original = self.app._('element.password_protected')
                is_protected = True
            else:
                _value_string_original = shinken_element_original.get(item_prop, '')
        elif mass_change:
            user = self.app.get_user_auth()
            placeholder = '%s [%s]' % (self.app._('element.default'), self.app._('element.none'))
            if self.app.frontend_cipher.match_protected_property(item_prop, item_type, user):
                _value_string_original = self.app._('element.password_protected')
                is_protected = True
        
        tooltip_data_disabled = ''
        if is_override and extra_class == 'shinken-disabled-and-unknown':
            tooltip_data_disabled = 'onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" shi-tip-type="shinken-warning-border" shi-tip-html="%s"' % (self.app._("element.cannot_edit_invalid_overriden_data"))
        
        to_return = []
        to_return.append(
            '''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'data', '', required, custom_selected, class_to_add, original=shinken_element_original, extra_class=extra_class, mass_change=mass_change))
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, has_delete_btn=True, mass_change=mass_change))
        
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, ""))
            to_return.append('''<td class="shinken-hidden  shinken-list-cell" data-filter="data_name" data-filter-value="%s"></td>''' % item_prop_label)
            to_return.append('''<td class="shinken-hidden  shinken-list-cell" data-filter="data_name_array" data-filter-value="%s"></td>''' % item_prop)
        
        _type_shinken_input = 'input' if is_protected else 'textarea'
        to_return.append('''        <td class="shinken-prop-cell-action">%s</td>''' % self.helper_component.get_input_html(
            _type_shinken_input, item_prop_value, item_prop=item_prop, placeholder=placeholder, type='text', add_lock=is_disabled,
            add_attribute=tooltip_data_disabled, extra_class_table="shinken-prop-cell-action-content", is_protected=is_protected))
        if not mass_change:
            if is_override:
                to_return.append('''    %s''' % self.helper_property.get_override_flat_value_html(_value_string_original, item_prop, shinken_element_original))
            else:
                to_return.append('''    %s''' % self.get_template_html())
        to_return.append('''</tr>''')
        
        return "".join(to_return)
    
    
    def get_line_checkbox_share_acl(self, item, item_prop, item_type, required=False, is_disabled=False, mass_change=False):
        
        item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        item_prop_name_final = item_prop
        _acl_share_prop = DEF_ITEMS[item_type]['class'].properties[item_prop]
        default = get_default_value(item_type, item_prop)
        
        if hasattr(_acl_share_prop, 'unpythonize'):
            default_unpythonize = _acl_share_prop.unpythonize(_acl_share_prop.default)
        else:
            default_unpythonize = default
        
        _value = "" if mass_change else item.get(item_prop, '')
        _value = _acl_share_prop.pythonize(_value)
        
        default_class = ""
        custom_selected = ""
        if _value in ('null', PROP_DEFAULT_VALUE):
            custom_selected = PROP_DEFAULT_VALUE
            default_class = 'shinken-active'
            _value = default if default is not None else '     '
        elif _value == '':
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
            _value = default if default is not None else '     '
        
        _value_all = '1' if _value == '11111' else '0'
        _value_all_default = '1' if default == '11111' else '0'
        
        class_to_add = 'shinken-disabled' if is_disabled else ''
        to_return = []
        to_return.append('''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'checkbox-multi', default, required, custom_selected, class_to_add, mass_change=mass_change))
        to_return.append('''    %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, mass_change=mass_change))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, default_unpythonize))
        to_return.append('''    <td class="shinken-prop-cell-action">''')
        to_return.append('''        <div class="shinken-prop-cell-action-content">''')
        to_return.append('''            <table class='shinken-prop-cell-action-table shinken-width-100'><tr class="shinken-prop-cell-action-row">''')
        to_return.append('''                %s''' % self.helper_object.get_check_box_cell_html(_value[4], default[4], 'read', class_to_add="authorization_share_cell"))
        to_return.append('''                %s''' % self.helper_object.get_check_box_cell_html(_value[3], default[3], 'create', class_to_add="authorization_share_cell"))
        to_return.append('''                %s''' % self.helper_object.get_check_box_cell_html(_value[2], default[2], 'organize', class_to_add="authorization_share_cell"))
        to_return.append('''                %s''' % self.helper_object.get_check_box_cell_html(_value[1], default[1], 'modify', class_to_add="authorization_share_cell"))
        to_return.append('''                %s''' % self.helper_object.get_check_box_cell_html(_value[0], default[0], 'delete', class_to_add="authorization_share_cell"))
        to_return.append('''                %s''' % self.helper_object.get_check_box_cell_html(_value_all, _value_all_default, 'all', class_to_add="authorization_share_cell"))
        to_return.append('''                %s ''' % self.get_property_btn_default_html(item_prop_name_final, default_unpythonize, default_class))
        if mass_change:
            to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_label())
        to_return.append('''            </tr></table>''')
        to_return.append('''        </div>''')
        to_return.append('''    </td>''')
        if not mass_change:
            to_return.append('''    %s''' % self.get_template_html())
        to_return.append('''</tr>''')
        return "".join(to_return)
    
    
    def get_multiple_line_object(self, item, item_prop, item_type, is_disabled=False, required=False, shinken_element_original=False, separator_line_saved=",", separator_key_value_saved=" ", separator_line_html=",", separator_key_value_html=" ",
                                 type=False, hide_import_key=0, mass_change=False):
        is_override = True if shinken_element_original else False
        class_to_add = 'shinken-disabled' if is_disabled else ''
        
        item_prop_name_final = self.helper_property.build_final_prop_name(shinken_element_original, item_prop, is_override)
        item_prop_label = self.app._("%s.%s" % (item_type, item_prop))
        
        if not type:
            type = item_prop
        
        _value = "" if mass_change else item.get(item_prop, '')
        
        custom_selected = ''
        lines_object = []
        if _value is None or _value == '':
            custom_selected = PROP_DEFAULT_OR_INHERITANCE
        elif _value == PROP_DEFAULT_VALUE or _value == 'null':
            custom_selected = PROP_DEFAULT_VALUE
            _value = ''
        else:
            if type == 'output_rules':
                from shinken.objects.resultmodulation import ModulationParser
                tokenizer = ModulationParser()
                try:
                    tokens = tokenizer.tokenize(_value)
                    lines_object.extend(tokenizer.parse(tokens))
                except ModulationSyntaxError:
                    pass
            elif type == TIME_PERIOD_TYPE.DAY:
                from shinkensolutions.time_period_parser import TimePeriodParser
                parser = TimePeriodParser()
                tokens = parser.tokenize_day(_value)
                parsed = parser.parse_day(tokens)
                _value = parsed['value']
                lines_object = parsed['defs']
                separator_line_saved = TimePeriodParser.SEPARATOR_DAY_DEF
            elif type == TIME_PERIOD_TYPE.ADVANCED:
                from shinkensolutions.time_period_parser import TimePeriodParser
                parser = TimePeriodParser()
                tokens = parser.tokenize_advanced(_value)
                lines_object = parser.parse_advanced(tokens)
            elif type == 'notes_multi_url':
                lines_object = [_current.split(separator_key_value_saved) for _current in _value.split(separator_line_saved)]
        
        is_set = 'true' if len(lines_object) else 'false'
        
        _value_string_original = ''
        if is_override:
            _value_string_original = shinken_element_original.get(item_prop, '')
        
        extra_attribute = 'data-separator-line-saved="%s" data-separator-key-value-saved="%s" data-separator-line-html="%s" data-separator-key-value-html="%s" data-line-type="%s"' % (
            separator_line_saved, separator_key_value_saved, separator_line_html, separator_key_value_html, type)
        
        _to_return = []
        _to_return.append(
            '''<tr %s>''' % self.helper_property.get_root_property_html(item, item_type, item_prop, 'multi-line-object', '', required, custom_selected, class_to_add, extra_attr=extra_attribute, original=shinken_element_original,
                                                                        hide_import_key=hide_import_key,
                                                                        is_set=is_set, mass_change=mass_change))
        if mass_change:
            _to_return.append('''    <td colspan="4">''')
        else:
            _to_return.append('''    <td colspan="2">''')
        _to_return.append('''        <div class="shinken-multi-line-object-div-container">''')
        _to_return.append('''            %s''' % self.get_multiple_line_object_main_line(item_prop_name_final, item_prop_label, _value, type, required, item_type, is_disabled=is_disabled, mass_change=mass_change))
        _to_return.append('''            <table class="shinken-width-100 shinken-table-content-part">''')
        _to_return.append('''                %s ''' % self.get_multi_line_header(item_prop))
        _to_return.append('''                <tbody class="shinken-multi-line-object-container">''')
        if len(lines_object):
            for _line in lines_object:
                _to_return.append('''                <tr class="shinken-multi-line-object-row">''')
                _to_return.append('''                    %s ''' % self.get_multi_line_object(item_prop_name_final, type, _line))
                _to_return.append('''                </tr>''')
        _to_return.append('''                </tbody>''')
        _to_return.append('''            </table>''')
        if type != TIME_PERIOD_TYPE.DAY:
            _to_return.append('''            <table class="shinken-width-100 shinken-table-cfg-format-part">''')
            _to_return.append('''                <tr>''')
            _to_return.append('''                    <td class="shinken-multi-line-cell shinken-cfg-title-cell"><div class="shinken-cfg-title">%s</div></td>''' % self.app._('multi-line-object.cfg_format'))
            _to_return.append('''                    <td class="shinken-multi-line-cell shinken-rules-value-if-inheritance-on">''')
            _to_return.append('''                       <div class="shinken-no-value">''')
            _to_return.append('''                           %s''' % self.get_shinken_copy_butons())
            _to_return.append('''                       </div>''')
            _to_return.append('''                       <div class="shinken-cfg-inherited-label-container shinken-parent-text-to-copy">''')
            _to_return.append('''                           %s''' % self.get_shinken_copy_butons())
            if is_override:
                _to_return.append('''                       <span class="shinken-multi-line-object-stringify-property">%s</span>''' % item_prop)
                _to_return.append('''                       <span class="shinken-multi-line-object-inherited-value">%s</span>''' % _value_string_original)
            
            _to_return.append('''                       </div>''')
            _to_return.append('''                    </td>''')
            _to_return.append('''                    <td class="shinken-multi-line-cell shinken-rules-value-if-inheritance-off">''')
            _to_return.append('''                        <div class="shinken-multi-line-object-stringify-container shinken-parent-text-to-copy">''')
            _to_return.append('''                            %s''' % self.get_shinken_copy_butons())
            _to_return.append('''                            <div class="shinken-cfg-text-container shinken-text-to-copy">''')
            _to_return.append('''                               <span class="shinken-multi-line-object-stringify-property">%s</span>''' % item_prop)
            _to_return.append('''                               <span class="shinken-multi-line-object-stringify-result" onfocusout="MANAGER.EventManager.onFocusOut(event)">%s</span>''' % _value)
            _to_return.append('''                            </div">''')
            _to_return.append('''                        </div>''')
            _to_return.append('''                    </td>''')
            _to_return.append('''                </tr>''')
            _to_return.append('''            </table>''')
        _to_return.append('''        </div>''')
        _to_return.append('''    </td>''')
        if not mass_change:
            if is_override:
                _to_return.append('''    %s''' % self.helper_property.get_override_flat_value_html(_value_string_original, item_prop, shinken_element_original))
            else:
                _to_return.append('''    %s''' % self.get_template_html())
        _to_return.append('''</tr>''')
        return "".join(_to_return)
    
    
    def get_property_group(self, type_separator, title=''):
        
        _to_return = []
        _to_return.append('''<tr class="shinken-property-separator" data-type="%s">''' % type_separator)
        _to_return.append('''    <td colspan="99">''')
        _to_return.append('''        <div class="shinken-property-separator-div">%s</div>''' % title)
        _to_return.append('''    </td>''')
        _to_return.append('''</tr>''')
        return "".join(_to_return)
    
    
    def get_property_sub_group_start(self, type_separator, title='', mass_change=False):
        
        _to_return = []
        _to_return.append('''<tr class="shinken-property-separator" data-type="%s">''' % type_separator)
        _to_return.append('''   <td colspan="99">''')
        _to_return.append('''        <div class="shinken-property-separator-sub-div">%s</div>''' % title)
        _to_return.append('''   </td>''')
        if not mass_change:
            _to_return.append('''   %s''' % self.get_template_html())
        _to_return.append('''</tr>''')
        _to_return.append('''<tr>''')
        _to_return.append('''   <td colspan="99">''')
        _to_return.append('''       <table class="shinken-prop-table shinken-sub-table">''')
        return "".join(_to_return)
    
    
    @staticmethod
    def get_property_sub_group_end():
        
        _to_return = []
        _to_return.append('''       </table>''')
        _to_return.append('''   </td>''')
        _to_return.append('''</tr>''')
        return "".join(_to_return)
    
    
    # *****************************   COMMON FUNCTIONS  ************************************************************#
    
    def get_multiple_line_object_main_line(self, item_prop, item_prop_label, value, type, required, item_type, is_disabled=False, mass_change=False):
        key_translate = 'multi-line-object.%s' % type
        _label_button_plus = self.app._(key_translate) if self.app.is_translate_existing(key_translate) else ''
        
        _to_return = []
        _to_return.append('''<table class="shinken-width-100 shinken-table-common-part">''')
        _to_return.append('''    <tr>''')
        _to_return.append('''        %s''' % self.helper_property.get_cell_label_property_html(item_prop_label, required=required, mass_change=mass_change))
        if mass_change:
            _to_return.append('''    %s''' % self.mass_change_helper.get_cell_mass_change_action_html(item_type, item_prop, ""))
        _to_return.append('''        <td class="shinken-prop-cell-action">''')
        _to_return.append('''            <table class="shinken-multi-line-object-table shinken-width-100">''')
        _to_return.append('''                <tr class="shinken-multi-line-object-action">''')
        _to_return.append('''                   <td class="shinken-lock-cell"><span class="shinkon-lock"></span></td>''')
        _to_return.append('''                   <td class="shinken-multi-line-object-cell" colspan="3">''')
        if type == TIME_PERIOD_TYPE.DAY:
            _to_return.append('''                   <table class="shinken-width-100"><tr>''')
            _to_return.append('''                       <td>''')
            _to_return.append(
                '''                           %s ''' % self.helper_component.get_input_html('input', value, item_prop=item_prop, type='text', extra_class_input="shinken-is-main-input", placeholder=app._('timeperiods.placeholder_value')))
            _to_return.append('''                       </td>''')
            _to_return.append('''                       <td class="shinken-time-period-line-add-button">''')
            _to_return.append('''                           <span class="shinken-multi-line-object-button-plus" onclick="MANAGER.EventManager.clickOnMultiLineObjectAdd('%s')">''' % item_prop)
            _to_return.append('''                               <i class="icon-plus"></i>''')
            _to_return.append('''                           </span>''')
            _to_return.append('''                       </td>''')
            _to_return.append('''                   </tr></table>''')
        else:
            _to_return.append('''                   <span class="shinken-multi-line-object-button-plus" onclick="MANAGER.EventManager.clickOnMultiLineObjectAdd('%s')">''' % item_prop)
            _to_return.append('''                       <i class="icon-plus"></i> %s''' % _label_button_plus)
            _to_return.append('''                   </span>''')
        _to_return.append('''                </td>''')
        _to_return.append('''                </tr>''')
        _to_return.append('''                <tr class="shinken-multi-line-object-template">''')
        _to_return.append('''                    %s ''' % self.get_multi_line_object(item_prop, type))
        _to_return.append('''                </tr>''')
        _to_return.append('''            </table>''')
        _to_return.append('''        </td>''')
        _to_return.append('''    </tr>''')
        _to_return.append('''</table>''')
        return "".join(_to_return)
    
    
    def get_multi_line_object(self, item_prop, type=False, line=None):
        if not type:
            type = item_prop
        
        if type == 'output_rules':
            if line:
                return self.get_output_rules_line(item_prop, state_init=line.getInState(), regex=line.getRegex(), state_final=line.out_state)
            else:
                return self.get_output_rules_line(item_prop)
        
        if type == 'notes_multi_url':
            return self.get_url_line(item_prop, line)
        
        if type == TIME_PERIOD_TYPE.ADVANCED:
            return self.get_key_value_line(item_prop, line, self.app._('timeperiods.placeholder_key'), self.app._('timeperiods.placeholder_value'))
        
        if type == TIME_PERIOD_TYPE.DAY:
            return self.get_time_period_day_line(item_prop, line)
        
        return 'TO BE DEFINED'
    
    
    def get_time_period_day_line(self, item_prop, line=False):
        value = line.value if line else ""
        week = int(line.week) if line else 1
        month = line.month if line else "None"
        flat = '%s %s' % (item_prop, line.flat_value) if line else ""
        
        popup_option_1 = PopUpInteger('week', week, -5, 5, 'timeperiods.label_week_%s', ignored_key=[0])
        
        selector_popup_1 = self.helper_component.get_selector_html(item_prop=item_prop, entries=popup_option_1.entries, select_label=True)
        
        popup_option = PopUpMonth(month)
        
        selector_popup = self.helper_component.get_selector_html(item_prop=item_prop, entries=popup_option.entries, select_label=True)
        
        _to_return = []
        _to_return.append('''<td class="shinken-period-option-line">''')
        _to_return.append('''   <table class="shinken-width-100"><tr>''')
        _to_return.append('''       <td class="shinken-week-line">''')
        _to_return.append('''           <div class="shinken-select-container" data-state-selector="0" data-name="week">%s</div>''' % selector_popup_1)
        _to_return.append('''       </td>''')
        _to_return.append('''       <td class="shinken-week-month-label">%s</td>''' % self.app._('element.from'))
        _to_return.append('''       <td class="shinken-month-line">''')
        _to_return.append('''           <div class="shinken-select-container" data-state-selector="0" data-name="month">%s</div>''' % selector_popup)
        _to_return.append('''       </td>''')
        _to_return.append('''   </tr></table>''')
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-value-line">''')
        _to_return.append('''   %s''' % self.helper_component.get_input_html('input', value, item_prop=item_prop, placeholder=app._('timeperiods.placeholder_value'), type='text'))
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-cfg-line-cell">''')
        _to_return.append(
            '''   <div class="shinken-cfg-line-value shinken-parent-text-to-copy" onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" shi-tip-html="%s">''' % app._('multi-line-object.cfg_format'))
        _to_return.append('''       %s''' % self.get_shinken_copy_butons())
        _to_return.append('''       <div class='shinken-cfg-file-value shinken-text-to-copy'>%s</div>''' % flat)
        _to_return.append('''   </div>''')
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-action-multi-object">''')
        _to_return.append('''   <div>''')
        _to_return.append('''       <div class="shinken-remove-btn" onclick="MANAGER.EventManager.clickOnDeleteLine(this)"><div class="shinkon-broom"></div></div>''')
        _to_return.append('''   </div>''')
        _to_return.append('''</td>''')
        return ' '.join(_to_return)
    
    
    def get_key_value_line(self, item_prop, line=False, placeholder_key="", placeholder_value=""):
        key = line.key if line else ""
        value = line.value if line else ""
        flat = line.flat_value if line else ""
        
        _to_return = []
        _to_return.append('''<td class="shinken-prop-cell-label"></td>''')
        _to_return.append('''<td class="shinken-key-line">''')
        _to_return.append('''   %s''' % self.helper_component.get_input_html('input', key, item_prop=item_prop, placeholder=placeholder_key, type='text'))
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-value-line">''')
        _to_return.append('''   %s''' % self.helper_component.get_input_html('input', value, item_prop=item_prop, placeholder=placeholder_value, type='text'))
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-cfg-line-cell">''')
        _to_return.append('''   <div class="shinken-cfg-line-value shinken-parent-text-to-copy"><div class='shinken-cfg-file-value shinken-text-to-copy'>%s</div></div>''' % flat)
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-action-multi-object">''')
        _to_return.append('''   <div>''')
        _to_return.append('''       <div class="shinken-remove-btn" onclick="MANAGER.EventManager.clickOnDeleteLine(this)"><div class="shinkon-broom"></div></div>''')
        _to_return.append('''   </div>''')
        _to_return.append('''</td>''')
        return ' '.join(_to_return)
    
    
    def get_url_line(self, item_prop, line):
        name = line[0] if line else ''
        url = line[1] if line else ''
        icon = line[2] if line else 'tag'
        popupping = line[3] if line else 'OPEN_IN_NEW_TAB'
        icons = IconsForUrlSelect(icon)
        
        selector_icons = self.helper_component.get_selector_html(item_prop=item_prop, entries=icons.entries, )
        
        popup_option = PopUpUrlSelect(popupping)
        
        selector_popup = self.helper_component.get_selector_html(item_prop=item_prop, entries=popup_option.entries, )
        
        _to_return = []
        _to_return.append('''<td class="shinken-name-urlmulti">''')
        _to_return.append('''   %s''' % self.helper_component.get_input_html('input', name, item_prop=item_prop, placeholder=app._('multi-line-object.enter_url_name'), type='text'))
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-url-urlmulti">''')
        _to_return.append('''   %s''' % self.helper_component.get_input_html('input', url, item_prop=item_prop, placeholder=app._('multi-line-object.enter_url'), type='text'))
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-icon-urlmulti">''')
        _to_return.append('''   <div>''')
        _to_return.append('''       <div class="shinken-select-container" data-state-selector="0" data-name="icon-url">%s</div>''' % selector_icons)
        _to_return.append('''   </div>''')
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-popup-urlmulti">''')
        _to_return.append('''   <div>''')
        _to_return.append('''       <div class="shinken-select-container" data-state-selector="0" data-name="popupping">%s</div>''' % selector_popup)
        _to_return.append('''   </div>''')
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-action-multi-object">''')
        _to_return.append('''   <div>''')
        _to_return.append('''       <div class="shinken-remove-btn" onclick="MANAGER.EventManager.clickOnDeleteLine(this)"><div class="shinkon-broom"></div></div>''')
        _to_return.append('''   </div>''')
        _to_return.append('''</td>''')
        return ' '.join(_to_return)
    
    
    def get_output_rules_line(self, item_prop, state_init='', regex='', state_final=0):
        
        entries_select_init = []
        entries_select_init.append(ComponentOption('shinken-matching', True, "", state_init == "", -1, label=app._('element.any_status')))
        entries_select_init.append(ComponentOption('shinken-matching', True, "OK", state_init == 0, -1, "OK"))
        entries_select_init.append(ComponentOption('shinken-matching', True, "WARNING", state_init == 1, -1, "WARNING"))
        entries_select_init.append(ComponentOption('shinken-matching', True, "CRITICAL", state_init == 2, -1, "CRITICAL"))
        entries_select_init.append(ComponentOption('shinken-matching', True, "UNKNOWN", state_init == 3, -1, "UNKNOWN"))
        
        selector_html_init = self.helper_component.get_selector_html(item_prop=item_prop, entries=entries_select_init, select_label=True)
        
        entries_select_final = []
        entries_select_final.append(ComponentOption('shinken-final', True, "OK", state_final == 0, -1, "OK"))
        entries_select_final.append(ComponentOption('shinken-final', True, "WARNING", state_final == 1, -1, "WARNING"))
        entries_select_final.append(ComponentOption('shinken-final', True, "CRITICAL", state_final == 2, -1, "CRITICAL"))
        entries_select_final.append(ComponentOption('shinken-final', True, "UNKNOWN", state_final == 3, -1, "UNKNOWN"))
        
        selector_html_final = self.helper_component.get_selector_html(item_prop=item_prop, entries=entries_select_final, select_label=True)
        
        _to_return = []
        _to_return.append('''<td class="shinken-status-match-cell-select">''')
        _to_return.append('''   <div>''')
        _to_return.append('''       <span class="shinken-if-label">''')
        _to_return.append('''           <span class="shinken-if-in-first-line text-center">%s</span>''' % self.app._('resultmodulations.if'))
        _to_return.append('''           <span class="shinken-if-not-in-first-line text-center">%s</span>''' % self.app._('resultmodulations.else_if'))
        _to_return.append('''       </span>''')
        _to_return.append('''       <div class="shinken-select-container" data-state-selector="0" data-name="shinken-matching">%s</div>''' % selector_html_init)
        _to_return.append('''       <span class="shinken-and-label">&</span>''')
        _to_return.append('''   </div>''')
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-regex-match-cell">''')
        _to_return.append('''   %s''' % self.helper_component.get_input_html('input', regex, item_prop=item_prop, placeholder=app._('resultmodulations.regular_expression'), type='text'))
        _to_return.append('''</td>''')
        _to_return.append('''<td class="shinken-status-final-cell">''')
        _to_return.append('''   <div>''')
        _to_return.append('''       <span class="shinken-then-label">%s</span>''' % (app._('resultmodulations.then')))
        _to_return.append('''       <div class="shinken-select-container" data-state-selector="0" data-name="shinken-final">%s</div>''' % selector_html_final)
        _to_return.append('''       <span class="shinken-remove-btn" onclick="MANAGER.EventManager.clickOnDeleteLine(this)"><span class="shinkon-broom"></span></span>''')
        _to_return.append('''   </div>''')
        _to_return.append('''</td>''')
        return ' '.join(_to_return)
    
    
    def get_multi_line_header(self, item_prop):
        _to_return = []
        _to_return.append('''<tr class="shinken-multi-line-object-header">''')
        if item_prop == 'output_rules':
            _to_return.append('''   <td class="shinken-multi-line-cell shinken-status-match-cell"><span class="shinken-state-init">%s</span></td>''' % self.app._('resultmodulations.status_match'))
            _to_return.append('''   <td class="shinken-multi-line-cell shinken-regex-match-cell"><span class="">%s</span></td>''' % self.app._('resultmodulations.output_re_match'))
            _to_return.append('''   <td class="shinken-multi-line-cell shinken-status-final-cell"><span class="shinken-state-final">%s</span></td>''' % self.app._('resultmodulations.status_result'))
        _to_return.append('''</tr>''')
        return "".join(_to_return)
    
    
    def get_template_html(self, extra_tooltip=''):
        return '<td class ="shinken-prop-cell-template" %s><div class=""><div class="from-template"></div></div></td>' % extra_tooltip
    
    
    def get_default_string_html(self, value, default_type):
        label = 'unknown'
        if default_type == 'default':
            label = self.app._('element.default')
        elif default_type == 'default_or_herited':
            label = self.app._('apply.value_not_set')
        elif default_type == 'forced':
            label = self.app._('element.forced_default')
        elif default_type == 'inherited':
            label = self.app._('element.inherited_from_template')
        return '<span class="shinken-default-label shinken-type-%s">%s</span> %s' % (default_type, label, value)
    
    
    def get_check_command_input_html(self, input_type, cmd_args, item_prop, _autocomplete, placeholder):
        input_html = []
        input_html.append('''<table class='shinken-check-command-input-container'>''')
        input_html.append('''   <tr>''')
        input_html.append('''       <td>''')
        input_html.append('''           <div>%(arg_label)s</div>''' % {'arg_label': 'Args'})
        input_html.append('''       </td>''')
        input_html.append('''       <td>''')
        input_html.append('''           <div>''')
        input_html.append('''               %s''' % self.helper_component.get_input_html(input_type, cmd_args, item_prop=item_prop, autocomplete=_autocomplete, placeholder=placeholder, type='text'))
        input_html.append('''           </div>''')
        input_html.append('''</td></tr></table>''')
        return ''.join(input_html)
    
    
    def get_boolean_btn_html(self, item_prop, btn_value, extra_class):
        _btn_label = self.app._('element.true') if btn_value == 1 else self.app._('element.false')
        _to_return = []
        _to_return.append('''<td class="shinken-btn-cell shinken-btn-user-cell">''')
        _to_return.append('''   <button type="button" ''')
        _to_return.append('''    class="shinken-radio-btn %s" ''' % extra_class)
        _to_return.append('''    value="%s" ''' % btn_value)
        _to_return.append('''    onclick="MANAGER.EventManager.clickOnRadioBtn(event)">''')
        _to_return.append('''       %s ''' % _btn_label)
        _to_return.append('''   </button> ''')
        _to_return.append('''</td> ''')
        return ''.join(_to_return)
    
    
    def get_property_btn_default_html(self, item_prop, default, extra_class):
        _to_return = []
        _to_return.append('''<td class="shinken-btn-cell shinken-btn-default-cell">''')
        _to_return.append('''   <button type="button" value="default_button" ''')
        _to_return.append('''    class="shinken-radio-btn %s" ''' % extra_class)
        _to_return.append('''    name="%s"''' % item_prop)
        _to_return.append('''    onclick="MANAGER.EventManager.clickOnRadioBtn(event)" >''')
        _to_return.append('''       <span class="shinken-default-radio-button shinken-btn-label-user">''')
        _to_return.append('''           <span class="shinken-boolean-title">''')
        _to_return.append('''               %s''' % self.get_default_string_html('', 'default'))
        _to_return.append('''               %s''' % self.get_default_string_html('', 'forced'))
        _to_return.append('''               <span class="shinken-default-value shinken-between-bracket">%s</span>''' % default)
        _to_return.append('''           </span>''')
        _to_return.append('''       </span>''')
        _to_return.append('''       <span class="shinken-btn-label-inheritance shinken-hidden">''')
        _to_return.append('''           %s''' % self.get_default_string_html('', 'inherited'))
        _to_return.append('''           <span class="shinken-inherited-value"></span>''')
        _to_return.append('''       </span>''')
        _to_return.append('''   </button> ''')
        _to_return.append('''</td> ''')
        
        return ''.join(_to_return)
    
    
    def get_boolean_btn_no_override_html(self, item_prop_final, extra_class, shinken_element_original, item_prop):
        _to_return = [
            '''<td class="shinken-btn-cell shinken-no-override-btn-cell">''',
            '''   <button type="button" value="__NO_OVERRIDE__" ''',
            '''    class="shinken-radio-btn shinken-no-override-btn %s" ''' % extra_class,
            '''    name="%s"''' % item_prop_final,
            '''    value="__NO_OVERRIDE__"''',
            '''    onclick="MANAGER.EventManager.clickOnRadioBtn(event)" >''',
            '''       <span class="shinken-default-radio-button">''',
            '''           <span class="">%s</span>''' % self._get_service_overrides_from(item_prop, shinken_element_original),
            '''       </span>''',
            '''   </button> ''',
            '''</td> '''
        ]
        return ''.join(_to_return)
    
    
    def get_range_btn_cells(self, item_prop_final, default_value, default_value_label, data_unit, override, shinken_element_original, item_prop, _original_value=''):
        _to_return = []
        _to_return.append('''<td class='shinken-cell-range-default-btn'>''')
        _to_return.append('''    <div class="shinken-range-btn-default-container">''')
        _to_return.append('''        <table class="shinken-width-100">''')
        if override:
            _to_return.append('''            %s ''' % self.get_range_btn_default_html(item_prop_final, default_value, default_value_label, data_unit))
            _to_return.append('''            %s ''' % self.get_range_btn_no_override_html(item_prop_final, item_prop, _original_value, '', shinken_element_original))
        else:
            _to_return.append('''            %s ''' % self.get_range_btn_default_html(item_prop_final, default_value, default_value_label, data_unit))
        
        _to_return.append('''        </table>''')
        _to_return.append('''    </div>''')
        _to_return.append('''</td>''')
        
        return ''.join(_to_return)
    
    
    def get_range_btn_default_html(self, item_prop, default_value, default_value_label, data_unit):
        _to_return = []
        _to_return.append('''<tr><td class="shinken-btn-cell shinken-btn-default-cell">''')
        _to_return.append('''   <button type="button" value="default_button" ''')
        _to_return.append('''    class="btn shinken-radio-btn" ''')
        _to_return.append('''    onclick="MANAGER.EventManager.clickOnRadioBtn(event)" >''')
        _to_return.append('''       <span class="shinken-btn-label-user">%s%s''' % (self.get_default_string_html('', 'default'), self.get_default_string_html('', 'forced')))
        _to_return.append('''           <span class="shinken-default-btn-default-label shinken-between-bracket">%s%s</span>''' % (default_value_label, data_unit))
        _to_return.append('''       </span>''')
        _to_return.append('''       <span class="shinken-btn-label-inheritance shinken-hidden">%s''' % self.get_default_string_html('', 'inherited'))
        _to_return.append('''           <span class="shinken-inherited-value"></span>''')
        _to_return.append('''       </span>''')
        _to_return.append('''   </button> ''')
        _to_return.append('''</td></tr>''')
        
        return ''.join(_to_return)
    
    
    def get_range_btn_no_override_html(self, item_prop_final, item_prop, original_value, extra_class, shinken_element_original):
        _to_return = []
        _to_return.append('''<tr><td>''')
        _to_return.append('''    <button type="button" value="__NO_OVERRIDE__" ''')
        _to_return.append('''     class="shinken-width-100 shinken-radio-btn shinken-no-override-btn %s" ''' % extra_class)
        _to_return.append('''     name="%s"''' % item_prop_final)
        _to_return.append('''     data-check-value="%s"''' % original_value)
        _to_return.append('''     value="__NO_OVERRIDE__"''')
        _to_return.append('''     onclick="MANAGER.EventManager.clickOnRadioBtn(event)" >''')
        _to_return.append('''        <span class="shinken-default-radio-button">''')
        _to_return.append('''            <span class="">%s</span>''' % self._get_service_overrides_from(item_prop, shinken_element_original))
        _to_return.append('''        </span>''')
        _to_return.append('''    </button> ''')
        _to_return.append('''</td></tr>''')
        return ''.join(_to_return)
    
    
    def get_modification_info(self, modification_info):
        info = {
            'last_modification': self.app._('apply.last_modification'),
            'from'             : self.app._('apply.from'),
            'at'               : self.app._('apply.at'),
            'date'             : 'date not set',
            'user_name'        : 'User not found'
        }
        
        if modification_info:
            info['date'] = self.print_date(modification_info.get('date', None))
            contact_link = modification_info.get('contact', None)
            if contact_link:
                contact = item_for_link(contact_link['links'][0], states=[ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION])
                info['user_name'] = contact.get_name()
        output = '''(%(last_modification)s: %(from)s %(user_name)s %(at)s %(date)s)''' % info
        
        return output
    
    
    # **********************************   HEADER COUNTER   ********************************************
    
    def get_counter_menu_number(self, name, type, configuration_stats):
        _to_return = ""
        if name == 'elements':
            if type == 'proposed':
                _to_return = configuration_stats['detail'][name][type][1]
            else:
                _to_return = configuration_stats['detail'][name][type] - configuration_stats['detail']['hosts'][type]
        elif name == 'clusters-gal':
            _to_return = configuration_stats['detail']['clusters'][type] + configuration_stats['detail']['clustertpls'][type]
        elif name == 'hosts-gal':
            if type == 'proposed':
                _to_return = configuration_stats['detail']['hosts'][type][1]
            else:
                _to_return = configuration_stats['detail']['hosttpls'][type] + configuration_stats['detail']['hostgroups'][type]
        elif name == 'checks-gal':
            _to_return = configuration_stats['detail']['serviceshosts'][type] + configuration_stats['detail']['serviceshosttpls'][type] + configuration_stats['detail']['servicesclusters'][type] + configuration_stats['detail']['servicesclustertpls'][
                type] + configuration_stats['detail']['servicetpls'][type]
        elif name == 'contacts-gal':
            _to_return = configuration_stats['detail']['contacts'][type] + configuration_stats['detail']['contactgroups'][type] + configuration_stats['detail']['contacttpls'][type] + configuration_stats['detail']['escalations'][type] + \
                         configuration_stats['detail']['notificationways'][type]
        elif name == 'modulations-gal':
            _to_return = configuration_stats['detail']['businessimpactmodulations'][type] + configuration_stats['detail']['macromodulations'][type] + configuration_stats['detail']['resultmodulations'][type]
        else:
            if type == 'proposed':
                _to_return = configuration_stats['detail'][name][type][1]
            else:
                _to_return = configuration_stats['detail'][name][type]
        if _to_return == 0:
            _to_return = ""
        return _to_return
    
    
    def get_counter_header_number(self, counter_name, configuration_stats):
        _to_return = ""
        if counter_name == 'working-area-edition':
            _to_return = configuration_stats['detail']['hosts']['working_area'][0] + configuration_stats['detail']['hosts']['working_area'][1]
        elif counter_name == 'working-area-rejected':
            _to_return = configuration_stats['detail']['hosts']['rejected'][0] + configuration_stats['detail']['hosts']['rejected'][1]
        elif counter_name == 'working-area-new':
            _to_return = configuration_stats['detail']['hosts']['new']
        elif counter_name == 'working-area-difference':
            _to_return = configuration_stats['detail']['hosts']['change']
        elif counter_name == 'stagging-area-new':
            _to_return = configuration_stats['overall']['new'] - configuration_stats['detail']['hosts']['new']
        elif counter_name == 'stagging-area-difference':
            _to_return = configuration_stats['overall']['change'] - configuration_stats['detail']['hosts']['change']
        elif counter_name == 'stagging-area-proposed':
            _to_return = configuration_stats['overall']['proposed'][1]
        elif counter_name == 'apply':
            _to_return = configuration_stats['overall']['to_apply']
        if _to_return == 0:
            _to_return = ""
        return _to_return
    
    
    def get_counter_header_label(self, counter_name, configuration_stats, number):
        _to_return = ""
        if counter_name == 'working-area-edition':
            _to_return = '''[<span class="shinken-counter-hover" %s>%s</span>/<span class="shinken-counter-hover">%s</span>]''' % (
                self.get_counter_header_link('my-working-area-edition'), configuration_stats['detail']['hosts']['working_area'][0], configuration_stats['detail']['hosts']['working_area'][1])
        elif counter_name == 'working-area-rejected':
            _to_return = '''[<span class="shinken-counter-hover" %s>%s</span>/<span class="shinken-counter-hover">%s</span>]''' % (
                self.get_counter_header_link('my-working-area-rejected'), configuration_stats['detail']['hosts']['rejected'][0], configuration_stats['detail']['hosts']['rejected'][1])
        elif counter_name == 'apply':
            _to_return = '''<span class="shinken-counter-hover">%s</span>''' % number
        else:
            _to_return = '''[<span class="shinken-counter-hover">%s</span>]''' % number
        return _to_return
    
    
    def _add_filter_to_dict(self, link_dict, key, value, only_contain=False):
        if value.startswith('!'):
            value = '!%s"%s"' % ('=' if not only_contain else '', value[1:])
        else:
            value = '%s"%s"' % ('=' if not only_contain else '', value)
        if not link_dict['filters'].get(key, None):
            link_dict['filters'][key] = [value]
        else:
            link_dict['filters'][key].append(value)
        return link_dict
    
    
    def get_counter_header_link(self, counter_name, type=False):
        _link = '/elements/'
        if type and type == 'change':
            type = 'difference'
        _arguments = {'add_to_link': 'elements', 'filters': {}}
        if counter_name == 'clusters-gal':
            _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.%s' % type))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.cluster'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.clustertpl'))
        elif counter_name == 'hosts-gal':
            if type == 'proposed':
                _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.host'))
                _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.hosttpl'))
                _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.hostgroup'))
                _arguments = self._add_filter_to_dict(_arguments, 'working_area_status', self.app._('element.PROPOSED'), only_contain=True)
            else:
                _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.%s' % type))
                _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.host'))
                _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.hosttpl'))
                _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.hostgroup'))
        elif counter_name == 'checks-gal':
            _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.%s' % type))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.servicescluster'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.serviceshosttpl'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.servicesclustertpl'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.serviceshost'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.servicetpl'))
        elif counter_name == 'contacts-gal':
            _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.%s' % type))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.escalation'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.contactgroup'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.contacttpl'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.contact'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.notificationway'))
        elif counter_name == 'modulations-gal':
            _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.%s' % type))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.businessimpactmodulation'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.macromodulation'))
            _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.resultmodulation'))
        elif counter_name == 'working-area-edition':
            _arguments['add_to_link'] = 'working_area/hosts'
            _arguments = self._add_filter_to_dict(_arguments, 'type', '!%s' % self.app._('element.new'))
            _arguments = self._add_filter_to_dict(_arguments, 'working_area_status', self.app._('element.WORKING'), only_contain=True)
        elif counter_name == 'my-working-area-edition':
            _arguments['add_to_link'] = 'my_working_area/hosts'
            _arguments = self._add_filter_to_dict(_arguments, 'type', '!%s' % self.app._('element.new'))
            _arguments = self._add_filter_to_dict(_arguments, 'working_area_status', self.app._('element.WORKING'), only_contain=True)
        elif counter_name == 'working-area-rejected':
            _arguments['add_to_link'] = 'working_area/hosts'
            _arguments = self._add_filter_to_dict(_arguments, 'working_area_status', self.app._('element.REJECTED'), only_contain=True)
        elif counter_name == 'my-working-area-rejected':
            _arguments['add_to_link'] = 'my_working_area/hosts'
            _arguments = self._add_filter_to_dict(_arguments, 'working_area_status', self.app._('element.REJECTED'), only_contain=True)
        elif counter_name == 'working-area-new':
            _arguments['add_to_link'] = 'working_area/hosts'
            _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.new'))
        elif counter_name == 'working-area-difference':
            _arguments['add_to_link'] = 'working_area/hosts'
            _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.difference'))
        elif counter_name == 'stagging-area-new':
            _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.new'))
        elif counter_name == 'stagging-area-difference':
            _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.difference'))
        elif counter_name == 'stagging-area-proposed':
            if not self.app.get_user_auth().is_admin():
                _arguments['add_to_link'] = 'hosts'
            _arguments = self._add_filter_to_dict(_arguments, 'working_area_status', self.app._('element.PROPOSED'), only_contain=True)
        elif counter_name == 'apply':
            return '''onclick="SHINKEN.HIGHWAY.goTo('',event);return false;"'''
        elif counter_name == 'elements':
            if type == 'proposed':
                _arguments = self._add_filter_to_dict(_arguments, 'working_area_status', self.app._('element.PROPOSED'), only_contain=True)
            else:
                _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.%s' % type))
        elif counter_name == 'hosts':
            if type == 'proposed':
                _arguments['add_to_link'] = 'hosts'
                _arguments = self._add_filter_to_dict(_arguments, 'working_area_status', self.app._('element.PROPOSED'), only_contain=True)
            else:
                _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.%s' % type))
                _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.hosttpl'))
                _arguments = self._add_filter_to_dict(_arguments, 'type_element', self.app._('type.hostgroup'))
        else:
            _arguments['add_to_link'] = counter_name
            _arguments = self._add_filter_to_dict(_arguments, 'type', self.app._('element.%s' % type))
        _computed_filters = []
        for filter in _arguments['filters'].iteritems():
            _computed_filters.append('%s:%s' % (filter[0], '|'.join(filter[1])))
        _final_filters = '~'.join(_computed_filters)
        _link = '%s%s?filter=%s' % (_link, _arguments['add_to_link'], urllib.quote(_final_filters.encode('utf-8')))
        _to_return = '''onclick="SHINKEN.HIGHWAY.goTo('%s',event);return false;"''' % _link
        return _to_return
    
    
    def get_counter_header_tooltip(self, counter_name, configuration_stats):
        if counter_name == 'working-area-edition':
            _to_return = self.app._("header.working_hover_text").format(configuration_stats['detail']['hosts']['working_area'][0], configuration_stats['detail']['hosts']['working_area'][1])
        elif counter_name == 'working-area-rejected':
            _to_return = self.app._("header.rejected_hover_text").format(configuration_stats['detail']['hosts']['rejected'][0], configuration_stats['detail']['hosts']['rejected'][1])
        else:
            _to_return = self.app._("header.%s" % counter_name)
        return _to_return
    
    
    def get_counter_header(self, counter_name, counter_label_key, configuration_stats):
        _id = "id-shinken-%s-counter" % counter_name
        _label = self.app._('%s' % counter_label_key)
        _number = self.get_counter_header_number(counter_name, configuration_stats)
        _number_label = ""
        if _number:
            _number_label = self.get_counter_header_label(counter_name, configuration_stats, _number)
        
        _to_return = []
        _to_return.append('''<td id="%s" class="shinken-navigation-info-td shinken-counter" data-number="%s"''' % (_id, _number))
        _to_return.append(''' onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" shi-tip-html="%s">''' % self.get_counter_header_tooltip(counter_name, configuration_stats))
        _to_return.append('''<span''')
        if counter_name == 'apply':
            _to_return.append(''' class="shinken-counter-label shinken-counter-hover-link" data-is-string="1">''')
            _to_return.append('''%s <span class="shinken-counter-extra-label">%s</span>''' % (_number_label, _label))
        else:
            _to_return.append(''' %s''' % self.get_counter_header_link(counter_name))
            _to_return.append(''' class="shinken-counter-label shinken-counter-hover-link" data-is-before="1" data-is-string="1">''')
            _to_return.append('''<span class="shinken-counter-extra-label">%s</span> %s''' % (_label, _number_label))
        _to_return.append('''</span>''')
        _to_return.append('''</td>''')
        return ''.join(_to_return)
    
    
    def get_counter_menu(self, name, type, configuration_stats, counter_link=False):
        _number = self.get_counter_menu_number(name, type, configuration_stats)
        _extra_dom_param = ""
        if counter_link == 'counter-parent':
            _extra_dom_param = 'id="id-shinken-%s-%s-counter"' % (name, type)
        elif counter_link:
            _extra_dom_param = 'data-parent="%s-%s"' % (counter_link, type)
        _to_return = []
        _to_return.append('''<span class="shinken-navigation-info-td shinken-counter" data-number="%s" data-counter-type="%s" %s>''' % (_number, type, _extra_dom_param))
        _to_return.append('''<span %s class='shinken-counter-label shinken-between-bracket shinken-counter-hover-link'>''' % self.get_counter_header_link(name, type))
        _to_return.append('''%s''' % _number)
        _to_return.append('''</span></span>''')
        return ''.join(_to_return)
    
    
    # *****************************   WINDOW OBJECT FUNCTIONS  ************************************************************#
    def getWindowTargetDiv(self, name, group_name_label, name_selected, extra_class='', extra_attr=''):
        _class_to_add = "" if name_selected == name else "shinken-hidden"
        _to_return = []
        _to_return.append('''id="id-shinken-window-%s%s-target"''' % (name, group_name_label))
        _to_return.append('''class ="shinken-tab-pane property-tab-pane %s %s"''' % (_class_to_add, extra_class))
        _to_return.append('''data-name="%s"''' % name)
        if extra_attr:
            _to_return.append('''%s''' % extra_attr)
        return ' '.join(_to_return)
    
    
    # *****************************   COPY TO CLIPBOARD  ************************************************************#
    def get_shinken_copy_butons(self):
        _to_return = []
        _to_return.append('''<table class='shinken-copy-buttons-container'>''')
        _to_return.append('''   <tr>''')
        _to_return.append('''       <td class="shinken-btn-copy-cell">''')
        _to_return.append(
            '''           <div class='shinken-btn-copy shinken-btn-copy-text' onclick='CLIPBOARD_SERVICE.setContent(this, CLIPBOARD_SERVICE.TYPE.TEXT)' data-status-copy='0' onmouseenter='SHINKEN_TOOLTIP.showTooltip(this);' onmouseleave='SHINKEN_TOOLTIP.hideTooltip();' shi-tip-html='%s' >''' % self.app._(
                'copy_service.tooltip_text_copy'))
        _to_return.append('''               <i class='shinken-copy-default-message shinicon shinicon-copy'></i>''')
        _to_return.append('''               <span class='shinken-copy-success-message'>%s</span>''' % self.app._('copy_service.copySuccess'))
        _to_return.append('''               <span class='shinken-copy-error-message'>%s</span>''' % self.app._('copy_service.copyFail'))
        _to_return.append('''           </div>''')
        _to_return.append('''       </td>''')
        _to_return.append('''       <td class="shinken-btn-edit-cell">''')
        _to_return.append(
            '''           <div class='shinken-btn-copy' onclick='MANAGER.EventManager.clickOnPaste(event)' data-status-copy='0' onmouseenter='SHINKEN_TOOLTIP.showTooltip(this);' onmouseleave='SHINKEN_TOOLTIP.hideTooltip();' shi-tip-html='%s' >''' % self.app._(
                'copy_service.tooltip_edition'))
        _to_return.append('''               <i class='shinken-copy-default-message shinken-icons-url shinken-icons-url-write'></i>''')
        _to_return.append('''           </div>''')
        _to_return.append('''       </td>''')
        _to_return.append('''   </tr>''')
        _to_return.append('''</table>''')
        
        return ' '.join(_to_return)
