#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import datetime
import httplib
import json
import optparse
import random
import ssl
import sys
import time
from shinkensolutions.shinken_time_helper import print_human_readable_period

from shinken.basemodule import ModuleState
from shinken.log import DEFAULT_LONG_FLUSH_STATS

API_VERSION = '2.12'

LONG_OUTPUT_BREAK = u'\n'
NEW_LINE = u'<br/><br/>'
BREAK_LINE = u'<br/>'
NEW_LINE_OUTPUT_AFTER_TABLE = u'<br/>'

EXECUTOR_LOAD_LIMIT = 0.95  # %

random.seed()


class ParseOptionError(SyntaxError):
    pass


class NotFoundException(BaseException):
    pass


class RaiseOnExitOptionParser(optparse.OptionParser):
    def exit(self, status=0, msg=None):
        raise ParseOptionError(msg)


class COLOR:
    BLACK = u'#000000'
    GREEN = u'#2A9A3D'
    RED = u'#FF0000'
    ORANGE = u'#f57700'
    GRAY = u'#808080'
    
    DEFAULT_COLOR = BLACK


class EXIT_STATUS:
    OK = 0
    WARNING = 1
    CRITICAL = 2
    UNKNOWN = 3


class Result(object):
    def __init__(self):
        self.status = EXIT_STATUS.OK
        self.warnings = []
        self.criticals = []
        self.outputs = []
        self.outputs_no_sort = []
        self.titles = []
        self.long_outputs = []
        self.perf_data = {}
        self.spare_info = ''
    
    
    def set_perf_data(self, perf_data):
        self.perf_data = perf_data
    
    
    def add_perf_data(self, name, value):
        self.perf_data[name] = value
    
    
    def set_logger_stats(self, data):
        if not data:
            return
        logger_stats = data.get('logger_stats', DEFAULT_LONG_FLUSH_STATS)
        is_too_long = logger_stats.get('is_too_long', False)
        write_duration = logger_stats.get('write_duration', 0.0)
        log_path = logger_stats.get('log_path', '')
        # If the logger is going well, bail out
        if not is_too_long:
            return
        
        self.add_check(EXIT_STATUS.WARNING, '%s - Writing logs on disk took too much time ( worth time was %.1fs during the last minute)<br>Path: "%s"' % (HTMLTag.color_text('WARNING', COLOR.ORANGE), write_duration, log_path))
    
    
    def set_spare_info(self, data, daemon_type=None):
        if not data:
            return
        daemon_type = daemon_type or data.get('daemon_type', 'Daemon')
        spare_message = u'<div style="' \
                        u'margin: 5px 5px 5px 1px;' \
                        u'background: #0095da;' \
                        u'color: #fff;' \
                        u'padding: 1px 7px;' \
                        u'border-radius: 5px;' \
                        u'box-sizing: border-box;' \
                        u'display: inline-block;' \
                        u'display: inline-block;' \
                        u'">SPARE</div>'
        active_message = u'<div style="' \
                         u'margin: 5px 5px 5px 1px;' \
                         u'background: #2a9a3d;' \
                         u'color: #fff;' \
                         u'padding: 1px 7px;' \
                         u'border-radius: 5px;' \
                         u'box-sizing: border-box;' \
                         u'display: inline-block;' \
                         u'display: inline-block;' \
                         u'">RUNNING</div><br>'
        idle_message = u"This {0} is currently idle because it is configured as a Spare and its main daemon is running well.<br>It will take over another {0} when a main {0} stops working".format(
            daemon_type.title())
        
        if data.get('spare', False):
            if data.get('activated', False):
                self.spare_info = ''.join((spare_message, active_message))
            else:
                self.spare_info = spare_message + BREAK_LINE
                self.hard_exit(EXIT_STATUS.OK, idle_message)
    
    
    def add_check(self, status=EXIT_STATUS.OK, output='', long_output='', title=False, no_new_line=False):
        if self.status < status:
            self.status = status
        
        if output:
            if not no_new_line:
                output = ''.join((output, BREAK_LINE))
            
            if title:
                self.titles.append(output)
            else:
                if status == EXIT_STATUS.CRITICAL:
                    self.criticals.append(output)
                elif status == EXIT_STATUS.WARNING:
                    self.warnings.append(output)
                else:
                    self.outputs.append(output)
                
                self.outputs_no_sort.append(output)
        if long_output:
            self.long_outputs.append(long_output)
    
    
    def add_title(self, title):
        self.add_check(output=title, title=True)
    
    
    def hard_exit(self, status, output, long_output=''):
        self.status = status
        self.outputs = [output]
        self.outputs_no_sort = []
        self.titles = []
        self.criticals = []
        self.warnings = []
        self.long_outputs = [long_output]
        self.perf_data = {}
        self.exit()
    
    
    def exit(self, sorted_by_level=True):
        global to_print
        
        if not self.titles:
            self.titles.append(BREAK_LINE)
        if sorted_by_level:
            warnings = ''
            criticals = ''
            if self.criticals:
                criticals = ''.join(('<div class="skn-ln">%s%s</div>' % (HTMLTag.color_text(u'=> ', COLOR.RED), i) for i in self.criticals))
                criticals = HTMLTag.tag_border(criticals, COLOR.RED)
            if self.warnings:
                warnings = ''.join(('<div class="skn-ln">%s%s</div>' % (HTMLTag.color_text(u'=> ', COLOR.ORANGE), i) for i in self.warnings))
                warnings = HTMLTag.tag_border(warnings, COLOR.ORANGE)
            
            output = u''.join((u''.join(self.titles), criticals, warnings, ''.join(self.outputs)))
        else:
            output = ''.join((''.join(self.titles), ''.join(self.outputs_no_sort)))
        if self.long_outputs:
            output = ''.join((output, LONG_OUTPUT_BREAK, ''.join(self.long_outputs)))
        
        # print "exit status[%s] output[%s] <br>" % (self.status, output)
        tag = ''
        if self.status == EXIT_STATUS.OK:
            tag = HTMLTag.color_text('[OK]', COLOR.GREEN)
        elif self.status == EXIT_STATUS.WARNING:
            tag = HTMLTag.color_text('[WARNING]', COLOR.ORANGE)
        elif self.status == EXIT_STATUS.CRITICAL:
            tag = HTMLTag.color_text('[CRITICAL]', COLOR.RED)
        elif self.status == EXIT_STATUS.UNKNOWN:
            tag = HTMLTag.color_text('[UNKNOWN]', COLOR.BLACK)
        
        if self.perf_data is None:
            self.perf_data = {}
        
        perfdata = Result._do_perfdata(self.perf_data)
        output = Utils.add_style(output)
        to_print = '%s%s %s' % (self.spare_info, tag, output.strip())
        if perfdata:
            to_print = '%s| %s' % (to_print, perfdata)
        try:
            to_print = to_print.encode('ascii', 'ignore')
        except UnicodeDecodeError:
            pass
        print to_print
        sys.exit(self.status)
    
    
    @staticmethod
    def _do_perfdata(perfs):
        return ' '.join(['%s=%s' % (k, v) for (k, v) in perfs.iteritems()])


class HTMLTag:
    used = False
    STYLE = u'''.skn-ctg{margin:1px;background:#DDD;padding:1px 7px;border-radius:5px;box-sizing:border-box;display:inline-block;border:1px solid} .skn-brd{margin:7px 2px;padding:5px 7px;border-radius:7px;box-sizing:border-box;border:2px solid;width:100%;word-break: break-word;}'''
    EXTRA_STYLE = u''
    EXTRA_CLASS = u''
    
    STATE_CRITICAL = u'CRITICAL'
    STATE_OK = u'OK'
    STATE_WARNING = u'WARNING'
    
    CRITICAL = u''
    OK = u''
    WARNING = u''
    
    STATE_COLOR = {
        u'FATAL'   : COLOR.RED,
        u'CRITICAL': COLOR.RED,
        u'OK'      : COLOR.GREEN,
        u'WARNING' : COLOR.ORANGE,
    }
    
    EXIT_STATUS_COLOR = {
        EXIT_STATUS.OK      : COLOR.GREEN,
        EXIT_STATUS.WARNING : COLOR.ORANGE,
        EXIT_STATUS.CRITICAL: COLOR.RED,
        EXIT_STATUS.UNKNOWN : COLOR.GRAY
    }
    
    
    @staticmethod
    def color_text(value, color=COLOR.BLACK, bold=True):
        # type: (basestring, basestring, bool) -> basestring
        _span = u'<span style="color:%(color)s;%(bold)s">%(value)s</span>'
        _info = {u'color': color, u'value': value, u'bold': ''}
        if bold:
            _info[u'bold'] = u'font-weight:bold;'
        return _span % _info
    
    
    @staticmethod
    def tag_border(value, color=COLOR.BLACK):
        HTMLTag.used = True
        _span = u'<table class="skn-brd %(extra_class)s" style="border-color:%(color)s; "><tr><td>%(value)s</tr></td></table>'
        return _span % {u'color': color, u'value': value, u'extra_class': HTMLTag.EXTRA_CLASS}
    
    
    @staticmethod
    def tag_value(value, color=COLOR.BLACK):
        HTMLTag.used = True
        _span = u'<span class="skn-ctg %(extra_class)s" style="color:%(color)s;border-color:%(color)s;">%(value)s</span>'
        return _span % {u'color': color, u'value': value, u'extra_class': HTMLTag.EXTRA_CLASS}
    
    
    @staticmethod
    def state_tag(state):
        return HTMLTag.tag_value(state, HTMLTag.STATE_COLOR.get(state, COLOR.DEFAULT_COLOR))
    
    
    @staticmethod
    def load_tag(load):
        if load == -1:
            return HTMLTag.tag_value('unavailable', COLOR.ORANGE)
        elif load == -2:
            return HTMLTag.tag_value('unreachable', COLOR.RED)
        elif load > EXECUTOR_LOAD_LIMIT:
            return HTMLTag.tag_value('No more CPU usable', COLOR.ORANGE)
        else:
            return HTMLTag.tag_value('Resources available', COLOR.BLACK)
    
    
    @staticmethod
    def ram_tag(ram_usage, max_ram_usage):
        if ram_usage == -1:
            return HTMLTag.tag_value('unavailable', COLOR.ORANGE)
        elif ram_usage == -2:
            return HTMLTag.tag_value('unreachable', COLOR.RED)
        elif ram_usage > max_ram_usage:
            return HTMLTag.tag_value('Limit reached', COLOR.RED)
        else:
            return HTMLTag.tag_value('normal', COLOR.BLACK)
    
    
    @staticmethod
    def cpu_queue_tag(cpu_running_queue, max_cpu_queue_per_cpu, nb_cpus):
        if cpu_running_queue == -1:
            return HTMLTag.tag_value('unavailable', COLOR.ORANGE)
        elif cpu_running_queue == -2:
            return HTMLTag.tag_value('unreachable', COLOR.RED)
        elif cpu_running_queue > (max_cpu_queue_per_cpu * nb_cpus):
            return HTMLTag.tag_value('Limit reached', COLOR.RED)
        else:
            return HTMLTag.tag_value('normal', COLOR.BLACK)


class HTMLList:
    used = False
    STYLE = u'.skn-ul{padding: 0; margin: 0 0 10px 25px;}'
    EXTRA_STYLE = u''
    
    
    @staticmethod
    def _list_item(item):
        return "<li>%s</li>" % item
    
    
    @staticmethod
    def _list_header(header):
        if header is None:
            return "<ul class='skn-ul'>"
        else:
            return "%s&nbsp;:<ul class='skn-ul'>" % header
    
    
    @staticmethod
    def _list_footer():
        return "</ul>"
    
    
    @staticmethod
    def header_list(header, items):
        HTMLList.used = True
        return '%(header)s%(items)s%(footer)s' % {
            'header': HTMLList._list_header(header),
            'items' : "".join(map(lambda item: HTMLList._list_item(item), items)),
            'footer': HTMLList._list_footer()}
    
    
    @staticmethod
    def simple_list(items):
        return HTMLList.header_list(None, items)


class HTMLTable:
    used = False
    STYLE = u'''.skn-ict,.skn-ict td,.skn-ict th{border:1px solid #000000 !important;border-collapse:collapse !important;word-break: break-all !important;color:#000000 !important} .skn-ict{width:100% !important;} .skn-ict th{background-color:#DDDDDD !important;padding:2px !important;word-break:break-word !important} .skn-ict td{padding:2px !important;width:auto !important;font-weight:normal !important;word-break:break-word !important;background-color:#FFFFFF !important}'''
    EXTRA_STYLE = u''
    
    
    @staticmethod
    def add_extra_style(extra_style):
        HTMLTable.EXTRA_STYLE = '%s %s' % (HTMLTable.EXTRA_STYLE, extra_style)
    
    
    @staticmethod
    def generate_class_uuid():
        return 'skn-tbl-%05d' % int(random.random() * 100000)
    
    
    @staticmethod
    def table(headers, lines, title=None, left_headers=None, compact_title=False, extra_tags='', all_col_same_width=True, class_uuid='', extra_style=''):
        HTMLTable.used = True
        class_uuid = class_uuid or HTMLTable.generate_class_uuid()
        extra_tags = '' or extra_tags
        _table = []
        
        if extra_style:
            HTMLTable.add_extra_style(' '.join(extra_style))
        
        if all_col_same_width:
            _extra_style = [
                '.%s .skn-ict{table-layout: fixed;}' % class_uuid,
            ]
        else:
            _extra_style = [
                '.%s .skn-ict .skn-lfh {width: 25%%;}' % class_uuid,
            ]
        
        HTMLTable.add_extra_style(' '.join(_extra_style))
        
        header_string = ''.join(('<th>%s</th>' % header for header in headers))
        
        values_string = []
        for i, line in enumerate(lines):
            line_string = ''.join(('<td>%s</td>' % l for l in line))
            if left_headers:
                line_string = '<th class="skn-lfh">%s</th>%s' % (left_headers[i], line_string)
            values_string.append('<tr>%s</tr>' % line_string)
        values_string = ''.join(values_string)
        
        if title:
            _table.append('<div class="skn-ich">%s:</div>' % title)
            if not compact_title:
                _table.append(BREAK_LINE)
        
        _table.append('<div class="%s"><table class="skn-ict" %s>' % (class_uuid, extra_tags))
        if header_string:
            _table.append('<tr>%s</tr>' % header_string)
        _table.append('%s' % values_string)
        _table.append('</table></div>')
        
        return ''.join(_table)


class Utils:
    
    @staticmethod
    def _http_get_conn(full_uri, timeout, use_ssl, ssl_version=ssl.PROTOCOL_TLSv1):
        if use_ssl:
            # If we are in SSL mode, do not look at certificate too much
            # NOTE: ssl.SSLContext is only availabe on last python 2.7 versions
            if hasattr(ssl, 'SSLContext'):
                ssl_context = ssl.SSLContext(ssl_version)
                ssl_context.check_hostname = False
                ssl_context.verify_mode = ssl.CERT_NONE
            else:
                ssl_context = None
            
            args = {}
            if ssl_context:
                args['context'] = ssl_context
            conn = httplib.HTTPSConnection(full_uri, timeout=timeout, **args)
        else:
            conn = httplib.HTTPConnection(full_uri, timeout=timeout)
        return conn
    
    
    # First try in HTTP and if fail from the server, retry in HTTPs
    @staticmethod
    def _request_get(base_uri, uri, use_ssl=False, timeout=3):
        if base_uri.startswith('http://'):
            use_ssl = False
            base_uri = base_uri[7:-1]
        elif base_uri.startswith('https://'):
            use_ssl = True
            base_uri = base_uri[8:-1]
        
        start_time = time.time()
        conn = Utils._http_get_conn(base_uri, timeout=timeout, use_ssl=use_ssl)
        
        conn.request("GET", uri)
        r1 = conn.getresponse()
        
        buf = r1.read()
        if r1.status == 400 and not use_ssl and 'sent a plain HTTP request' in buf:
            return Utils._request_get(base_uri, uri, use_ssl=True, timeout=timeout)
        if r1.status == 404:
            conn.close()
            raise NotFoundException()
        if r1.status != 200:
            conn.close()
            raise Exception(buf)
        return buf, (time.time() - start_time)
    
    
    # First try in HTTP and if fail from the server, retry in HTTPs
    @staticmethod
    def request_get(result, base_uri, uri, use_ssl=False, raise_exp=False, timeout=3):
        try:
            return Utils._request_get(base_uri, uri, use_ssl, timeout=timeout)
        except NotFoundException:
            raise NotFoundException
        except Exception as e:
            if raise_exp:
                raise
            else:
                msg = str(e)
                if e.__class__.__name__ == 'timeout':
                    msg = 'The request timed out (%ss)' % timeout
                result.hard_exit(EXIT_STATUS.CRITICAL, 'Cannot connect to %s%s with exception : <br/> %s' % (base_uri, uri, msg))
    
    
    @staticmethod
    def print_time(_time):
        return datetime.datetime.fromtimestamp(_time).strftime('%x %X')
    
    
    @staticmethod
    def add_style(output):
        html_items = (HTMLList, HTMLTable, HTMLTag)
        
        active_item = []
        for html_item in html_items:
            if html_item.used:
                active_item.append(html_item.STYLE)
                active_item.append(html_item.EXTRA_STYLE)
                html_item.used = False
        
        if active_item:
            output = u'<style type="text/css">%s</style>%s' % (''.join(active_item), output)
        return output
    
    
    @staticmethod
    def print_human_readable_number(number):
        if sys.version_info < (2, 7):
            return str(int(number))
        else:
            return '{:,}'.format(int(number)).replace(',', ' ')
    
    
    @staticmethod
    def print_human_readable_size(size):
        if isinstance(size, basestring):
            try:
                size = int(size)
            except ValueError:
                return size
        
        if not isinstance(size, (float, int)):
            return size
        
        if size < 1024:
            return '%s octets' % size
        elif 1024 <= size < (1024 * 1024):
            return '%.2f Ko' % (size / 1024)
        elif (1024 * 1024) <= size < (1024 * 1024 * 1024):
            return '%.2f Mo' % (size / 1024 / 1024)
        else:
            return '%.2f Go' % (size / 1024 / 1024 / 1024)
    
    
    @staticmethod
    def print_human_readable_period(time_period, time_format='auto'):
        return print_human_readable_period(time_period, time_format)
    
    
    @staticmethod
    def print_percent(_value, _total):
        return '%0.2f' % ((_value / _total) * 100.0)


class ShinkenUtils:
    
    @staticmethod
    def request_get_daemon(result, daemon_type, base_uri, uri, use_ssl=False, timeout=3):
        try:
            return Utils.request_get(result, base_uri, uri, use_ssl, timeout=timeout)
        except NotFoundException:
            result.hard_exit(EXIT_STATUS.CRITICAL, 'Cannot connect to uri : "%s%s". %sThe uri doesn\'t exists (error 404)' % (base_uri, uri, BREAK_LINE))
        except Exception as e:
            result.hard_exit(EXIT_STATUS.CRITICAL, 'Cannot connect to %s daemon at %s' % (daemon_type, base_uri), 'cause by : %s' % e)
    
    
    @staticmethod
    def add_module_info(result, data):
        module_infos = data.get('modules_info', {})
        if not module_infos:
            return ''
        
        lines = []
        status = EXIT_STATUS.OK
        for module_info in module_infos:
            submodules = module_info.get('modules', [])
            submodules_info = '-'
            if submodules:
                sub = ['%s : %s' % (i['name'], HTMLTag.state_tag(i['status'])) for i in submodules]
                submodules_info = HTMLList.simple_list(sub)
            
            last_restart = module_info.get('last_restart', {})
            last_restart = Utils.print_time(last_restart['timestamp']) if last_restart else ''
            module_state = HTMLTag.STATE_WARNING if module_info.get('nb_restart', 0) and module_info['status'] == ModuleState.OK else module_info['status']
            
            if module_info['status'] != 'OK':
                status = EXIT_STATUS.WARNING
            
            lines.append((module_info['name'], module_info['type'], HTMLTag.state_tag(module_state), module_info.get('nb_restart', 0), last_restart, submodules_info))
        
        long_output = HTMLTable.table(('Name', 'Type', 'Status', 'Restart in the last 2h', 'Last restart date', 'Submodules'), lines, 'Module info')
        result.add_check(status=status, long_output=long_output)
    
    
    @staticmethod
    def add_http_error_count_message(result, data):
        http_errors_count = data.get('http_errors_count', {})
        if http_errors_count:
            result.add_check(EXIT_STATUS.WARNING, "Some API calls between daemons failed in the last 24 hours (%d errors). Please look at your daemon logs for more details about these errors.%s" % (sum(http_errors_count.values()), BREAK_LINE))
    
    
    @staticmethod
    def add_warning_module_restart(result, data):
        modules_info = data.get('modules_info', [])
        
        restart_module = []
        for module_info in modules_info:
            module_restarts = module_info.get('restarts', [])
            if len(module_restarts) <= 0:
                continue
            
            # Manage restarts stored in the old format : [ts1, ts2, ts3]
            if isinstance(module_restarts[-1], float):
                module_restarts = [{'timestamp': ts, 'reason': "The reason for this restart was not saved"} for ts in module_restarts]
            
            limit_dt = datetime.datetime.now() - datetime.timedelta(minutes=120)
            restart_count = len(filter(lambda i: datetime.datetime.fromtimestamp(i['timestamp']) > limit_dt, module_restarts))
            module_info['nb_restart'] = restart_count
            module_info['last_restart'] = module_restarts[-1]
            if restart_count > 0:
                restart_module.append('The module %s has restarted [%s times]. A restart is removed from count after 2h.' % (module_info['name'], restart_count))
        
        if restart_module:
            result.add_check(EXIT_STATUS.WARNING, output=HTMLList.header_list('Some modules have restarted since the last 2h', restart_module))
    
    
    @staticmethod
    def check_arbiter_connection(result, uri, daemon_type, timeout=3):
        data, _ = ShinkenUtils.request_get_daemon(result, daemon_type, uri, '/arbiter_traces_get', timeout=timeout)
        data = json.loads(data)
        if len(data) == 1:
            # Correct number of arbiter (one).
            arbiter = data[0]
            current_time = int(time.time())
            insert_time = arbiter['insert_time']
            diff_time_with_arbiter = arbiter.get('diff_time_with_arbiter', 0)
            expire_period = arbiter.get('expire_period', 2 * arbiter.get('check_interval', 60))
            expire_time = insert_time + expire_period
            expire_in = expire_time - current_time
            diff_time_info = '( and no time shift )'
            if abs(arbiter.get('diff_time_with_arbiter', 0)) > 30:
                output = 'Correct connection from arbiter "%s" but a time shift of %d seconds' % (data[0]['name'], diff_time_with_arbiter)
                result.hard_exit(EXIT_STATUS.CRITICAL, output)
            elif expire_in < 0:
                output = 'Missed connection from arbiter "%s" since %i seconds %s%s' % (data[0]['name'], current_time - insert_time, diff_time_info, BREAK_LINE)
                result.add_check(EXIT_STATUS.WARNING, output)
                return
        elif len(data) == 0:
            # Not yet contacted by an arbiter.
            result.hard_exit(EXIT_STATUS.WARNING, 'Daemon has not been contacted by an arbiter for now.')
        else:
            _arbiters = []
            # More than one arbiter, problem.
            current_time = int(time.time())
            for _arbiter in data:
                # Compute and print the time in wich the arbiter entry will expire
                insert_time = _arbiter['insert_time']
                expire_period = _arbiter.get('expire_period', 2 * _arbiter.get('check_interval', 60))
                expire_time = insert_time + expire_period
                expire_in = expire_time - current_time
                _arbiters.append('Arbiter: %s (%s)\t This arbiter will expire in %s seconds.' % (_arbiter['name'], _arbiter['uri'], expire_in))
            
            output = HTMLList.header_list('Arbiters conflicts', _arbiters)
            result.hard_exit(EXIT_STATUS.CRITICAL, output)
    
    
    @staticmethod
    def _check_versions(daemon_type, daemon_api_version, daemon_version, arbiter_version, shinken_supervisor_version):
        # Let's check first the API version
        if daemon_api_version and daemon_api_version != API_VERSION:
            if daemon_version and shinken_supervisor_version:
                return 'Your %s is alive but this daemon (%s) and the Shinken installation (%s) that monitors it are not in the same version.' % (daemon_type, daemon_version, shinken_supervisor_version)
            return 'Your %s is alive but not up to date. Please update.' % daemon_type
        
        if not daemon_version or not daemon_api_version:
            return 'Your %s is alive but not up to date. Please update.' % daemon_type
        
        if daemon_type != 'synchronizer' and not arbiter_version:
            return 'Your %s is alive but not up to date. Please update.' % daemon_type
        
        # the synchronizer has no arbiter_version because it doesn't use 'put_conf' but it's always on the same machine as the arbiter, so we don't need to test it
        if daemon_type != 'synchronizer':
            if daemon_version != arbiter_version:
                return 'Your %s is alive but its version (%s) is not the same as its arbiter (%s). Please update.' % (daemon_type, daemon_version, arbiter_version)
        
        return None
    
    
    @staticmethod
    def minimal_check(result, data, daemon_type, shinken_supervisor_version):
        have_conf = data.get('have_conf', False)
        if not have_conf:
            output = 'No configuration given by an Arbiter for now.'
            result.hard_exit(EXIT_STATUS.WARNING, output)
        
        # If the daemon have his type in get_raw_stats, we compare it with the wanted daemon_type. Else, We pass because lower API_VERSION can be checked
        checked_daemon_type = data.get('daemon_type', None)
        if checked_daemon_type and checked_daemon_type != daemon_type:
            output = 'The daemon being checked is not a %s. This daemon is a %s. Please check the connection information (address, port) in the check configuration.' % (daemon_type, checked_daemon_type)
            result.hard_exit(EXIT_STATUS.WARNING, output)
        
        daemon_version = data.get('daemon_version', None)
        arbiter_version = data.get('arbiter_version', None)
        daemon_api_version = data.get('api_version', None)
        output = ShinkenUtils._check_versions(daemon_type, daemon_api_version, daemon_version, arbiter_version, shinken_supervisor_version)
        if output:
            result.hard_exit(EXIT_STATUS.WARNING, output)
        
        result.set_spare_info(data, daemon_type)
        result.set_logger_stats(data)  # if logger is too slow, will be WARNING


HTMLTag.CRITICAL = HTMLTag.state_tag(HTMLTag.STATE_CRITICAL)
HTMLTag.WARNING = HTMLTag.state_tag(HTMLTag.STATE_WARNING)
HTMLTag.OK = HTMLTag.state_tag(HTMLTag.STATE_OK)
