#!/bin/bash

if [[ ! -f patch.conf ]]; then
    printf "\033[31m Error: please launch the patch from its directory\033[0m\n"
    exit 2
fi

if [[ $EUID != 0 ]]; then
    printf "\033[31m Error: please launch this script as root\033[0m\n"
    exit 2
fi

. patch.conf

function display_help() {
    printf "install-patch [options]\n"
    printf "This script will install the patch PATCH${PATCH_VERSION} for the shinken version v${SHINKEN_VERSION}-release.${PATCH_LANG,,}\n"
    printf "Options:\n"
    printf "  --help          -h    : will display this help\n"
    printf "  --force               : removes the need to hit a key when required during installation. All daemons will be restarted\n"
    printf "\n"
}

force=""

while [ $# -ge 1 ]; do
    case "$1" in
    -h | --help)
        display_help # Call your function
        # no shifting needed here, we're done.
        exit 0
        ;;
    --force)
        #  It's better to assign a string, than a number like "verbose=1"
        #  because if you're debugging script with "bash -x" code like this:
        #    if [ "$verbose" ] ...
        #  You will see:
        #    if [ "verbose" ] ...
        force="--force"
        shift
        ;;
    --) # End of all options
        shift
        break
        ;;
    -*)
        echo "Error: Unknown option: $1" >&2
        display_help
        exit 1
        ;;
    *) # No more options
        break
        ;;
    esac
done

if [[ ! -f common.sh ]]; then
    printf "\033[31m Error: please launch the patch from its directory\033[0m\n"
    exit 2
fi

. common.sh

if [[ "$force" != "" ]]; then
    printf -- "\n${YELLOW}NOTE: FORCE MODE ENABLED${RESET}\n"
fi

run_python common.py --check-version

restore_all_patch_valid_rights

# NO SPACE ALLOWED in patch name
print_step "Patch information" "1" "10"
init_patch_name_and_location "$PATCH_ID"
display_acceptance_banner "$force"

# If we already did apply the patch, we do exit
run_python common.py --patch-is-not-installed

print_step "Stopping Shinken daemons" "2" "10"
run_python common.py --stop "${force}"

print_step "Revert old patch" "3" "10"
if [[ ${PATCH_TO_REVERT} == "" ]]; then
    printf -- "\n${YELLOW}There is no patch to revert${RESET}\n"
else
    for patch_id in ${PATCH_TO_REVERT}; do
        run_python common.py --restore-after-patch --id "${patch_id}"
    done
fi

print_step "Backup files" "4" "10"
run_python common.py --backup-before-patch

print_step "Apply patch" "5" "10"
run_python common.py --apply-patch-files

print_step "Apply new selinux rules" "6" "10"
run_python common.py --remove-selinux-rules --selinux-rules-path "selinux/updated-rules"
run_python common.py --apply-selinux-rules --selinux-rules-path "selinux/all-rules"

print_step "Looking if rpms need to be updated and check system parameters" "7" "10"
run_python common.py --update-rpms-and-system --rpms-path "rpms"

# NOTE: to remove for a target < 02.07.00
shinken-context-manage --install-patch --data-type "$PATCH_CATEGORIES" --patch-name "$PATCH_NAME"

print_step "Executing sanatize" "8" "10"
run_python common.py --sanatize "${force}"

print_step "Restart Shinken daemons" "9" "10"
run_python common.py --start "${force}"

print_step "Restart external daemons : ${EXTERNAL_DAEMONS}" "10" "10"
if [[ ${EXTERNAL_DAEMONS} == "" ]]; then
    printf -- "\n${YELLOW}There is no external daemons to restart${RESET}\n\n"
fi

for external_daemon in ${EXTERNAL_DAEMONS}; do
    run_python common.py --restart --service-list "${external_daemon}" "${force}"
done

# the ,, in the variable means "to lower"
printf -- "\n$YELLOW Your shinken installation was patched from version v${SHINKEN_VERSION}-release.${PATCH_LANG,,} to PATCH${PATCH_VERSION}${RESET}\n\n"
