#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import errno
import fnmatch
import os
import shutil

from shinken.log import logger


class COLOR:
    GREY = '\033[90m'
    BLUE = '\033[94m'
    MAGENTA = '\033[94m'
    YELLOW = '\033[93m'
    RED = '\033[31m'
    RESET = '\033[0m'


def create_tree(_path):
    abs_path = os.path.abspath(_path)
    parent = os.path.dirname(abs_path)
    
    if os.path.exists(abs_path):
        return
    elif os.path.exists(parent):
        os.makedirs(abs_path)
        return
    else:
        create_tree(parent)
        create_tree(_path)


def find_files(directory, pattern):
    for root, dirs, files in os.walk(directory):
        for basename in files:
            if fnmatch.fnmatch(basename, pattern):
                filename = os.path.join(root, basename)
                yield filename


def find_folder(directory, pattern):
    for root, dirs, files in os.walk(directory):
        for basename in dirs:
            if fnmatch.fnmatch(basename, pattern):
                filename = os.path.join(root, basename)
                yield filename


def copy_recursively(_path, dest, debug=False):
    files_to_copy = os.walk(_path)
    for _root, _dirs, _files in files_to_copy:
        
        absolute_path = _root.split('%s%s' % (_path, os.sep))
        absolute_path = absolute_path[1] if len(absolute_path) > 1 else ''
        
        for _dir in _dirs:
            create_tree(os.path.join(dest, absolute_path, _dir))
        for _file in _files:
            copy_file(os.path.join(_root, _file), os.path.join(dest, absolute_path, _file), debug=debug)


def copy_file(_source, _dest, deleted_files=None, debug=False):
    if os.name == 'nt':
        _dest = u'\\\\?\\%s' % os.path.abspath(_dest)
    else:
        _dest = os.path.abspath(_dest)
    
    if os.path.exists(_dest):
        logger.info('File [%s] has been overwritten.' % _dest.strip(u'\\\\?\\'))
    
    if debug:
        logger.debug('Copy file [%s] \t==>\t [%s]' % (_source, _dest.strip(u'\\\\?\\')))
    
    file_path = os.path.dirname(_dest)
    create_tree(file_path)
    try:
        shutil.copyfile(_source, _dest)
    except IOError:
        if deleted_files is not None:
            deleted_files.append(_source)
        else:
            logger.warning('The file %s can\'t be copied.' % _source)


def touch_file(filename):
    with open(filename, 'a'):
        os.utime(filename, None)


def mkdir_p(path):
    try:
        os.makedirs(path)
    except OSError as exc:  # Python >2.5
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise
