#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import cPickle
import glob
import optparse
import os
import shutil
import sys
import time

try:
    import pwd
    import grp
except ImportError:
    pwd, grp = None, None

from shinkensolutions.localinstall import get_local_instances_for_type
from shinkensolutions.system_tools import run_command

try:
    from shinkensolutions.system_tools import run_command_with_return_code
except ImportError:
    import subprocess
    from shinken.util import make_unicode
    
    
    def run_command_with_return_code(command):
        _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
        ret = _process.communicate()
        return _process.returncode, make_unicode(ret[0][:-1]), make_unicode(ret[1][:-1])

# The patch_lib is added on build_patch, do not fix these error
from lib.lib_patch.patch_new_hash_list import PatchNewHashList, PatchNewHashListException
from lib.lib_patch.misc import create_tree, copy_recursively, mkdir_p, COLOR, ask_user, print_info, print_error

PATCH_DAT = {}
with open(u'patch.dat', u'rb') as f:
    PATCH_DAT = cPickle.loads(f.read())

DAEMON_ORDER = {
    u'gatherer'    : 0,
    u'scheduler'   : 1,
    u'poller'      : 2,
    u'reactionner' : 3,
    u'broker'      : 4,
    u'receiver'    : 5,
    u'synchronizer': 6,
    u'arbiter'     : 7,
}

PYTHON_MINOR_VERSION = sys.version_info[1]  # cannot use .minor because python2.6 do not have it
WEBUI_PATH = u'/var/lib/shinken/modules/webui/htdocs/ui'

FILES_DESTS = {
    u'shinken_lib'             : {u'path': u'/usr/lib/python2.%s/site-packages/shinken' % PYTHON_MINOR_VERSION},
    u'shinken_solutions_lib'   : {u'path': u'/usr/lib/python2.%s/site-packages/shinkensolutions' % PYTHON_MINOR_VERSION},
    u'modules'                 : {u'path': u'/var/lib/shinken/modules'},
    u'var_lib_shinken'         : {u'path': u'/var/lib/shinken'},
    u'graphite'                : {u'path': u'/opt/graphite'},
    u'shinken_libexec'         : {u'path': u'/var/lib/shinken/libexec/', u'user': u'shinken', u'rights': 0755},
    u'shinken_user_lib'        : {u'path': u'/var/lib/shinken-user/', u'user': u'shinken'},
    u'etc_shinken'             : {u'path': u'/etc/shinken', u'user': u'shinken'},
    u'etc_shinken_user'        : {u'path': u'/etc/shinken-user', u'user': u'shinken'},
    u'etc_shinken_user_example': {u'path': u'/etc/shinken-user-example', u'user': u'shinken', u'rights': 0555},
    u'root_etc'                : {u'path': u'/etc'},
    u'webui'                   : {u'path': WEBUI_PATH},
    u'shinken_pack'            : {u'path': u'/etc/shinken/packs', u'user': u'shinken'},
    u'init.d'                  : {u'path': u'/etc/init.d', u'rights': 0755},
    u'usr_sbin'                : {u'path': u'/usr/sbin', u'rights': 0755},
    u'site-packages'           : {u'path': u'/usr/lib/python2.%s/site-packages/' % PYTHON_MINOR_VERSION},
}

LOG_FILE_PATCHNEW_NAME = u'files_patchnew_created.log'
SANATIZE_LOG_NAME = u'sanatize.log'


#####################################################################
# FILE DISPATCHER
#####################################################################
def _get_files_and_destinations(root_dir=None, mapping=FILES_DESTS):
    if root_dir is None:
        root_dir = os.path.join(os.getcwd(), u'files')
    
    to_dispatch_files = []
    for source_dir, destination_entry in mapping.iteritems():
        if source_dir == u'webui':
            continue
        if source_dir == u'shinken_pack':
            continue
        
        destination_dir = destination_entry[u'path']
        destination_user = destination_entry.get(u'user', None)
        destination_rights = destination_entry.get(u'rights', None)
        
        for root, dir_names, file_names in os.walk(os.path.join(root_dir, source_dir)):
            for filename in file_names:
                if filename == u'._for_git' or filename.endswith(u'~'):
                    continue
                
                local_file_path = os.path.join(root, filename)
                
                from_root_path = local_file_path[len(os.path.join(root_dir, source_dir)) + 1:]
                #                destination_file = os.path.join(destination_dir, source_dir, filename)
                
                destination_file = os.path.join(destination_dir, from_root_path)
                inside_files_path = local_file_path[len(root_dir) + 1:]
                entry = {
                    u'local_file_path'  : local_file_path,  # u'/root/XXX...XXX-PATCH01/files/modules/simple-log/module.py'
                    u'destination_file' : destination_file,  # u'/var/lib/shinken/modules/simple-log/module.py'
                    u'inside_files_path': inside_files_path,  # u'modules/simple-log/module.py'
                    u'from_files_path'  : source_dir,  # u'modules'
                    u'user'             : destination_user,  # shinken or None
                    u'rights'           : destination_rights,  # int 0755 or None
                }
                to_dispatch_files.append(entry)
    return to_dispatch_files


def is_config_file_managed(file_path):
    # type: (unicode) -> bool
    
    managed_paths = [
        os.path.join(os.path.sep, u'etc', u'shinken'),  # include also /etc/shinken-user
        os.path.join(os.path.sep, u'opt', u'graphite', u'conf')
    ]
    
    exclude_paths = (
        os.path.join(os.path.sep, u'etc', u'shinken', u'packs', u'shinken'),
        os.path.join(os.path.sep, u'etc', u'shinken', u'packs', u'shinken-core'),
        os.path.join(os.path.sep, u'etc', u'shinken', u'_default'),
        os.path.join(os.path.sep, u'etc', u'shinken', u'external'),
        os.path.join(os.path.sep, u'etc', u'shinken-user-example'),  # The user-example must be overwrite but not the shinken-user
        os.path.join(os.path.sep, u'etc', u'shinken-skeletons'),  # The skeletons must be overwrite
    )
    
    is_managed = False
    
    for managed_path in managed_paths:
        if file_path.startswith(managed_path):
            is_managed = True
            for exclude_path in exclude_paths:
                if file_path.startswith(exclude_path):
                    return False
    return is_managed


def manage_file_patchnew(hash_list_manager, file_path, local_file_path, patchnew_list):
    # type: (PatchNewHashList, unicode, unicode, list) -> unicode
    
    file_path_with_patchnew = u'%s%s' % (file_path, u'.patchnew')
    
    if not is_config_file_managed(file_path):
        return file_path
    
    if not os.path.exists(file_path):
        if file_path == u'/etc/shinken/daemons/gathererd.ini':
            # The gathererd.ini must be added if missing
            return file_path
        patchnew_list.append(file_path_with_patchnew)
        return file_path_with_patchnew
    
    hash_list = hash_list_manager.get_hash_list_for_file(local_file_path)
    
    if not hash_list:
        return file_path
    
    with open(file_path, u'r') as _file:
        user_file_hash = PatchNewHashList.get_hash_for_file(_file.read())
    
    if user_file_hash in hash_list:
        return file_path
    
    patchnew_list.append(file_path_with_patchnew)
    return file_path_with_patchnew


def backup_before_patch(backup_dir):
    print_info(u' - Backup the files before applying the patch', color=COLOR.YELLOW)
    to_dispatch_files = _get_files_and_destinations()
    for entry in to_dispatch_files:
        destination_file = entry[u'destination_file']
        if not os.path.exists(destination_file):
            print_info(u'   * No need to backup the file %s as it does not exists' % destination_file)
            continue
        inside_files_path = entry[u'inside_files_path']
        backup_path = os.path.join(backup_dir, inside_files_path)
        # If the backup is already existing, it's an error, we should NOT lost it
        if os.path.exists(backup_path):
            do_exit_error(u'The backup file %s is already existing, cannot erase it' % backup_path)
        print_info(u'   * Backup %s to %s' % (destination_file, backup_path))
        backup_path_directory = os.path.dirname(backup_path)
        try:
            mkdir_p(backup_path_directory)
            shutil.copy2(destination_file, backup_path)
        except Exception as exp:
            do_exit_error(u'Cannot save the file %s to %s: %s' % (destination_file, backup_path, exp))
    # maybe we will have no file to backup, so force one file to be set
    pth = os.path.join(backup_dir, u'installed')
    
    file_content = u'Installed at: %s' % int(time.time())
    
    with open(pth, u'w') as _install_file:
        _install_file.write(file_content.encode(u'utf-8'))
    
    _patch_directory = os.path.abspath(os.path.dirname(__file__))
    # For the webui, we backup all the folder because, all the folder will be replaced
    if os.path.exists(os.path.join(_patch_directory, u'files', u'webui')):
        print_info(u'   * Backup webui')
        copy_webui_and_packs(WEBUI_PATH, os.path.join(backup_dir, u'webui'), delete_before=False)
    
    # For the shinken pack, we backup all the folder
    if os.path.exists(os.path.join(_patch_directory, u'files', u'shinken_pack')):
        print_info(u'   * Backup shinken packs')
        copy_webui_and_packs(u'/etc/shinken/packs/shinken', os.path.join(backup_dir, u'shinken_pack', u'shinken'), delete_before=False)
        copy_webui_and_packs(u'/etc/shinken/packs/shinken-core', os.path.join(backup_dir, u'shinken_pack', u'shinken-core'), delete_before=False)
    
    # We have to backup the to_dispatch_files variable to know what files was installed during patch (to revert it in next patch)
    with open(os.path.join(backup_dir, u'file_mapping.dat'), u'wb') as file_mapping:
        file_mapping.write(cPickle.dumps(to_dispatch_files))
    
    do_exit()


def revert_patch( patch_id_to_revert=None):
    from shinkensolutions.localinstall import _context
    
    if patch_id_to_revert:
        installed_patch = [patch_id_to_revert]
    else:
        installed_patch = []
    
        for install in _context[u'installation_history']:
            install_type = install.get(u'type')
            patch_version = install.get(u'patch_name', u'NO PATCH NAME')
            
            if install_type == u'UPDATE':
                #reset patchs if new Update done, in order to not unpatch old version
                installed_patch = []
            
            elif install_type == u'PATCH':
                installed_patch.append(patch_version)
            
            elif install_type == u'UNPATCH':
                installed_patch.reverse()
                try:
                    installed_patch.remove(patch_version)
                except ValueError:
                    raise Exception(u'The patch %s was uninstall without being installed.' % patch_version)
                installed_patch.reverse()
        
    if len(installed_patch) == 0:
        print_info(u'   * There is no patch to revert')
    else:
        installed_patch.reverse()
        for patch_id in installed_patch:
            backup_dir = get_backup_dir(patch_id)
            if not os.path.exists(os.path.join(backup_dir, u'installed')):
                print_info(u' - The patch %s is not installed' % patch_id)
                continue

            print_info(u' - Restoring the files for the patch %s' % patch_id, color=COLOR.YELLOW)
            to_dispatch_files = None
            file_mapping_path = os.path.join(backup_dir, u'file_mapping.dat')
            
            if os.path.exists(file_mapping_path):
                try:
                    with open(file_mapping_path, u'rb') as file_mapping:
                        to_dispatch_files = cPickle.loads(file_mapping.read())
                except Exception:
                    pass
            
            if to_dispatch_files is None:
                to_dispatch_files = _get_files_and_destinations(backup_dir)
            
            for entry in to_dispatch_files:
                destination_file = entry[u'destination_file']
                user = entry[u'user']
                rights = entry[u'rights']
                inside_files_path = entry[u'inside_files_path']
                file_backup_path = os.path.join(backup_dir, inside_files_path)
                
                if is_config_file_managed(destination_file):
                    # Don't touch the user config files !!!!!
                    continue
                if not os.path.exists(file_backup_path):
                    print_info(u'   * File %s was not present before patch %s. Remove it' % (destination_file, patch_id))
                    try:
                        os.unlink(destination_file)
                    except Exception as exp:
                        print_info(u'The file %s is already deleted' % destination_file)
                    continue
                
                print_info(u'   * Restoring %s to %s' % (file_backup_path, destination_file))
                try:
                    mkdir_p(os.path.dirname(destination_file))
                    shutil.move(file_backup_path, destination_file)
                    if user is not None:
                        uid = pwd.getpwnam(user).pw_uid
                        gid = grp.getgrnam(user).gr_gid
                        os.chown(destination_file, uid, gid)
                    if rights is not None:
                        os.chmod(destination_file, rights)
                except Exception as exp:
                    do_exit_error(u'Cannot restore the file %s to %s: %s' % (file_backup_path, destination_file, exp))
            
            web_ui_dir = os.path.join(backup_dir, u'webui')
            if os.path.exists(web_ui_dir):
                print_info(u'   * Restoring webui')
                copy_webui_and_packs(web_ui_dir, WEBUI_PATH)
            
            if os.path.exists(os.path.join(backup_dir, u'shinken_pack')):
                print_info(u'   * Restoring shinken and shinken-core packs')
                copy_webui_and_packs(os.path.join(backup_dir, u'shinken_pack', u'shinken'), u'/etc/shinken/packs/shinken')
                copy_webui_and_packs(os.path.join(backup_dir, u'shinken_pack', u'shinken-core'), u'/etc/shinken/packs/shinken-core')
            
            # At the end, should clean what we put in the backup dir from the files (do not erase all as maybe the patch set something else into it)
            for files_dest_root in FILES_DESTS.keys():
                files_dest_backup_dir = os.path.join(backup_dir, files_dest_root)
                if os.path.exists(files_dest_backup_dir):
                    print_info(u'   * Cleaning backup dir %s' % files_dest_backup_dir)
                    try:
                        shutil.rmtree(files_dest_backup_dir)
                    except Exception as exp:
                        do_exit_error(u'Cannot clean backup directory %s: %s' % (files_dest_backup_dir, exp))
            # Also remove the installed file, as it it no more need
            pth = os.path.join(backup_dir, u'installed')
            
            if os.path.exists(pth):
                os.unlink(pth)
            if os.path.exists(file_mapping_path):
                os.unlink(file_mapping_path)
            
            return_code, _, _ = run_command_with_return_code(u'which shinken-context-manage')
            if return_code == 0:
                run_command(u'''shinken-context-manage --revert-patch --data-type "installation" --patch-name "%s" 2> /dev/null''' % patch_id)
            

def apply_patch_files(log_directory_path):
    print_info(u' - Applying patch files', color=COLOR.YELLOW)
    try:
        hash_list_manager = PatchNewHashList(u'patch_new_hash_list.dat')
    except PatchNewHashListException:
        do_exit_error(u'Cannot load patch_new_hash_list.dat. Exiting')
    
    patch_new_list = []
    to_dispatch_files = _get_files_and_destinations()
    for entry in to_dispatch_files:
        inside_files_path = entry[u'inside_files_path']
        local_file_path = entry[u'local_file_path']
        destination_file = manage_file_patchnew(hash_list_manager, entry[u'destination_file'], entry[u'inside_files_path'], patch_new_list)
        user = entry[u'user']
        rights = entry[u'rights']
        print_info(u'   * Copying %s to %s' % (inside_files_path, destination_file))
        dest_dir = os.path.dirname(destination_file)
        try:
            mkdir_p(dest_dir)
            shutil.copy2(local_file_path, destination_file)
            if user is not None:
                uid = pwd.getpwnam(user).pw_uid
                gid = grp.getgrnam(user).gr_gid
                os.chown(destination_file, uid, gid)
                os.chown(dest_dir, uid, gid)
                print_info(u'     - setting as owner %s/%s' % (user, user))
            if rights is not None:
                os.chmod(destination_file, rights)
                print_info(u'     - setting rights as %o' % rights)
        except Exception as exp:
            do_exit_error(u'Cannot save the file %s to %s: %s' % (local_file_path, destination_file, exp))
    _patch_directory = os.path.abspath(os.path.dirname(__file__))
    if os.path.exists(os.path.join(_patch_directory, u'files', u'webui')):
        print_info(u'   * Copying webui')
        copy_webui_and_packs(os.path.join(_patch_directory, u'files', u'webui'), WEBUI_PATH)
    
    if os.path.exists(os.path.join(_patch_directory, u'files', u'shinken_pack')):
        print_info(u'   * Copying shinken and shinken-core packs')
        copy_webui_and_packs(os.path.join(_patch_directory, u'files', u'shinken_pack', u'shinken'), u'/etc/shinken/packs/shinken')
        copy_webui_and_packs(os.path.join(_patch_directory, u'files', u'shinken_pack', u'shinken-core'), u'/etc/shinken/packs/shinken-core')
    
    if patch_new_list:
        patch_new_log_file_path = os.path.join(log_directory_path, LOG_FILE_PATCHNEW_NAME)
        file_content_list = [
            u'Some configuration files have been delivered with the extension ".patchnew" because the original file has been modified or deleted.'
            u'These new files will not be taken into account by Shinken and they allow you to compare the changes made using the "diff" command :'
        ]
        for patch_new_file in patch_new_list:
            file_content_list.append(u'- %s' % patch_new_file)
        
        last_file_patch_new = patch_new_list[-1]
        original_last_file, _ = os.path.splitext(patch_new_list[-1])
        file_content_list.append(u'\nHere is an example of how to use the diff command : diff -u %s %s\n\n' % (original_last_file, last_file_patch_new))
        
        file_content = u'\n'.join(file_content_list)
        
        with open(patch_new_log_file_path, u'w') as patch_new_log_file:
            patch_new_log_file.write(file_content.encode(u'utf-8'))
    
    do_exit()


#####################################################################
# Miscellaneous
#####################################################################
def do_exit_error(msg):
    print_error(msg)
    exit(2)


def do_exit(msg=None, color=COLOR.BLUE):
    if msg:
        print u'\n%s%s%s\n' % (color, msg, COLOR.RESET)
    exit(0)


def get_backup_dir(patch_id):
    backup_dir = u'/var/lib/shinken/backup/shinken-patch-%s' % patch_id
    return backup_dir


def copy_file(_source, _dest):
    file_path = os.path.dirname(_dest)
    create_tree(file_path)
    try:
        shutil.copyfile(_source, _dest)
    except IOError as err:
        do_exit_error(err.message)


#####################################################################
# GET DAEMONS INFOS
#####################################################################
def _daemon_type_is_enabled(daemon_type):
    if daemon_type == u'gatherer':
        # The gatherer is not managed in get_local_instances_for_type, but it must always be enabled if it's installed
        return os.path.exists(u'/etc/init.d/shinken-gatherer')
    id_daemons_enabled = [daemon_id for daemon_id, daemon_enabled in get_local_instances_for_type(daemon_type) if daemon_enabled]
    return bool(len(id_daemons_enabled) > 0)


#####################################################################
# MANAGE DAEMONS START/STOP
#####################################################################
def request_start_all_daemons(force):
    daemons_needed = PATCH_DAT.get(u'RESTART_DAEMONS', [])
    if not daemons_needed:
        do_exit(u'No need to start any daemon')
    
    daemons_to_start = sorted([daemon_type.strip() for daemon_type in daemons_needed if daemon_type.strip() and _daemon_type_is_enabled(daemon_type)], key=lambda d: DAEMON_ORDER.get(d, 99))
    
    if not daemons_to_start:
        do_exit(u'No need to start any daemon')
    
    if force:
        for daemon_type in daemons_to_start:
            _manage_service(u'shinken-%s' % daemon_type, u'start')
        do_exit()
    
    ask_msg = u'This patch need to start such daemons : %s.\nDo you want to start all these daemons now ?' % u', '.join(daemons_to_start)
    user_choice = ask_user(ask_msg)
    
    if user_choice in (u'o', u'y'):
        for daemon_type in daemons_to_start:
            _manage_service(u'shinken-%s' % daemon_type, u'start')
        do_exit()
    else:
        do_exit_error(u'No daemon has been started. In order for the patch to be effective, you must restart the following daemons : %s' % u', '.join(daemons_to_start))


def request_stop_all_daemons(force):
    daemons_needed = PATCH_DAT.get(u'RESTART_DAEMONS', [])
    if not daemons_needed:
        do_exit(u'No need to stop any daemon')
    
    daemons_to_stop = sorted([daemon_type.strip() for daemon_type in daemons_needed if daemon_type.strip() and _daemon_type_is_enabled(daemon_type)], key=lambda d: DAEMON_ORDER.get(d, 99))
    
    if not daemons_to_stop:
        do_exit(u'No need to stop any daemon')
    
    if force:
        for daemon_type in daemons_to_stop:
            _manage_service(u'shinken-%s' % daemon_type, u'stop')
        do_exit()
    
    ask_msg = u'This patch need to stop such daemons : %s.\nDo you want to stop all these daemons now ?' % u', '.join(daemons_to_stop)
    user_choice = ask_user(ask_msg)
    
    if user_choice in (u'o', u'y'):
        for daemon_type in daemons_to_stop:
            _manage_service(u'shinken-%s' % daemon_type, u'stop')
    else:
        do_exit_error(u'No daemon was stopped. You must stop the daemon to apply the patch')


def request_restart_services(service, force):
    if not service:
        do_exit(u'No need to restart any service')
    
    if force:
        _manage_service(service, u'restart')
        do_exit()
    
    ask_msg = u'This patch need to restart the %s service.\nDo you want to restart this service now ?' % service
    user_choice = ask_user(ask_msg)
    
    if user_choice in (u'o', u'y'):
        _manage_service(service, u'restart')
    else:
        do_exit(u'The service %s was not restarted.')


def _manage_service(service, action):
    print run_command(u'service %s %s' % (service, action))


#####################################################################
# SANATIZE
#####################################################################
def run_sanatize(log_path):
    sanatize_list = PATCH_DAT.get(u'SANATIZE_LIST', [])
    if not sanatize_list:
        do_exit(u'There is no sanatize to run into this patch', color=COLOR.YELLOW)
    
    sanatize_log = os.path.join(log_path, SANATIZE_LOG_NAME)
    for sanatize_name in sanatize_list:
        
        print run_command(u'python2 /var/lib/shinken/libexec/tools/sanatize-data.py -r %s -L %s' % (sanatize_name, sanatize_log))
    do_exit()


#####################################################################
# PATCHS INSTALLATION
#####################################################################
def patch_is_installed():
    _backup_dir = u'/var/lib/shinken/backup/shinken-patch-%s' % PATCH_DAT[u'PATCH_ID']
    # If there is no a file in the backup directory, means that it was already installed
    backup_files = os.listdir(_backup_dir) if os.path.exists(_backup_dir) else []
    if len(backup_files) == 0:
        do_exit_error(u'The patch %s is not installed. The backup directory %s is void' % (PATCH_DAT[u'PATCH_ID'], _backup_dir))
    do_exit(u' - Checking if the patch is already installed: OK')


def patch_is_not_installed():
    _backup_dir = u'/var/lib/shinken/backup/shinken-patch-%s' % PATCH_DAT[u'PATCH_ID']
    # If there is no a file in the backup directory, means that it was already installed
    backup_files = os.listdir(_backup_dir) if os.path.exists(_backup_dir) else []
    if len(backup_files) != 0:
        do_exit_error(u'The patch %s is already installed. The backup directory %s is not void' % (PATCH_DAT[u'PATCH_ID'], _backup_dir))
    do_exit(u' - Checking if the patch is not already installed: OK')


def check_version():
    shinken_version_in_patch = PATCH_DAT.get(u'SHINKEN_VERSION', u'')
    try:
        from shinkensolutions.localinstall import VERSION as shinken_version_installed
        if shinken_version_installed != shinken_version_in_patch:
            do_exit_error(u'This patch is designed to be install only on version %s' % shinken_version_in_patch)
    
    except ImportError:
        do_exit_error(u'This patch must be installer on a existing shinken installation')


#####################################################################
# WEBUI
#####################################################################
def copy_webui_and_packs(source, dest, delete_before=True):
    if delete_before:
        try:
            shutil.rmtree(dest)
        except OSError:
            pass
    copy_recursively(source, dest)


#####################################################################
# Selinux
#####################################################################
def apply_selinux_rules(path):
    selinux_pattern = os.path.join(path, u'*.pp')
    selinux_files = glob.glob(selinux_pattern)
    if len(selinux_files) == 0:
        do_exit(u'There is no selinux rules to update into this patch', color=COLOR.YELLOW)
    
    _backup_dir = os.path.join(get_backup_dir(PATCH_DAT[u'PATCH_ID']), u'selinux_rules')
    os.mkdir(_backup_dir)
    
    for root, dir_names, file_names in os.walk(path):
        for rule in file_names:
            if rule.endswith(u'.pp'):
                rule_path = os.path.join(os.path.abspath(root), rule)
                rule_name = rule.strip(u'.pp')
                src_rule_path = rule_path.replace(u'.pp', u'.te')
                # checkmodule -M -m -o /tmp/$rule_name.mod dependencies/selinux-rules/$rule_name.te 2>/dev/null >/dev/null
                run_command(u'''checkmodule -M -m -o /tmp/%s.mod %s''' % (rule_name, src_rule_path))
                # semodule_package -o  /tmp/$rule_name.pp -m /tmp/$rule_name.mod
                run_command(u'''semodule_package -o  /tmp/%s.pp -m /tmp/%s.mod''' % (rule_name, rule_name))
                # semodule -i /tmp/$rule_name.pp
                run_command(u'''semodule -i /tmp/%s.pp''' % rule_name)
                print_info(u'Rule %s applied' % rule_name, color=COLOR.YELLOW)
                # We need to backup the files for revert it if necessary. The .pp is sufficient because for revert we need only its name
                shutil.copy2(rule_path, _backup_dir)


def backup_configuration(backup_path):
    if _daemon_type_is_enabled(u'synchronizer'):
        return_code, _, std_err = run_command_with_return_code(u'''shinken-backup --configuration --user --output-name "backup-pre-update-version-%s" --output-directory "%s" ''' % (PATCH_DAT[u'PATCH_ID'], backup_path))
        if return_code == 2:
            do_exit_error(u'An error has occurred during the pre-update backup :\n%s\n\nThe update is aborted, please contact your Support.' % std_err)
        
        do_exit(u'Backup is available at : %s' % backup_path, color=COLOR.YELLOW)
    else:
        do_exit(u'There is no Synchronizer, no need to backup configuration', color=COLOR.YELLOW)


def remove_selinux_rules(path):
    is_in_revert = False
    if not path:
        # No specific folder ? ok take the backup dir
        is_in_revert = True
        path = os.path.join(get_backup_dir(PATCH_DAT[u'PATCH_ID']), u'selinux_rules')
    
    selinux_pattern = os.path.join(path, u'*.pp')
    selinux_files = glob.glob(selinux_pattern)
    if len(selinux_files) == 0:
        if is_in_revert and os.path.exists(path):
            shutil.rmtree(path)
        do_exit(u'There is no selinux rules to remove into this patch', color=COLOR.YELLOW)
    
    for root, dir_names, file_names in os.walk(path):
        for rule in file_names:
            if rule.endswith(u'.pp'):
                rule_name = rule.strip(u'.pp')
                run_command(u'''semodule -r %s''' % rule_name)
                print_info(u'Rule %s removed' % rule_name)
    
    # If we are in a revert-patch.sh/py (hope
    if is_in_revert and os.path.exists(path):
        shutil.rmtree(path)


#####################################################################
# RPMs & System
#####################################################################
def update_rpms_and_system(rpm_path, log_path):
    # Try to detect 6 or 7:
    minor = PYTHON_MINOR_VERSION
    rpm_pattern = os.path.join(rpm_path, u'%s.X' % minor, u'*.rpm')
    rpm_files = glob.glob(rpm_pattern)
    if len(rpm_files) != 0:
        print_info(u'Updating rpm files', color=COLOR.YELLOW)
        for rpm_file in rpm_files:
            print_info(u' - %s' % rpm_file, color=COLOR.BLUE)
        return_code, stdout, stderr = run_command_with_return_code(u'''source ./lib/common.sh;dorpminstall "%s" 'Patch %s' ''' % (rpm_pattern, PATCH_DAT[u'PATCH_ID']))
        if return_code == 2:
            try:
                with open(os.path.join(log_path, u'shinken.enterprise.install.detail.log'), u'r') as error_log:
                    error = error_log.read()
            except (OSError, IOError) :
                if stdout or stderr:
                    error = u'%s\n%s' % (stdout, stderr)
                else:
                    error = u'This error was not logged.'
                
            do_exit_error(u'Cannot install RPM :\n%s\n\nThis patch was not installed.' % error)
    else:
        print_info(u'There is no RPMs to update into this patch', color=COLOR.YELLOW)
    
    print_info(u'Checking system parameters', color=COLOR.YELLOW)
    print_info(u' - Check jemalloc configuration', color=COLOR.BLUE)
    run_command(u'''source ./lib/common.sh;assert_jemalloc_memory ''')


#####################################################################
# Run !!!!
#####################################################################
def parse_opts():
    parser = optparse.OptionParser(u'Common lib for patch')
    # Common
    parser.add_option(u'--id', dest=u'patch_id', action=u'store', default=u'', help=u'patch id')
    parser.add_option(u'--force', dest=u'force', action=u'store_true', default=False, help=u'bypass all answers')
    parser.add_option(u'--log-path', dest=u'log_path', action=u'store', default=u'', help=u'log path')
    # Restart daemon
    parser.add_option(u'--start', dest=u'start_all_daemons', action=u'store_true', default=False, help=u'request for start all daemons')
    parser.add_option(u'--stop', dest=u'stop_all_daemons', action=u'store_true', default=False, help=u'request for stop all daemons')
    # Restart external service
    parser.add_option(u'--restart-services', dest=u'restart_services', action=u'store_true', default=False, help=u'request for start all daemons')
    parser.add_option(u'--service-list', dest=u'service_list', action=u'store', default=u'', help=u'list of daemons required')
    # Patch installation
    parser.add_option(u'--patch-is-installed', dest=u'patch_is_installed', action=u'store_true', default=False, help=u'check if patch is already installed')
    parser.add_option(u'--patch-is-not-installed', dest=u'patch_is_not_installed', action=u'store_true', default=False, help=u'check if patch is not already installed')
    parser.add_option(u'--check-version', dest=u'check_version', action=u'store_true', default=False, help=u'check if patch will be installed on the good shinken version')
    # File dispatcher
    parser.add_option(u'--backup-before-patch', dest=u'backup_before_patch', action=u'store_true', default=False, help=u'Backup before install patch')
    parser.add_option(u'--revert-patch', dest=u'revert_patch', action=u'store_true', default=False, help=u'Restore after patch')
    parser.add_option(u'--apply-patch-files', dest=u'apply_patch_files', action=u'store_true', default=False, help=u'Apply the patch')
    # Backup_configuration
    parser.add_option(u'--backup-config', dest=u'backup_configuration', action=u'store_true', default=False, help=u'Backup configuration before doing some things')
    parser.add_option(u'--backup-path', dest=u'backup_path', action=u'store', default=u'', help=u'Path for store configuration backup')
    # Sanatize
    parser.add_option(u'--sanatize', dest=u'sanatize', action=u'store_true', default=False, help=u'run sanatize')
    # Selinux
    parser.add_option(u'--remove-selinux-rules', dest=u'remove_selinux_rules', action=u'store_true', default=False, help=u'Remove selinux rules')
    parser.add_option(u'--apply-selinux-rules', dest=u'apply_selinux_rules', action=u'store_true', default=False, help=u'Remove selinux rules')
    parser.add_option(u'--selinux-rules-path', dest=u'selinux_rules_path', action=u'store', default=u'', help=u'Selinux rules to add/remove')
    # RPMs
    parser.add_option(u'--update-rpms-and-system', dest=u'update_rpms_and_system', action=u'store_true', default=False, help=u'Update rpms and system')
    parser.add_option(u'--rpms-path', dest=u'rpms_path', action=u'store', default=u'', help=u'Top level path for rpms')
    
    return parser.parse_args()


def main():
    opts, args = parse_opts()
    
    # Restart daemon
    if opts.stop_all_daemons:
        request_stop_all_daemons(opts.force)
    elif opts.start_all_daemons:
        request_start_all_daemons(opts.force)
    
    # Restart services
    elif opts.restart_services:
        request_restart_services(opts.service_list, opts.force)
    
    # Patch installation
    elif opts.patch_is_installed:
        patch_is_installed()
    elif opts.patch_is_not_installed:
        patch_is_not_installed()
    elif opts.check_version:
        check_version()
    
    # File dispatcher
    elif opts.backup_before_patch:
        backup_dir = get_backup_dir(PATCH_DAT[u'PATCH_ID'])
        backup_before_patch(backup_dir)
    elif opts.revert_patch:
        revert_patch(opts.patch_id)
    
    elif opts.apply_patch_files:
        if not opts.log_path:
            do_exit_error(u'need a log path with option --log-path')
        apply_patch_files(opts.log_path)
    
    # Sanatize
    elif opts.sanatize:
        if not opts.log_path:
            do_exit_error(u'need a sanatize log file path with option --log-path')
        run_sanatize(opts.log_path)
    
    # SELinux rules
    elif opts.remove_selinux_rules:
        remove_selinux_rules(opts.selinux_rules_path)
    
    elif opts.apply_selinux_rules:
        apply_selinux_rules(opts.selinux_rules_path)
    
    # RPM & system
    elif opts.update_rpms_and_system:
        update_rpms_and_system(opts.rpms_path, opts.log_path)
    
    # backup configuration before start apply patch
    elif opts.backup_configuration:
        if not opts.backup_path:
            do_exit_error(u'need a backup configuration path')
        backup_configuration(opts.backup_path)


if __name__ == u'__main__':
    try:
        main()
    except KeyboardInterrupt:
        do_exit_error(u'Patch not installed.')
