#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) shinken-solutions team:
#
# This module is a sample for third party developers to bootstrap
# their own broker module.
#
# This module is a worker based one. It will allow to have N workers that will
# automatically manage 1/N oh the hosts of your realm(s).


from shinken.brokermoduleworker import BrokerModuleWorker as BrokerExportWorker

from datetime import datetime, timedelta
import json
from collections import namedtuple

from .controller_sla.sla_info import SLAInfo
from .controller_sla.sla_reader import SlaReader
from shinken.webui import bottlewebui as bottle
from shinken.webui.bottlewebui import response
from shinken.webui.cherrypybackend import CherryPyServerHTTP

Date = namedtuple('Date', ['yday', 'year'])


def datetime_to_sla_date(date_time):
    # type: (datetime) -> Date
    tm = date_time.timetuple()
    return Date(tm.tm_yday, tm.tm_year)


class BrokerModuleSlaApi_Worker(BrokerExportWorker):
    module_configuration = None
    port = None
    host = None
    use_ssl = None
    ssl_key = None
    ssl_cert = None
    sla_reader = None
    sla_info = None
    srv = None
    logger_init = None
    
    
    def init_worker(self, module_configuration, **kwargs):
        self.module_configuration = module_configuration
        
        self.port = int(getattr(module_configuration, 'port', '7790'))
        self.host = getattr(module_configuration, 'host', '0.0.0.0')
        self.use_ssl = getattr(module_configuration, 'use_ssl', '0') == '1'
        self.ssl_key = getattr(module_configuration, 'ssl_key', '')
        self.ssl_cert = getattr(module_configuration, 'ssl_cert', '')
        
        self.sla_reader = None
        self.sla_info = None
        self.srv = None
        self.logger_init = self.logger.get_sub_part('INITIALISATION')
        
        self.init()
    
    
    def worker_main(self, **kwargs):
        self._http_start()
    
    
    def init(self):
        self.logger_init.info('Initialization of the module')
        self.sla_reader = SlaReader(self.logger_init, self.module_configuration)
        self.sla_info = SLAInfo(self.sla_reader.sla_database_connection, self.logger_init)
        self._init_http_server()
    
    
    def _init_http_server(self):
        self.logger_init.info('Init http server')
        self.logger_init.info('   port       : [%s]' % self.port)
        self.logger_init.info('   host       : [%s]' % self.host)
        self.logger_init.info('   use_ssl    : [%s]' % self.use_ssl)
        self.logger_init.info('   ssl_key    : [%s]' % self.ssl_key)
        self.logger_init.info('   ssl_cert   : [%s]' % self.ssl_cert)
        try:
            # instantiate a new Bottle object, don't use the default one otherwise all module will share the same
            app = bottle.Bottle()
            app = self._init_routes(app)
            self.srv = app.run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=self.use_ssl, ssl_key=self.ssl_key, ssl_cert=self.ssl_cert)
        except Exception, e:
            self.logger_init.error('Fail to start http server with Exception : %s' % str(e))
            raise
        self.logger_init.info('Server loaded')
    
    
    def do_stop(self):
        self.stop_listener()
    
    
    def stop_listener(self):
        self.srv.stop()
        self.logger.info('Stop http server')
    
    
    def _http_start(self):
        # Now block and run
        self.logger.info('Starting http server')
        self.srv.start()
    
    
    def _init_routes(self, app):
        def sla_api():
            response.content_type = 'application/json'
            
            host_name = bottle.request.GET.get('host_name', '')
            check_name = bottle.request.GET.get('check_name', '')
            raw_date = bottle.request.GET.get('date', '')
            end_date = bottle.request.GET.get('until', '')
            if raw_date:
                start_date = datetime.strptime(raw_date, '%d_%m_%Y')
            else:
                start_date = datetime(datetime.now().year, datetime.now().month, datetime.now().day)
            if end_date:
                end_date = datetime.strptime(end_date, '%d_%m_%Y')
            else:
                end_date = start_date
            
            if not host_name:
                return 'no host name'
            
            uuid = self.sla_info.get_uuid(host_name, check_name)
            
            date_range = [start_date + timedelta(d) for d in range((end_date - start_date).days + 1)]
            return json.dumps([self.sla_reader.read_one_sla(uuid, datetime_to_sla_date(cur_day)) for cur_day in date_range])
        
        
        app.route('/shinken/v1/sla_api', callback=sla_api, method='GET')
        return app
