#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import os
import random
import time

from component.sla_archive import SLAArchive
from component.sla_common import shared_data
from component.sla_component_manager import ComponentManager
from component.sla_compute_percent_sla import ComputePercentSla
from component.sla_database import SLADatabase
from component.sla_database_connection import SLADatabaseConnection
from component.sla_error_handler import error_handler
from component.sla_info import SLAInfo
from shinken.basesubprocess import BaseSubProcess
from shinken.log import LoggerFactory, PART_INITIALISATION
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.subprocess_helper.error_handler import ERROR_LEVEL
from shinkensolutions.date_helper import Date, get_now, date_now, get_previous_date, get_diff_date, get_start_of_day, get_end_of_day, timestamp_from_date
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration, read_float_in_configuration

DEFAULT_LOOP_SPEED = 60  # in sec
SIZE_CHUNK_TO_ARCHIVE = 10000
TIME_BETWEEN_TWO_CHUNKS = 0.1
KEEP_RAW_SLA_DAY = 7
PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD = '/var/lib/shinken/tmp/manual_archive_to_rebuild'


class Archivator(BaseSubProcess):
    component_manager = None  # type: ComponentManager
    sla_database_connection = None  # type: SLADatabaseConnection
    sla_info = None  # type: SLAInfo
    sla_database = None  # type: SLADatabase
    sla_archive = None  # type: SLAArchive
    compute_percent_sla = None  # type: ComputePercentSla
    
    
    def __init__(self, daemon_name, module_name, conf):
        # type: (str, str, ShinkenModuleDefinition) -> None
        super(Archivator, self).__init__('archive', father_name=daemon_name, loop_speed=10, only_one_process_by_class=True, error_handler=error_handler)
        self.conf = conf
        self.module_name = module_name
        self.logger = LoggerFactory.get_logger(conf.get_name())
        
        self.total_sla_archived = 0
        self.last_taking_lock = 0
        self.logger.info(PART_INITIALISATION, 'Creating new archive process name : %s' % self.get_process_name())
    
    
    def on_init(self):
        # type: () -> None
        global SIZE_CHUNK_TO_ARCHIVE, TIME_BETWEEN_TWO_CHUNKS, KEEP_RAW_SLA_DAY
        self.component_manager = ComponentManager(self.logger)
        self.sla_database_connection = SLADatabaseConnection(self.conf, self.component_manager)
        self.sla_info = SLAInfo(self.conf, self.component_manager, self.sla_database_connection)
        self.sla_database = SLADatabase(self.conf, self.component_manager, self.sla_database_connection, self.sla_info)
        self.compute_percent_sla = ComputePercentSla(self.conf, self.component_manager)
        self.sla_archive = SLAArchive(self.conf, self.component_manager, self.sla_info, self.compute_percent_sla, self.sla_database)
        
        SIZE_CHUNK_TO_ARCHIVE = read_int_in_configuration(self.conf, 'size_chunk_to_archive', SIZE_CHUNK_TO_ARCHIVE)
        TIME_BETWEEN_TWO_CHUNKS = read_float_in_configuration(self.conf, 'time_between_two_chunks', TIME_BETWEEN_TWO_CHUNKS)
        KEEP_RAW_SLA_DAY = read_int_in_configuration(self.conf, 'keep_raw_sla_day', KEEP_RAW_SLA_DAY)
        
        self.logger.info(PART_INITIALISATION, 'Parameter load for archiving sla')
        self.logger.info(PART_INITIALISATION, '   - size_chunk_to_archive --- :[%s]' % SIZE_CHUNK_TO_ARCHIVE)
        self.logger.info(PART_INITIALISATION, '   - time_between_two_chunks - :[%s]' % TIME_BETWEEN_TWO_CHUNKS)
        self.logger.info(PART_INITIALISATION, '   - keep_raw_sla_day -------- :[%s]' % KEEP_RAW_SLA_DAY)
        self.logger.info(PART_INITIALISATION, '   - Initialisation of the archive process done')
        
        self.component_manager.init()
    
    
    def get_process_name(self):
        return '%s [ - Module: %s - %s]' % (self.father_name, self.module_name, self.name)
    
    
    def get_logger_name(self):
        return [self.father_name.replace('shinken-', '').strip(), self.module_name, '%s (pid:%s)' % (self.name, os.getpid())]
    
    
    def get_lock_name(self):
        return ' '.join(self.get_logger_name())
    
    
    def _take_lock(self, date):
        lock_name = self.get_lock_name()
        sla_status = self.sla_database.find_raw_sla_status(date)
        
        my_lock = next(([ln, le] for ln, le in sla_status.get('archive_locks', []) if ln == lock_name), None)
        # We refresh the lock at most every 15sec
        _now = get_now()
        if not my_lock or _now - my_lock[1] > 15:
            # We remove lock older than 60sec
            sla_status['archive_locks'] = [[ln, le] for ln, le in sla_status.get('archive_locks', []) if ln != lock_name and _now - le < 60]
            sla_status['archive_locks'].append([lock_name, _now])
            self.sla_database.save_raw_sla_status(date, sla_status)
    
    
    def _check_lock(self, date, sla_status):
        lock_name = self.get_lock_name()
        other_archive_locks = [[ln, le] for ln, le in sla_status.get('archive_locks', []) if ln != lock_name and get_now() - le < 30]
        if other_archive_locks:
            error_message = 'Fail to archive collection %s (for date %s) because it was lock by [%s] and i am [%s] (an other process try to archive this day)' % (
                SLADatabaseConnection.get_sla_collection_name(date),
                self.logger.format_sla_date(date),
                ','.join((i for i, _ in other_archive_locks)),
                lock_name
            )
            error_handler.handle_exception(error_message, 'archive is lock', self.logger, level=ERROR_LEVEL.ERROR)
            return False
        else:
            return True
    
    
    def _archive_collection(self, date):
        start_time = time.time()
        done_archive = False
        if not self.sla_database.collection_exist(date):
            return
        
        self.logger.info('Archiving the day %s' % self.logger.format_sla_date(date))
        
        sla_status = self.sla_database.find_raw_sla_status(date)
        if not sla_status:
            invalide_col_name = 'invalide_%s_%s_%s' % (date.yday, date.year, get_now())
            self.sla_database.rename_collection(date, invalide_col_name)
            error_message = 'We fail to archive raw sla of the %s because it not have the aliveness module information (sla_status). ' \
                            'We save your data in mongo collection %s it will be keep %s day. ' \
                            'Please contact your dedicated support before this time with the log of the broker and a shinken-backup of the sla.' % (
                                self.logger.format_sla_date(date), invalide_col_name, KEEP_RAW_SLA_DAY)
            error_handler.handle_exception(error_message, 'missing sla_status', self.logger, level=ERROR_LEVEL.ERROR)
            return True
        
        if not self._check_lock(date, sla_status):
            return True
        self._take_lock(date)
        
        to_archives = self.sla_info.get_all_uuids()
        already_archive = self.sla_database.list_archives_uuid(date)
        if len(already_archive):
            self.logger.info('Next %s element SLA archive for the day %s will be skip because we already found a archive for them.' % (len(already_archive), self.logger.format_sla_date(date)))
            to_archives = list(set(to_archives) - set(already_archive))
        nb_to_archives = len(to_archives)
        
        if nb_to_archives > 0:
            self._stats_archive_start(start_time, nb_to_archives, timestamp_from_date(date))
        nb_elements_archived = 0
        
        nb_chunk_to_archives = nb_to_archives / SIZE_CHUNK_TO_ARCHIVE + 1
        start_of_day = get_start_of_day(date)
        end_of_day = get_end_of_day(date)
        inactive_ranges = self.sla_archive.compute_inactive_ranges(date, start_of_day, end_of_day)
        current_chunk_to_archives = 0
        
        todo_archives = []
        
        self.logger.info('%s items to archive in %s blocks ( max size block : %s )' % (nb_to_archives, nb_chunk_to_archives, SIZE_CHUNK_TO_ARCHIVE))
        while current_chunk_to_archives < nb_chunk_to_archives:
            t0 = time.time()
            max_range = min(nb_to_archives, (current_chunk_to_archives + 1) * SIZE_CHUNK_TO_ARCHIVE)
            min_range = min(nb_to_archives, current_chunk_to_archives * SIZE_CHUNK_TO_ARCHIVE)
            chunk_to_archives = to_archives[min_range:max_range]
            self.logger.info('     -  block %s: archiving %s elements' % (current_chunk_to_archives, len(chunk_to_archives)))
            
            for item_uuid in chunk_to_archives:
                # self.logger.debug('     -  block %s: archiving element %s' % (current_chunk_to_archives, item_uuid))
                archive = self.sla_archive.build_archive(date, item_uuid, inactive_ranges=inactive_ranges)
                if archive is not None:
                    done_archive = True
                    nb_elements_archived += 1
                    shared_data.set_sla_archived_during_current_archive(nb_elements_archived)
                    todo_archives.append(archive)
                    self._take_lock(date)
            
            if len(todo_archives) != 0:
                t1 = time.time()
                self.sla_database.save_archives(todo_archives, update=False)
                self.logger.info('     -  block %s: insert %s archive in mongo done in %s' % (current_chunk_to_archives, len(todo_archives), self.logger.format_chrono(t1)))
            
            self._take_lock(date)
            self.logger.info('     -  block %s: archiving block elements done in %s' % (current_chunk_to_archives, self.logger.format_chrono(t0)))
            todo_archives = []
            current_chunk_to_archives += 1
            time.sleep(TIME_BETWEEN_TWO_CHUNKS)
        
        if not shared_data.get_already_archived():
            has_been_archive_col_name = 'has_been_archive_%s_%s' % (date.yday, date.year)
            self.sla_database.rename_collection(date, has_been_archive_col_name)
        
        self.logger.info(' %s entries archive for the day %s in %ss nb block %s' % (nb_to_archives, self.logger.format_sla_date(date), self.logger.format_chrono(start_time), nb_chunk_to_archives))
        return done_archive
    
    
    def loop_turn(self):
        self.component_manager.tick()
        # wait +/- 2min (do not have strong computation for some modules at the same time for example)
        self.loop_speed = DEFAULT_LOOP_SPEED + random.randint(0, 60)
        t0 = time.time()
        now = get_now()
        _date_now = date_now()
        tm = time.localtime(now)
        done_archive = False
        
        # we wait 00h05 to make a archive so brok with data at 23h59 which come at 00h01 have time to be manage
        if tm.tm_hour == 0 and tm.tm_min < 5:
            self.logger.debug('Archiving  %s wait for 00h05' % self.logger.format_time(now))
            self.loop_speed = 5 * 60
            return
        
        done_archive = self._manual_archive_collection()
        
        date_to_archive = _date_now
        for day_count in range(1, 7):
            date_to_archive = get_previous_date(date_to_archive)
            _done_archive = self._archive_collection(date_to_archive)
            done_archive = done_archive or _done_archive
        
        self._remove_old_raw_sla_collections()
        
        if done_archive:
            time_taken = time.time() - t0
            self._stats_archive_end(time_taken)
    
    
    def _manual_archive_collection(self):
        # Sometime we need to rebuild manually SLA.
        # We can use script : sla_recompute_data_for_one_day.py
        # Or collection in file PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD will be rebuild one time
        done_archive = False
        if not os.path.exists(PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD):
            return done_archive
        
        with open(PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD) as f:
            to_archive_lines = f.readlines()
        
        for to_archive_line in to_archive_lines:
            has_been_archive = False
            is_already_archive = ''
            if to_archive_line.startswith('has_been_archive_'):
                has_been_archive = True
                to_archive_line = to_archive_line.replace('has_been_archive_', '')
                is_already_archive = '(is already archived)'
            tmp_to_archive_line = to_archive_line.split('_')
            date_to_archive = Date(int(tmp_to_archive_line[0]), int(tmp_to_archive_line[1]))
            
            old_has_been_archive = shared_data.get_already_archived()
            shared_data.set_already_archived(has_been_archive)
            self.logger.info('Start manual archive for %s %s' % (self.logger.format_sla_date(date_to_archive), is_already_archive))
            _done_archive = self._archive_collection(date_to_archive)
            shared_data.set_already_archived(old_has_been_archive)
            done_archive = done_archive or _done_archive
        
        os.remove(PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD)
        return done_archive
    
    
    def _remove_old_raw_sla_collections(self):
        _date_now = date_now()
        for collection_name in self.sla_database.list_name_collections()[:]:
            if collection_name.startswith('has_been_archive_') or collection_name.startswith('invalide_'):
                split = collection_name.replace('has_been_archive_', '').replace('invalide_', '')
                split = split.split('_')
                collection_date = Date(int(split[0]), int(split[1]))
                must_del_archive = not (0 < get_diff_date(_date_now, collection_date) <= KEEP_RAW_SLA_DAY)
                
                if must_del_archive:
                    self.sla_database.drop_collections(collection_name)
                    self.logger.info('Collection %s with raw data of the %s can be deleted because it was %s days old, value from keep_raw_sla_day.' % (collection_name, self.logger.format_sla_date(collection_date), KEEP_RAW_SLA_DAY))
    
    
    def _stats_archive_start(self, start_time, nb_to_archive, progression_date):
        if not shared_data.get_archive_in_progress():
            if shared_data.get_latest_archive_start_time() != 0:
                shared_data.set_previous_archive_start_time(shared_data.get_latest_archive_start_time())
            shared_data.set_latest_archive_start_time(start_time)
            shared_data.set_archive_in_progress(True)
        
        shared_data.set_total_sla_current_archive(nb_to_archive)
        self.total_sla_archived += nb_to_archive
        shared_data.set_sla_archived_during_current_archive(0)
        shared_data.set_archive_progression_date(progression_date)
    
    
    def _stats_archive_end(self, time_taken):
        if shared_data.get_latest_archive_execution_time() != 0:
            shared_data.set_previous_archive_execution_time(shared_data.get_latest_archive_execution_time())
            shared_data.set_previous_archive_sla_archived(shared_data.get_total_sla_archived())
        
        shared_data.set_total_sla_archived(self.total_sla_archived)
        self.total_sla_archived = 0
        
        shared_data.set_latest_archive_execution_time(time_taken)
        shared_data.set_archive_in_progress(False)
        shared_data.set_total_unique_elements_valid(False)
