#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import time
import uuid

from component.sla_common import STATUS, RAW_SLA_KEY, FutureState
from component.sla_component_manager import ComponentManager
from component.sla_database import SLADatabase
from component.sla_database_connection import SLADatabaseConnection, BulkSla
from component.sla_error_handler import error_handler
from shinken.brokermoduleworker import BrokerModuleWorker
from shinken.check import NO_END_VALIDITY
from shinken.subprocess_helper.error_handler import ERROR_LEVEL
from shinken.misc.fast_copy import fast_deepcopy, NO_COPY
from shinken.misc.type_hint import Dict
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.date_helper import date_from_timestamp, get_now, get_previous_date, get_next_date, date_now, compare_date, DATE_COMPARE, Date, get_start_of_day, get_end_of_day
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration, read_bool_in_configuration, read_list_in_configuration

_cache_write_downtime = set()

LOOP_SPEED = 1  # in sec
MARGIN_SLA_INACTIVE = 30  # in sec
INITIAL_FUTURE_STATES_NEXT_CLEAN = 60 * 60  # in sec
PERIOD_FUTURE_STATES_CLEAN = 86400  # in sec


class LOG_PART_RAW_DATA(object):
    INITIALISATION = 'INITIALISATION'
    FUTURE_STATE = 'REPORT STATE'


class KEY_CACHE(object):
    START = 0
    END = 1
    STATUS = 2
    ID = 3
    DATE = 4


class BrokHandlerModuleWorker(BrokerModuleWorker):
    component_manager = None  # type: ComponentManager
    sla_database_connection = None  # type: SLADatabaseConnection
    sla_database = None  # type: SLADatabase
    minimal_time_before_an_element_become_missing_data = 0  # type: int
    
    _stats_sla_current_minute = (0, 0)
    
    store_output = True
    store_long_output = True
    list_of_stored_output_status = set()
    
    bulks = {}  # type: Dict[Date, BulkSla]
    future_states = {}  # type: Dict[str, FutureState]
    future_states_must_be_save = False
    future_states_last_report = False
    update_sla_state_cache = {}
    future_states_next_clean = 0
    
    _last_manage_broks = 0
    
    
    def init_worker(self, conf):
        # type: (ShinkenModuleDefinition) -> None
        global MARGIN_SLA_INACTIVE
        start_time = time.time()
        self.logger.info(LOG_PART_RAW_DATA.INITIALISATION, 'Staring new worker.')
        
        MARGIN_SLA_INACTIVE = read_int_in_configuration(conf, 'time_before_shinken_inactive', MARGIN_SLA_INACTIVE)
        self.component_manager = ComponentManager(self.logger)
        self.sla_database_connection = SLADatabaseConnection(conf, self.component_manager)
        # We don't give a SLAInfo to SLADatabase because we don't need to query archive here and we don't want the thread of SLAInfo
        self.sla_database = SLADatabase(conf, self.component_manager, self.sla_database_connection, None)
        
        self._stats_sla_current_minute = (0, 0)
        
        # Stored output configuration part
        self.store_output = read_bool_in_configuration(conf, 'store_output', True)
        self.store_long_output = read_bool_in_configuration(conf, 'store_long_output', True)
        self.list_of_stored_output_status = read_list_in_configuration(conf, 'list_of_stored_output_status', '')
        self.list_of_stored_output_status = set([STATUS.STATUS_MAP[i] for i in self.list_of_stored_output_status if i in STATUS.STATUS_MAP])
        self.minimal_time_before_an_element_become_missing_data = read_int_in_configuration(conf, 'minimal_time_before_an_element_become_missing_data', 0)
        
        # Internal attribute part
        self.bulks = {}
        self.future_states = {}  # type: Dict[str, FutureState]
        self.future_states_must_be_save = False
        self.future_states_last_report = False
        self.update_sla_state_cache = {}
        self.future_states_next_clean = time.time() + INITIAL_FUTURE_STATES_NEXT_CLEAN
        
        self._last_manage_broks = 0
        
        self.component_manager.init()
        self._load_future_states()
        
        self.logger.info(LOG_PART_RAW_DATA.INITIALISATION, 'Parameter load for build raw sla')
        self.logger.info(LOG_PART_RAW_DATA.INITIALISATION, '   - time_before_shinken_inactive - :[%s]' % MARGIN_SLA_INACTIVE)
        self.logger.info(LOG_PART_RAW_DATA.INITIALISATION, '   - store_output ----------------- :[%s]' % self.store_output)
        self.logger.info(LOG_PART_RAW_DATA.INITIALISATION, '   - store_long_output ------------ :[%s]' % self.store_long_output)
        self.logger.info(LOG_PART_RAW_DATA.INITIALISATION, '   - list_of_stored_output_status - :[%s]' % self.list_of_stored_output_status)
        
        self.logger.info(LOG_PART_RAW_DATA.INITIALISATION, 'New worker start in %s.' % self.logger.format_chrono(start_time))
    
    
    def manage_service_check_result_brok(self, new_info):
        # self.logger.debug( 'manage_service_check_result_brok')
        self._handle_brok(new_info)
    
    
    def manage_host_check_result_brok(self, new_info):
        # self.logger.debug( 'manage_host_check_result_brok')
        self._handle_brok(new_info)
    
    
    def manage_update_service_status_brok(self, new_info):
        # self.logger.debug( 'manage_update_service_status_brok')
        self._handle_brok(new_info)
    
    
    def manage_update_host_status_brok(self, new_info):
        # self.logger.debug( 'manage_update_host_status_brok')
        self._handle_brok(new_info)
    
    
    def manage_initial_service_status_brok(self, new_info):
        # self.logger.debug( 'manage_initial_service_status_brok')
        self._handle_brok(new_info)
    
    
    def manage_initial_host_status_brok(self, new_info):
        # self.logger.debug( 'manage_initial_host_status_brok')
        self._handle_brok(new_info)
    
    
    def worker_main(self):
        while not self.interrupted:
            self._tick()
            self.interruptable_sleep(LOOP_SPEED)
    
    
    def _handle_brok(self, brok, ):
        new_info_data = brok.data
        item_uuid = new_info_data['instance_uuid']
        
        check_time = int(new_info_data['creation_date'])
        state_id = new_info_data['state_id']
        # before first check we force STATUS_MISSING_DATA in state_id
        if new_info_data['state'] == 'PENDING':
            state_id = STATUS.MISSING_DATA
        elif new_info_data['state'] == 'UNREACHABLE':
            state_id = STATUS.UNKNOWN
        
        if '-' not in item_uuid:
            # Cluster case
            if new_info_data.get('got_business_rule', False):
                state_id = new_info_data['bp_state']
            # Host case
            elif state_id == STATUS.WARN:
                state_id = STATUS.CRIT
        
        output = new_info_data['output']
        long_output = new_info_data['long_output']
        
        if not self.store_output or (len(self.list_of_stored_output_status) > 0 and state_id not in self.list_of_stored_output_status):
            output = None
            long_output = None
        if not self.store_long_output:
            long_output = None
        
        downtime = new_info_data.get('in_scheduled_downtime', None)
        active_downtime_uuids = new_info_data.get('active_downtime_uuids', None)
        inherited_downtime = new_info_data.get('in_inherited_downtime', None)
        
        # Invalid or old brok, skip this
        if downtime is None:
            return
        
        dt_value = 0
        if downtime:
            dt_value = 1
            downtimes = new_info_data.get('downtimes', None)
            self._create_downtime(active_downtime_uuids, downtimes, brok)
        if inherited_downtime:
            dt_value = 2
        
        ack = 0
        if new_info_data['problem_has_been_acknowledged'] and not new_info_data['in_inherited_acknowledged']:
            ack = 1
        elif new_info_data['in_inherited_acknowledged']:
            ack = 2
        ack_uuid = ''
        
        if new_info_data.get('got_business_rule', False):
            check_time = get_now()
        
        if new_info_data['acknowledge_id'] is not None:
            ack_uuid = new_info_data['acknowledge_id']
        if new_info_data['acknowledgement'] is not None:
            self._create_acknowledge(new_info_data['acknowledgement'], item_uuid)
        
        state_validity_period = new_info_data['state_validity_period']
        
        flapping = new_info_data['is_flapping']
        partial_flapping = new_info_data.get('is_partial_flapping', False)
        partial_ack = new_info_data.get('is_partial_acknowledged', False)
        partial_dt = new_info_data.get('in_partial_downtime', False)
        
        # self.logger.debug( 'brok-[%s] [%s] - check_time[%s] state[%s-%s]:for[%s]' % (brok.type, item_uuid, print_time(check_time), state_id, new_info_data['state'], state_validity_period))
        self._update_sla(check_time, state_id, flapping, partial_flapping, dt_value, partial_dt, ack, partial_ack, ack_uuid, state_validity_period, item_uuid, output, long_output, active_downtime_uuids)
    
    
    def _do_in_worker_manage_broks_thread_loop(self):
        try:
            super(BrokHandlerModuleWorker, self)._do_in_worker_manage_broks_thread_loop()
            now = time.time()
            if now - self._last_manage_broks > LOOP_SPEED:
                self._check_report_raw_sla()
                self._execute_bulks()
                self._last_manage_broks = now
        except Exception as e:
            error_handler.handle_exception('Fatal error caused by : %s' % e, e, self.logger, ERROR_LEVEL.FATAL)
            raise
    
    
    def _execute_bulks(self):
        for date, bulk in self.bulks.iteritems():
            insert_cmp, update_cmp = bulk.bulks_execute()
            self.logger.info('%s raw data update in database for the day %s' % (insert_cmp + update_cmp, self.logger.format_sla_date(date)))
        
        # We keeps in bulks only the bulks of yesterday, today, tomorrow
        bulks_keys_before_filter = self.bulks.keys()
        date = date_now()
        to_keeps = [get_previous_date(date), date, get_next_date(date)]
        self.bulks = dict(filter(lambda (key, _bulk): key in to_keeps, self.bulks.iteritems()))
        bulks_keys_after_filter = self.bulks.keys()
        if len(bulks_keys_before_filter) != len(bulks_keys_after_filter):
            self.logger.info('Size of bulks cache have change [%s]/[%s]' % (bulks_keys_before_filter, bulks_keys_after_filter))
    
    
    def _tick(self):
        try:
            if self.future_states_must_be_save:
                self._save_future_states()
            
            if self.future_states_next_clean > time.time():
                self._clean_future_states()
        except Exception as e:
            error_handler.handle_exception('Fatal error caused by : %s' % e, e, self.logger, ERROR_LEVEL.FATAL)
            raise
    
    
    def _create_acknowledge(self, acknowledge, item_uuid):
        acknowledge_id = acknowledge.id
        prev_acknowledge = self.sla_database.find_acknowledge(acknowledge_id)
        if prev_acknowledge is None:
            # ok there is not such acknowledge before, save it
            acknowledge_entry = {'_id': acknowledge_id, 'item_uuid': item_uuid}
            for k in acknowledge.__class__.properties:
                acknowledge_entry[k] = getattr(acknowledge, k)
            self.sla_database.save_acknowledge(acknowledge_entry)
            self.logger.debug('Saving new acknowledge: %s' % acknowledge_entry)
    
    
    def _create_downtime(self, uuid_downtimes, downtimes, brok):
        if not uuid_downtimes:
            return
        for uuid_downtime in uuid_downtimes[:]:
            if uuid_downtime in _cache_write_downtime:
                continue
            
            prev_downtime = self.sla_database.find_downtime(uuid_downtime)
            if prev_downtime:
                _cache_write_downtime.add(uuid_downtime)
                continue
            
            downtime = next((i for i in downtimes if i.uuid == uuid_downtime), None) if downtimes else None
            if not downtime:
                uuid_downtimes.remove(uuid_downtime)
                self.logger.error('This brok referencing the downtime uuid [%s] but we never got a downtime with this uuid' % uuid_downtime)
                brok_data = brok.data
                in_scheduled_downtime = brok_data.get('in_scheduled_downtime', None)
                inherited_downtime = brok_data.get('in_inherited_downtime', None)
                item_name = '%s-%s' % (brok.data['host_name'], brok.data.get('service_description', ''))
                self.logger.error(
                    'Fail to save this downtime from the brok [%s] of [%s] ( brok data => in_scheduled_downtime:[%s] inherited_downtime:[%s] downtimes:[%s] )' % (brok.type, item_name, in_scheduled_downtime, inherited_downtime, downtimes))
                continue
            
            downtime_entry = {
                '_id'    : uuid_downtime,
                'author' : downtime.author,
                'comment': downtime.comment,
            }
            _cache_write_downtime.add(uuid_downtime)
            self.sla_database.save_downtime(downtime_entry)
            self.logger.debug('Saving new downtime: %s' % downtime_entry)
    
    
    def _reset_cache(self):
        self.update_sla_state_cache = {}
    
    
    def _get_bulk_raw_sla(self, date):
        bulk = self.bulks.get(date, None)
        if bulk is None:
            bulk = self.sla_database.build_raw_sla_bulk(date, self._reset_cache)
            self.bulks[date] = bulk
        return bulk
    
    
    def _report_future_state(self, bulk, item_uuid, future_state, start_of_day, end_of_day):
        to_report = future_state.to_report
        to_report_left = 0
        
        if to_report == NO_END_VALIDITY:
            end_time = NO_END_VALIDITY
            to_report_left = NO_END_VALIDITY
        else:
            end_time = to_report + start_of_day
            if end_time > end_of_day:
                to_report_left = end_time - end_of_day
                end_time = end_of_day
        
        _id = uuid.uuid4().hex
        sla_to_save = {
            '_id'                         : _id,
            RAW_SLA_KEY.UUID              : item_uuid,
            RAW_SLA_KEY.CONTEXT_AND_STATUS: future_state.state,
            RAW_SLA_KEY.START             : start_of_day,
            RAW_SLA_KEY.END               : end_time,
            RAW_SLA_KEY.OUTPUT            : future_state.output,
            RAW_SLA_KEY.LONG_OUTPUT       : future_state.long_output,
        }
        
        if future_state.ack_uuid:
            sla_to_save[RAW_SLA_KEY.ACK_UUID] = future_state.ack_uuid
        if future_state.active_downtime_uuids:
            sla_to_save[RAW_SLA_KEY.DOWNTIMES_UUID] = future_state.active_downtime_uuids
        
        bulk.insert(sla_to_save)
        self.logger.debug('report_future_state for item:[%s] with CONTEXT_AND_STATUS:[%s] start:[%s] end:[%s] ack_uuid:[%s] downtimes_uuid:[%s]' %
                          (item_uuid, future_state.state, self.logger.format_time(start_of_day), self.logger.format_time(end_time), future_state.ack_uuid, future_state.active_downtime_uuids))
        return to_report_left
    
    
    def _have_state_in_cache(self, item_uuid, cache_status_value, date, new_value_timestamp):
        sla_state_cache = self.update_sla_state_cache.get(item_uuid, None)
        
        have_state_in_cache = \
            sla_state_cache and \
            sla_state_cache[KEY_CACHE.STATUS] == cache_status_value and \
            sla_state_cache[KEY_CACHE.DATE] == date and \
            new_value_timestamp - sla_state_cache[KEY_CACHE.END] < MARGIN_SLA_INACTIVE * 2
        
        return have_state_in_cache
    
    
    def _update_sla(self, new_value_timestamp, status, flapping, partial_flapping, dt, partial_dt, ack, partial_ack, ack_uuid, state_validity_period, item_uuid, output, long_output, active_downtime_uuids):
        stats_start_execution = time.time()
        date = date_from_timestamp(new_value_timestamp)
        end_of_day = get_end_of_day(date)
        start_time = int(new_value_timestamp)
        end_time = new_value_timestamp + max(self.minimal_time_before_an_element_become_missing_data, state_validity_period)
        active_downtime_uuids = tuple(active_downtime_uuids) if active_downtime_uuids else tuple()
        cache_status_value = len(active_downtime_uuids) * 10000000 + partial_dt * 1000000 + partial_ack * 100000 + partial_flapping * 10000 + flapping * 1000 + status * 100 + dt * 10 + ack
        bulk = self._get_bulk_raw_sla(date)
        future_state = self.future_states.get(item_uuid, None)
        
        if future_state and future_state.date == date:
            start_of_day = get_start_of_day(date)
            self._report_future_state(bulk, item_uuid, future_state, start_of_day, end_of_day)
            del self.future_states[item_uuid]
            self.future_states_must_be_save = True
        
        if state_validity_period == NO_END_VALIDITY:
            end_time = NO_END_VALIDITY
            self.future_states[item_uuid] = FutureState(get_next_date(date), NO_END_VALIDITY, cache_status_value, output, long_output, ack_uuid, active_downtime_uuids)
            self.future_states_must_be_save = True
        elif end_time > end_of_day:
            to_report = end_time - end_of_day
            end_time = end_of_day
            self.future_states[item_uuid] = FutureState(get_next_date(date), to_report, cache_status_value, output, long_output, ack_uuid, active_downtime_uuids)
            self.future_states_must_be_save = True
        elif future_state and future_state.date != date:
            del self.future_states[item_uuid]
            self.future_states_must_be_save = True
        
        # self.logger.debug( 'update_sla[%s] -> [%s]:[%s]' % (item_uuid, print_time(new_value_timestamp), cache_status_value))
        
        if self._have_state_in_cache(item_uuid, cache_status_value, date, new_value_timestamp):
            sla_state_cache = self.update_sla_state_cache[item_uuid]
            if sla_state_cache[KEY_CACHE.END] == end_time:
                # self.logger.debug( 'update_sla[%s] -> broks with same time and state send in double' % item_uuid)
                return
            # self.logger.debug( 'update_sla[%s] -> item state [%s] in cache' % (item_uuid, cache_status_value))
            
            _id = sla_state_cache[KEY_CACHE.ID]
            bulk.update({'_id': _id}, {'$set': {RAW_SLA_KEY.END: end_time}})
            sla_state_cache[KEY_CACHE.END] = end_time
        else:
            _id = uuid.uuid4().hex
            sla_to_save = {
                '_id'                         : _id,
                RAW_SLA_KEY.UUID              : item_uuid,
                RAW_SLA_KEY.CONTEXT_AND_STATUS: cache_status_value,
                RAW_SLA_KEY.START             : start_time,
                RAW_SLA_KEY.END               : end_time,
                RAW_SLA_KEY.OUTPUT            : output,
                RAW_SLA_KEY.LONG_OUTPUT       : long_output,
            }
            
            # self.logger.debug( 'update_sla[%s] -> item state [%s] NOT in cache [%s,%s]' %
            #              (item_uuid, cache_status_value, self.logger.format_time_as_sla(sla_to_save[RAW_SLA_KEY.START]), self.logger.format_time_as_sla(sla_to_save[RAW_SLA_KEY.END])))
            
            if ack:
                sla_to_save[RAW_SLA_KEY.ACK_UUID] = ack_uuid
            if active_downtime_uuids:
                sla_to_save[RAW_SLA_KEY.DOWNTIMES_UUID] = active_downtime_uuids
            
            bulk.insert(sla_to_save)
            self.update_sla_state_cache[item_uuid] = [sla_to_save[RAW_SLA_KEY.START], sla_to_save[RAW_SLA_KEY.END], cache_status_value, _id, date]
        self._stats_sla_current_minute = (self._stats_sla_current_minute[0] + 1, self._stats_sla_current_minute[1] + (time.time() - stats_start_execution))
    
    
    def _check_report_raw_sla(self):
        time_start = time.time()
        nb_items_report = 0
        tm = time.localtime(get_now())
        date = Date(tm.tm_yday, tm.tm_year)
        
        here_are_no_report_done = not self.future_states_last_report
        last_report_was_in_the_past = compare_date(date, self.future_states_last_report) == DATE_COMPARE.IS_BEFORE
        wait_after_00h07 = tm.tm_hour > 0 or tm.tm_min > 7
        if not ((here_are_no_report_done or last_report_was_in_the_past) and wait_after_00h07):
            return
        self.logger.debug(LOG_PART_RAW_DATA.FUTURE_STATE, 'start report_raw_sla for the date:[%s]' % str(date))
        
        start_of_day = get_start_of_day(date)
        end_of_day = get_end_of_day(date)
        bulk = self._get_bulk_raw_sla(date)
        new_future_states = {}  # type: Dict[str, FutureState]
        for item_uuid, future_state in self.future_states.iteritems():
            _compare = compare_date(date, future_state.date)
            if _compare == DATE_COMPARE.IS_AFTER:
                new_future_states[item_uuid] = future_state
                continue
            if _compare == DATE_COMPARE.IS_BEFORE:
                # report from the past are just drop
                continue
            
            to_report_left = self._report_future_state(bulk, item_uuid, future_state, start_of_day, end_of_day)
            nb_items_report += 1
            self.logger.debug(LOG_PART_RAW_DATA.FUTURE_STATE, 'report state of [%s] to the date [%s] and it left to report:[%s]' % (item_uuid, date, to_report_left))
            if to_report_left:
                new_future_states[item_uuid] = FutureState(get_next_date(date), to_report_left, future_state.state, future_state.output, future_state.long_output, future_state.ack_uuid, future_state.active_downtime_uuids)
        
        self.future_states = new_future_states
        self.future_states_must_be_save = True
        self.future_states_last_report = date
        
        self.logger.info(LOG_PART_RAW_DATA.FUTURE_STATE, 'We have report the state of %s elements in the day %s in %s' % (nb_items_report, self.logger.format_sla_date(date), self.logger.format_chrono(time_start)))
    
    
    def _save_future_states(self):
        future_states_to_save = fast_deepcopy(self.future_states, additional_dispatcher={FutureState: NO_COPY})
        self.future_states_must_be_save = False
        self.sla_database.save_future_states(future_states_to_save)
        self.logger.debug(LOG_PART_RAW_DATA.FUTURE_STATE, 'future_states have been save')
    
    
    def _load_future_states(self):
        start_time = time.time()
        self.future_states = self.sla_database.load_future_states()
        self.logger.info(LOG_PART_RAW_DATA.INITIALISATION, 'Load previous state of %s elements done in %s.' % (len(self.future_states), self.logger.format_chrono(start_time)))
    
    
    def _clean_future_states(self):
        # At 00:xx there are archive, the db is working hard so we report the clean to the next hour
        tm = time.localtime(get_now())
        if tm.tm_hour == 0:
            self.future_states_next_clean = time.time() + 60 * 60
            return
        
        self.future_states_next_clean = time.time() + PERIOD_FUTURE_STATES_CLEAN
        self.sla_database.clean_future_states()
    
    
    def ask_sla_count(self):
        return self._stats_sla_current_minute
