#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.
import os
import re
import sys
import time
import datetime
import json
import threading
from collections import OrderedDict, namedtuple
from shinken.log import logger
from shinken.misc.filter import only_related_to, is_authorized, get_all_my_visible_items
from shinken.misc.sorter import hst_srv_sort

mydir = os.path.abspath(os.path.dirname(__file__))
sys.path.insert(0, mydir)
from filters import hst_properties, sst_properties, hp_properties, sp_properties
from filters import match_hst, match_sst, match_hp, match_sp, match_contact, match_root_problem, match_business_impact
from filters import match_host, match_hg, match_parent, match_realm, match_service, match_tag

### Will be populated by the UI with it's own value
FRONTEND_DATE_FORMAT = '%Y-%m-%d %H:%M:%S'
DEFAULT_SORT = {'tag': 'host_name', 'value': 'asc'}
app = None

DateFilter = namedtuple('DateFilter', ['operator', 'delay'])


class DateFilterOperator(object):
    NEWER = 'newer'
    OLDER = 'older'


class StaticCache(object):
    lock = threading.RLock()
    part_configuration_id = -1


# TODO put in helper after patch
def _get_state(item):
    if item.__class__.my_type == 'host' and item.got_business_rule:
        return item.bp_state
    elif item.state == 'OK' or item.state == 'UP':
        return 0
    elif item.state == 'WARNING':
        return 1
    elif item.state == 'CRITICAL' or item.state == 'DOWN':
        return 2
    else:
        return 3


# TODO put in helper after patch
def _get_context(elt):
    clsname = elt.__class__.my_type
    is_host = (clsname == 'host')
    is_service = (clsname == 'service')
    context = 'NOTHING', 'NOTHING'
    
    # DISABLED is for hosts, service can't be like this because an host disabled got no service
    if is_host and 'disabled' in elt.tags:
        context = 'DISABLED', 'DISABLED'
    elif elt.downtimes and next((dt for dt in elt.downtimes if dt.ref == elt and dt.is_in_effect), None):
        context = 'DOWNTIME', 'DOWNTIME'
    elif elt.is_in_inherited_downtime():
        context = 'INHERITED-DOWNTIME', 'DOWNTIME'
    elif elt.in_partial_downtime:
        context = 'PARTIAL-DOWNTIME', 'DOWNTIME'
    elif elt.acknowledgement and not elt.acknowledgement.automatic:
        context = 'ACKNOWLEDGED', 'ACKNOWLEDGED'
    elif (elt.acknowledgement and elt.acknowledgement.automatic) or (is_service and elt.host.acknowledgement and not elt.host.acknowledgement.automatic):
        context = 'INHERITED-ACKNOWLEDGED', 'ACKNOWLEDGED'
    elif elt.is_partial_acknowledged:
        context = 'PARTIAL-ACKNOWLEDGED', 'ACKNOWLEDGED'
    elif elt.is_flapping:
        context = 'FLAPPING', 'FLAPPING'
    elif elt.got_business_rule and elt.is_partial_flapping:
        context = 'PARTIAL-FLAPPING', 'FLAPPING'
    return context


FILTER_TYPE = {
    'display_name'        : 'string',
    'business_impact'     : 'int',
    'contact_groups'      : 'array',
    'contacts'            : 'array',
    'status'              : 'int',
    'context'             : 'array',
    'status_since'        : 'date',
    'last_check'          : 'date',
    'is_status_confirmed' : 'bool',
    'attempts'            : 'string',
    'next_check'          : 'date',
    'is_root_problem'     : 'bool',
    'output'              : 'string',
    'long_output'         : 'string',
    'perf_data'           : 'string',
    'notification_period' : 'string',
    'notification_options': 'array',
    'type'                : 'array',
    'host_name'           : 'string',
    'service_description' : 'string',
    'realm'               : 'array',
    'address'             : 'string',
    'host_templates'      : 'array',
    'host_groups'         : 'array',
    
}


# Yes this function is ugly but do not refactor without check performance first
def _get_value_for_filter(item, filter_name):
    if filter_name in item.filter_cache_static:
        return item.filter_cache_static[filter_name]
    
    if filter_name == 'business_impact':
        return item.business_impact
    if filter_name == 'status':
        return _get_state(item)
    if filter_name == 'context':
        return set((_get_context(item)[1].lower(),))
    if filter_name == 'status_since':
        return int(item.last_state_change)
    if filter_name == 'last_check':
        return int(item.last_chk)
    if filter_name == 'next_check':
        return item.next_chk
    if filter_name == 'is_status_confirmed':
        return item.state_type == 'HARD'
    if filter_name == 'attempts':
        return '%s/%s' % (item.attempt, item.max_check_attempts)
    if filter_name == 'is_root_problem':
        return item.is_problem
    if filter_name == 'output':
        return item.output.lower()
    if filter_name == 'long_output':
        return item.long_output.lower()
    if filter_name == 'perf_data':
        return item.perf_data.lower()


########################################################################################################################
#  ____  _____ ____  ____  _____ ____    _  _____ _____ ____    ____   _    ____ _____   ____ _____  _    ____ _____
# |  _ \| ____|  _ \|  _ \| ____/ ___|  / \|_   _| ____|  _ \  |  _ \ / \  |  _ \_   _| / ___|_   _|/ \  |  _ \_   _|
# | | | |  _| | |_) | |_) |  _|| |     / _ \ | | |  _| | | | | | |_) / _ \ | |_) || |   \___ \ | | / _ \ | |_) || |
# | |_| | |___|  __/|  _ <| |__| |___ / ___ \| | | |___| |_| | |  __/ ___ \|  _ < | |    ___) || |/ ___ \|  _ < | |
# |____/|_____|_|   |_| \_\_____\____/_/   \_\_| |_____|____/  |_| /_/   \_\_| \_\|_|   |____/ |_/_/   \_\_| \_\|_|
#
########################################################################################################################


#  ops: list of the ops possible for this parameter
#  desc: label used for display
#  listed: list on the type select
filter_properties = {
    'hst'            : {'ops': ['='], 'desc': 'list.host_status', 'listed': False, 'obj': 'int', 'f': match_hst},
    'hp'             : {'ops': ['='], 'desc': 'list.host_properties', 'listed': False, 'obj': 'int', 'f': match_hp},
    'sst'            : {'ops': ['='], 'desc': 'list.check_status', 'listed': False, 'obj': 'int', 'f': match_sst},
    'sp'             : {'ops': ['='], 'desc': 'list.check_properties', 'listed': False, 'obj': 'int', 'f': match_sp},
    
    'host'           : {'ops': ['~', '!~'], 'desc': 'list.f_host', 'obj': 'host', 'f': match_host},
    'tag'            : {'ops': ['=', '!=', '~', '!~'], 'desc': 'list.f_host_template', 'obj': 'tag', 'f': match_tag},
    'service'        : {'ops': ['~', '!~'], 'desc': 'list.f_check', 'obj': 'service', 'f': match_service},
    
    'hostgroup'      : {'ops': ['=', '!='], 'desc': 'list.f_hostgroup', 'obj': 'hostgroup', 'f': match_hg, 'options': []},
    'parent'         : {'ops': ['=', '!='], 'desc': 'list.f_link', 'obj': 'host', 'f': match_parent},
    'realm'          : {'ops': ['=', '!='], 'desc': 'list.f_realm', 'obj': 'realm', 'f': match_realm, 'options': []},
    
    'is root problem': {'ops'    : ['='], 'desc': 'list.f_root_problem', 'obj': 'other', 'f': match_root_problem,
                        'options': [{'value': '1', 'display': 'list_option.true'},
                                    {'value': '0', 'display': 'list_option.false'},
                                    ]},
    'business impact': {'ops'    : ['>=', '<=', '>', '<', '=', '!='], 'desc': 'list.f_business_impact', 'obj': 'int', 'f': match_business_impact,
                        'options': [{'value': 5, 'display': 'list_option.top'},
                                    {'value': 4, 'display': 'list_option.important'},
                                    {'value': 3, 'display': 'list_option.production'},
                                    {'value': 2, 'display': 'list_option.normal'},
                                    {'value': 1, 'display': 'list_option.testing'},
                                    {'value': 0, 'display': 'list_option.dev'},
                                    ]
                        },
    'contact'        : {'ops': ['=', '!='], 'desc': 'list.f_contact', 'obj': 'contact', 'f': match_contact, 'options': []},
    
}

# For exporting into javascript, we remove all functions from this dict
clean_filter_properties = {}
for (k, v) in filter_properties.iteritems():
    d = {}
    clean_filter_properties[k] = d
    for (k2, v2) in v.iteritems():
        if not callable(v2):
            d[k2] = v2

quick_searches = [
    {
        'name'   : 'hdown',
        'path'   : '/all?inp-F1-K1-type=hst&sel-F1-K1-op=%3D&inp-F1-K1-value=6&inp-F1-K2-type=hp&sel-F1-K2-op=%3D&inp-F1-K2-value=67594&inp-F1-K3-type=sst&sel-F1-K3-op=%3D&inp-F1-K3-value=0&inp-F1-K4-type=sp&sel-F1-K4-op=%3D&inp-F1-K4-value=0',
        'display': 'Down Hosts',
    },
    
    {
        'name'   : 'rootpbs',
        'path'   : '/all?inp-F1-K1-type=hst&sel-F1-K1-op=%3D&inp-F1-K1-value=15&inp-F1-K2-type=hp&sel-F1-K2-op=%3D&inp-F1-K2-value=0&inp-F1-K3-type=sst&sel-F1-K3-op=%3D&inp-F1-K3-value=31&inp-F1-K4-type=sp&sel-F1-K4-op=%3D&inp-F1-K4-value=0&sel-F1-K5-type=is+root+problem&sel-F1-K5-op=%3D&inp-F1-K5-value=1',
        'display': 'Root Problems'
    },
]


# DEPRECATED : use get_list_elements
def get_problems():
    return get_view('problems')


# DEPRECATED : use get_list_elements
def get_all():
    return get_view('all')


# DEPRECATED : use get_list_elements
def get_list_api_part(part):
    d = get_view(part, start=0, end=100000, sort_by='sort_full_name')
    items = d['pbs']
    
    elements = [_api_list_element_dict(e) for e in items]
    return json.dumps(elements)


# DEPRECATED : use get_list_elements
def get_clean_filter_properties():
    return json.dumps(clean_filter_properties)


# DEPRECATED : use get_list_elements
def _analyse_arg(filters, k, v):
    print 'GROK analysing', k, v
    elts = k.split('-')
    if len(elts) != 4:
        print 'INVALID ARG FOR FILTERING', k, v
        return None
    FN = elts[1]
    KN = elts[2]
    part = elts[3]
    if part not in ['type', 'op', 'value']:
        print 'INVALID ARG FOR FILTERING', k, v
        return None
    print 'GROK:', FN, KN, part
    f = _get_F(filters, FN)
    print 'GROK UPDATing F', f
    k = _get_K(f, KN)
    
    print 'GROK: K is reached', k
    
    # Type setting is quite easy, it must be a valid filter_properties entry
    if part == 'type':
        if len(v) == 0:
            print 'INVALID ARG FOR FILTERING', k, v
            return None
        t = v[0]
        print 'GROK: trying to set type', t
        if not t in filter_properties:
            print 'INVALID ARG FOR FILTERING', k, v
            return None
        k['type'] = t
        return
    
    # op can be void. If so, use the first possible entry
    if part == 'op':
        t = k['type']
        # e = filter_properties[t]
        op = '='
        if len(v) > 0:
            op = v[0]
        # Hook some values that must be transformed
        op = {'&gt;': '>', '&gt;=': '>=', '&lt;': '<', '&lt;=': '<='}.get(op, op)
        
        # Maybe the op is known BEFORE the type (arg orders is not sure), so
        # we cannot look at filter_properties possibles values
        
        k['op'] = op
        return
    
    # We lok at the obj entry of the properties, and try to objectify the value
    if part == 'value':
        t = k['type']
        e = filter_properties[t]
        val = ''
        if len(v) > 0:
            val = v[0]
        print 'VAL', val
        print 'T', t
        obj = e.get('obj', None)
        if not obj:
            k['value'] = val
            return
        if obj == 'int':
            k['value'] = int(val)
            return
        if obj == 'bool':
            if val.lower() in ['1', 'true', 'on']:
                k['value'] = True
            else:
                k['value'] = False
            return
        k['value'] = val
        return


# DEPRECATED : use get_list_elements
def _gef_default_F(F):
    return {'name': F, 'value': [
        {'kn': 'K1', 'fn': F, 'type': 'hst', 'op': '=', 'value': 15},
        {'kn': 'K2', 'fn': F, 'type': 'hp', 'op': '=', 'value': 0},
        {'kn': 'K3', 'fn': F, 'type': 'sst', 'op': '=', 'value': 31},
        {'kn': 'K4', 'fn': F, 'type': 'sp', 'op': '=', 'value': 0},
    ]}


# DEPRECATED : use get_list_elements
# BIG OR statement, and inside a F: AND rule
def _get_default_filters(FN):
    F = _gef_default_F(FN)
    filters = [F]
    return filters


# DEPRECATED : use get_list_elements
def _get_F(filters, FN):
    for f in filters:
        if f['name'] == FN:
            return f
    # oups, still not there? ok create it
    f = _gef_default_F(FN)
    filters.append(f)
    return f


# DEPRECATED : use get_list_elements
def _get_K(f, KN):
    fn = ''
    for k in f['value']:
        fn = k['fn']
        if k['kn'] == KN:
            return k
    # Still there? we give a simple new k that means 'all'
    k = {'kn': KN, 'fn': fn, 'type': 'host', 'op': '~', 'value': ''}
    f['value'].append(k)
    return k


# DEPRECATED : use get_list_elements
def _do_filter_F(pack, f):
    res = set(pack)
    
    for k in f['value']:
        t = k['type']
        op = k['op']
        value = k['value']
        e = filter_properties.get(t, None)
        if not e:
            continue
        f = e.get('f', None)
        # If no filtering already coded, bypass it
        if not f:
            continue
        
        to_del = set()
        print 'GROK: APPLy FILTER', t, op, value, to_del
        for i in res:
            r = f(app, i, op, value)
            if not r:
                to_del.add(i)
        print 'GROK: do_filter_F len to_del', len(to_del)
        # Now remove all bad item from the current result
        res = res.difference(to_del)
        print 'GROK: do_filter_F len res afger difference', len(to_del)
    print 'GROK: do_filter_F len result', len(res)
    return res


# DEPRECATED : use get_list_elements
def _do_filter(pack, filters):
    print 'GROK: do_filter START len result', len(pack)
    res = set()
    # If not current filter, give all :)
    if len(filters) == 0:
        return list(pack)
    tmp = []
    for f in filters:
        p = _do_filter_F(pack, f)
        tmp.append(p)
    for p in tmp:
        res = res | p
    print 'GROK: do_filter len result', len(res)
    return list(res)


# DEPRECATED : use get_list_elements
# Sort hosts and services by impact, states and co
def _srv_sort_full_name(s1, s2):
    # by full name...
    return cmp(s1.get_full_name().upper().replace(' ', 'SPACE').replace('-', 'TIRET').replace('.', 'POINT'), s2.get_full_name().upper().replace(' ', 'SPACE').replace('-', 'TIRET').replace('.', 'POINT'))


# DEPRECATED : use get_list_elements
# Our View code.
# We will get different data from all and problems but it's mainly filtering changes
def get_view(page, show_ack_on_problems=False, start=None, end=None, sort_by=None, show_elt_unauthorized=False):
    user = app.get_user_auth()
    if not user or not user.is_admin:
        app.bottle.redirect('/')
    
    logger.debug('get_view request.get %s' % app.request.GET.__dict__)
    toolbar = app.request.GET.get('toolbar', '')
    search = app.request.GET.getall('global_search')
    start_get = app.request.GET.get('start', '0')
    end_get = app.request.GET.get('end', '')
    if sort_by is None:
        sort_by = app.request.GET.get('sort_by', '')
    
    # Look for the toolbar pref
    tool_pref = app.get_user_preference(user, 'toolbar')
    # If void, create an empty one
    if not tool_pref:
        app.set_user_preference(user, 'toolbar', 'show')
        tool_pref = 'show'
    
    logger.debug('Toolbar tool_pref %s, toolbar %s' % (tool_pref, toolbar))
    if toolbar != tool_pref and len(toolbar) > 0:
        preference = app.set_user_preference(user, 'toolbar', toolbar)
        logger.debug('Need to change user prefs for Toolbar %s' % preference)
    tool_pref = app.get_user_preference(user, 'toolbar')
    
    # We will keep a trace of our filters
    filtersold = {}
    ts = ['hst_srv', 'hg', 'realm', 'htag', 'ack', 'downtime', 'crit', 'output', 'long_output', 'host', 'check', 'duration', 'type']
    for t in ts:
        filtersold[t] = []
    
    # Most of the case, search will be a simple string, if so
    # make it a list of this string
    if isinstance(search, basestring):
        search = [search]
    
    # Force utf8 encoding  for parameters too
    search = [s.decode('utf8', 'ignore') for s in search]
    
    # Load the columns that this user want
    cols = app.get_user_preference(user, 'tab_cols')
    if not cols:
        app.set_user_preference(user, 'cols', '0')
        cols = '0'
    cols = int(json.loads(cols))
    
    tab_nb_elts = app.get_user_preference(user, 'tab_nb_elts')
    if not tab_nb_elts:
        app.set_user_preference(user, 'tab_nb_elts', '30')
        tab_nb_elts = '30'
    tab_nb_elts = int(json.loads(tab_nb_elts))
    
    # Load the bookmarks
    bookmarks_r = app.get_user_preference(user, 'bookmarks')
    if not bookmarks_r:
        app.set_user_preference(user, 'bookmarks', '[]')
        bookmarks_r = '[]'
    
    bookmarks_ro = app.get_common_preference('bookmarks')
    if not bookmarks_ro:
        bookmarks_ro = '[]'
    
    bookmarksro = json.loads(bookmarks_ro)
    bookmarks = json.loads(bookmarks_r)
    
    # We want to limit the number of elements if not given
    if start is None:
        start = int(start_get)
    if end is None:
        end = int(('%d' % tab_nb_elts) if end_get == '' else end_get)
    
    items = []
    if page == 'problems':
        items = app.datamgr.get_all_problems(to_sort=False, get_acknowledged=show_ack_on_problems)
    elif page == 'all':
        items = app.datamgr.get_all_hosts_and_services()
    else:  # WTF?!?
        app.bottle.redirect('/problems')
    
    # Filter with the user interests
    # TODO mark filter elements.
    if not show_elt_unauthorized:
        items = only_related_to(items, user, app)
    
    # Now parse the GET input to find some filters :)
    filters = []
    
    args = {}
    for (_, args) in app.request.GET.__dict__.iteritems():
        pass
    
    logger.debug('GROK1 %s' % args)
    for (k, v) in args.iteritems():
        logger.debug('GROK:' + str(k) + str(type(k)) + str(v) + str(type(v)))
        if k.startswith('inp-'):
            _k = 'sel-' + k[len('inp-'):]
            # Skip it, there is a sel- one
            if _k in args:
                continue
        if k.startswith('inp-') or k.startswith('sel-'):
            # Special case: inp-F1-K5-value is useless if sel-F1-K5-type
            _analyse_arg(filters, k, v)
    
    logger.debug('filter use %s' % filters)
    logger.debug('element size before filter %s' % len(items))
    items = _do_filter(items, filters)
    logger.debug('element size after filter %s' % len(items))
    
    logger.debug('search use %s' % search)
    
    # Ok, if need, appli the search filter
    for s in search:
        s = s.strip()
        if not s:
            continue
        
        print 'SEARCHING FOR', s
        print 'Before filtering', len(items)
        
        elts = s.split(':', 1)
        t = 'hst_srv'
        if len(elts) > 1:
            t = elts[0]
            s = elts[1]
        
        print 'Search for type %s and pattern %s' % (t, s)
        if not t in filtersold:
            filtersold[t] = []
        filtersold[t].append(s)
        
        if t == 'hst_srv':
            # We compile the pattern
            pat = re.compile(s, re.IGNORECASE)
            new_items = []
            for i in items:
                if pat.search(i.get_full_name()):
                    new_items.append(i)
                    continue
            
            items = new_items
        
        if t == 'host':
            _s = s.lower()
            new_items = []
            for i in items:
                if i.__class__.my_type == 'host':
                    if _s in i.host_name.lower():
                        new_items.append(i)
                else:
                    if _s in i.host.host_name.lower():
                        new_items.append(i)
            items = new_items
        
        if t == 'check':
            _s = s.lower()
            new_items = []
            for i in items:
                if i.__class__.my_type == 'service':
                    if _s in i.service_description.lower():
                        new_items.append(i)
            items = new_items
        
        if t == 'parents':
            _s = s.lower()
            new_items = []
            for i in items:
                _o = i
                if i.__class__.my_type == 'service':
                    _o = i.host
                my_parents = getattr(_o, 'my_parents', None)
                if my_parents is None:  # oh first related build for this host? do it so :)
                    my_parents = set()
                    _ = [my_parents.add(p.get_name().lower()) for p in _o.parents if p.__class__.my_type == 'host']
                    _ = [my_parents.add(p.get_name().lower()) for p in _o.parent_dependencies if p.__class__.my_type == 'host']
                    _ = [my_parents.add(p.get_name().lower()) for p in _o.child_dependencies if p.__class__.my_type == 'host']
                    _o.my_parents = list(my_parents)
                    _o.my_parents.sort()
                # Now this is build for the good elements, check if the current search match (or not)
                for rname in _o.my_parents:
                    if _s == rname:
                        new_items.append(i)
                    continue
            
            items = new_items
        
        if t == 'root-problem':
            b = (s == '1')
            items = [i for i in items if i.is_problem == b]
        
        if t == 'root-problem-with-parent':
            new_items = []
            for i in items:
                if i.is_problem:
                    new_items.append(i)
                    if (i.__class__.my_type == 'service') and not (i.host in new_items):
                        new_items.append(i.host)
            items = new_items
        
        if t == 'acknowledged':
            b = (s == '1')
            items = [i for i in items if i.problem_has_been_acknowledged == b]
        
        if t == 'downtime':
            b = (s == '1')
            items = [i for i in items if i.in_scheduled_downtime == b]
        
        if t == 'hostgroups':
            _s = s.lower()
            new_items = []
            for i in items:
                _o = i
                if i.__class__.my_type == 'service':
                    _o = i.host
                vhgnames = [p for p in _o.hostgroups if _s == p.get_name().lower()]
                if len(vhgnames) != 0:
                    new_items.append(i)
            items = new_items
        
        if t == 'servicegroups':
            _s = s.lower()
            new_items = []
            for i in items:
                _o = i
                if i.__class__.my_type == 'host':  # skip hosts
                    continue
                vhgnames = [p for p in _o.servicegroups if _s == p.get_name().lower()]
                if len(vhgnames) != 0:
                    new_items.append(i)
            items = new_items
        
        if t == 'contacts':
            _s = s.lower()
            new_items = []
            for i in items:
                vnames = [c for c in i.contacts if _s == c.get_name().lower()]
                if len(vnames) != 0:
                    new_items.append(i)
            items = new_items
        
        if t == 'type':
            _s = s.lower()
            if _s == 'host':
                items = [i for i in items if i.__class__.my_type == _s and not i.got_business_rule]
            elif _s == 'cluster':
                items = [i for i in items if i.got_business_rule]
            else:
                items = [i for i in items if i.__class__.my_type == 'service']
        
        if t == 'hg':
            hg = app.datamgr.get_hostgroup(s)
            print 'And a valid hg filtering for', s
            items = [i for i in items if hg in i.get_hostgroups()]
        
        if t == 'realm':
            r = app.datamgr.get_realm(s)
            print 'Add a realm filter', r
            items = [i for i in items if i.get_realm() == r]
        
        if t == 'output':
            _s = s.lower()
            items = [i for i in items if _s in i.output.lower()]
        
        if t == 'long_output':
            _s = s.lower()
            items = [i for i in items if _s in i.long_output.lower()]
        
        if t == 'business-impact':
            _s = int(s)
            items = [i for i in items if _s == i.business_impact]
        
        if t == 'display-name':
            _s = s.lower()
            new_items = []
            for i in items:
                _o = i
                if i.__class__.my_type == 'service':
                    _o = i.host
                if _s in _o.display_name.lower():
                    new_items.append(i)
            items = new_items
        
        if t == 'address':
            _s = s.lower()
            new_items = []
            for i in items:
                _o = i
                if i.__class__.my_type == 'service':
                    _o = i.host
                if _s in _o.address.lower():
                    new_items.append(i)
            items = new_items
        
        if t == 'tag':
            _items = []
            for elt in items:
                for tag in elt.get_host_tags():
                    if tag.lower() == s:
                        _items.append(elt)
            items = _items
        if t == 'attempts':
            items = [i for i in items if i.attempt >= int(s)]
        
        if t == 'state-type':
            items = [i for i in items if s in i.state_type == s]
        
        if t == 'duration':
            more_recent = True
            v = None
            if s.startswith('>'):
                more_recent = False
                v = int(s[1:])
            if v is None:
                v = int(s)
            _past = int(time.time()) - v
            if more_recent:
                items = [i for i in items if i.last_state_change > _past]
            else:
                items = [i for i in items if i.last_state_change < _past]
        
        if t == 'last-check':
            more_recent = True
            v = None
            if s.startswith('>'):
                more_recent = False
                v = int(s[1:])
            if v is None:
                v = int(s)
            _past = int(time.time()) - v
            if more_recent:
                items = [i for i in items if i.last_chk > _past]
            else:
                items = [i for i in items if i.last_chk < _past]
        
        if t == 'ack':
            print 'Got an ack filter', s
            if s == 'false':
                # First look for hosts, so ok for services, but remove problem_has_been_acknowledged elements
                items = [i for i in items if i.__class__.my_type == 'service' or not i.problem_has_been_acknowledged]
                # Now ok for hosts, but look for services, and service hosts
                items = [i for i in items if i.__class__.my_type == 'host' or (not i.problem_has_been_acknowledged and not i.host.problem_has_been_acknowledged)]
            if s == 'true':
                # First look for hosts, so ok for services, but remove problem_has_been_acknowledged elements
                items = [i for i in items if i.__class__.my_type == 'service' or i.problem_has_been_acknowledged]
                # Now ok for hosts, but look for services, and service hosts
                items = [i for i in items if i.__class__.my_type == 'host' or (i.problem_has_been_acknowledged or i.host.problem_has_been_acknowledged)]
        
        if t == 'downtime':
            if s == 'false':
                # First look for hosts, so ok for services, but remove problem_has_been_acknowledged elements
                items = [i for i in items if i.__class__.my_type == 'service' or not i.in_scheduled_downtime]
                # Now ok for hosts, but look for services, and service hosts
                items = [i for i in items if i.__class__.my_type == 'host' or (not i.in_scheduled_downtime and not i.host.in_scheduled_downtime)]
            if s == 'true':
                # First look for hosts, so ok for services, but remove problem_has_been_acknowledged elements
                items = [i for i in items if i.__class__.my_type == 'service' or i.in_scheduled_downtime]
                # Now ok for hosts, but look for services, and service hosts
                items = [i for i in items if i.__class__.my_type == 'host' or (i.in_scheduled_downtime or i.host.in_scheduled_downtime)]
        
        if t == 'summary':
            # This one can be quite consuming, so compute items summary only once
            summaries = [(i, app.helper.get_summary(i)['summary']) for i in items]
            if s == 'NO-DOWNTIME':
                items = [i for (i, _summary) in summaries if _summary != 'DOWNTIME']
            elif s == 'DOWNTIME':
                items = [i for (i, _summary) in summaries if _summary == 'DOWNTIME']
            elif s == 'NO-ACKNOWLEDGED':
                items = [i for (i, _summary) in summaries if _summary not in ['ACKNOWLEDGED', 'PARTIAL-ACKNOWLEDGED']]
            elif s == 'ACKNOWLEDGED':
                items = [i for (i, _summary) in summaries if _summary in ['ACKNOWLEDGED', 'PARTIAL-ACKNOWLEDGED']]
            elif s == 'NO-FLAPPING':
                items = [i for (i, _summary) in summaries if _summary not in ['FLAPPING', 'PARTIAL-FLAPPING']]
            elif s == 'FLAPPING':
                items = [i for (i, _summary) in summaries if _summary in ['FLAPPING', 'PARTIAL-FLAPPING']]
        
        if t == 'crit':
            print 'Add a criticity filter', s
            items = [i for i in items if (i.__class__.my_type == 'service' and i.state_id == 2) or (i.__class__.my_type == 'host' and i.state_id == 1)]
        
        if t == 'state':
            if s == 'OK-UP':
                items = [i for i in items if i.state_id == 0]
            if s == 'WARNING':
                items = [i for i in items if i.__class__.my_type == 'service' and i.state_id == 1]
            if s == 'CRITICAL-DOWN':
                items = [i for i in items if i.state_id == 2 or (i.__class__.my_type == 'host' and i.state_id == 1)]
            if s == 'UNKNOWN-PENDING-UNREACHEABLE':
                items = [i for i in items if i.state_id == 3 or i.state == 'PENDING']
        
        print 'After filtering for', t, s, 'we got', len(items)
    
    # Now sort it!
    if sort_by == 'sort_full_name':
        items.sort(_srv_sort_full_name)
    else:
        items.sort(hst_srv_sort)
    
    total = len(items)
    # If we overflow, came back as normal
    if start > total:
        start = 0
        end = tab_nb_elts
    
    navi = app.helper.get_navi(total, start, step=tab_nb_elts)
    items = items[start:end]
    
    # Update filter properties with realm and such things for the select
    realms = [r for r in app.datamgr.get_realms()]
    realms.sort()
    clean_filter_properties['realm']['options'] = [{'value': r, 'display': r.capitalize()} for r in realms]
    # now hostggroups
    hgs = [hg.get_name() for hg in app.datamgr.get_hostgroups()]
    hgs.sort()
    clean_filter_properties['hostgroup']['options'] = [{'value': hg, 'display': hg.capitalize()} for hg in hgs]
    
    # Contacts
    cs = [c.get_name() for c in app.datamgr.get_contacts()]
    cs.sort()
    clean_filter_properties['contact']['options'] = [{'value': c, 'display': c.capitalize()} for c in cs]
    
    # Create the F1 filter only if there are no other filters send from the user (like F2 and so on)
    if len(filters) == 0:
        filters = _get_default_filters('F1')
    
    return {'app'           : app, 'pbs': items, 'user': user, 'navi': navi, 'search': '&'.join(search), 'page': page, 'filters': filtersold, 'bookmarks': bookmarks, 'bookmarksro': bookmarksro, 'toolbar': tool_pref, 'filters2': filters,
            'hst_properties': hst_properties, 'hp_properties': hp_properties, 'sst_properties': sst_properties, 'sp_properties': sp_properties, 'filter_properties': clean_filter_properties, 'cols': cols, 'tab_nb_elts': tab_nb_elts,
            'quick_searches': quick_searches, 'start': start, 'end': end, 'total': total, 'sort_by': sort_by}


########################################################################################################################
#  ____  _____ ____  ____  _____ ____    _  _____ _____ ____    ____   _    ____ _____   _____ _   _ ____
# |  _ \| ____|  _ \|  _ \| ____/ ___|  / \|_   _| ____|  _ \  |  _ \ / \  |  _ \_   _| | ____| \ | |  _ \
# | | | |  _| | |_) | |_) |  _|| |     / _ \ | | |  _| | | | | | |_) / _ \ | |_) || |   |  _| |  \| | | | |
# | |_| | |___|  __/|  _ <| |__| |___ / ___ \| | | |___| |_| | |  __/ ___ \|  _ < | |   | |___| |\  | |_| |
# |____/|_____|_|   |_| \_\_____\____/_/   \_\_| |_____|____/  |_| /_/   \_\_| \_\|_|   |_____|_| \_|____/
#
########################################################################################################################


def api_get_pbs_widget(item_uuid):
    user = app.get_user_auth()
    nb_elements = int(app.request.GET.get('nb_elements', '10'))
    element = app.helper.get_elt_from_uuid(item_uuid)
    
    if not element:
        return app.abort(404, '[api_get_pbs_widget] Element [%s] is not found' % item_uuid, True)
    
    element_data = {
        'uuid'      : item_uuid,
        'unsight'   : not is_authorized(element, user, app),
        'is_unknown': False
    }
    
    element_type = element.__class__.my_type
    if element_type == 'service':
        element_data['type'] = 'check'
        element_data['service_description'] = element.service_description
        element_data['host_name'] = element.host.host_name
    elif element_type == 'host' and element.got_business_rule:
        element_data['type'] = 'cluster'
        element_data['host_name'] = element.host_name
    else:
        element_data['type'] = 'host'
        element_data['host_name'] = element.host_name
    
    problems_list = []
    for problem_index, problem_uuid in enumerate(element.source_problems):
        if problem_index + 1 > nb_elements:
            break
        
        logger.debug('[api_get_pbs_widget] look for [%s]' % problem_uuid)
        item_pb = app.helper.get_elt_from_uuid(problem_uuid)
        
        if not is_authorized(item_pb, user, app):
            problems_list.append(app.helper.get_unauthorized_item(item_pb))
        else:
            item_pb_data = {
                'display_name': item_pb.get_full_name(),
                'status'      : app.helper.get_summary(item_pb)['status'],
                'unsight'     : False,
                'is_unknown'  : False
            }
            item_pb_type = item_pb.__class__.my_type
            
            context = 'NOTHING'
            if item_pb.is_acknowledged() and not item_pb.is_flapping and not item_pb.is_in_downtime():
                context = 'ACKNOWLEDGED'
            elif item_pb.is_flapping and not item_pb.is_in_downtime():
                context = 'FLAPPING'
            elif item_pb.is_in_downtime():
                context = 'DOWNTIME'
            
            item_pb_data['context'] = context
            if item_pb_type == 'service':
                item_pb_data['uuid'] = '%s-%s' % (item_pb.host.uuid, item_pb.uuid)
                item_pb_data['uuid_parent'] = item_pb.host.uuid
                item_pb_data['host_name'] = item_pb.host.host_name
                item_pb_data['type'] = 'check'
                item_pb_data['service_description'] = item_pb.service_description
            else:
                item_pb_data['uuid'] = item_pb.uuid
                item_pb_data['host_name'] = item_pb.host_name
                
                if item_pb.got_business_rule:
                    item_pb_data['type'] = 'cluster'
                else:
                    item_pb_data['type'] = item_pb_type
            
            problems_list.append(item_pb_data)
    
    return json.dumps({'element': element_data, 'children_list': problems_list})


def _build_static_sort_key(_value):
    return [int(s) if s.isdigit() else s.lower() for s in re.split(r'(\d+)', _value.encode('utf-8', 'ignore') if hasattr(_value, 'encode') else str(_value))]


def _init_static_cache(item):
    item_type = 'host'
    if item.__class__.my_type == 'service':
        item_type = 'check'
    elif item.got_business_rule:
        item_type = 'cluster'
    
    list_cache_static = {
        'display_name'        : item.display_name,
        'contact_groups'      : [cg.strip() for cg in getattr(item, 'contact_groups', '') if cg.strip()],
        'contacts'            : [c.get_name() for c in item.contacts],
        'max_attempts'        : item.max_check_attempts,
        'notification_period' : item.notification_period.get_name() if item.notification_period else '',
        'notification_options': item.notification_options,
        'type'                : item_type,
        'is_unknown'          : False,
    }
    
    if item_type == 'check':
        list_cache_static['host_name'] = item.host.host_name
        list_cache_static['service_description'] = item.service_description
        list_cache_static['realm'] = item.host.realm
        list_cache_static['uuid'] = '%s-%s' % (item.host.uuid, item.uuid)
        list_cache_static['uuid_parent'] = item.host.uuid
        list_cache_static['host_groups'] = [i.get_name() for i in item.host.hostgroups]
    else:
        if item_type == 'host':
            list_cache_static['address'] = item.address
        
        list_cache_static['host_name'] = item.host_name
        list_cache_static['realm'] = item.realm
        list_cache_static['host_templates'] = [i for i in item.tags if i != 'shinken-host']
        list_cache_static['host_groups'] = [i.get_name() for i in item.hostgroups]
        list_cache_static['uuid'] = item.uuid
    item.list_cache_static = list_cache_static
    
    filter_cache_static = {
        'display_name'        : list_cache_static['display_name'].lower(),
        'contact_groups'      : set([i.lower() for i in list_cache_static['contact_groups']]),
        'contacts'            : set([i.lower() for i in list_cache_static['contacts']]),
        'notification_period' : list_cache_static['notification_period'].lower(),
        'notification_options': set(list_cache_static['notification_options']),
        'type'                : set((item_type,)),
        'host_name'           : list_cache_static['host_name'].lower(),
        'realm'               : set((list_cache_static['realm'].lower(),)),
        'host_groups'         : set([i.lower() for i in list_cache_static['host_groups']])
    }
    if list_cache_static['type'] == 'host':
        filter_cache_static['service_description'] = ''
        filter_cache_static['address'] = item.address.lower()
        filter_cache_static['host_templates'] = set([i.lower() for i in list_cache_static['host_templates']])
    if list_cache_static['type'] == 'cluster':
        filter_cache_static['service_description'] = ''
        filter_cache_static['address'] = ''
        filter_cache_static['host_templates'] = set([i.lower() for i in list_cache_static['host_templates']])
    if list_cache_static['type'] == 'check':
        filter_cache_static['service_description'] = item.service_description.lower()
        filter_cache_static['address'] = item.host.address.lower()
        filter_cache_static['host_templates'] = set([i.lower() for i in item.host.tags if i != 'shinken-host'])
    
    item.filter_cache_static = filter_cache_static
    
    sort_cache_static = {}
    for _tag, _value in filter_cache_static.iteritems():
        sort_cache_static[_tag] = _build_static_sort_key(_value)
    
    item.sort_cache_static = sort_cache_static


def _api_list_element_dict(item):
    list_cache_var = getattr(item, 'list_cache_var', {})
    if not list_cache_var:
        list_cache_var['business_impact'] = item.business_impact
        list_cache_var['status'] = _get_state(item)
        list_cache_var['context'], _ = _get_context(item)
        list_cache_var['status_since'] = int(item.last_state_change)
        list_cache_var['is_status_confirmed'] = (item.state_type == 'HARD')
        list_cache_var['attempts'] = item.attempt
        list_cache_var['is_root_problem'] = item.is_problem
        list_cache_var['output'] = item.output
        list_cache_var['long_output'] = item.long_output
        list_cache_var['perf_data'] = item.perf_data
        list_cache_var['got_business_rule'] = item.got_business_rule
        list_cache_var['in_flapping'] = item.is_flapping
        list_cache_var['inherited_flapping'] = item.inherited_flapping
        list_cache_var['in_partial_flapping'] = item.is_partial_flapping
        list_cache_var['notes_multi_url'] = item.notes_multi_url
        list_cache_var['active_checks_enabled'] = item.active_checks_enabled
        
        app.helper.get_ack_on_item(item, list_cache_var)
        app.helper.get_downtime_on_item(item, list_cache_var)
        
        if not item.got_business_rule:
            list_cache_var['next_check'] = item.next_chk
            list_cache_var['last_check'] = item.last_chk
        
        list_cache_var.update(item.list_cache_static)
        item.list_cache_var = list_cache_var
    
    return list_cache_var


def _build_var_sort_key(item, tag):
    _value = ''
    if tag == 'problem_has_been_inherited_acknowledged':
        _value = 0
        if item.__class__.my_type == 'service':
            if item.host.acknowledgement:
                _value = 1
        elif item.acknowledgement and item.acknowledgement.automatic:
            _value = 2
    elif tag == 'next_check':
        if item.got_business_rule:
            _value = -1
        else:
            _value = item.next_chk
    elif tag == 'last_check':
        if item.got_business_rule:
            _value = -1
        else:
            _value = item.last_chk
    elif tag == 'business_impact':
        _value = item.business_impact
    elif tag == 'status':
        _value = str(_get_state(item))
    elif tag == 'context':
        _value = _get_context(item)[1]
    elif tag == 'status_since':
        _value = int(item.last_state_change)
    elif tag == 'is_status_confirmed':
        _value = (item.state_type == 'HARD')
    elif tag == 'attempts':
        _value = item.attempt
    elif tag == 'is_root_problem':
        _value = item.is_problem
    elif tag == 'output':
        _value = item.output
    elif tag == 'long_output':
        _value = item.long_output
    elif tag == 'perf_data':
        _value = item.perf_data
    elif tag == 'got_business_rule':
        _value = item.got_business_rule
    elif tag == 'problem_has_been_acknowledged':
        _value = bool(item.acknowledgement and not item.acknowledgement.automatic)
    elif tag == 'in_flapping':
        _value = item.is_flapping
    elif tag == 'inherited_flapping':
        _value = item.inherited_flapping
    elif tag == 'in_partial_flapping':
        _value = item.is_partial_flapping
    
    return [int(s) if s.isdigit() else s.lower() for s in re.split(r'(\d+)', _value.encode('utf-8', 'ignore') if hasattr(_value, 'encode') else str(_value))]


def _sort_list_api_part_all(sorts, items):
    for sort in sorts:
        tag = sort['tag']
        reverse = (sort['value'] == 'desc')
        items.sort(key=lambda item: item.sort_cache_static[tag] if tag in item.sort_cache_static else _build_var_sort_key(item, tag), reverse=reverse)


def _parse_filter(get_filters):
    filters = []
    for get_filter in get_filters:
        filter = {}
        for search_type in get_filter.split('~'):
            search_type = search_type.replace('&#x7e;', '~')
            split = search_type.split(':', 1)
            filter_tag = split[0].lower()
            filter_value = split[1].lower().split('^^')
            if not filter_value:
                continue
            compare_type = FILTER_TYPE[filter_tag]
            if compare_type == 'int':
                filter_value = set([int(v) for v in filter_value])
            elif compare_type == 'bool':
                filter_value = set([('true' == v) for v in filter_value])
            elif compare_type == 'array':
                filter_value = set(filter_value)
            elif compare_type == 'date':
                val = filter_value[0]
                if val.startswith('latest') or val.startswith('old') or val.startswith('in_more_than') or val.startswith('in_less_than'):
                    operator, delay = val.split('|')
                    delay = int(delay)
                    operator = DateFilterOperator.NEWER if operator == 'latest' or operator == 'in_more_than' else DateFilterOperator.OLDER
                    filter_value = [DateFilter(operator, delay)]
            filter[filter_tag] = filter_value
        filters.append(filter)
    return filters


def _filter_elements_with_search(items, _filters):
    new_elts = []
    
    for item in items:
        add_element = False
        for _filter in _filters:
            if _apply_filter(_filter, item):
                add_element = True
                break
        
        if add_element:
            new_elts.append(item)
    
    return new_elts


def _apply_filter(_filter, item):
    match_filter = True
    for (filter_tag, filter_values) in _filter.iteritems():
        value = _get_value_for_filter(item, filter_tag)
        compare_type = FILTER_TYPE[filter_tag]
        # 0 is a value
        if value != 0 and not value:
            match_filter = False
            break
        
        _match_tag = False
        if compare_type == 'bool' or compare_type == 'int':
            _match_tag = value in filter_values
        elif compare_type == 'array':
            _match_tag = bool(filter_values & value)
        elif compare_type == 'string':
            _match_tag = filter_values[0] in value
        elif compare_type == 'date':
            filter_value = filter_values[0]
            if isinstance(filter_value, DateFilter):
                if filter_value.operator == DateFilterOperator.NEWER:
                    _match_tag = filter_value.delay >= time.time() - value
                else:
                    _match_tag = filter_value.delay <= time.time() - value
            else:
                value = datetime.datetime.fromtimestamp(value).strftime(FRONTEND_DATE_FORMAT)
                _match_tag = filter_value in value
        else:
            logger.warning('unknown filter type : %s' % compare_type)
            _match_tag = False
        
        if not _match_tag:
            match_filter = False
            break
    return match_filter


def _list_all_values(items):
    all_values = {
        'host_templates'     : set(),
        'realm'              : set(),
        'host_groups'        : set(),
        'contact_groups'     : set(),
        'contacts'           : set(),
        'notification_period': set()
    }
    update_realm = all_values['realm'].add
    update_host_template = all_values['host_templates'].update
    update_host_group = all_values['host_groups'].update
    update_contact_group = all_values['contact_groups'].update
    update_contacts = all_values['contacts'].update
    update_notification_period = all_values['notification_period'].add
    
    for item in items:
        list_cache_static = item.list_cache_static
        update_contact_group(list_cache_static['contact_groups'])
        update_contacts(list_cache_static['contacts'])
        update_notification_period(list_cache_static['notification_period'])
        if not list_cache_static['type'] == 'check':
            update_host_template(list_cache_static['host_templates'])
            update_host_group(list_cache_static['host_groups'])
            update_realm(list_cache_static['realm'])
    
    for (_type, values) in all_values.iteritems():
        all_values[_type] = list(values)
    return all_values


def get_visualisation_list():
    thread_id = '%s' % threading.current_thread().name
    
    full_start_time = time.time()
    get_range = app.request.GET.get('range', '').strip()
    get_sorts = app.request.GET.get('sort', '').strip()
    configuration_id_in_client = app.request.GET.get('configuration_id', '-1').strip()
    flattened_view = app.request.GET.get('flatten_list', '0') == '1'
    
    get_filters = []
    cnt = 0
    filters_json = app.request.body.getvalue()
    filters_dict = json.loads(filters_json)
    get_filter = filters_dict.get('filter%s' % cnt, '')
    while get_filter:
        get_filters.append(get_filter.strip())
        cnt += 1
        get_filter = filters_dict.get('filter%s' % cnt, '')
    
    full_filter_for_log = get_filters[:]
    full_sort_for_log = get_sorts[:]
    page = None
    page_size = None
    filters = []
    sorts = []
    
    start_time = time.time()
    user = app.get_user_auth()
    
    log_prefix = '%-22s ] [ user= %s ] [ get_visualisation_list' % (thread_id, user.uuid)
    
    items = app.datamgr.get_all_hosts_and_services()
    
    configuration_id = ''
    part_configuration_id = ''
    last_part_configuration_incarnation = app.datamgr.get_last_part_configuration_incarnation()
    if last_part_configuration_incarnation:
        configuration_id = last_part_configuration_incarnation.get_uuid()
        part_configuration_id = '%s-%s' % (last_part_configuration_incarnation.get_uuid(), last_part_configuration_incarnation.get_part_id())
    full_nb_items_in_broker = len(items)
    
    logger.log_perf(start_time, log_prefix, 'get_all_hosts_and_services len [%s]' % full_nb_items_in_broker)
    
    start_time = time.time()
    with StaticCache.lock:
        if StaticCache.part_configuration_id != part_configuration_id:
            for i in items:
                _init_static_cache(i)
            StaticCache.part_configuration_id = part_configuration_id
    logger.log_perf(start_time, log_prefix, '_init_static_cache len [%s]' % full_nb_items_in_broker)
    
    # Filter with authorization
    start_time = time.time()
    items = get_all_my_visible_items(user, app)[:]
    nb_element_total = len(items)
    logger.log_perf(start_time, log_prefix, 'only_related_to')
    
    start_time = time.time()
    if get_range:
        param_start = get_range.split('~')[0]
        param_nb = get_range.split('~')[1]
        page = int(param_start.split(':')[1])
        page_size = int(param_nb.split(':')[1])
    logger.log_perf(start_time, log_prefix, 'page %s page_size %s' % (page, page_size))
    
    start_time = time.time()
    if get_filters:
        filters = _parse_filter(get_filters)
    logger.log_perf(start_time, log_prefix, 'filters %s' % filters)
    
    start_time = time.time()
    if get_sorts:
        for get_sort in get_sorts.split('~'):
            split = get_sort.split(':', 1)
            sort_tag = split[0].lower()
            sort_value = split[1].lower()
            sorts.append({'tag': sort_tag, 'value': sort_value})
    else:
        sorts.append(DEFAULT_SORT)
    logger.log_perf(start_time, log_prefix, 'sorts %s' % sorts)
    
    # Values for selector
    all_values = {}
    if configuration_id_in_client != configuration_id:
        start_time = time.time()
        all_values = _list_all_values(items)
        logger.log_perf(start_time, log_prefix, '_list_all_values')
    
    start_time = time.time()
    hosts = {}
    nb_element_root_problems = 0
    for item in items:
        if item.__class__.my_type == 'service':
            if item.host is None:
                logger.error('A service without host have been found id: %s  service_description: %s' % (item.instance_uuid, item.service_description))
                continue
        else:
            hosts[item.host_name] = item
        if item.is_problem:
            nb_element_root_problems += 1
    logger.log_perf(start_time, log_prefix, 'count nb_element_root_problems')
    
    # Filter with filter parameters
    timestamp_backend_processing_start = int(time.time())
    start_time = time.time()
    if filters:
        filters_contain_root_problems = 'true' in filters[0].get('is_root_problem', set(['false']))
        if filters_contain_root_problems:
            items = _filter_root_problems(filters, items)
        
        # normal filters
        items = _filter_elements_with_search(items, filters)
    logger.log_perf(start_time, log_prefix, '_filter_elements_with_search ')
    
    nb_items_post_filter = len(items)
    
    if flattened_view:
        start_time = time.time()
        if sorts != [DEFAULT_SORT] or get_filters:
            _sort_list_api_part_all(sorts, items)
        logger.log_perf(start_time, log_prefix, 'sort all (flattened view)')
        
        _pages = []
        if page_size:
            _pages = [{'start': start_index, 'end': min(start_index + page_size, len(items))} for start_index in range(0, len(items), page_size)]
        elements = items
    else:
        start_time = time.time()
        host_element = OrderedDict((i.host_name, i) for i in items if (i.__class__.my_type != 'service'))
        check_element = OrderedDict()
        for i in items:
            if i.__class__.my_type == 'service':
                check_host_name = i.host.host_name
                if check_host_name not in check_element:
                    check_element[check_host_name] = []
                check_element[check_host_name].append(i)
                
                # If the host of the check isn't in host_element we add it
                # We don't want orphan check
                if check_host_name not in host_element:
                    host = hosts[check_host_name]
                    host_element[host.host_name] = host
        logger.log_perf(start_time, log_prefix, 'append host')
        
        # Sort result
        start_time = time.time()
        host_element = host_element.values()
        if sorts != [DEFAULT_SORT] or get_filters:
            _sort_list_api_part_all(sorts, host_element)
        logger.log_perf(start_time, log_prefix, 'sort host')
        
        start_time = time.time()
        elements = []
        _pages = []
        start_page = 0
        counter = 0
        current_page = 0
        for host in host_element:
            elements.append(host)
            counter += 1
            check_for_host = check_element.get(host.host_name, [])
            if check_for_host:
                if sorts != [DEFAULT_SORT] and (page is None or page == current_page):
                    _sort_list_api_part_all(sorts, check_for_host)
                elements.extend(check_for_host)
                counter += len(check_for_host)
            if counter >= page_size:
                _pages.append({'start': start_page, 'end': start_page + counter})
                start_page = start_page + counter
                counter = 0
                current_page += 1
        if counter != 0:
            _pages.append({'start': start_page, 'end': start_page + counter})
        
        logger.log_perf(start_time, log_prefix, 'sort checks')
    
    # Result pagination
    start_time = time.time()
    nb_element_return = len(elements)
    log_pages = ''
    if page is not None and _pages:
        log_pages = '%s / %s' % (page + 1, len(_pages))
        elements = elements[_pages[page]['start']:_pages[page]['end']]
        elements = [_api_list_element_dict(e) for e in elements]
    logger.log_perf(start_time, log_prefix, 'Result pagination')
    
    ret = {
        'elements'                          : elements,
        'nb_element_filter'                 : nb_element_return,
        'nb_element_total'                  : nb_element_total,
        'nb_element_root_problems'          : nb_element_root_problems,
        'configuration_id'                  : configuration_id,
        'pages'                             : _pages,
        'timestamp_backend_processing_start': timestamp_backend_processing_start,
        'timestamp_backend_processing_end'  : int(time.time())
    }
    if all_values:
        ret['all_values'] = all_values
    
    start_time = time.time()
    ret = json.dumps(ret)
    logger.log_perf(start_time, log_prefix, 'json.dumps')
    
    if not full_filter_for_log:
        full_filter_for_log = ''
    if not full_sort_for_log:
        full_sort_for_log = ''
    
    logger.log_perf(full_start_time, log_prefix, 'elements:[ in broker= %s filtered= %s total= %s in page= %s ] page:[ %s ] filter:[ %s ] sort:[ %s ]' % (
        full_nb_items_in_broker, nb_items_post_filter, nb_element_return, len(elements), log_pages, full_filter_for_log, full_sort_for_log), min_time=0)
    return ret


def _filter_root_problems(filters, items):
    items = _filter_elements_with_search(items, [{'is_root_problem': set(['true'])}])
    # remove the occurrence of is_root_problem in filter
    # special cassdedi for Mr Bmar
    for filter in filters[:]:
        filter.pop('is_root_problem', None)
    return items


def get_pbs_widget():
    pass


widget_display_name = 'Root Problems'
widget_desc = '''Show the list of root problems of an element'''
widget_default_options = [
    {'name': 'search', 'value': '', 'type': 'host', 'label': 'Select Host/Cluster', 'required': True},
    {'name': 'nb_elements', 'value': 10, 'type': 'int', 'label': 'Max number of elements to show', 'required': True},
]

pages = {
    get_problems               : {'routes': ['/problems'], 'view': 'problems', 'static': True, 'wrappers': ['auth']},
    get_all                    : {'routes': ['/all'], 'view': 'problems', 'static': True, 'wrappers': ['auth']},
    
    get_pbs_widget             : {
        'routes'             : ['/widget/problems'],
        'view'               : 'widget_problems',
        'static'             : True,
        'widget'             : ['dashboard'],
        'widget_display_name': widget_display_name,
        'widget_desc'        : widget_desc,
        'widget_name'        : 'problems',
        'widget_picture'     : '/static/tabular/img/widget_it_problems.png',
        'widget_size'        : {'width': 4, 'height': 5},
        'widget_options'     : widget_default_options,
        'widget_favoritable' : False,
        'old_style'          : False,
        'wrappers'           : []
    },
    
    get_clean_filter_properties: {'routes': ['/api/filter-properties'], 'wrappers': ['json']},
    api_get_pbs_widget         : {'routes': ['/api/widget/problems/:item_uuid'], 'static': True, 'wrappers': ['json']},
    get_list_api_part          : {'routes': ['/api/list-elements/:part'], 'wrappers': ['auth', 'json']},
    get_visualisation_list     : {'routes': ['/api/list-elements-all'], 'wrappers': ['auth', 'json'], 'method': 'POST'},
    
}
