import os
import time

from .log import logger, get_section_string

# Typing
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, List, Optionnal
    from .check import Check
    from threading import RLock

LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX = os.environ.get('SHINKEN_LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX_FLAG', '0') == '1'
JOB_EXECUTION_FAST_INDEX = get_section_string('JOB-EXECUTION FAST INDEX')


class ActionsContainer(object):
    def __init__(self, job_type_str):
        self.checks_n_actions_lock = None  # type: Optionnal[RLock]
        self._time_index = {}  # type: Dict[int, List[Check]]
        self._job_type_str = job_type_str  # type: str  #  For logging
    
    
    def load_lock(self, checks_n_actions_lock):
        # type: (RLock) -> None
        self.checks_n_actions_lock = checks_n_actions_lock
    
    
    def reset(self):
        with self.checks_n_actions_lock:
            self._time_index.clear()
    
    
    # A job did have (or have a new) time_to_go, so:
    # * maybe it was already inserted in an index epoch, if so, remove it from there
    # * insert it in the new epoch of the job
    def do_index_job_execution(self, job):
        # type: (Check) -> None
        # In this section, we are looking at index so we need its lock
        with self.checks_n_actions_lock:
            self.delete_job_from_index_seconds(job)  # be sure to delete old index entry
            
            if not job.can_be_indexed_for_execution():
                if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                    logger.debug('%s execution %s was refused in scheduling because its current status "%s" is not "scheduled"' % (JOB_EXECUTION_FAST_INDEX, job.get_printable_name(), job.status))
                return
            time_to_start = job.get_time_to_be_start()
            # NOTE: we are doing a try/catch instead of a "time_to_start in" for PERF reasons (max 1 keyerror / second)
            try:
                self._time_index[time_to_start].append(job)
            except KeyError:
                self._time_index[time_to_start] = [job]
            if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                logger.debug('%s execution %s was inserted at time (%s). Current time len=%s' % (JOB_EXECUTION_FAST_INDEX, job.get_printable_name(), time_to_start, len(self._time_index[time_to_start])))
            # as we did insert the job, let the job know it (NOTE: in the lock)
            job.set_current_index_at_epoch()
    
    
    # NOTE: need the check_n_actions lock
    def get_sorted_past_job_execution_index_seconds(self):
        with self.checks_n_actions_lock:
            now = int(time.time())
            past_seconds = [second for second in self._time_index.iterkeys() if second <= now]
            past_seconds.sort()  # only sort usefull seconds
        return past_seconds
    
    
    def get_jobs_at_second(self, second):
        # type: (Int) -> List[Check]
        with self.checks_n_actions_lock:
            return self._time_index.get(second, [])
    
    
    # Fast clean of job already checks that they are at the same seconds
    def clean_jobs_from_index_seconds(self, jobs_to_del, second):
        # type: (List[Check], int) -> None
        with self.checks_n_actions_lock:
            # Let the job know they are no more indexed
            for job in jobs_to_del:
                job.unset_current_index_at_epoch()
            
            jobs_at_t = self._time_index[second]
            # Now we did get jobs, clean the index
            _nb_deleted_jobs = len(jobs_to_del)
            if len(jobs_at_t) == _nb_deleted_jobs:  # we did consume all jobs, we can clean this entry directly
                del self._time_index[second]  # FASTER than remove jobs one by one in a list
                if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                    logger.debug('%s second %s was totally cleaned (all %s remaining %ss are taken)' % (JOB_EXECUTION_FAST_INDEX, second, _nb_deleted_jobs, self._job_type_str))
            else:  # there are still some, just remove what we did take
                for job in jobs_to_del:
                    jobs_at_t.remove(job)
                if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                    logger.debug('%s second %s is remaining (%s %ss are taken / %s remaining)' % (JOB_EXECUTION_FAST_INDEX, second, _nb_deleted_jobs, self._job_type_str, len(jobs_at_t)))
    
    
    def delete_job_from_index_seconds(self, job):
        # type: (Check) -> None
        with self.checks_n_actions_lock:
            previous_index_epoch = job.get_current_index_at_epoch()
            if previous_index_epoch is not None:  # if it already as in the index, try to remove it
                try:
                    self._time_index[previous_index_epoch].remove(job)
                except (KeyError, ValueError):
                    logger.error('The %s %s was expected to be indexed at time %s but is is not (new time=%s)' % (self._job_type_str, job.get_printable_name(), previous_index_epoch, job.get_time_to_be_start()))
                # No more in the index, can clean he job entry
                job.unset_current_index_at_epoch()
    
    
    # We can have bugs in the job indexing: the some jobs can stay in index and not
    # be deleted (bug fix soon, 20 janv 2021), but what ever, if it happen again, we must detect/correct it
    # if possible. So we will look at all past time, and delete zombie jobs we are founding
    # NOTE: as we will lookup lot of jobs, we should not run every second
    def cleanup_old_forgotten_zombies(self, scheduling_chapter):
        # type: (str) -> None
        start = time.time()
        nb_clean = 0
        nb_bad_epoch = 0
        
        _log_prefix = '%s %s' % (scheduling_chapter, JOB_EXECUTION_FAST_INDEX)
        
        with self.checks_n_actions_lock:
            after_lock_aquired = time.time()
            now = int(time.time())
            epochs = list(self._time_index.keys())  # we MUST have a copy here, NOT a generator (yes Py3, I talk about you)
            # Only look at past epochs, as future epochs cannot have zombie, or at least, we will clean them in the future
            past_epochs = [epoch for epoch in epochs if epoch <= now - 1]  # now - 1 so we did remove zombies from here
            if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                logger.debug('%s Looking for %s time index for %ss (on a total of %s seconds)' % (_log_prefix, len(past_epochs), self._job_type_str, len(epochs)))
            for epoch in past_epochs:
                jobs_at_epoch = self._time_index[epoch]  # type: List[Check]
                if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                    logger.debug('%s [now=%s] LOOK AT CLEANING %s, that have %s %ss' % (_log_prefix, now, epoch, len(jobs_at_epoch), self._job_type_str))
                
                nb_jobs_at_epoch = len(jobs_at_epoch)
                
                if nb_jobs_at_epoch == 0:  # already void entry, can freely clean it
                    if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                        logger.debug('%s [now=%s] The second %s was already void, cleaning the second entry' % (_log_prefix, now, epoch))
                    del self._time_index[epoch]
                    continue  # next epoch
                # Not void, look if need clean
                to_del = [job for job in jobs_at_epoch if job.is_zombie()]  # get ALL zombies
                nb_to_del = len(to_del)
                nb_clean += nb_to_del  # keep a trace for the final log
                if nb_to_del == 0:  # no zombie, we are clean
                    if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                        logger.debug('%s [now=%s] The second %s is clean' % (_log_prefix, now, epoch))
                    continue
                nb_bad_epoch += 1  # this epoch was rotten, we are sure now
                if nb_to_del == nb_jobs_at_epoch:  # ALL the time is zombie, delete the time entry
                    if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                        logger.debug('%s [now=%s] The second %s is full of zombies (%s) cleaning the second entry' % (_log_prefix, now, epoch, nb_to_del))
                    del self._time_index[epoch]
                    continue  # next epoch
                if LOG_SCHEDULER_JOB_EXECUTION_FAST_INDEX:
                    logger.debug('%s [now=%s] The second %s have some zombies (%s) cleaning the second entry (total this second=%s)' % (_log_prefix, now, epoch, nb_to_del, len(jobs_at_epoch)))
                # There are some jobs that are zombies
                for job in to_del:
                    jobs_at_epoch.remove(job)
        
        # Now log things
        lock_aquired_time = after_lock_aquired - start
        total_time = time.time() - start
        if nb_clean != 0:
            logger.error('%s [PERF: lock aquired in %.3fs, done in %.3fs] Did remove %s forgotten execution from %s past seconds indexed (total number of seconds inside=%s)' % (
                _log_prefix, lock_aquired_time, total_time, nb_clean, nb_bad_epoch, len(epochs)))
        else:
            logger.info('%s [PERF: lock aquired in %.3fs, done in %.3fs] No job to remove (total number of seconds inside=%s) ' % (_log_prefix, lock_aquired_time, total_time, len(epochs)))
