#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

"""
 This is the class of the dispatcher. Its role is to dispatch
 configurations to other elements like schedulers, reactionner,
 pollers, receivers and brokers. It is responsible for high availability part. If an
 element dies and the element type has a spare, it sends the config of the
 dead one to the spare
"""

import time
import random
import threading
from numbers import Number

from .satellitelink import SatelliteLink
from .log import LoggerFactory

logger = LoggerFactory.get_logger()
logger_perf = logger.get_sub_part(u'PERF', part_name_size=4)
logger_is_alive = logger.get_sub_part(u'IS ALIVE CHECK')
logger_dispatch = logger.get_sub_part(u'DISPATCH')
logger_configuration = logger_dispatch.get_sub_part(u'CONFIGURATION')

# Typing
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, List, NoReturn, Dict
    from shinken.daemons.arbiterdaemon import Arbiter
    from shinken.objects.config import Config
    from .configuration_incarnation import ConfigurationIncarnation
    from .objects.realm import Realm, Realms
    from .pollerlink import PollerLinks
    from .reactionnerlink import ReactionnerLinks
    from .receiverlink import ReceiverLinks
    from .brokerlink import BrokerLinks
    from .schedulerlink import SchedulerLinks
    from .arbiterlink import ArbiterLinks

# Always initialize random :)
random.seed()

# CHAPTER_INITIAL_DAEMONS_CHECK = get_chapter_string('INITIAL DAEMONS CHECK')
logger = LoggerFactory.get_logger()
logger_initial_daemon_check = logger.get_sub_part('INITIAL DAEMONS CHECK')


# Dispatcher Class
class Dispatcher:
    pollers = None  # type: Optional[PollerLinks]
    reactionners = None  # type: Optional[ReactionnerLinks]
    receivers = None  # type: Optional[ReceiverLinks]
    schedulers = None  # type: Optional[SchedulerLinks]
    brokers = None  # type: Optional[BrokerLinks]
    arbiters = None  # type: Optional[ArbiterLinks]
    realms = None  # type: Optional[Realms]
    
    
    # Load all elements, set them as not assigned
    # and add them to elements, so loop will be easier :)
    def __init__(self, conf, arbiter, arbiter_trace, configuration_incarnation):
        # type: (Config, Arbiter, Dict, ConfigurationIncarnation) -> NoReturn
        self.arbiter = arbiter
        self.arbiter_trace = arbiter_trace
        # Pointer to the whole conf
        self.conf = conf
        self.realms = conf.realms  # type: List[Realm]
        # Direct pointer to important elements for us
        self.configuration_incarnation = configuration_incarnation
        self.configuration_dispatch__initial_daemons_check__max_duration = self.conf.configuration_dispatch__initial_daemons_check__max_duration  # type: int
        
        for sat_type in ('arbiters', 'schedulers', 'reactionners', 'brokers', 'receivers', 'pollers'):
            setattr(self, sat_type, getattr(self.conf, sat_type))
            
            # for each satellite, we look if current arbiter have a specific
            # satellitemap value setted for this satellite if so, we give this map to
            # the satellite (used to build satellite URI later)
            if arbiter is None:
                continue
            
            key = sat_type[:-1] + '_name'  # i.e: schedulers -> scheduler_name
            for satellite in getattr(self, sat_type):
                sat_name = getattr(satellite, key)
                satellite.set_arbiter_satellitemap(arbiter.satellitemap.get(sat_name, {}))
        
        # Some flag about dispatch need or not
        self.first_dispatch_done = False
        
        # Compute for receivers & brokers the list of ALL realms they can be talk about so the receiver/broker
        # configuration will be able to have it
        for realm in self.realms:
            realm.compute_accessible_realms()
        
        # Prepare the satellites confs
        for satellite in self._get_satellites_but_not_the_schedulers():
            satellite.prepare_for_conf()
        
        # Some properties must be given to satellites from global
        # configuration, like the max_plugins_output_length to pollers
        parameters = {'max_plugins_output_length': self.conf.max_plugins_output_length}
        for poller in self.pollers:
            poller.add_global_conf_parameters(parameters)
        # and for all types, give some parameters too
        for s in self._get_satellites_but_not_the_schedulers():
            parameters = {'human_timestamp_log': self.conf.human_timestamp_log}
            s.add_global_conf_parameters(parameters)
    
    
    def load_new_configuration_incarnation(self, configuration_incarnation):
        self.configuration_incarnation = configuration_incarnation
    
    
    def _get_all_shards(self):
        shards = []
        for realm in self.realms:
            for shard in realm.shards.itervalues():
                shards.append(shard)
        return shards
    
    
    def _get_satellites_but_not_the_schedulers(self):
        r = []
        for _type in (self.pollers, self.reactionners, self.brokers, self.receivers):
            for daemon in _type:
                r.append(daemon)
        return r
    
    
    def _get_all_satellites(self):
        r = self._get_satellites_but_not_the_schedulers()
        r.extend(self.schedulers)
        return r
    
    
    def get_satellite_connections(self):
        sats_infos = []
        for sat in self._get_all_satellites():
            sat_info = {'address': sat.address,
                        'port'   : sat.port,
                        'type'   : sat.my_type,
                        'name'   : getattr(sat, sat.my_type + '_name'),
                        'proto'  : 'https' if sat.use_ssl else 'http',
                        }
            
            sat_info['uri'] = '%s://%s:%i/' % (sat_info['proto'], sat_info['address'], sat_info['port'])
            sats_infos.append(sat_info)
        
        return sats_infos
    
    
    # checks alive elements
    def check_alive(self):
        t0 = time.time()
        self.arbiter_trace['arbiter_time'] = time.time()
        _threads = []
        for elt in self._get_all_satellites():
            t = threading.Thread(None, target=elt.update_infos, name='do_one_update_infos')
            t.daemon = True
            t.start()
            _threads.append(t)
        # Now wait for all to finish before going too far
        for t in _threads:
            t.join()
        
        for arb in self.arbiters:
            # If not me, but not the master too
            if arb != self.arbiter and arb.spare:
                arb.update_infos()
        logger_perf.info('Time to check if all elements are alive: [ %.3f ]s' % (time.time() - t0))
    
    
    def print_initial_listing(self):
        for realm in self.realms:
            realm.print_initial_listing()
    
    
    def initial_daemons_check(self):
        every_one_else = self._get_all_satellites()  # all but the other arbiter spare
        
        logger_initial_daemon_check.info('The initial daemon check will start to check all %d daemons, for a maximum of %ds' % (len(every_one_else), self.configuration_dispatch__initial_daemons_check__max_duration))
        
        threads_checks_in_progress = []
        for satellite in every_one_else:
            t = threading.Thread(None, target=satellite.initial_daemon_check, name='initial_daemons_check::%s' % satellite.get_name(), args=(self.configuration_dispatch__initial_daemons_check__max_duration,))
            t.daemon = True
            t.start()
            threads_checks_in_progress.append((satellite, t))
        
        every_one_else_names = set([satellite.get_name() for satellite in every_one_else])
        
        # We will wait for max_initial_daemons_check_time (30s by default)
        start = time.time()
        last_progress_log = start
        progress_log_step = 10  # we will print progress every 10s
        while len(threads_checks_in_progress) > 0:
            now = time.time()
            # Time largely elapsed
            if abs(now - start) > self.configuration_dispatch__initial_daemons_check__max_duration + 10:  # abs= protect against back in time
                logger_initial_daemon_check.error('Some checks seems to be still running, skiping result.')
                break
            # Every 10s, we print progress, to show what is still running, so if the start is slow, we will quickly see
            # why
            if abs(now - last_progress_log) > progress_log_step:
                running_since = abs(now - start)
                last_progress_log = now
                still_in_progress_names = set(['%s' % satellite.get_name() for (satellite, thread) in threads_checks_in_progress])
                finished_names = list(every_one_else_names - still_in_progress_names)
                finished_names.sort()
                still_in_progress_names = list(still_in_progress_names)
                still_in_progress_names.sort()
                
                logger_initial_daemon_check.info('The initial daemon check is running since %ds:' % running_since)
                logger_initial_daemon_check.info('   - Finished checks : %s' % (', '.join(finished_names)))
                logger_initial_daemon_check.info('   - Still checking  : %s' % (', '.join(still_in_progress_names)))
            
            # Now really checks threads
            threads_checks_in_progress_copy = threads_checks_in_progress[:]  # make a copy
            for (satellite, thread) in threads_checks_in_progress_copy:
                if not thread.is_alive():
                    threads_checks_in_progress.remove((satellite, thread))
                    satellite.logger_initial_check.debug('-> initial check finished')
                    continue
            
            time.sleep(0.1)  # don't hammer CPU
        
        logger_initial_daemon_check.info('The initial daemons check did finish in: %.3fs' % (time.time() - start))
    
    
    def assert_inventories_dispatch(self):
        t0 = time.time()
        
        _threads = []
        # First broker inventory as they can take lot of time
        for realm in self.realms:
            t = threading.Thread(None, target=realm.assert_brokers_inventories, name='assert_brokers_inventories')
            t.daemon = True
            t.start()
            _threads.append(t)
        # Now wait for all to finish before going too far
        for t in _threads:
            t.join()
        
        _threads = []
        # Then receiver inventory as they can take lot of time
        for realm in self.realms:
            t = threading.Thread(None, target=realm.assert_receivers_inventories, name='assert_receivers_inventories')
            t.daemon = True
            t.start()
            _threads.append(t)
        # Now wait for all to finish before going too far
        for t in _threads:
            t.join()
        
        _threads = []
        # Then receiver host name mapping
        for realm in self.realms:
            t = threading.Thread(None, target=realm.assert_receivers_host_mapping, name='assert_receivers_host_mapping')
            t.daemon = True
            t.start()
            _threads.append(t)
        # Now wait for all to finish before going too far
        for t in _threads:
            t.join()
        
        logger_perf.info('Time to check (and if need send inventories): [ %.3f ]s' % (time.time() - t0))
    
    
    def _check_other_arbiters_dispatch(self):
        logger_configuration.debug('Checking that others arbiters configuration are sync with us')
        # We are a spare, we should not talk to the master
        if self.arbiter.spare:
            logger.debug('We are an arbiter spare so we do not force dispatch to the other arbiter.')
            return
        
        self.arbiter_trace['arbiter_time'] = time.time()
        # Check if the other arbiter has a conf, but only if I am a master
        for arb in self.arbiters:
            #  In all cases, ss the master, I assert that the spare are not running
            arb.do_not_run()
            
            # Check configuration, but only if we can connect to it
            if arb.is_alive() and arb.reachable and arb != self.arbiter:
                if not arb.is_managing_the_valid_configuration():
                    if not hasattr(self.conf, 'whole_conf_pack'):
                        logger.error('CRITICAL: the arbiter try to send a configuration but it is not a MASTER one?? Look at your configuration.')
                        continue
                    
                    arb.put_conf_to_arbiter_spare(self.conf, self.arbiter_trace)
    
    
    def _check_schedulers_dispatch(self):
        logger_configuration.debug('Checking that all shards are managed by a scheduler, and by master schedulers before spare ones')
        # We check for confs to be dispatched on alive scheds. If not dispatched, need dispatch :)
        # and if dispatch on a failed node, remove the association, and need a new dispatch
        for realm in self.realms:
            realm.check_schedulers_dispatch(self.first_dispatch_done)
    
    
    # Check if all active items are still alive
    def check_dispatch(self):
        before = time.time()
        # Check that the other arbiters
        self._check_other_arbiters_dispatch()
        logger_perf.info('Time to check if other arbiter (if exists) is managing the valid configuration [ %.3f ]s ' % (time.time() - before))
        
        before = time.time()
        self._check_schedulers_dispatch()
        logger_perf.info('Time to check if schedulers are managing the valid shards as expected [ %.3f ]s ' % (time.time() - before))
    
    
    # Imagine a world where... oh no, wait...
    # Imagine a master got the conf and the network is down a spare takes it (good :) ). Like the Empire, the master
    # strikes back! It was still alive! (like Elvis). It still got conf and is running! not good!
    # Bad dispatch: a link that has a conf but I do not allow this so I ask it to wait a new conf and stop kidding.
    def check_bad_dispatch(self):
        before = time.time()
        logger_configuration.debug('Checking daemons that manage a shard/configuration but should not have (like rogue daemons)')
        for realm in self.realms:
            realm.check_bad_dispatch(self.first_dispatch_done)
        logger_perf.info('Time to check that sleeping schedulers are really sleeping [ %.3f ]s' % (time.time() - before))
    
    
    # Manage the dispatch
    # REF: doc/shinken-conf-dispatching.png (3)
    def dispatch(self):
        logger_configuration.debug('Start to dispatch shards & satellites configurations')
        # Ok, we pass at least one time in dispatch, so now errors are True errors
        self.first_dispatch_done = True
        
        t0 = time.time()
        threads = []
        for realm in self.realms:
            thread = threading.Thread(None, target=realm.dispatch_schedulers, name='dispatch_schedulers_%s' % realm.get_name())
            thread.daemon = True
            thread.start()
            threads.append(thread)
        for thread in threads:
            thread.join()
        logger_perf.info('Time to send shards to all schedulers if they need [ %.3f ]s' % (time.time() - t0))
        
        arbiters_cfg = {}
        for arb in self.arbiters:
            arbiters_cfg[arb.id] = arb.give_satellite_cfg()
        
        # We put the satellites conf with the "new" way so they see only what we want
        t0 = time.time()
        threads = []
        for realm in self.realms:
            thread = threading.Thread(None, target=realm.dispatch_satellites, name='dispatch_satellites_%s' % realm.get_name(), args=(arbiters_cfg,))
            thread.daemon = True
            thread.start()
            threads.append(thread)
        for thread in threads:
            thread.join()
        logger_perf.info('Time to send configurations to all daemons (but not the schedulers) if need [ %.3f ]s' % (time.time() - t0))
    
    
    # interface to run the do_disable_previous_run_daemons into a thread
    def disable_previous_run_daemons(self, last_run_alive_daemons):
        t = threading.Thread(None, target=self._do_disable_previous_run_daemons, name='do_disable_previous_run_daemons', args=(last_run_alive_daemons,))
        t.daemon = True
        t.start()
    
    
    # Find the new disabled daemons and ask them to stop the work
    # They are now disabled, we don't need their work anymore
    def _do_disable_previous_run_daemons(self, last_run_alive_daemons):
        # convert the current daemon dot dict and str representation like stored in the retention
        current_daemons = {}
        for _type, daemons in self.conf.get_all_daemons().iteritems():
            current_daemons[_type] = [d.give_satellite_cfg() for d in daemons]
            for current_daemon in current_daemons[_type]:
                # generate a pseudo unique key depending on the daemon
                unique_key = u"%s:%s" % (current_daemon['address'], current_daemon['port'])
                current_daemon['unique_key'] = unique_key
        
        # we need to keep only the satellites that are not anymore in conf
        all_daemons_to_stop = []
        for _type, last_run_daemons in last_run_alive_daemons.iteritems():
            for last_run_daemon in last_run_daemons:
                # generate a pseudo unique key depending on the daemon
                unique_key = u"%s:%s" % (last_run_daemon['address'], last_run_daemon['port'])
                daemon_presently_running = any(unique_key == d['unique_key'] for d in current_daemons[_type])
                # the daemon was running previously but not anymore running
                if not daemon_presently_running:
                    all_daemons_to_stop.append(last_run_daemon)
        
        # We now need to convert the config (dict representation) into SatelliteLink and ask them to stop
        for daemon_to_stop in all_daemons_to_stop:
            # we need to have each key as list to instantiate a SatelliteLink
            for k, v in daemon_to_stop.iteritems():
                if v is True:
                    daemon_to_stop[k] = ['1', ]
                elif v is False:
                    daemon_to_stop[k] = ['0', ]
                elif isinstance(v, Number):
                    daemon_to_stop[k] = [unicode(v), ]
                if not isinstance(daemon_to_stop[k], list):
                    daemon_to_stop[k] = [v, ]
            sat_link = SatelliteLink(daemon_to_stop)
            sat_link.fill_default()
            sat_link.pythonize()
            sat_link.wait_new_conf()
            logger_configuration.info("DISABLED DAEMON  The element [%s] at [%s] was managed previously but have been disabled. We are asking it to go in sleep mode." % (daemon_to_stop['name'], sat_link.uri))
