#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import hashlib

from action import Action, ACTION_TYPES
from shinken.autoslots import AutoSlots
from shinken.log import logger
from shinken.property import IntegerProp, StringProp, FloatProp, BoolProp

""" TODO: Add some comment about this class for the doc"""


class EventHandler(Action):
    # AutoSlots create the __slots__ with properties and
    # running_properties names
    __metaclass__ = AutoSlots
    
    my_type = ACTION_TYPES.EVENTHANDLER
    
    properties = {
        'is_a'            : StringProp(default=ACTION_TYPES.EVENTHANDLER),
        'type'            : StringProp(default=''),
        '_in_timeout'     : StringProp(default=False),
        'status'          : StringProp(default=''),
        'exit_status'     : StringProp(default=3),
        'output'          : StringProp(default=''),
        'long_output'     : StringProp(default=''),
        't_to_go'         : StringProp(default=0),
        'original_t_to_go': IntegerProp(default=0),
        'execution_time'  : FloatProp(default=0),
        'u_time'          : FloatProp(default=0.0),
        's_time'          : FloatProp(default=0.0),
        'env'             : StringProp(default={}),
        'perf_data'       : StringProp(default=''),
        'sched_id'        : IntegerProp(default=0),
        'timeout'         : IntegerProp(default=10),
        'check_time'      : IntegerProp(default=0),
        'command'         : StringProp(default=''),
        'module_type'     : StringProp(default='fork'),
        'reactionner_tag' : StringProp(default='None'),
        'executor_id'     : StringProp(default='none'),
        'shell_execution' : BoolProp(default=0),
        'command_name'    : StringProp(default='MISSING_NAME'),
        'average_cpu_time': IntegerProp(default=0),
    }
    
    
    # id = 0  #Is common to Actions
    def __init__(self, command, id=None, ref=None, timeout=10, env={}, module_type='fork', reactionner_tag='None', shell_execution=False, command_name='MISSING_NAME'):
        super(EventHandler, self).__init__(id, command, command_name, timeout, module_type)
        
        self.is_a = ACTION_TYPES.EVENTHANDLER
        self.type = ''
        self.status = 'scheduled'
        
        self.ref = ref
        self._in_timeout = False
        self.timeout = timeout
        self.exit_status = 3
        self.command = command
        self.output = ''
        self.long_output = ''
        self.check_time = 0
        self.execution_time = 0
        self.u_time = 0
        self.s_time = 0
        self.cpu_time = 0
        self.perf_data = ''
        self.env = {}
        self.module_type = module_type
        self.reactionner_tag = reactionner_tag
        self.shell_execution = shell_execution
        
        if self.ref is not None:
            logger.debug('[EVENTHANDLER] creating the event handler %s for the command %s on the objects %s' % (self.id, command_name, self.ref.get_full_name()))
        else:
            logger.debug('[EVENTHANDLER] creating a copy object of the event handler %s (%s), the copy that will be take by a reactionner' % (id, command_name))
    
    
    # return a copy of the check but just what is important for execution
    # So we remove the ref and all
    def copy_shell(self):
        # We create a dummy check with nothing in it, just defaults values
        return self.minimal_copy_for_exec(EventHandler('', id=self.id, shell_execution=self.shell_execution, command_name=self.command_name))
    
    
    # <TMI!!>
    def get_outputs(self, out, max_plugins_output_length):
        elts = out.split('\n')
        # For perf data
        elts_line1 = elts[0].split('|')
        # First line before | is output
        self.output = elts_line1[0]
        # After | is perfdata
        if len(elts_line1) > 1:
            self.perf_data = elts_line1[1]
        # The others lines are long_output
        if len(elts) > 1:
            self.long_output = '\n'.join(elts[1:])
    
    
    # </TMI!!>
    
    def __str__(self):
        return "Check %s status:%s command:%s" % (self.id, self.status, self.command)
    
    
    # Action::get_hash
    def get_hash(self):
        if self.command_name:
            cmd = self.command_name
            if not isinstance(cmd, str):
                cmd = cmd.encode('utf8', 'ignore')
            return hashlib.md5(cmd).hexdigest()
        else:
            return None
    
    
    # Call by pickle to dataify the comment
    # because we DO NOT WANT REF in this pickleisation!
    def __getstate__(self):
        cls = self.__class__
        # id is not in *_properties
        res = {'id': self.id}
        for prop in cls.properties:
            if hasattr(self, prop):
                res[prop] = getattr(self, prop)
        
        if hasattr(self, 'worker_id'):
            res['worker_id'] = getattr(self, 'worker_id')
        if hasattr(self, 'executor_id'):
            res['executor_id'] = getattr(self, 'executor_id')
        return res
    
    
    # Inverted function of getstate
    def __setstate__(self, state):
        cls = self.__class__
        self.id = state['id']
        super(EventHandler, self).__init__(self.id, '', '', '', '')
        for prop in cls.properties:
            if prop in state:
                setattr(self, prop, state[prop])
        if not getattr(self, 'module_type', None):
            self.module_type = 'fork'
        # s_time, u_time, cpu_time are added between 1.2 and 1.4
        if not hasattr(self, 'cpu_time'):
            self.u_time = 0
            self.s_time = 0
            self.cpu_time = 0
        
        if not hasattr(self, 'worker_id'):
            self.worker_id = -1
        if not hasattr(self, 'executor_id'):
            self.executor_id = 'none'
