#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

import cPickle
import errno
import os
import shutil
import sys
import uuid
from ctypes import c_int
from multiprocessing import RLock, Value
from shinken.misc.type_hint import Dict

# Can not import shinkensolutions.system_tools because this class is use in logger


if sys.platform.startswith('win'):
    import tempfile
    
    SHARE_ITEM_SYNC_FILE_PATH = os.path.join(tempfile.gettempdir(), u'share_item')
    
    
    def set_ownership(path, user='shinken', group='shinken'):
        pass
    
    
    def rename_file(source, target):
        if os.path.exists(target):
            os.remove(target)
        os.rename(source, target)
else:
    SHARE_ITEM_SYNC_FILE_PATH = '/dev/shm/share_item'
    from pwd import getpwnam
    from grp import getgrnam
    
    
    def set_ownership(path, user='shinken', group='shinken'):
        try:
            uid = getpwnam(user).pw_uid
        except KeyError:
            raise OSError(errno.EINVAL, 'This user doesn\'t exists', user)
        try:
            gid = getgrnam(group).gr_gid
        except KeyError:
            raise OSError(errno.EINVAL, 'This group doesn\'t exists', group)
        os.chown(path, uid, gid)
    
    
    def rename_file(source, target):
        os.rename(source, target)

INTERNAL_KEY = ('_lock', '_current_version', '_loaded_version', '_data', '_in_update_context', '_instance_id', '_sync_file_path', '_is_clean')


class ShareItem(object):
    
    def __init__(self):
        self._lock = RLock()
        self._current_version = Value(c_int, 0)
        self._loaded_version = 0
        self._data = {}
        self._in_update_context = False
        self._instance_id = uuid.uuid4().hex
        share_type = type(self)
        share_type_name = share_type.__name__
        self._sync_file_path = os.path.join(SHARE_ITEM_SYNC_FILE_PATH, share_type_name, '%s.%s-%s' % (share_type.__module__, share_type_name, self._instance_id))
        if getattr(share_type, '_is_clean', False):
            sub_folder = os.path.join(SHARE_ITEM_SYNC_FILE_PATH, share_type_name)
            shutil.rmtree(sub_folder)
            share_type._is_clean = True
    
    
    def __str__(self):
        return 'ShareItem as %s [%s]' % (type(self).__name__, self._data)
    
    
    def __repr__(self):
        return self.__str__()
    
    
    def __getattr__(self, key):
        if key in INTERNAL_KEY:
            return super(ShareItem, self).__getattribute__(key)
        self._reload()
        try:
            return self._data[key]
        except KeyError:
            raise AttributeError('%s object has no attribute \'%s\'' % (type(self).__name__, key))
    
    
    def __setattr__(self, key, value):
        if key in INTERNAL_KEY:
            return super(ShareItem, self).__setattr__(key, value)
        self._reload()
        self._data[key] = value
        if not self._in_update_context:
            self._save()
    
    
    def __enter__(self):
        self._in_update_context = True
    
    
    def __exit__(self, _type, value, traceback):
        self._in_update_context = False
        self._save()
    
    
    def __del__(self):
        # The sync file may not exit so if del fail we skip this error
        try:
            os.remove(self._sync_file_path)
        except:
            pass
    
    
    def get_all_attr(self):
        # type: () -> Dict
        self._reload()
        return self._data
    
    
    def clear_all_attr(self):
        # type: () -> None
        self._data.clear()
        self._save()
    
    
    def _reload(self):
        with self._lock:
            if self._current_version.value > self._loaded_version:
                with open(self._sync_file_path, 'r') as data_file:
                    self._data = cPickle.load(data_file)
                self._loaded_version = self._current_version.value
    
    
    def _save(self):
        with self._lock:
            tmp_path = '%s.tmp' % self._sync_file_path
            if not os.path.exists(SHARE_ITEM_SYNC_FILE_PATH):
                os.mkdir(SHARE_ITEM_SYNC_FILE_PATH, 0o777)
                set_ownership(SHARE_ITEM_SYNC_FILE_PATH)
            
            sub_folder = os.path.join(SHARE_ITEM_SYNC_FILE_PATH, type(self).__name__)
            if not os.path.exists(sub_folder):
                os.mkdir(sub_folder, 0o777)
                set_ownership(sub_folder)
            
            with open(tmp_path, 'w') as sync_file:
                cPickle.dump(self._data, sync_file)
                sync_file.flush()
                os.fsync(sync_file.fileno())
            
            set_ownership(tmp_path)
            rename_file(tmp_path, self._sync_file_path)
            self._current_version.value += 1
