#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


import time

from shinken.http_client import HTTPExceptions
from shinken.log import logger, get_chapter_string, get_section_string
# Typing
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.property import BoolProp, IntegerProp, StringProp
from shinken.satellitelink import SatelliteLink, SatelliteLinks
from shinken.util import to_mb_size
from .configuration_incarnation import ConfigurationIncarnation, SCHEDULER_MANAGED_CONFIGURATION_KEYS

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from .objects.shard import Shard
    from .objects.realm import Realm

CHAPTER_CONFIGURATION = get_chapter_string('CONFIGURATION')


# IMPORTANT: When adding a new property, think about the retention in the arbiter SPARE!
class SchedulerLink(SatelliteLink):
    """Please Add a Docstring to describe the class here"""
    
    id = 0
    
    # Ok we lie a little here because we are a mere link in fact
    my_type = 'scheduler'
    
    # typing
    conf = None  # type: Optional[Shard]
    realm = None  # type: Optional[Realm]
    is_active = False
    
    properties = SatelliteLink.properties.copy()
    properties.update({
        'scheduler_name'    : StringProp(fill_brok=['full_status']),
        'port'              : IntegerProp(default='7768', fill_brok=['full_status']),
        'weight'            : IntegerProp(default='1', fill_brok=['full_status']),
        'skip_initial_broks': BoolProp(default='0', fill_brok=['full_status']),
    })
    
    running_properties = SatelliteLink.running_properties.copy()
    running_properties.update({
        'conf'                 : StringProp(default=None),
        'external_commands'    : StringProp(default=[]),
        'current_satellites'   : StringProp(default={'broker': [], 'reactionner': [], 'poller': [], 'receiver': []}),
        'managed_configuration': StringProp(default={SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID          : None,
                                                     SCHEDULER_MANAGED_CONFIGURATION_KEYS.IS_ACTIVE         : False,
                                                     SCHEDULER_MANAGED_CONFIGURATION_KEYS.SCHEDULER_NAME    : '',
                                                     SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIG_INCARN_DUMP: {}}),
        'is_active'            : BoolProp(default=False),  # should be running a shard or not
    })
    
    
    def get_name(self):
        return getattr(self, 'scheduler_name', 'unknown')
    
    
    def run_external_commands(self, commands):
        if self.con is None:
            self.create_connection()
        if not self.alive:
            return None
        logger.debug("[SchedulerLink] Sending %d commands" % len(commands))
        try:
            self.con.post('run_external_commands', {'cmds': commands})
        except HTTPExceptions as exp:
            self.con = None
            logger.debug(exp)
            return False
    
    
    def register_to_my_realm(self):
        self.realm.schedulers.append(self)
    
    
    def give_satellite_cfg(self):
        return {'port'               : self.port,
                'address'            : self.address,
                'name'               : self.get_name(),
                'instance_id'        : self.id,
                'active'             : self.is_managing_a_shard(),
                'use_ssl'            : self.use_ssl,
                'hard_ssl_name_check': self.hard_ssl_name_check,
                'timeout'            : self.timeout,
                'data_timeout'       : self.data_timeout,
                }
    
    
    def is_managing_a_shard(self):
        return self.conf is not None
    
    
    def get_managed_shard_id(self):
        if not self.is_managing_a_shard():
            return None
        return self.conf.get_id()
    
    
    # Some parameters can give as 'overridden parameters' like use_timezone
    # so they will be mixed (in the scheduler) with the standard conf sent by the arbiter
    def get_override_configuration(self):
        r = {}
        properties = self.__class__.properties
        for prop, entry in properties.iteritems():
            if entry.override:
                r[prop] = getattr(self, prop)
        return r
    
    
    def check_post_02_08_02_satellite_communication(self):
        b = hasattr(self, 'managed_configuration')
        if not b:
            logger.info('The scheduler %s load from the retention is too old, must wait for a new configuration from the arbiter master' % self.get_name())
        return b
    
    
    def _do_update_managed_configuration(self):
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return
            
            try:
                tab = self.con.get('get_currently_managed_configuration')
                
                self.logger_configuration.debug("manage -> %s" % tab)
                
                if tab == {}:
                    self.already_have_conf = False
                    self._reset_managed_configuration()
                    return
                
                self.already_have_conf = True
                # We can update our list now
                self.managed_configuration = tab
            except HTTPExceptions as exp:
                self.logger_configuration.warning("[%s][%s] Call to know what the daemon is currently managing did fail: %s" % (self._get_my_realm_name(), self.get_name(), exp))
    
    
    def is_managing_any_configuration(self):
        return self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIG_INCARN_DUMP] != {}
    
    
    def is_managing_a_inactive_configuration(self):
        return not self.managed_configuration['is_active']
    
    
    def is_managing_the_valid_configuration_incarnation(self):
        current_configuration_incarnation = self.get_current_managed_configuration()
        do_manage_same_configuration_incarnation = self.configuration_incarnation.is_equal(current_configuration_incarnation)
        return do_manage_same_configuration_incarnation
    
    
    def is_managing_valid_shard_id_and_incarnation(self, expected_shard_id):
        current_shard_id = self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID]
        # If it's not even the valid incarnation, it must be wrong
        if not self.is_managing_the_valid_configuration_incarnation():
            self.logger_configuration.debug(
                'COMPARING managed shards (%s/%s) with expected one (%s/%s) => the incarnation are not the same' % (
                    current_shard_id, self.get_current_managed_configuration(), expected_shard_id, self.configuration_incarnation))
            return False
        
        do_manage_valid_shard = expected_shard_id == current_shard_id
        self.logger_configuration.debug('COMPARING managed shards (%s/%s) with expected one (%s/%s) => %s' % (
            current_shard_id, self.get_current_managed_configuration(), expected_shard_id, self.configuration_incarnation, do_manage_valid_shard))
        return do_manage_valid_shard
    
    
    # Return True if the satellite said to managed a configuration
    def assert_manage_shard_id(self, realm_name, expected_shard_id):
        # type: (str, int) -> bool
        with self.big_lock:
            do_manage_valid_shard = self.is_managing_valid_shard_id_and_incarnation(expected_shard_id)
            if do_manage_valid_shard:
                return True
            
            current_shard_id = self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID]
            current_configuration_incarnation = self.get_current_managed_configuration()
            # Oups, not managing the good one!
            self.logger_configuration.get_sub_part(realm_name).error("MISMATCHED SCHEDULER    Scheduler did not managed the shard [%d/%s] as expected. Managed shard by the scheduler [shard_id=%s/configuration incarnation=%s]" % (
                expected_shard_id, self.configuration_incarnation, current_shard_id, current_configuration_incarnation))
            self.unassign_shard()
            return False
    
    
    def _get_conf_package(self, serialized_shard, override_conf, satellites_for_sched, activated):
        conf_package = {
            'name'                     : self.get_name(),
            'conf'                     : serialized_shard,
            'override_conf'            : override_conf,
            'modules'                  : self.modules,
            'satellites'               : satellites_for_sched,
            'instance_name'            : self.scheduler_name,
            'skip_initial_broks'       : self.skip_initial_broks,
            'realm'                    : self.realm.get_name(),
            'activated'                : activated,
            'spare'                    : self.spare,
            'arbiter_trace'            : self.arbiter_trace,
            'configuration_incarnation': self.configuration_incarnation,
        }
        
        return conf_package
    
    
    def _do_send_shard_to_scheduler(self, shard):
        # REF: doc/shinken-conf-dispatching.png (3)
        # REF: doc/shinken-scheduler-lost.png (2)
        override_conf = self.get_override_configuration()
        satellites_for_sched = self.realm.get_satellites_links_for_scheduler()
        serialized_shard = shard.get_serialized_configuration()
        shard_size = len(serialized_shard)
        self.arbiter_trace['arbiter_time'] = time.time()  # update arbiter time with the current time
        
        # Prepare the conf before sending it
        conf_package = self._get_conf_package(serialized_shard=serialized_shard,
                                              override_conf=override_conf,
                                              satellites_for_sched=satellites_for_sched,
                                              activated=True)
        # Active scheduler have a bonus property
        conf_package['shard_id'] = shard.get_id()
        
        t1 = time.time()
        is_sent = self.put_conf(conf_package)
        sent_time = time.time() - t1
        sent_speed = shard_size / sent_time
        
        if not is_sent:
            self.logger_configuration.error('SHARD NOT SENT   shard [%d] dispatching error' % shard.id)
            return False  # sent was failed
        self.logger_configuration.info("SHARD ASSIGNED TO SCHEDULER   Shard [%s/%s] is sent to scheduler, done in %.2fs (size=%.3fMB speed=%.3fMB/s)" % (
            shard.id, self.configuration_incarnation, sent_time, to_mb_size(shard_size), to_mb_size(sent_speed)))
        return True  # sent was OK
    
    
    def create_and_put_active_scheduler_configuration(self, shard):
        # type (Shard, dict, ConfigurationIncarnation) -> bool
        self.logger_configuration.debug('SENDING SHARD   Trying to assign shard %s/%s ' % (shard.id, self.configuration_incarnation))
        
        if self.conf is not None:
            if self.conf == shard:  # already the same
                return True
            # Not the same!
            self.logger_configuration.info('SCHEDULER ALREADY SET   The scheduler do not need a shard, it already manage shard %s' % (self.conf.get_id()))
            return False
        
        # Maybe we are already managing the shard (like a spare that take over the job)
        already_manage_this_shard = self.is_managing_valid_shard_id_and_incarnation(shard.get_id())
        if already_manage_this_shard:
            self.logger_configuration.info('SHARD SENT TO SCHEDULER   Dispatch OK of shard [%d/%s]: was not need as the scheduler is already managing this shard' % (
                shard.id, self.configuration_incarnation))
        else:
            is_sent = self._do_send_shard_to_scheduler(shard)
            if not is_sent:
                return  # already logged as error
        
        self.conf = shard
        self.is_active = True  # We are a scheduler with a shard
        
        # Let the shard now it's assigned to us
        shard.assign_to_scheduler(self)
        
        # We update all data for this scheduler
        self.managed_configuration = {
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID          : shard.get_id(),
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.IS_ACTIVE         : self.is_active,
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.SCHEDULER_NAME    : self.get_name(),
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIG_INCARN_DUMP: self.configuration_incarnation.dump_as_json(),
        }
        return True
    
    
    # AT START: we are checking if the scheduler do not have any configuration
    # from a previous run. If there is one, ask it to go to wait mode
    def assert_no_previous_run_configuration(self):
        if self.conf is not None:
            return
        
        if not self.reachable:
            return
        
        # Maybe we are an arbiter spare that take over the job, and the scheduler already manage a valid configuration_incarnation
        # so maybe we will skip to send the same shard again
        if self.is_managing_the_valid_configuration_incarnation():
            self.logger_configuration.info('ALREADY VALID INCARNATION  The scheduler have already the valid configuration incarnation %s. We are not asking it to drop its configuration.' % self.configuration_incarnation)
            return
        
        # At start, we already did update info
        if self.is_managing_any_configuration():
            self._go_drop_shard_and_sleep(is_error=False)  # at boot, it's just an INFO
    
    
    def wait_new_conf(self):
        r = super(SchedulerLink, self).wait_new_conf()
        self._reset_managed_configuration()
        self.is_active = False
        return r
    
    
    # Asking a scheduler to go to sleep, but maybe it's an ERROR, maybe not (at start it is normal, so INFO)
    def _go_drop_shard_and_sleep(self, is_error):
        old_shard_id = self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID]
        old_configuration_incarnation = self.get_current_managed_configuration()
        txt = "DAEMON WITH OLD SHARD   The scheduler have a shard from a previous arbiter (shard_id=%s/%s). We are asking it to save it in retention and wait for a new configuration." % (old_shard_id, old_configuration_incarnation)
        log_f = self.logger_configuration.error if is_error else self.logger_configuration.info
        log_f(txt)  # noqa => typing is lost here even if both methods are the same
        self.wait_new_conf()
    
    
    def assert_no_invalid_configuration(self):
        if self.conf is None and self.reachable:
            if not self.is_active:  # sleeping spare, should have an inactive conf
                if not self.is_managing_a_inactive_configuration():
                    err = "IDLE DAEMON WITH SHARD   The scheduler have a shard but should not. We are asking it to wait for a new configuration."
                    logger.error(err)
                    self.wait_new_conf()
                else:
                    self.logger_configuration.debug('The scheduler have a valid spare configuration')
            else:  # other unused scheduler
                if self.is_managing_any_configuration():
                    self._go_drop_shard_and_sleep(is_error=True)  # this one is not normal
    
    
    def unassign_shard(self):
        if self.conf is None:
            return
        self.logger_configuration.info('The scheduler is unassigned of the shard %s' % (self.conf.get_id()))
        self.conf.unassign_scheduler()
        self.conf = None
        self.is_active = False  # will be able to manage any conf
    
    
    def _reset_managed_configuration(self):
        self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID] = None
        self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.IS_ACTIVE] = False
        self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.SCHEDULER_NAME] = self.get_name()
        self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIG_INCARN_DUMP] = {}
    
    
    def get_current_managed_configuration(self):
        if not self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIG_INCARN_DUMP]:
            return None
        current_configuration_incarnation = ConfigurationIncarnation.create_from_json(self.managed_configuration[SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIG_INCARN_DUMP])
        return current_configuration_incarnation
    
    
    def create_and_put_inactive_scheduler_configuration(self, reason):
        if not self.alive or not self.reachable or self.conf is not None:
            return
        
        conf_package = self._get_conf_package(serialized_shard=None,  # no conf
                                              override_conf={},  # so no override
                                              satellites_for_sched={},  # no other daemons
                                              activated=False)  # you are inactive
        
        t1 = time.time()
        
        is_sent = self.put_conf(conf_package)
        
        # It can be:
        # * spare = spare daemon that just go as spare
        # * idle  = NOT spare daemon but need to sleep a bit until the dispatch can come
        conf_type_sent = 'spare' if self.spare else 'idle'
        
        if not is_sent:
            self.logger_configuration.error('%s SHARD NOT SENT   %s configuration dispatching error for scheduler' % (conf_type_sent.upper(), conf_type_sent))
            return
        self.logger_configuration.info('%s SCHEDULER SET AS %s   The scheduler is assigned a %s configuration (in %.2fs) because %s' % (
            get_section_string(conf_type_sent.upper()), conf_type_sent.upper(), conf_type_sent, (time.time() - t1), reason))
        self.is_active = False  # We are a scheduler without a shard
        # We save what we did send
        self.managed_configuration = {
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID          : 0,
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.IS_ACTIVE         : self.is_active,
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.SCHEDULER_NAME    : self.get_name(),
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIG_INCARN_DUMP: self.configuration_incarnation.dump_as_json()
        }
    
    
    def _update_managed_configuration(self):
        self._do_update_managed_configuration()
        self._refresh_current_satellites()
    
    
    def _refresh_current_satellites(self):
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return
            
            try:
                current_satellites = self.con.get('get_current_satellites')
                self.logger_configuration.debug("have satellites [%s]" % current_satellites)
                
                if current_satellites is None:  # HTTP fail? will be detected by ping
                    return
                self.current_satellites = current_satellites
            except HTTPExceptions as exp:
                self.logger_configuration.warning("Call to know what the daemon is currently managing did fail: %s" % exp)
    
    
    def assert_only_allowed_brokers(self, allowed_broker_names):
        to_remove = []
        current_brokers = self.current_satellites['broker']
        for current_broker in current_brokers:
            if current_broker not in allowed_broker_names:
                to_remove.append(current_broker)
        # Maybe there is no problem
        if len(to_remove) == 0:
            return
        
        # Maybe the scheduler did gone, will be detected in the ping
        if self.con is None:
            return
        
        self.logger_configuration.info('Removing old brokers %s from the scheduler as they are no more need currently.' % (','.join(to_remove)))
        
        try:
            self.con.post('satellites_to_remove', {'to_remove': {'broker': to_remove}})
            # Now we did remove some satellites, refresh the list so we are up to date
            # and we are OK with what the daemon REALLY did ^^
            self._refresh_current_satellites()
        except HTTPExceptions as exp:
            self.logger_configuration.warning("Cannot remove old brokers %s in the scheduler. We will retry in the next seconds: %s" % (','.join(to_remove), exp))


class SchedulerLinks(SatelliteLinks):
    name_property = "scheduler_name"
    inner_class = SchedulerLink
