#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import threading
import time
import uuid
import traceback
import pymongo
import urllib
import shinkensolutions.shinkenjson as json

from shinken.log import logger
from ..dao.def_items import ITEM_TYPE, ITEM_STATE
from pymongo.errors import BulkWriteError


# WARNING: si vous modifiez ces numéros il faut PREVOIR ABSOLUMENT UNE UPDATE EN DATABASE
# vous etes prevenus ^^
class ANALYZER_JOB_STATUS(object):
    PENDING = 1
    RUNNING = 2
    OK = 3
    ERROR = 4
    ERROR_TIMEOUT = 5
    ERROR_AUTH = 6


class AnalyzerJob(object):
    def __init__(self, object_type=ITEM_TYPE.HOSTS, object_uuid='', object_name='', object_state='stagging', job_parameters=None, analyzer_name='__NO_NAME__', overload_parameters={}, launch_batch_uuid=''):
        self.uuid = uuid.uuid4().get_hex()
        self.object_type = object_type
        self.object_uuid = object_uuid
        self.creation_time = int(time.time())
        self.status = ANALYZER_JOB_STATUS.PENDING
        self.start_time = 0
        self.end_time = 0
        self.analyze_result = {}
        self.analyze_log = ''
        self.analyzer_name = analyzer_name
        self.job_parameters = job_parameters
        self.object_name = object_name
        self.object_state = object_state
        self.launch_batch_uuid = launch_batch_uuid
        if self.job_parameters is None:
            self.job_parameters = {}
    
    
    def to_json(self):
        return {
            '_id'              : self.uuid,
            'object_type'      : self.object_type,
            'object_uuid'      : self.object_uuid,
            'creation_time'    : self.creation_time,
            'status'           : self.status,
            'start_time'       : self.start_time,
            'end_time'         : self.end_time,
            'analyze_result'   : self.analyze_result,
            'analyze_log'      : self.analyze_log,
            'job_parameters'   : self.job_parameters,
            'analyzer_name'    : self.analyzer_name,
            'object_name'      : self.object_name,
            'object_state'     : self.object_state,
            'launch_batch_uuid': self.launch_batch_uuid,
            
        }
    
    
    def from_json(self, data):
        self.uuid = data['_id']
        self.object_type = data['object_type']
        self.object_uuid = data['object_uuid']
        self.job_parameters = data['job_parameters']
        self.analyzer_name = data['analyzer_name']
        self.creation_time = data['creation_time']
        self.status = data['status']
        self.start_time = data['start_time']
        self.end_time = data['end_time']
        self.analyze_result = data['analyze_result']
        self.analyze_log = data['analyze_log']
        self.object_name = data['object_name']
        self.object_state = data['object_state']
        self.launch_batch_uuid = data['launch_batch_uuid']
    
    
    def set_error(self, error_txt, error_type=ANALYZER_JOB_STATUS.ERROR):
        self.status = error_type
        self.analyze_log = error_txt
        self.end_time = int(time.time())
    
    
    def start_analyze(self):
        self.status = ANALYZER_JOB_STATUS.RUNNING
        self.start_time = int(time.time())
    
    
    def set_analyze_result(self, result):
        self.end_time = int(time.time())
        self.analyze_result = result
        self.status = ANALYZER_JOB_STATUS.OK
    
    
    def set_object_name(self, object_name):
        self.object_name = object_name
    
    
    def has_job_parameter(self, part, property):
        return property in self.job_parameters.get(part, {})
    
    
    def get_job_parameter(self, part, property):
        return self.job_parameters[part][property]


class AnalyzerController(object):
    def __init__(self, app):
        self.app = app
        self.analyzers = [source for source in app.sources if source.my_type == 'analyzer']
        self._ = app._
        
        self.max_workers = 16
        
        # Main thread that will listen for order and spawn new worker threads
        self.main_thread = None
        self.must_run = True
        self.main_lock = threading.RLock()
        
        # Threads that will really scan the hosts
        self.worker_threads = {}
        
        # The results
        self.hosts = {}
        self.hosts_lock = threading.RLock()
        
        # We will have a bunch of locks for address, so we will have only one scan at the same time
        self.__in_progress_analyzes_by_address = {}
        self.__in_progress_analyzes_by_address_lock = threading.RLock()
        
        self.analyze_jobs_collection = None
    
    
    # Main thread that will reap jobs to do, and get back results to analyzers
    # it need a mongodb collection for keep this between restarts
    def _start_main_thread(self):
        self.analyze_jobs_collection = self.app.mongodb_db.analyze_jobs
        self.analyze_jobs_collection.ensure_index([('object_uuid', pymongo.ASCENDING)], name='object_uuid')
        self.analyze_jobs_collection.ensure_index([('object_uuid', pymongo.ASCENDING), ('creation_time', pymongo.ASCENDING)], name='object_uuid_creation_time')
        self.analyze_jobs_collection.ensure_index([('status', pymongo.ASCENDING), ('creation_time', pymongo.ASCENDING)], name='status_creation_time')
        
        self.analyze_jobs_collection.update_many({'status': ANALYZER_JOB_STATUS.RUNNING}, {"$set": {"status": ANALYZER_JOB_STATUS.ERROR}})
        logger.info('[analyzer] Starting analyzer main thread')
        self.main_thread = threading.Thread(None, target=self._main_loop, name='server-analyzer main thread')
        self.main_thread.daemon = True
        self.main_thread.start()
    
    
    def is_alive(self):
        return self.main_thread.is_alive()
    
    
    def _update_job_database_entry(self, job):
        job_entry = job.to_json()
        self.analyze_jobs_collection.save(job_entry)
    
    
    def _check_for_finished_threads(self):
        to_del = []
        for (job_uuid, thread_data) in self.worker_threads.iteritems():
            thread, job = thread_data
            if not thread.is_alive():
                thread.join(1)
                to_del.append((job_uuid, job))
        if len(to_del) != 0:
            logger.debug('[analyzer] %d worker threads did finished' % len(to_del))
        for (job_uuid, job) in to_del:
            logger.debug('[analyzer] The job %s is finish.' % job_uuid)
            # Allow a new thread to run
            del self.worker_threads[job_uuid]
            # And save the final job state in the database
            self._update_job_database_entry(job)
    
    
    def _spawn_new_worker_threads(self):
        all_jobs_to_do = list(self.analyze_jobs_collection.find({'status': ANALYZER_JOB_STATUS.PENDING}).sort([('creation_time', -1), ('object_name', -1)]).limit(self.max_workers))
        
        queue_len = len(all_jobs_to_do)
        if queue_len == 0:  # no job to do, bail out
            return all_jobs_to_do
        
        current_job_numbers = len(self.worker_threads)
        allowed_slots = max(0, self.max_workers - current_job_numbers)
        logger.debug('[analyzer] %d worker slots are available currently' % allowed_slots)
        if allowed_slots == 0:  # no more allowed slot, exit
            return []
        
        will_spawn = min(queue_len, allowed_slots)
        logger.debug('[analyzer] Will spawn this turn %d jobs' % will_spawn)
        for i in range(will_spawn):
            # Depop an element (FIFO) and also clean it from the queue
            # NOTE: mongo save a dict, we need a real object
            job_data = all_jobs_to_do.pop()
            job = AnalyzerJob()
            job.from_json(job_data)
            self._spawn_worker_thread(job)
    
    
    def _main_loop(self):
        t0 = time.time()
        while self.must_run:
            try:
                self._check_for_finished_threads()
                self._spawn_new_worker_threads()
                nb_jobs = len(self.worker_threads)
                # Display this log only each 3sec
                if nb_jobs and time.time() - t0 > 3:
                    logger.info('[analyzer] Number of jobs in progress: %d' % nb_jobs)
                    t0 = time.time()
                time.sleep(0.5)
            except:
                err = 'The Analyzer thread did fail on an unknown error: %s. Your daemon will exit.' % traceback.format_exc()
                logger.error(err)
                return
    
    
    def _spawn_worker_thread(self, job):
        t = threading.Thread(None, target=self._do_host_analyze, name='server-analyzer worker (job %s)' % job.uuid, args=(job,))
        t.daemon = True
        t.start()
        self.worker_threads[job.uuid] = (t, job)
    
    
    def find_analyzer(self, analyzer_name):
        for analyzer in self.analyzers:
            if analyzer.get_name() == analyzer_name:
                return analyzer
        return None
    
    
    def set_enabled(self, analyzer_name, to_is_enabled):
        # NOTE: Do not care about not found, the synchronizer did already did this job
        analyzer = self.find_analyzer(analyzer_name)
        did_change = analyzer.set_enabled(to_is_enabled)
        if not to_is_enabled:  # if we are disabling the analyzer, drop all pending jobs
            self._remove_all_pending_jobs(analyzer_name)
            # and force stop all running ones
            analyzer.stop_all_analyzes()
        return did_change
    
    
    # We need to look at our sources, and especially the analyzers ones, and when enabled, ask to them to init their modules
    def start_analyzers(self):
        self._start_main_thread()
        for analyzer in self.analyzers:
            module = analyzer.get_module()
            try:
                logger.info('Trying to start the analyzer module: %s ' % module.get_name())
                module.init_analyzer()
                analyzer.state = 'OK'
            except Exception, exp:
                logger.warning('The instance %s raised an exception %s. I disabled it, and set it to restart later' % (module.get_name(), str(exp)))
                self.app.modules_manager.did_crash(module, 'The instance %s raised an exception %s.' % (module.get_name(), str(exp)))
                analyzer.state = 'CRITICAL'
    
    
    def add_hosts_to_analyze(self, analyzer_name, host_states_str, host_uuids_str, overload_parameters, launch_batch_uuid):
        source_enabled = self.app.get_api_sources_from_backend().get('server-analyzer', {}).get('enabled', False)
        result = {'jobs': [], 'source_is_enabled': source_enabled}
        if not source_enabled:
            return result
        host_uuids = host_uuids_str.split(',')
        host_states = host_states_str.split(',')
        bulk_ops = self.analyze_jobs_collection.initialize_unordered_bulk_op()
        for index, host_uuid in enumerate(host_uuids):
            host_state = host_states[index]
            job = AnalyzerJob(object_uuid=host_uuid, analyzer_name=analyzer_name, object_state=host_state, job_parameters=overload_parameters, launch_batch_uuid=launch_batch_uuid)
            job_db_entry = job.to_json()
            bulk_ops.insert(job_db_entry)
            result['jobs'].append((host_uuid, job.uuid))
        if result['jobs']:
            try:
                bulk_ops.execute()
            except BulkWriteError as bwe:
                logger.error("[analyzer] save [%s] items in collection [%s] failed." % (len(host_uuids), 'analyze_jobs'))
                logger.error("BulkWriteError.details [%s]" % bwe.details)
                raise
        return result
    
    
    def get_analyze_jobs_result(self, launch_batch_uuid):
        request = {'launch_batch_uuid': launch_batch_uuid}
        return list(self.analyze_jobs_collection.find(request))
    
    
    # stop an analyze (by leaving the analyze page for example)
    def stop_analyze_batch(self, launch_batch_uuid):
        # get all jobs to to stop, delete the pending and set the running in error
        request_pending = {'launch_batch_uuid': launch_batch_uuid, 'status': ANALYZER_JOB_STATUS.PENDING}
        self.analyze_jobs_collection.delete_many(request_pending)
        # and set the running as error
        running_pending = {'launch_batch_uuid': launch_batch_uuid, 'status': ANALYZER_JOB_STATUS.RUNNING}
        self.analyze_jobs_collection.update_many(running_pending, {'$set': {'status': ANALYZER_JOB_STATUS.ERROR}}, upsert=True)
        return True
    
    
    def _get_inherited_item(self, item_state, item_type, item_id):
        item = None
        conn = self.app.get_synchronizer_syncui_connection()
        # NOTE: internal calls are protected
        params = urllib.urlencode({
            'private_key': self.app.get_private_http_key(),
        })
        conn.request("GET", "/internal/item_by_id/%s/%s/%s?%s" % (item_state, item_type, item_id, params))
        response = conn.getresponse()
        if response.status != 200:
            item = None
        else:
            buf = response.read()
            item = json.loads(buf)
            conn.close()
        return item
    
    
    # THREAD function, that will take a job and run it on the good analyzer
    # and will manage all the life of the job object. When it will finish, the main thread will detect it and
    # check&save the job entry in the database
    def _do_host_analyze(self, job):
        # We start to work, let the object know it and when
        job.start_analyze()
        
        # and before run, check if all our inputs are OK
        analyzer = self.find_analyzer(job.analyzer_name)
        if analyzer is None:
            job.set_error('Cannot find the analyzer %s' % job.analyzer_name)
            return
        
        host_uuid = job.object_uuid
        object_state = job.object_state
        if object_state == 'discovery':
            # this is a special case, we can't find the host in any cache, we just have the ip as uuid
            host = {
                '_id'      : host_uuid,
                'host_name': host_uuid,
                'address'  : host_uuid,
            }
        else:
            # currently in 2.5.0 only the staging is managed to launch the analyser (we cannot do the wroking area as we don't have the new elements with inheritance, should be ok on
            # v2.6.0)
            if object_state != ITEM_STATE.STAGGING:
                job.set_error('Only the staging is managed by the analyser')
                return
            
            # Try to look at the element in stagging, but if cannot find it, maybe the element is in
            # the working_area
            host = self._get_inherited_item(object_state, ITEM_TYPE.HOSTS, host_uuid)
            if host is None:
                host = self._get_inherited_item(ITEM_STATE.WORKING_AREA, ITEM_TYPE.HOSTS, host_uuid)
        
        if host is None:  # did disapear or wrong? no luck
            job.set_error('Cannot find the host %s' % host_uuid)
            return
        
        host_name = host.get('host_name', '')
        if not host_name:
            job.set_error('The host %s do not have a valid name' % host_uuid)
            return
        
        job.set_object_name(host_name)
        # Let the others know the job is in progress and the final host we did found
        self._update_job_database_entry(job)
        
        # We should avoid analyze the same host multiple time in parallel. We prefer to avoid to lock on the distant node
        # currently, so we are doing a in-synchronizer locking system
        address = host.get('address', '')
        if not address:
            job.set_error(self.app._('analyzer.no_address') % (host_uuid, host_name))
            return
        
        # We are managing a counter of current scan of the same address, lock so we have
        # only one at a time, but be sure to clean the entry when current number of instance is ==0
        while True:  # will be break as soon as the address is not in analyze
            with self.__in_progress_analyzes_by_address_lock:
                # Maybe there is no current analyze in progress, create the entry
                if address not in self.__in_progress_analyzes_by_address:
                    self.__in_progress_analyzes_by_address[address] = True
                    break
            # address is stil present, sleep a bit and retry
            # NOTE: sleep OUTSIDE the with lock!
            time.sleep(1)
        
        # All is well, we are sure to be the only analyze on this address
        try:
            analyzer.analyze_host(job, host)
        except Exception, exp:
            err = 'The analyzer %s did fail to analyze host %s: %s' % (analyzer.get_name(), host_uuid, traceback.format_exc())
            logger.error(err)
            job.set_error(err)
            return
        finally:
            # IMPORTANT: unset this address in the analyze in progress so other analyzes on the same address can be started
            with self.__in_progress_analyzes_by_address_lock:
                del self.__in_progress_analyzes_by_address[address]
    
    
    def _remove_all_pending_jobs(self, analyzer_name):
        self.analyze_jobs_collection.remove({'status': ANALYZER_JOB_STATUS.PENDING, 'analyzer_name': analyzer_name})
