#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.


import base64
import httplib
import ssl

from .source import Source, Sources
from ...dao.def_items import ITEM_STATE
from ...dao.dataprovider.dataprovider_mongo import DataProviderMongo
from shinken.property import StringProp


class Listener(Source):
    # NOTE: DO NOT SET id HERE. We want to use the same id as the Source class
    my_type = 'listener'
    
    running_properties = Source.running_properties.copy()
    running_properties.update({
        'type': StringProp(default='listener'),
    })
    
    
    def __init__(self, params, skip_useless_in_configuration=False):
        super(Listener, self).__init__(params)
        self.last_actions = {}
        self.last_actions_for_current_import = {}
    
    
    def get_name(self):
        return getattr(self, 'listener_name', '')
    
    
    def get_configuration_fields(self):
        return self.get_module().get_configuration_fields()
    
    
    # Before we go into the sources, we need to hook some properties
    # to behave really like a source, especially the Sources.find() will
    # need a source_name property, so fake it
    def prepare_before_being_merge_into_sources(self):
        self.source_name = self.get_name()
    
    
    def _get_my_module_class(self):
        from shinkensolutions.api.synchronizer.source.abstract_module.listener_module import ListenerModule
        return ListenerModule
    
    
    def get_module(self):
        module = getattr(self, 'module', None)
        if module is None:
            for inst in self.modules:
                # And only modules that inherit from Listener module are allowed to be listener
                if not isinstance(inst, self._get_my_module_class()):
                    continue
                self.module = inst
                break
            if self.module is None:
                raise Exception('We cannot find a valid module')
        return self.module
    
    
    # For listener, we are doing the source job, but also look at our modules
    def set_enabled(self, b):
        r = super(Listener, self).set_enabled(b)
        try:
            mod = self.get_module()
            if b:
                mod.start_listener()
            else:
                mod.stop_listener()
        except NotImplementedError:
            pass
        return r
    
    
    def restart(self):
        mod = self.get_module()
        mod.stop_listener()
        if self.enabled:
            mod.start_listener()
    
    
    # keep the history of the actions done until the next merge
    def _update_last_action(self, action_type, items_type, data):
        if data:
            if not items_type in self.last_actions_for_current_import:
                self.last_actions_for_current_import[items_type] = {}
            if not action_type in self.last_actions_for_current_import[items_type]:
                self.last_actions_for_current_import[items_type][action_type] = []
            self.last_actions_for_current_import[items_type][action_type].append(data)
    
    
    def get_listener_headers(self, login, password):
        _listener_headers = getattr(self, '_listener_headers', None)
        if _listener_headers is None:
            # base64 encode the username and password
            auth = base64.encodestring('%s:%s' % (login, password)).replace('\n', '')
            _listener_headers = {
                'Authorization': 'Basic %s' % auth,
                'Content-type' : 'application/json',
            }
        return _listener_headers
    
    
    def get_listener_connection(self, use_ssl, port):
        if use_ssl:
            if hasattr(ssl, 'SSLContext'):
                ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1)
                ssl_context.check_hostname = False
                ssl_context.verify_mode = ssl.CERT_NONE
                http_conn = httplib.HTTPSConnection('localhost', port, timeout=3, context=ssl_context)
            else:
                http_conn = httplib.HTTPSConnection('localhost', port, timeout=3)
        else:
            http_conn = httplib.HTTPConnection('localhost', port, timeout=3)
        return http_conn
    
    
    def remove_source_item(self, item_type, item_id):
        provider_mongo = DataProviderMongo(self.app.mongodb_db, self.app.database_cipher)
        source_item = provider_mongo.find_item_by_id(item_id, item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name())
        if source_item:
            self.modules[0].remove_source_item(item_type, source_item)
            provider_mongo.delete_item(source_item, item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name())
    
    
    def callback_about_delete_elements(self, items_type, data=None, import_needed=True):
        # We have deleted an item, let the UI know it
        self.unmerged_new_nb -= 1
        # keep the history of the actions done until the next merge
        self._update_last_action('delete', items_type, data)
        # and launch the merge
        if import_needed:
            self.set_force_import()
    
    
    def callback_about_new_elements(self, items_type, data=None):
        # We have a new item, let the UI know it
        self.unmerged_new_nb += 1
        # keep the history of the actions done until the next merge
        self._update_last_action('new', items_type, data)
        # and launch the merge
        self.set_force_import()
    
    
    def callback_about_update_elements(self, items_type, data=None):
        # keep the history of the actions done until the next merge
        self._update_last_action('update', items_type, data)
        # Launch the merge
        self.set_force_import()
    
    
    def do_import(self, need_types):
        super(Listener, self).do_import(need_types)
        # Total number of items in self.objects (sum of all items for each item type)
        self.last_actions = self.last_actions_for_current_import
        self.last_actions_for_current_import = {}
        output_warn = False
        if self.output.endswith(self.app._('validator.output_warning_elements')):
            output_warn = True
        if self.unmerged_new_nb > 1:
            self.output = self.app._('%s.output_successful' % self.my_type) % self.unmerged_new_nb
        elif self.unmerged_new_nb == 0:
            self.output = self.app._('%s.output_null_successful' % self.my_type)
        elif self.unmerged_new_nb == 1:
            self.output = self.app._('%s.output_one_successful' % self.my_type) % self.unmerged_new_nb
        elif self.unmerged_new_nb == -1:
            self.output = self.app._('%s.output_delete_one_successful' % self.my_type) % abs(self.unmerged_new_nb)
        elif self.unmerged_new_nb < -1:
            self.output = self.app._('%s.output_delete_successful' % self.my_type) % abs(self.unmerged_new_nb)
        _to_add = self.app._('common.save_in_database') % self.nb_elements
        self.output = '%s %s' % (self.output, _to_add)
        
        if output_warn:
            self.output = '%s<br/>%s' % (self.output, self.app._('validator.output_warning_elements'))
        
        self.source_import_output = self.output
        # Import done ; all items found in the source will be part of the next merge so we can reset the counter for the next import
        self.unmerged_new_nb = 0
    
    
    def done_import(self):
        done_import = super(Listener, self).done_import()
        done_import['output'] = self.output
        if self.last_actions:
            if self.my_type == "listener":
                nb_of_request = 0
                for items in self.last_actions['hosts'].items():
                    nb_of_request += len(items[1])
                done_import['run_detail'] = "%s %s" % (nb_of_request, self.app._('source.request') if nb_of_request <= 1 else self.app._('source.requests'))
            done_import['last_actions'] = self.last_actions.copy()
            self.last_actions = {}
        
        return done_import


class Listeners(Sources):
    name_property = 'listener_name'
    inner_class = Listener
