#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import logger
from shinken.misc.fast_copy import get_copier, NO_COPY
from shinken.misc.type_hint import Optional, Any
from ...dao.def_items import SERVICE_OVERRIDE


class SourceInfoProperty(object):
    UNIQUE_TYPE = 'UNIQUE'  # Single value field
    SET_TYPE = 'SET'  # Unique values list without order consideration
    ORDERED_TYPE = 'ORDERED'  # Unique values, but where order should be kept
    
    
    @staticmethod
    def from_dict(from_dict, separator):
        prop = SourceInfoProperty(separator)
        prop.__restore(from_dict)
        return prop
    
    
    def __str__(self):
        return '%s' % self.as_dict()
    
    
    def __eq__(self, other):
        return self.__key == other.__key and \
               self.__type == other.__type and \
               self.__separator == other.__separator and \
               self.__force == other.__force and \
               self.__value == other.__value and \
               self.__modif_auto == other.__modif_auto
    
    
    def __init__(self, separator, property_name='', field_type=UNIQUE_TYPE, force=False):
        self.__key = property_name
        self.__type = field_type
        self.__separator = separator
        self.__force = force
        self.__value = []
        self.__modif_auto = None
    
    
    def update(self, value, source):
        """
        :param value: str
        :param source: str
        :return: None
        """
        
        if self.__type == SourceInfoProperty.UNIQUE_TYPE:
            self.__value = [(source, value)]
        else:  # SET_TYPE
            old_v = set(v for _, v in self.__value)
            self.__value.extend([(source, v.strip()) for v in value.split(self.__separator) if v.strip() not in old_v])
    
    
    def key(self):
        # type: () -> unicode
        return self.__key
    
    
    def value(self):
        """
        :return: [tuple]
        """
        return self.__value
    
    
    def force(self):
        """
        :return: bool
        """
        return self.__force
    
    
    def type(self):
        return self.__type
    
    
    def modif_auto(self):
        return self.__modif_auto
    
    
    def as_dict(self):
        return {
            'property_key'       : self.__key,
            'property_type'      : self.__type,
            'property_value'     : [(x, list(y) if isinstance(y, set) else y) for (x, y) in self.__value],
            'property_force'     : self.__force,
            'property_modif_auto': self.__modif_auto,
        }
    
    
    def __restore(self, restore_dict):
        self.__key = restore_dict['property_key']
        self.__type = restore_dict['property_type']
        self.__value = restore_dict['property_value']
        self.__force = restore_dict.get('property_force', False)
        self.__modif_auto = restore_dict.get('property_modif_auto', None)
    
    
    def as_string(self, user, pretty=False):
        if not self.__value:
            return ''
        if self.__type == SourceInfoProperty.UNIQUE_TYPE:
            (source, _) = self.__value[0]
            return '' if source is None else source
        else:
            if pretty:
                source_names = set([a for (a, _) in self.__value])
                return_strings = []
                for source in source_names:
                    source_values = [value for (source_value, value) in self.__value if source_value == source]
                    if self.__key == SERVICE_OVERRIDE:
                        from ...front_end.helper import display_service_override_value
                        for i, value in enumerate(source_values):
                            source_values[i], _ = display_service_override_value(value, user=user)
                    return_strings.append('<tr><td class=\'source-name\'>%s%s</td><td><ul>%s</li></td></tr>' % (source, ' [FORCE] ' if self.__force else '', ''.join(['<li>%s</li>' % s for s in source_values])))
                return_text = '<table><tbody>%s</tbody></table>' % ''.join(return_strings)
            else:
                source_names = set([a for (a, _) in self.__value])
                return_strings = []
                for source in source_names:
                    source_values = [value for (source_value, value) in self.__value if source_value == source]
                    if self.__key == SERVICE_OVERRIDE:
                        from ...front_end.helper import display_service_override_value
                        for i, value in enumerate(source_values):
                            source_values[i], _ = display_service_override_value(value, user=user)
                    return_strings.append('%s%s<br>(%s)' % ('[FORCE] ' if self.__force else '', source, self.__separator.join(source_values)))
                return_text = '\n'.join(return_strings)
            return return_text
    
    
    def get_origin_of_value(self, value=None):
        # type: (Any) -> Optional[unicode]
        if self.__type == self.UNIQUE_TYPE or value is None:
            previous_value = self.value()
            if previous_value:
                return previous_value[0][0]
        return None
    
    
    @staticmethod
    def fast_copy_source_info_property(source_info_prop_to_copy, memo, dispatcher, ignore_warnings):
        y = memo.get(id(source_info_prop_to_copy))
        if y:
            return y
        
        new_prop = SourceInfoProperty(source_info_prop_to_copy.__separator, property_name=source_info_prop_to_copy.__key, field_type=source_info_prop_to_copy.__type, force=source_info_prop_to_copy.__force)
        new_prop.__modif_auto = source_info_prop_to_copy.__modif_auto
        
        cp = get_copier(source_info_prop_to_copy.__value, dispatcher)
        if cp == NO_COPY:
            new_prop.__value = source_info_prop_to_copy.__value
        elif cp:
            new_prop.__value = cp(source_info_prop_to_copy.__value, memo, dispatcher, ignore_warnings)
        elif not ignore_warnings:
            logger.warning('no dispatcher found for type [%s]. no copy done. ' % type(source_info_prop_to_copy.__value))
        
        memo[id(source_info_prop_to_copy)] = new_prop
        return new_prop
