#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.objects.config import Config
from .abstract_component import AbstractComponent
from .configuration_component import ConfigurationComponent
from .translate_component import TranslateComponent
from .logger_component import LoggerComponent
from ..dao.validators.rules import RulesComponent

try:
    from ordereddict import OrderedDict
except ImportError:
    from collections import OrderedDict

ComponentType = str


class COMPONENT(object):
    CONFIGURATION = 'configuration'
    TRANSLATE = 'translate'
    LOGGER = 'logger'
    RULE = 'rule'


class ComponentManager(object):
    
    def __init__(self):
        self.components = OrderedDict()  # type: OrderedDict[ComponentType, AbstractComponent]
        self._ready = False
    
    
    def _build(self, conf, host_name, http_start_time):
        # type: (Config , str,  int) -> None
        configuration_component = ConfigurationComponent(conf, host_name, http_start_time)
        self.components[COMPONENT.CONFIGURATION] = configuration_component
        
        logger_component = LoggerComponent()
        self.components[COMPONENT.LOGGER] = logger_component
        
        translate_component = TranslateComponent(configuration_component, logger_component)
        self.components[COMPONENT.TRANSLATE] = translate_component
        
        rule_component = RulesComponent()
        self.components[COMPONENT.RULE] = rule_component
    
    
    def init(self):
        for component_type, component in self.components.iteritems():
            component.init()
        
        self._ready = True
    
    
    def get_component(self, component_type):
        # type: (ComponentType) -> AbstractComponent
        if not self._ready:
            raise Exception('ComponentManager is not ready')
        return self.components[component_type]
    
    
    def is_ready(self):
        # type: () -> bool
        return self._ready
    
    
    def get_translate_component(self):
        # type: () -> TranslateComponent
        return self.get_component(COMPONENT.TRANSLATE)
    
    
    def get_configuration_component(self):
        # type: () -> ConfigurationComponent
        return self.get_component(COMPONENT.CONFIGURATION)


component_manager = ComponentManager()
