#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from .empty_object import EmptyObject
from .html.tooltip import Tooltip
from ...component.component_manager import component_manager
from ...dao.helpers import escape_XSS

if TYPE_CHECKING:
    from ...component.translate_component import TranslatePart
    from shinken.misc.type_hint import Union, NoReturn, Dict, Optional


class ShinkenObject(object):
    
    def __init__(self, translate=None):
        # type: (TranslatePart) -> NoReturn
        self.translate = translate if translate else component_manager.get_translate_component().translator()
        self.tooltip = EmptyObject()  # type: Union[EmptyObject,Tooltip]
        self.url_param_key = ''  # type: unicode
        self._data = {}  # type: Dict
        self._attribute = {}  # type: Dict
        self._class = ''  # type: unicode
        self._class_container = ''  # type: unicode
        self._class_content = ''  # type: unicode
    
    
    @staticmethod
    def wrap_in_middle_vertical_align_container(content):
        # type: (unicode) -> unicode
        return u'''<div class="shinken-display-table"><div class="shinken-display-table-cell">%s</div></div>''' % content
    
    
    # ****************************************************** GETTER SETTER  ***************************************
    def add_class_container(self, to_set):
        # type: (unicode) -> NoReturn
        if not to_set:
            return
        if self._class_container:
            self._class_container = u'%s %s' % (self._class_container, to_set)
        else:
            self._class_container = to_set
    
    
    def add_class_content(self, to_set):
        # type: (unicode)-> NoReturn
        if not to_set:
            return
        if self._class_content:
            self._class_content = u'%s %s' % (self._class_content, to_set)
        else:
            self._class_content = to_set
    
    
    def add_class(self, to_set):
        # type: (unicode) -> NoReturn
        if not to_set:
            return
        if self._class:
            self._class = u'%s %s' % (self._class, to_set)
        else:
            self._class = to_set
    
    
    def add_data(self, key, value):
        # type: (str, Union[str, int])-> NoReturn
        self._data[key] = value
    
    
    def add_attribute(self, key, value):
        # type: (str, str)-> NoReturn
        self._attribute[key] = value
    
    
    # ****************************************************** OTHER  ***************************************
    def add_tooltip(self):
        # type: () -> Tooltip
        self.tooltip = Tooltip()
        return self.tooltip
    
    
    def add_url_param_key(self, key_name):
        # type: (str) -> NoReturn
        self.url_param_key = key_name
    
    
    @staticmethod
    def escape_xss(value):
        return escape_XSS(value)
    
    
    # ****************************************************** MISC  ***************************************
    def _(self, key):
        # type: (unicode) -> unicode
        return self.translate.translate(key)
    
    
    @staticmethod
    def get_http_start_time():
        return component_manager.get_configuration_component().http_start_time
    
    
    # ****************************************************** HTML  ***************************************
    def get_object_tag_html(self, class_to_add='', extra_attribute=None):
        # type: ( str, Optional[Dict])-> str
        if self._class or class_to_add:
            class_to_add = "%s %s" % (self._class, class_to_add)
            class_to_add = class_to_add.strip()
        if class_to_add:
            return '''class="%s" %s %s''' % (class_to_add, self.get_data_html(), self.get_attribute_html(extra_attribute))
        return '''%s %s''' % (self.get_data_html(), self.get_attribute_html())
    
    
    def get_data_html(self):
        # type: ()-> str
        return ' '.join(['data-%s="%s"' % (k, v) for k, v in self._data.iteritems()])
    
    
    def get_attribute_html(self, extra_attribute=None):
        # type: (Optional[Dict])-> str
        _to_return = ['%s="%s"' % (k, v) for k, v in self._attribute.iteritems()]
        if extra_attribute:
            _to_return.extend(['%s="%s"' % (k, v) for k, v in extra_attribute.iteritems()])
        return ' '.join(_to_return)
    
    
    @staticmethod
    def get_html_tooltip(text='', message_type='', placement=""):
        # type: (str, str, str) -> str
        _attribute_type = '''shi-tip-type="%s"''' % message_type if message_type else ''
        _attribute_placement = '''shi-tip-placement="%s"''' % placement if placement else ''
        return '''onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" %s shi-tip-html="%s" %s''' % (_attribute_type, text.replace('&', '&amp;'), _attribute_placement)
    
    
    @staticmethod
    def get_html_lock():
        # type: () -> str
        return '''<td class="shinken-lock-cell"><span class="shinkon-lock"></span></td>'''
    
    
    def get_html(self):
        # type: () -> str
        return ''


class ShinkenObjectContainer(ShinkenObject):
    
    def __init__(self, html):
        super(ShinkenObjectContainer, self).__init__()
        self.html_to_return = html
    
    
    def get_html(self):
        # type: () -> str
        return self.html_to_return
