PROPERTY.CONST.VALIDATOR = {
    MESSAGES: {
        required       : _( "element.tooltip_required" ),
        url            : _( "element.tooltip_url" ),
        url_netloc     : _( "element.tooltip_url_netloc" ),
        notes_multi_url: _( "element.tooltip_missing_url" ),
        forbidden      : _( "element.tooltip_forbidden" ),
        replace        : _( "element.tooltip_replace" )
    },
    REGEXS  : {
        xss_exclude_characters                       : /[<>&"'/]/,
        xss_exclude_characters_url                   : /["' ]/,
        url                                          : /(ftp|https?):\/\/(\w+:{0,1}\w*@)?(\S+)(:[0-9]+)?(\/|\/([\w#!:.?+=&%@\-\/]))?/,
        note_url                                     : /^https?:\/\/(\w+(:\w+)?@)?[^"':/ ]+(:[0-9A-Za-z_$]+)?(\/|\/([\w#!:.?+=&%@\-\/$]+))?$/,
        restrict                                     : /[`~!$%^&*\"\'|<>?,()=/+\\]/,
        restrict_check                               : /[`~!%^&*\"\'|<>?,()=+]/, // The $ and / is allowed
        restrict_check_exclude                       : /[~!%^&\"\'|<>?()=+]/, // The $ and / is allowed
        restrict_host_group_name                     : /[~%^*\"\'<>?=]/,
        restrict_data                                : /[^0-9a-zA-Z_-]/,
        restrict_positive_integer                    : /^[1-9][0-9]*$/,
        restrict_positive_integer_or_zero            : /^[0-9][0-9]*$/,
        restrict_complex_expression_seps             : /[()&|!]/,
        restrict_filter_expression_seps              : /[()&|!"]/,
        restrict_positive_decimal_zero_to_one_hundred: /^(?:100(?:[.,]00?0?)?|0(?:[.,]\d\d?\d?)?|[1-9]\d?(?:[.,]\d\d?\d?)?)$/,
        restrict_filter_operators                    : /[&|]/,
        restrict_filter_modifiers                    : /[=*><]/g,
        compatible_filter_modifiers                  : /[*><]/g,
        key_data                                     : /[$]KEY[$]/,
        time_period_format                           : /^([0-9][0-9]:[0-9][0-9]\-[0-9][0-9]:[0-9][0-9]\,*){1,}$/
    },
    TYPE    : {
        SHINKEN_ELEMENT_NAME                         : {
            NAME : "shinken_element_name",
            VALUE: [ PROPERTY.CONST.CUSTOM_SELECT.DEFAULT, PROPERTY.CONST.CUSTOM_SELECT.NONE, 'shinken-service', 'shinken-host', 'shinken-cluster' ]
        },
        NO_INVISIBLE_CHAR                        : {
            NAME: "invisible_char"
        },
        NOT_EMPTY                                : {
            NAME: "not_empty"
        },
        NOT_NULL                                 : {
            NAME: "not_null"
        },
        RESULT_OK                                : {
            NAME: "result_ok"
        },
        REQUIRED                                 : {
            NAME: "required"
        },
        DFE_START                                    : {
            NAME   : "dfe_start",
            VALUE  : PROPERTY.CONST.DATA.IDENTIFIER_DATA_LOCAL,
            MESSAGE: _( 'validator_js.dfe_starts_not_by_' )
        },
        URL                                      : {
            NAME: "url"
        },
        NOTES_MULTI_URL                          : {
            NAME: "notes_multi_url"
        },
        RESTRICT                                 : {
            NAME   : "restrict",
            MESSAGE: '`~!$%^&*"\'|<>?,()=/+'
        },
        XSS_BLOCK                                : {
            NAME   : "xss_block",
            MESSAGE: '<>&"\'/'
        },
        XSS_BLOCK_URL                            : {
            NAME   : "xss_block_url",
            MESSAGE: "\" ' " + _( 'element.and_space_char' )
        },
        XSS_REPLACE                              : {
            NAME   : "xss_replace",
            MESSAGE: '<>&"\'/'
        },
        RESTRICT_CHECK                           : {
            NAME   : "restrict_check",
            MESSAGE: '`~!%^&*"\'|<>?,()=+'
        },
        RESTRICT_CHECK_EXCLUDE                   : {
            NAME   : "restrict_check_exclude",
            MESSAGE: '~!%^&"\'|<>?()=+'
        },
        RESTRICT_HOST_GROUP_NAME                 : {
            NAME   : "restrict_host_group_name",
            MESSAGE: '~%^*"\'<>?='
        },
        RESTRICT_DATA                            : {
            NAME: "restrict_data"
        },
        RESTRICT_DATA_DFE                        : {
            NAME: "restrict_data_dfe"
        },
        RESTRICT_POSITIVE_INTEGER                : {
            NAME: "restrict_positive_integer"
        },
        RESTRICT_POSITIVE_INTEGER_OR_ZERO        : {
            NAME: "restrict_positive_integer_or_zero"
        },
        RESTRICT_POSITIVE_DECIMAL_ZERO_TO_ONE_HUNDRED: {
            NAME: "restrict_positive_decimal_zero_to_one_hundred"
        },
        RESTRICT_INTEGER_WITH_MINUS_ONE          : {
            NAME     : "restrict_integer_with_minus_one",
            MINUS_ONE: '-1'
        },
        RESTRICT_HOST_NOTIFICATION_OPTIONS       : {
            NAME          : "restrict_host_notification_options",
            VALIDE_OPTIONS: ['w', 'c', 'd', 'u', 'r', 'f', 's', 'n']
        },
        RESTRICT_CHECK_NOTIFICATION_OPTIONS      : {
            NAME          : "restrict_check_notification_options",
            VALIDE_OPTIONS: ['d', 'w', 'u', 'c', 'r', 'f', 's', 'n']
        },
        RESTRICT_FLAP_DETECTION_OPTIONS_FOR_CHECK: {
            NAME            : "restrict_flap_detection_options_check",
            VALIDE_OPTIONS  : ['o', 'w', 'c', 'u'],
            REQUIRED_OPTIONS: ['o']
        },
        RESTRICT_FLAP_DETECTION_OPTIONS_FOR_HOST : {
            NAME            : "restrict_flap_detection_options_host",
            VALIDE_OPTIONS  : ['o', 'd', 'u'],
            REQUIRED_OPTIONS: ['o']
        },
        RESTRICT_ESCALATION_OPTIONS              : {
            NAME          : "restrict_escaltion_options",
            VALIDE_OPTIONS: ['d', 'u', 'r', 'w', 'c']
        },
        RESTRICT_DFE_VALUE                       : {
            NAME: 'restrict_dfe_value'
        },
        RESTRICT_TIMEPERIOD_DAYS                 : {
            NAME: "restrict_timeperiod_days"
        },
        SELECT_INPUT_WITH_CONDITION_FOCUS            : {
            NAME: "select_input_with_condition_focus"
        },
        SELECT_INPUT_WITH_CONDITION_NOT_FOCUS        : {
            NAME: "select_input_with_condition_not_focus"
        },
        RESTRICT_COMPLEX_EXPRESSION                  : {
            NAME: "restrict_complex_expression"
        },
        IP_RANGE                                 : {
            NAME: "ip_range_validator"
        },
        PORT_RANGE                               : {
            NAME: "port_range_validator"
        },
        MAXIMUM                                  : {
            NAME           : "maximum_date_validator",
            DATE_MAX_MINUTE: (5 * 365 + 2) * 1440,
            DATE_MAX_SECOND: ((5 * 365 + 2) * 1440) * 60
        }
    }
};

SHINKEN.TOOLS.SHINKEN_VALIDATOR = {
    init_with_name           : function ( validation_keys, value ) {
        var _prop = MANAGER.__instance_property_distributor.getElement( validation_keys.prop_name );
        return this.init( _prop, value, validation_keys.component_name );
    },
    init                     : function ( property, value, component_name ) {
        if ( !property ) {
            return new OBJECT.Messages();
        }
        
        var _validations_type = [];
        var name_property     = property.is_override ? property.override_prop_name : property.name;
        switch ( property.type ) {
            case PROPERTY.CONST.TYPES.STRING_INPUT:
            case PROPERTY.CONST.TYPES.DATA:
            case PROPERTY.CONST.TYPES.MULTI_LINE_OBJECT:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.NO_INVISIBLE_CHAR );
                break;
        }
        switch ( name_property ) {
            case PROPERTY.CONST.NAME.SERVICE_EXCLUDES:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_CHECK_EXCLUDE );
                break;
            case PROPERTY.CONST.NAME.SERVICE_DESCRIPTION:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_CHECK );
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.SHINKEN_ELEMENT_NAME );
                break;
            case PROPERTY.CONST.NAME.NAME:
                if ( MANAGER.__instance_element_distributor.current_element.type === OBJECT.CONST.ELEMENT.TYPE.CHECK_TPLS ) {
                    _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_CHECK );
                    _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.SHINKEN_ELEMENT_NAME );
                }
                else {
                    _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT );
                    _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.SHINKEN_ELEMENT_NAME );
                }
                break;
            case PROPERTY.CONST.NAME.HOST_NAME:
                switch ( MANAGER.__instance_element_distributor.current_element.type ) {
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_CLUSTER:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_HOST:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_TPLS:
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_HOST_GROUP_NAME );
                        break;
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_TPL_CLUSTER:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_TPL_HOST:
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_HOST_GROUP_NAME );
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_COMPLEX_EXPRESSION );
                        break;
                    default:
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT );
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.SHINKEN_ELEMENT_NAME );
                        break;
                }
                break;
            case PROPERTY.CONST.NAME.COMMAND_NAME:
            case PROPERTY.CONST.NAME.PREFIX:
            case PROPERTY.CONST.NAME.SYNC_NAME:
            case PROPERTY.CONST.NAME.BUSINESS_IMPACT_MODULATION_NAME:
            case PROPERTY.CONST.NAME.CONTACT_NAME:
            case PROPERTY.CONST.NAME.CONTACTGROUP_NAME:
            case PROPERTY.CONST.NAME.ESCALATION_NAME:
            case PROPERTY.CONST.NAME.NOTIFICATIONWAY_NAME:
            case PROPERTY.CONST.NAME.MACROMODULATION_NAME:
            case PROPERTY.CONST.NAME.RESULTMODULATION_NAME:
            case PROPERTY.CONST.NAME.TIMEPERIOD_NAME:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT );
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.SHINKEN_ELEMENT_NAME );
                break;
            case PROPERTY.CONST.NAME.HOSTGROUP_NAME:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_HOST_GROUP_NAME );
                break;
            case PROPERTY.CONST.NAME.DUPLICATE_FOREACH_NAME:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_DATA );
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_DATA_DFE );
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.DFE_START );
                break;
            case PROPERTY.CONST.NAME.DEFAULT_VALUE:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_DFE_VALUE );
                break;
            case PROPERTY.CONST.NAME.ADDRESS:
            case PROPERTY.CONST.NAME.PACK:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.XSS_BLOCK );
                break;
            case PROPERTY.CONST.NAME.NOTE_URL:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.XSS_BLOCK_URL );
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.URL );
                break;
            case PROPERTY.CONST.NAME.NOTE_MULTI_URL:
                switch ( component_name ) {
                    case "shinken-name-urlmulti":
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.XSS_BLOCK );
                        break;
                    case "shinken-url-urlmulti":
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.XSS_BLOCK_URL );
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.NOTES_MULTI_URL );
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.URL );
                        break;
                }
                break;
            case PROPERTY.CONST.NAME.NOTIFICATION_INTERVAL:
                switch ( MANAGER.__instance_element_distributor.current_element.type ) {
                    case OBJECT.CONST.ELEMENT.TYPE.ESCALATIONS:
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_INTEGER_WITH_MINUS_ONE );
                        break;
                    default:
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_POSITIVE_INTEGER_OR_ZERO );
                        property.validation_max_value = PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM.DATE_MAX_MINUTE;
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM );
                        break;
                }
                break;
            case PROPERTY.CONST.NAME.FIRST_NOTIFICATION_DELAY:
            case PROPERTY.CONST.NAME.LAST_NOTIFICATION_TIME:
            case PROPERTY.CONST.NAME.FIRST_NOTIFICATION_TIME:
                property.validation_max_value = PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM.DATE_MAX_MINUTE;
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM );
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_POSITIVE_INTEGER_OR_ZERO );
                break;
            case PROPERTY.CONST.NAME.CHECK_INTERVAL:
            case PROPERTY.CONST.NAME.RETRY_INTERVAL:
                property.validation_max_value = PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM.DATE_MAX_MINUTE;
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM );
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_POSITIVE_INTEGER );
                break;
            case PROPERTY.CONST.NAME.WARNING_THRESHOLD_CPU_USAGE:
            case PROPERTY.CONST.NAME.FRESHNESS_THRESHOLD:
                property.validation_max_value = PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM.DATE_MAX_SECOND;
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM );
            case PROPERTY.CONST.NAME.DEFINITION_ORDER:
            case PROPERTY.CONST.NAME.MAX_CHECK_ATTEMPTS:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_POSITIVE_INTEGER );
                break;
            case PROPERTY.CONST.NAME.SLA_WARNING_THRESHOLD:
            case PROPERTY.CONST.NAME.SLA_CRITICAL_THRESHOLD:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_POSITIVE_DECIMAL_ZERO_TO_ONE_HUNDRED );
                break;
            case PROPERTY.CONST.NAME.CHECK_RUNNING_TIMEOUT:
            case PROPERTY.CONST.NAME.TIMEOUT:
                property.validation_max_value = PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM.DATE_MAX_SECOND;
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM );
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_INTEGER_WITH_MINUS_ONE );
                break;
            case PROPERTY.CONST.NAME.HOST_NOTIFICATION_OPTIONS:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_HOST_NOTIFICATION_OPTIONS );
                break;
            case PROPERTY.CONST.NAME.SERVICE_NOTIFICATION_OPTIONS:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_CHECK_NOTIFICATION_OPTIONS );
                break;
            case PROPERTY.CONST.NAME.FLAP_DETECTION_OPTIONS:
                switch ( MANAGER.__instance_element_distributor.current_element.type ) {
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_CLUSTER:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_TPL_CLUSTER:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_HOST:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_TPL_HOST:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_TPLS:
                    case OBJECT.CONST.ELEMENT.TYPE.CLUSTERS:
                    case OBJECT.CONST.ELEMENT.TYPE.CLUSTER_TPLS:
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_FLAP_DETECTION_OPTIONS_FOR_CHECK );
                        break;
                    default:
                        if ( property.is_override ) {
                            _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_FLAP_DETECTION_OPTIONS_FOR_CHECK );
                        }
                        else {
                            _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_FLAP_DETECTION_OPTIONS_FOR_HOST );
                        }
                        break;
                }
                break;
            case PROPERTY.CONST.NAME.NOTIFICATION_OPTIONS:
                switch ( MANAGER.__instance_element_distributor.current_element.type ) {
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_CLUSTER:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_TPL_CLUSTER:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_HOST:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_ON_TPL_HOST:
                    case OBJECT.CONST.ELEMENT.TYPE.CHECK_TPLS:
                        _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_CHECK_NOTIFICATION_OPTIONS );
                        break;
                    default:
                        if ( property.is_override ) {
                            _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_CHECK_NOTIFICATION_OPTIONS );
                        }
                        else {
                            _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_HOST_NOTIFICATION_OPTIONS );
                        }
                        break;
                }
                break;
            // TODO A réactiver quand le ticket 3703 sera effectué
            //case PROPERTY.CONST.NAME.ESCALATION_OPTIONS:
            //    _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_ESCALATION_OPTIONS );
            //    break;
            case PROPERTY.CONST.NAME.TIMEPERIOD_DAYS.MONDAY:
            case PROPERTY.CONST.NAME.TIMEPERIOD_DAYS.TUESDAY:
            case PROPERTY.CONST.NAME.TIMEPERIOD_DAYS.WEDNESDAY:
            case PROPERTY.CONST.NAME.TIMEPERIOD_DAYS.THURSDAY:
            case PROPERTY.CONST.NAME.TIMEPERIOD_DAYS.FRIDAY:
            case PROPERTY.CONST.NAME.TIMEPERIOD_DAYS.SATURDAY:
            case PROPERTY.CONST.NAME.TIMEPERIOD_DAYS.SUNDAY:
            case PROPERTY.CONST.NAME.ADVANCED_TIME_PERIODS:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_TIMEPERIOD_DAYS );
                break;
            case PROPERTY.CONST.NAME.IP_RANGE:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.IP_RANGE );
                break;
            case PROPERTY.CONST.NAME.PORT_RANGE:
                _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.PORT_RANGE );
                break;
        }
        
        if ( !MANAGER.__instance_element_distributor.current_element.is_creation && property.is_required && property.component_btn_inheritance.state !== COMPONENT.CONST.BTN_STATE.ON ) {
            switch ( property.type ) {
                case PROPERTY.CONST.TYPES.MULTI_SELECT:
                case PROPERTY.CONST.TYPES.SINGLE_SELECT:
                case PROPERTY.CONST.TYPES.SINGLE_SELECT_WITH_INPUT:
                    _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.NOT_NULL );
                    break;
                default:
                    _validations_type.push( PROPERTY.CONST.VALIDATOR.TYPE.NOT_EMPTY );
                    break;
            }
        }
        return this._compute_validation( _validations_type, value, null, property );
    },
    _compute_validation         : function ( _validations_type, value_original, _to_return, property ) {
        if ( !_to_return ) {
            _to_return = new OBJECT.Messages();
        }
        var value = value_original;
        if ( value === "null" ) {
            value = "";
        }
        var _validation_type;
        var _validation_type_name;
        for ( var i = 0; i < _validations_type.length; i++ ) {
            _validation_type      = _validations_type[ i ];
            _validation_type_name = _validation_type.NAME;
            switch ( _validation_type_name ) {
                case PROPERTY.CONST.VALIDATOR.TYPE.SHINKEN_ELEMENT_NAME.NAME:
                    if ( SHINKEN.TOOLS.ARRAY.contains( PROPERTY.CONST.VALIDATOR.TYPE.SHINKEN_ELEMENT_NAME.VALUE, value_original ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'validator_js.shinken_name_forbidden' ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.REQUIRED.NAME:
                    if ( !value ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( PROPERTY.CONST.VALIDATOR.MESSAGES.required, _( 'element.this_field' ) ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.NOTES_MULTI_URL.NAME:
                    if ( !value ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, PROPERTY.CONST.VALIDATOR.MESSAGES.notes_multi_url );
                        return _to_return;
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.URL.NAME:
                    if ( value ) {
                        if ( PROPERTY.CONST.VALIDATOR.REGEXS.note_url.test( value ) ) {
                            if ( !MANAGER.__instance_macro_distributor.hasMacro( value ) && !DOM.Service.hasHrefHost( value ) ) {
                                _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, PROPERTY.CONST.VALIDATOR.MESSAGES.url_netloc );
                            }
                        }
                        else {
                            _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, PROPERTY.CONST.VALIDATOR.MESSAGES.url_netloc );
                        }
                        if ( SHINKEN.TOOLS.STRING.startsWith( value, 'http://' ) ) {
                            if ( value.length < 8 ) {
                                _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, PROPERTY.CONST.VALIDATOR.MESSAGES.url );
                            }
                        }
                        else if ( SHINKEN.TOOLS.STRING.startsWith( value, 'https://' ) ) {
                            if ( value.length < 9 ) {
                                _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, PROPERTY.CONST.VALIDATOR.MESSAGES.url );
                            }
                        }
                        else {
                            var _split_command = MANAGER.__instance_macro_distributor.splitCommand( value );
                            if ( _split_command.length && typeof _split_command[ 0 ] === 'object' ) {
                            
                            }
                            else {
                                _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, PROPERTY.CONST.VALIDATOR.MESSAGES.url );
                            }
                        }
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT.NAME:
                    if ( value && value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( PROPERTY.CONST.VALIDATOR.MESSAGES.forbidden, _validation_type.MESSAGE ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.XSS_BLOCK.NAME:
                    if ( value && value.match( PROPERTY.CONST.VALIDATOR.REGEXS.xss_exclude_characters ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( PROPERTY.CONST.VALIDATOR.MESSAGES.forbidden, _validation_type.MESSAGE ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.XSS_BLOCK_URL.NAME:
                    if ( value && value.match( PROPERTY.CONST.VALIDATOR.REGEXS.xss_exclude_characters_url ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( PROPERTY.CONST.VALIDATOR.MESSAGES.forbidden, _validation_type.MESSAGE ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.XSS_REPLACE.NAME:
                    if ( value && value.match( PROPERTY.CONST.VALIDATOR.REGEXS.xss_exclude_characters ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, SHINKEN.TOOLS.STRING.format( PROPERTY.CONST.VALIDATOR.MESSAGES.replace, _validation_type.MESSAGE ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_CHECK_EXCLUDE.NAME:
                    if ( value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict_check_exclude ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( PROPERTY.CONST.VALIDATOR.MESSAGES.forbidden, _validation_type.MESSAGE ) );
                    }
                    if ( value ) {
                        var _split = value.split( ',' );
                        for ( var j = 0, _size_j = _split.length; j < _size_j; j++ ) {
                            if ( !_split[ j ].trim() ) {
                                _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'element.tooltip_service_excludes_empty_part' ), j + 1 ) );
                            }
                        }
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_CHECK.NAME:
                    value = value.replace( PROPERTY.CONST.VALIDATOR.REGEXS.key_data, "" );
                    if ( value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict_check ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( PROPERTY.CONST.VALIDATOR.MESSAGES.forbidden, _validation_type.MESSAGE ) );
                    }
                    else if ( value.match( '[$]' ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, _( 'element.tooltip_dollar' ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_HOST_GROUP_NAME.NAME:
                    if ( value && value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict_host_group_name ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( PROPERTY.CONST.VALIDATOR.MESSAGES.forbidden, _validation_type.MESSAGE ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_DATA.NAME:
                    if ( value ) {
                        if ( value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict_data ) ) {
                            _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'element.tooltip_no_special_chars' ) );
                        }
                        else if ( !SHINKEN.TOOLS.STRING.isUpperCase( value ) ) {
                            _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, _( 'element.tooltip_not_in_upper_case' ) );
                        }
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_DATA_DFE.NAME:
                    if ( value ) {
                        if ( SHINKEN.TOOLS.ENCRYPTION.isProtectedField( value ) ) {
                            _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, SHINKEN.TOOLS.STRING.format( _( 'element.tooltip_dfe_key_encrypted' ), value ) );
                        }
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.DFE_START.NAME:
                    if ( value ) {
                        if ( SHINKEN.TOOLS.STRING.startsWith( value, PROPERTY.CONST.VALIDATOR.TYPE.DFE_START.VALUE ) ) {
                            _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, PROPERTY.CONST.VALIDATOR.TYPE.DFE_START.MESSAGE );
                        }
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_POSITIVE_INTEGER_OR_ZERO.NAME:
                    if ( value && !value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict_positive_integer_or_zero ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'element.tooltip_no_letter_2' ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_POSITIVE_DECIMAL_ZERO_TO_ONE_HUNDRED.NAME:
                    if ( value && !value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict_positive_decimal_zero_to_one_hundred ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'element.tooltip_no_letter_decimal_1_100' ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_POSITIVE_INTEGER.NAME:
                    if ( value && !value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict_positive_integer ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'element.tooltip_no_letter' ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.MAXIMUM.NAME:
                    if ( value && value > property.validation_max_value ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'element.tooltip_invalid_time' ), property.validation_max_value ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_INTEGER_WITH_MINUS_ONE.NAME:
                    if ( value && !value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict_positive_integer ) && value !== _validation_type.MINUS_ONE ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'element.tooltip_no_letter_except_minus_one' ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_CHECK_NOTIFICATION_OPTIONS.NAME:
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_HOST_NOTIFICATION_OPTIONS.NAME:
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_FLAP_DETECTION_OPTIONS_FOR_CHECK.NAME:
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_FLAP_DETECTION_OPTIONS_FOR_HOST.NAME:
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_ESCALATION_OPTIONS.NAME:
                    if ( value ) {
                        _to_return = this._validation_options( value, _validation_type );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_DFE_VALUE.NAME:
                    if ( value ) {
                        var _validator_dfe = App.DFE.validatorDFE( App.DFE.splitDFE( PROPERTY.CONST.NAME.DEFAULT_VALUE, value, true ) );
                        _to_return         = _validator_dfe._messages.hasError() ? _validator_dfe._messages : _validator_dfe._dfe_elem[ 0 ]._messages;
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_TIMEPERIOD_DAYS.NAME:
                    if ( value ) {
                        _to_return = this._validate_timeperiod_day( value );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.RESTRICT_COMPLEX_EXPRESSION.NAME:
                    if ( value ) {
                        _to_return.addMessages( this._validate_complex_expression( value ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.NOT_EMPTY.NAME:
                    if ( !SHINKEN.TOOLS.STRING.trimExtented( value ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'element.tooltip_not_empty' ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.NO_INVISIBLE_CHAR.NAME:
                    if ( SHINKEN.TOOLS.STRING.contains( value, SHINKEN.TOOLS.STRING.CONST.INVISIBLE_CHAR ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, SHINKEN.TOOLS.STRING.format( _( 'element.tooltip_no_invisible_characters' ), SHINKEN.TOOLS.STRING.replaceAll( value, SHINKEN.TOOLS.STRING.CONST.INVISIBLE_CHAR, "<span class='shinken-highlight-data-user'>\\u200b</span>" ) ) );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.NOT_NULL.NAME:
                    switch ( value ) {
                        case PROPERTY.CONST.CUSTOM_SELECT.DEFAULT:
                        case PROPERTY.CONST.CUSTOM_SELECT.DEFAULT_FORCED:
                        case PROPERTY.CONST.CUSTOM_SELECT.NONE:
                            _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'element.tooltip_not_empty' ) );
                            break;
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.SELECT_INPUT_WITH_CONDITION_FOCUS.NAME:
                    _to_return = SHINKEN.TOOLS.SHINKEN_VALIDATOR_COMPLEX._validate_filter_input_with_dropdown_focus( value );
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.SELECT_INPUT_WITH_CONDITION_NOT_FOCUS.NAME:
                    _to_return = SHINKEN.TOOLS.SHINKEN_VALIDATOR_COMPLEX._validate_filter_input_with_dropdown_not_focus( value );
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.IP_RANGE.NAME:
                    if ( SHINKEN.TOOLS.STRING.trimExtented( value ) ) {
                        _to_return = this._validate_ip_ranges_input( value );
                    }
                    break;
                case PROPERTY.CONST.VALIDATOR.TYPE.PORT_RANGE.NAME:
                    _to_return = this._validate_port_range( value );
                    break;
            }
        }
        return _to_return;
    },
    //**************************************   OPTIONS          **************************************//
    _validation_options      : function ( value, validation_type ) {
        var _to_return    = new OBJECT.Messages();
        var _split_values = value.split( ',' );
        var _current;
        var _size         = _split_values.length;
        
        for ( var i = 0; i < _size; i++ ) {
            _current = _split_values[ i ];
            if ( !_current ) {
                _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'element.tooltip_invalide_space_notification_options' ) );
            }
            else if ( !SHINKEN.TOOLS.ARRAY.contains( validation_type.VALIDE_OPTIONS, _current ) ) {
                _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'element.tooltip_invalide_notification_options' ), SHINKEN.TOOLS.STRING.cleanXss(_current) ) );
            }
        }
        for ( var i = 0; i < _size - 1; i++ ) {
            _current = _split_values[ i ];
            if ( _current && SHINKEN.TOOLS.ARRAY.contains( _split_values.slice( i + 1 ), _current ) ) {
                _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'element.tooltip_duplicate_notification_options' ), SHINKEN.TOOLS.STRING.cleanXss(_current) ) );
            }
        }
        if ( _to_return.hasMessage() ) {
            _to_return.footer = SHINKEN.TOOLS.STRING.format( _( 'element.tooltip_invalide_notification_options_footer' ), validation_type.VALIDE_OPTIONS.join( ',' ) );
        }
        if ( validation_type.REQUIRED_OPTIONS ) {
            for ( var i = 0, _size_i = validation_type.REQUIRED_OPTIONS.length; i < _size_i; i++ ) {
                _current = validation_type.REQUIRED_OPTIONS[ i ];
                if ( !SHINKEN.TOOLS.ARRAY.contains( _split_values, _current ) ) {
                    _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'element.tooltip_missing_required_notification_options' ), _current ) );
                }
            }
        }
        
        return _to_return;
    },
    //**************************************   IP RANGE     **************************************//
    _validate_ip_ranges_input: function ( value ) {
        var _to_return = new OBJECT.Messages();
        
        var _split = value.split( ' ' );
        for ( var i = 0, _size_i = _split.length; i < _size_i; i++ ) {
            this._validate_ip_ranges_part( _split[ i ], _to_return );
        }
        return _to_return;
    },
    _validate_ip_ranges_part : function ( value, messages ) {
        if ( !value ) {
            messages.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'help-discovery.validator-ip-too-much-space' ) );
            return;
        }
        var _split = value.split( '-' );
        var _size  = _split.length;
        switch ( _size ) {
            case 0:
                return;
            case 1:
                _split = value.split( '/' );
                _size  = _split.length;
                switch ( _size ) {
                    case 1:
                        this._validate_ip_range( value, messages );
                        return;
                    case 2:
                        this._validate_ip_range( _split[ 0 ], messages );
                        if ( !_split[ 1 ] ) {
                            messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), value ) );
                        }
                        else {
                            this._validate_ip_range_digit( _split[ 1 ], messages, 32, 16 );
                        }
                        return;
                    default:
                        messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), value ) );
                        break;
                }
                return;
            case 2:
                var _is_valid       = this._validate_ip_range( _split[ 0 ], messages );
                var _last_digit_int = parseInt( _split[ 0 ].split( "." )[ 3 ] );
                if ( !_split[ 1 ] ) {
                    messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), value ) );
                }
                else if ( _is_valid && _last_digit_int > _split[ 1 ] ) {
                    messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-not-inferior-to' ), _last_digit_int, "<span class='shinken-highlight-data-user shinken-between-bracket'>" + _split[ 1 ] + "</span>" ) );
                    return;
                }
                else {
                    this._validate_ip_range_digit( _split[ 1 ], messages );
                    return;
                }
                break;
            default:
                messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), value ) );
                break;
            
        }
    },
    _validate_ip_range       : function ( ip_range, messages ) {
        var _split = ip_range.split( "." );
        var _size  = _split.length;
        
        if ( _size !== 4 ) {
            messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), ip_range ) );
            return false;
        }
        var _to_return = true;
        for ( var i = 0; i < _size; i++ ) {
            if ( !_split[ i ] ) {
                messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), ip_range ) );
                _to_return = false;
            }
        }
        
        for ( var i = 0; i < _size; i++ ) {
            if ( !this._validate_ip_range_digit( _split[ i ], messages ) ) {
                _to_return = false;
            }
        }
        return _to_return;
    },
    _validate_ip_range_digit : function ( ip_range_digit, messages, high_limit, low_limit ) {
        if ( !high_limit ) {
            high_limit = 255;
        }
        if ( !SHINKEN.TOOLS.STRING.isInteger( ip_range_digit ) ) {
            messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), ip_range_digit ) );
            return false;
        }
        if ( parseInt( ip_range_digit ) > high_limit ) {
            messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-not-inferior-to' ), ip_range_digit, high_limit ) );
            return false;
        }
        else if ( low_limit && parseInt( ip_range_digit ) < low_limit ) {
            messages.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-not-superior-to' ), ip_range_digit, low_limit ) );
            return false;
        }
        return true;
    },
    //**************************************   PORT RANGE      **************************************//
    _validate_port_range     : function ( value ) {
        var _to_return = new OBJECT.Messages();
        var _split     = value.split( ',' );
        var _size      = _split.length;
        
        if ( _size == 0 ) {
            return _to_return;
        }
        
        for ( var i = 0; i < _size; i++ ) {
            var _rangeSplit = _split[ i ].split( '-' );
            var _rangeSize  = _rangeSplit.length;
            switch ( _rangeSize ) {
                case 1:
                    if ( _rangeSplit[ 0 ].length == 0 ) {
                        if ( _size == 1 ) {
                            return _to_return;
                        }
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, _( 'help-discovery.validator-ip-range-empty' ) );
                        break;
                    }
                    if ( !SHINKEN.TOOLS.STRING.isInteger( _rangeSplit[ 0 ] ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), _split[ i ] ) );
                        break;
                    }
                    this._validate_ip_range_digit( _rangeSplit[ 0 ], _to_return, 65535, 1 );
                    break;
                case 2:
                    if ( !SHINKEN.TOOLS.STRING.isInteger( _rangeSplit[ 0 ] ) || !SHINKEN.TOOLS.STRING.isInteger( _rangeSplit[ 1 ] ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), _split[ i ] ) );
                        break;
                    }
                    if ( parseInt( _rangeSplit[ 0 ] ) >= parseInt( _rangeSplit[ 1 ] ) ) {
                        _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-not-inferior-to' ), _rangeSplit[ 0 ], "<span class='shinken-highlight-data-user shinken-between-bracket'>" + _rangeSplit[ 1 ] + "</span>" ) );
                    }
                    this._validate_ip_range_digit( _rangeSplit[ 0 ], _to_return, 65535, 1 );
                    this._validate_ip_range_digit( _rangeSplit[ 1 ], _to_return, 65535, 1 );
                    break;
                default:
                    _to_return.addMessage( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( 'help-discovery.validator-ip-range-wrong-format' ), _split[ i ] ) );
                    break;
            }
        }
        return _to_return;
    },
    //**************************************   TIME PERIOD      **************************************//
    _validate_timeperiod_day : function ( value ) {
        var _to_return          = new OBJECT.Messages();
        var timeperiod_splitted = DATE_TIME.splitTimeperiodDay( value, _to_return );
        _to_return.addMessages( this._analyseTimeperiod( timeperiod_splitted ) );
        if ( SHINKEN.TOOLS.STRING.contains( value, SHINKEN.TOOLS.STRING.CONST.WHITE_SPACE ) ) {
            _to_return.addMessage( COMPONENT.CONST.STATUS.WARNING, _( "timeperiods.timeperiod_no_spaces_in_ranges" ) );
        }
        return _to_return;
    },
    _analyseTimeperiod       : function ( timeperiod_splitted ) {
        var _size      = timeperiod_splitted.length;
        var _to_return = [];
        var _current_message;
        for ( var i = 0; i < _size; i++ ) {
            _current_message = this._analyseTimeperiodRange( timeperiod_splitted[ i ] );
            if ( _current_message ) {
                _to_return.push( _current_message );
            }
        }
        
        if ( !_to_return.length && _size > 1 ) {
            var _current_range = timeperiod_splitted[ 0 ];
            var last_end_time  = _current_range[ 1 ];
            for ( var i = 1; i < _size; i++ ) {
                _current_range = timeperiod_splitted[ i ];
                if ( DATE_TIME.isValidRangeTimeperiod( last_end_time, _current_range[ 0 ] ) ) {
                    last_end_time = _current_range[ 1 ];
                }
                else {
                    _to_return.push( new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, _( "element.tooltip_timeperiod_invalid" ) ) );
                }
            }
        }
        
        return _to_return;
    },
    _analyseTimeperiodRange  : function ( range ) {
        var _size = range.length;
        if ( _size !== 2 ) {
            return new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, _( "element.tooltip_timeperiod_two_hour_missing" ) );
        }
        var _current_message;
        for ( var i = 0; i < _size; i++ ) {
            _current_message = this._analyseTimeperiodTime( range[ i ] );
            if ( _current_message ) {
                return _current_message;
            }
        }
        if ( !DATE_TIME.isValidRangeTimeperiod( range[ 0 ], range[ 1 ] ) ) {
            return new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( "element.tooltip_timeperiod_superior_hour" ), DATE_TIME.getTimeperiodLabel( PROPERTY.CONST.TIMEPERIOD.FORMAT_LABEL.RANGE, range ) ) );
        }
        
    },
    _analyseTimeperiodTime   : function ( time ) {
        if ( time.invalid ) {
            return new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, _( 'element.tooltip_timeperiod_time_format_invalid' ) );
        }
        if ( time.hour === "24" && time.min === "00" ) {
            return;
        }
        var _hour_is_valid = DATE_TIME.isValid( DATE_TIME.FORMAT.TYPE_VALIDATION.HOUR, time.hour );
        var _min_is_valid  = DATE_TIME.isValid( DATE_TIME.FORMAT.TYPE_VALIDATION.MIN, time.min );
        if ( !_hour_is_valid || !_min_is_valid ) {
            return new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, SHINKEN.TOOLS.STRING.format( _( "element.tooltip_timeperiod_time_invalid" ), DATE_TIME.getTimeperiodLabel( PROPERTY.CONST.TIMEPERIOD.FORMAT_LABEL.HOUR, time ) ) );
        }
    },
    //**************************************   COMPLEX EXPRESSIONS    ********************************//
    _validate_complex_expression: function ( value ) {
        var _error_messages = [];
        
        
        this._parse_complex_expression( _error_messages, value );
        return _error_messages;
    },
    _parse_complex_expression   : function ( error_messages, value ) {
        // No separator ; this is a operand, so no error there
        if ( value[ 0 ] === '!' ) {
            value = value.substring( 1 );
        }
        if ( !value.match( PROPERTY.CONST.VALIDATOR.REGEXS.restrict_complex_expression_seps ) ) {
            return;
        }
        
        var _in_part                = false;
        var _tmp                    = "";
        var _stacked_par            = 0;
        var _start_of_subexpression = true;
        
        for ( var i = 0; i < value.length; i++ ) {
            switch ( value[ i ] ) {
                case ',':
                case '|':
                case '&':
                case '+':
                    if ( _start_of_subexpression ) {
                        error_messages.push( new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, _( "validator.servicehosttpls_complex_expression_two_successive_operators" ) ) );
                    }
                    _start_of_subexpression = true;
                    if ( _in_part ) {
                        _tmp += value[ i ];
                    }
                    else {
                        _tmp = SHINKEN.TOOLS.STRING.trim( _tmp );
                        if ( _tmp ) {
                            this._parse_complex_expression( error_messages, _tmp );
                        }
                        _tmp = "";
                    }
                    break;
                case '(':
                    _start_of_subexpression = true;
                    _stacked_par += 1;
                    _in_part                = true;
                    _tmp                    = SHINKEN.TOOLS.STRING.trim( _tmp );
                    if ( _stacked_par === 1 && _tmp ) {
                        error_messages.push( new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, _( "validator.servicehosttpls_complex_expression_missing_operator" ) ) );
                    }
                    if ( _stacked_par > 1 ) {
                        _tmp += value[ i ];
                    }
                    break;
                case ')':
                    
                    _stacked_par -= 1;
                    if ( _stacked_par < 0 || _start_of_subexpression ) {
                        error_messages.push( new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, _( "validator.servicehosttpls_complex_expression_too_many_closing_parenthesis" ) ) );
                        _tmp = "";
                        break;
                    }
                    _start_of_subexpression = false;
                    
                    if ( _stacked_par === 0 ) {
                        _tmp = SHINKEN.TOOLS.STRING.trim( _tmp );
                        this._parse_complex_expression( error_messages, _tmp );
                        _in_part = false;
                        _tmp     = "";
                    }
                    else {
                        _tmp += value[ i ];
                    }
                    break;
                case '!':
                    if ( !_start_of_subexpression ) {
                        error_messages.push( new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, _( "validator.servicehosttpls_complex_expression_bang" ) ) );
                    }
                    break;
                default:
                    _start_of_subexpression = false;
                    _tmp += value[ i ];
            }
        }
        _tmp = SHINKEN.TOOLS.STRING.trim( _tmp );
        if ( _tmp ) {
            this._parse_complex_expression( error_messages, _tmp );
        }
        if ( _stacked_par > 0 ) {
            error_messages.push( new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, _( "validator.servicehosttpls_complex_expression_too_many_opening_parenthesis" ) ) );
        }
        if ( _start_of_subexpression ) {
            error_messages.push( new OBJECT.Message( COMPONENT.CONST.STATUS.ERROR, _( "validator.servicehosttpls_complex_expression_ends_with_operator" ) ) );
        }
    }
};


