#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import base64
import hashlib
import os

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.service import PREFIX_LINK_UUID, PREFIX_LINK_DFE_KEY
from shinken.runtime_stats.memory_stats import memory_stats
from shinken.util import force_memory_trimming
from shinkensolutions.service_override_parser import parse_service_override_property_list_errors, unparse_service_override
from ...business.item_controller.item_running_state import ItemContext, ACTIONS
from ...business.source.source_controller import make_item_hash
from ...business.sync_ui_common import syncuicommon
from ...dao import DataException
from ...dao.datamanagerV2 import FALLBACK_USER
from ...dao.dataprovider.dataprovider_mongo import DataProviderMongo
from ...dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE, WORKING_AREA_STATUS, SERVICE_OVERRIDE, HISTORY_ACTION, METADATA
from ...dao.helpers import get_inherited_without_default, unescape_XSS
from ...dao.items import get_item_instance
from ...dao.items.mixins import _get_item_from_link
from ...dao.transactions.transactions import DBTransaction
from ...synchronizerdaemon import transaction_protecter

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from ...synchronizerdaemon import Synchronizer

app = None  # type: Optional[Synchronizer]


def import_done():
    logger.debug('[import_done] start update sync-ui process data post import')
    warning_message = app.datamanagerV2.reload_after_import()
    syncuicommon.reset_configuration_stats_cache()
    return {'message': warning_message}


def reset_configuration_stats():
    syncuicommon.reset_configuration_stats_cache()
    return ''


def get_inherited_item(item_state, item_type, item_id):
    item = app.datamanagerV2.find_item_by_id(item_id=item_id, item_type=item_type, item_state=item_state)
    if item:
        item = get_inherited_without_default(syncuicommon.app.datamanagerV2, item, flatten_states=[item_state])
    else:
        app.abort(404, 'Item (%s-%s-%s)was not found' % (item_id, item_type, item_state))
    return item


def flatten_conf():
    item_states = app.request.GET.get('states', None)
    if not item_states:
        app.abort(500, 'flatten_conf need states param')
    
    logger.debug('[INTERNAL] call flatten_conf in internal with parameters [%s]' % (item_states))
    res = {}
    item_states = item_states.split(',')
    for item_type in DEF_ITEMS.iterkeys():
        res[item_type] = []
    
    items = app.datamanagerV2.find_merge_state_items(item_type=ITEM_TYPE.ELEMENTS, item_states=item_states)
    for item in items:
        item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
        res[item_type].append(item.get_raw_item(flatten_links=item_states))
    
    return res


def get_conf():
    item_states = app.request.GET.get('states', None)
    if not item_states:
        app.abort(500, 'get_existing_conf need states param')
    
    logger.debug('[INTERNAL] call get_conf in internal with parameters [%s]' % (item_states))
    res = {}
    item_states = item_states.split(',')
    for item_state in item_states:
        res[item_state] = {}
        for item_type in DEF_ITEMS.iterkeys():
            res[item_state][item_type] = []
    
    items = app.datamanagerV2.find_items(item_type=ITEM_TYPE.ELEMENTS, item_state=item_states)
    for item in items:
        item_state = METADATA.get_metadata(item, METADATA.STATE)
        item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
        res[item_state][item_type].append(item.get_raw_item(flatten_links=None, keep_links=False))
    
    return res


def get_whole_configuration():
    original_item_states = app.request.GET.get('states', [ITEM_STATE.PRODUCTION])
    FLATTEN_STATES = {
        ITEM_STATE.WORKING_AREA: (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING),
        ITEM_STATE.PRODUCTION  : (ITEM_STATE.PRODUCTION,),
        ITEM_STATE.PREPROD     : (ITEM_STATE.PREPROD,),
        'default'              : (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA),
    }
    keep_so_links_to_disabled = app.request.GET.get('keep_overrides_on_disabled_checks', '1').lower() in ('1', "true", "yes")
    item_types = app.request.GET.get('types', DEF_ITEMS.keys())
    
    if not original_item_states:
        app.abort(500, 'flatten_conf need states param')
    # if asking for 'propose' items we give propose in working area + staging items
    item_states = [ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA] if original_item_states == 'propose' else original_item_states
    
    if not isinstance(item_types, (list, tuple)):
        item_types = item_types.split(',')
    
    if not isinstance(item_states, (list, tuple)):
        item_states = item_states.split(',')
    
    # Search last specified states first
    item_states.reverse()
    
    # If we want changes, we need to apply diffs to each item and not fetch items in the 'changes' state
    if ITEM_STATE.CHANGES in item_states:
        diffs_to_apply = True
        for_arbiter = False
        item_states.remove(ITEM_STATE.CHANGES)
        if not keep_so_links_to_disabled:
            logger.warning("[get-whole-configuration] Since you are asking for changes you cannot also have the overrides on disabled checks removed")
    else:
        diffs_to_apply = False
        for_arbiter = True
    
    logger.debug('[INTERNAL] call get_whole_configuration original_item_states[%s] item_states[%s]' % (original_item_states, item_states))
    whole_configuration = {}
    for item_type in item_types:
        item_class = DEF_ITEMS[item_type]['table']
        if item_class not in whole_configuration:
            whole_configuration[item_class] = {}
        
        for item_state in item_states:
            flatten_links = FLATTEN_STATES.get(item_state, FLATTEN_STATES['default'])
            
            # Select propose to staging items if the original_item_states is propose
            optional_where = {'work_area_info.status': WORKING_AREA_STATUS.PROPOSED} if item_state == ITEM_STATE.WORKING_AREA and original_item_states == 'propose' else {}
            
            items = app.datamanagerV2.find_items(item_type=item_type, item_state=item_state, where=optional_where)
            for item in items:
                # removing 'submit_del_item' for propose item
                if original_item_states == 'propose' and item.get_work_area_status(ITEM_STATE.STAGGING) == WORKING_AREA_STATUS.PROPOSED and item['work_area_info'].get('submit_del_item', False):
                    continue
                
                if item['_id'] in whole_configuration[item_class]:
                    continue
                
                link_names_to_remove = []
                if diffs_to_apply and item_state != ITEM_STATE.NEW:
                    raw_item = item.apply_all_diffs(flatten_links=flatten_links)
                    if item.get(SERVICE_OVERRIDE) and not keep_so_links_to_disabled:
                        logger.warning('|get-whole configuration] For item %s-%s-%s, overrides on disabled checks will not be removed' % (item.get_state(), item.get_type(), item['_id']))
                elif item.get(SERVICE_OVERRIDE, None) and not keep_so_links_to_disabled:
                    # Remove service overrides pointing to disabled checks
                    for so_link in item[SERVICE_OVERRIDE]['links']:
                        if '@link' in so_link['check_link']:
                            check = _get_item_from_link(so_link['check_link'], flatten_links, item)
                            if not check.is_enabled():
                                link_names_to_remove.append('%s%s%s%s' % (PREFIX_LINK_UUID, so_link['check_link']['_id'], PREFIX_LINK_DFE_KEY, so_link.get('dfe_key', '-')))
                            if check.is_dfe() and 'dfe_key' not in so_link:
                                link_names_to_remove.append('%s%s%s%s' % (PREFIX_LINK_UUID, so_link['check_link']['_id'], PREFIX_LINK_DFE_KEY, so_link.get('dfe_key', '-')))
                        if so_link['check_link'].get('name', '').startswith('override-conflict-'):
                            link_names_to_remove.append('%s%s%s' % ('override-conflict-%s' % so_link['check_link'].get('name', ''), PREFIX_LINK_DFE_KEY, so_link.get('dfe_key', '-')))
                    
                    raw_item = item.get_raw_item(flatten_links=flatten_links, for_arbiter=for_arbiter)
                else:
                    raw_item = item.get_raw_item(flatten_links=flatten_links, for_arbiter=for_arbiter)
                whole_configuration[item_class][item['_id']] = raw_item
                whole_configuration[item_class][item['_id']].pop('work_area_info', None)
                whole_configuration[item_class][item['_id']].pop('last_modification', None)
                
                if link_names_to_remove and SERVICE_OVERRIDE in whole_configuration[item_class][item['_id']]:
                    overrides = parse_service_override_property_list_errors(whole_configuration[item_class][item['_id']][SERVICE_OVERRIDE])
                    service_overrides = overrides['parsed_overrides']
                    for name in link_names_to_remove:
                        service_overrides.pop(name, None)
                    whole_configuration[item_class][item['_id']][SERVICE_OVERRIDE] = unparse_service_override(service_overrides)
                
                # Here we have to escape the data save in XSS safe mode to get the real value
                for key, data in raw_item.iteritems():
                    if key.startswith('_') and key not in (u'_SE_UUID', u'_SE_UUID_HASH', u'_id', u'_SYNC_KEYS'):
                        whole_configuration[item_class][item['_id']][key] = unescape_XSS(data)
    for item_class in whole_configuration.iterkeys():
        whole_configuration[item_class] = whole_configuration[item_class].values()
    return whole_configuration


def apply_all_diffs():
    messages = []
    for item_type in DEF_ITEMS.iterkeys():
        logger.debug('[apply_all_diffs] Will apply diffs for [%s]' % (item_type))
        item_changes = app.datamanagerV2.find_items(item_type, ITEM_STATE.STAGGING, where={'have_change': True})
        for item_change in item_changes:
            raw_item_change = item_change.apply_all_diffs()
            _validation = syncuicommon.validator.validate(item_type, raw_item_change)
            if _validation['has_messages']:
                messages.extend(_validation['messages'])
            
            if not _validation['has_critical']:
                app.datamanagerV2.save_item(raw_item_change, user=None, item_type=item_type, item_state=ITEM_STATE.STAGGING)
    
    return {'data': messages}


def put_in_production():
    response = {'code': 200, 'msg': ''}
    user_id = app.request.GET.get('user_id', '')
    user = app.datamanagerV2.find_item_by_id(user_id, ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
    if not user:
        response['code'] = 500
        response['msg'] = 'user with id [%s] not found' % user_id
        return response
    if not user.is_admin():
        response['code'] = 403
        response['msg'] = 'The user %s is not a Shinken admin' % user.get_name()
        return response
    if not user.is_enabled():
        response['code'] = 403
        response['msg'] = 'The user %s is disabled' % user.get_name()
        return response
    
    item_type = app.request.GET.get('item_type', '')
    if not item_type:
        response['code'] = 500
        response['msg'] = 'no item_type was set on request'
    if item_type != ITEM_TYPE.HOSTS:
        # For the moment, put_in_prod manage only hosts
        response['code'] = 500
        response['msg'] = 'This item_type is not yet implemented'
    
    _filter = app.request.GET.get('filter', '')
    where = {}
    if _filter and len(_filter.split(':')) == 2:
        _filter_split = _filter.split(':')
        if _filter_split[0] != 'sources':
            response['code'] = 500
            response['msg'] = 'invalid filter'
            return response
        
        where[_filter_split[0]] = _filter_split[1]
    else:
        response['code'] = 500
        response['msg'] = 'invalid filter'
        return response
    
    return app.state_controller.put_in_prod(where, user, item_type)


def delete_in_production():
    response = {'code': 200, 'msg': ''}
    user_id = app.request.GET.get('user_id', '')
    user = app.datamanagerV2.find_item_by_id(user_id, ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
    if not user:
        response['code'] = 500
        response['msg'] = 'user with id [%s] not found' % user_id
        return response
    if not user.is_admin():
        response['code'] = 403
        response['msg'] = 'The user %s is not a Shinken admin' % user.get_name()
        return response
    if not user.is_enabled():
        response['code'] = 403
        response['msg'] = 'The user %s is disabled' % user.get_name()
        return response
    
    item_type = app.request.GET.get('item_type', '')
    if not item_type:
        response['code'] = 500
        response['msg'] = 'no item_type was set on request'
        return response
    if item_type != ITEM_TYPE.HOSTS:
        # For the moment, put_in_prod manage only hosts
        response['code'] = 500
        response['msg'] = 'This item_type is not yet implemented'
        return response
    
    uuids = app.request.GET.get('uuids', '').split(',')
    
    return app.state_controller.delete_in_prod(uuids, user, item_type)


def force_trusted_source_behaviour():
    response = {'imported': {}, 'changed': {}, 'code': 200, 'msg': ''}
    user_name = app.request.GET.get('user_name', '')
    if user_name == 'shinken-core':
        user = get_item_instance(ITEM_TYPE.CONTACTS, FALLBACK_USER)
    else:
        user = app.datamanagerV2.find_item_by_name(user_name, ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
        if not user:
            response['code'] = 500
            response['msg'] = app._('source.user_id_not_found') % user_name
            return response
        if not user.is_admin():
            response['code'] = 403
            response['msg'] = app._('source.user_not_shinken_admin') % user.get_name()
            return response
        if not user.is_enabled():
            response['code'] = 403
            response['msg'] = app._('source.user_disabled') % user.get_name()
            return response
    
    _filter = app.request.GET.get('filter', '')
    where = {}
    if _filter and len(_filter.split(':')) == 2:
        _filter_split = _filter.split(':')
        if _filter_split[0] != 'sources':
            response['code'] = 500
            response['msg'] = app._('source.filtre_invalide') % _filter
            return response
        
        where[_filter_split[0]] = _filter_split[1]
    else:
        response['code'] = 500
        response['msg'] = app._('source.filtre_invalide') % _filter
        return response
    
    with DBTransaction(app.datamanagerV2):
        imported = response['imported']
        for item_type in DEF_ITEMS.iterkeys():
            imported[item_type] = 0
            items = app.datamanagerV2.find_items(item_type=item_type, item_state=ITEM_STATE.NEW, where=where)
            for item in items:
                # Insert into staging, and remove from new
                app.datamanagerV2.save_item(item, user=user, item_type=item_type, item_state=ITEM_STATE.STAGGING)
                app.datamanagerV2.delete_item(item, user=user, item_type=item_type, item_state=ITEM_STATE.NEW, action=HISTORY_ACTION.IMPORT)
                imported[item_type] += 1
        logger.debug("[force_trusted_source_behaviour] import new item done")
        
        changed = response['changed']
        where['have_change'] = True
        for item_type in DEF_ITEMS.iterkeys():
            changed[item_type] = 0
            if item_type == ITEM_TYPE.HOSTS:
                items = app.datamanagerV2.find_merge_state_items(item_type=item_type, item_states=(ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING), where=where)
            else:
                items = app.datamanagerV2.find_items(item_type=item_type, item_state=ITEM_STATE.STAGGING, where=where)
            
            for item in items:
                context = ItemContext(app, item['_id'], item.get_type(), action=ACTIONS.VALIDATE_CHANGES, user=user, bypass_work_area=True)
                validation, item_name = app.state_controller.validate_changes(context)
                if validation['has_messages']:
                    for msg in validation['messages']:
                        logger.error('[force_trusted_source_behaviour] applied difference on [%s in %s named : %s] failed with error : [%s]' % (item.get_type(), item.get_state(), item.get_name(), msg))
                    response['code'] = 506
                    response['msg'] = app._('source.applied_difference_fail') % (item.get_type(), item.get_state(), item.get_name(), ','.join(validation['messages']))
                    return response
                else:
                    changed[item_type] += 1
        
        logger.debug("[force_trusted_source_behaviour] apply change item done")
    
    reset_configuration_stats()
    return response


def validate_all_elements():
    response = {}
    with DBTransaction(app.datamanagerV2):
        for item_type in DEF_ITEMS.keys():
            response[item_type] = 0
            logger.debug("[validate_all_elements] Type %s" % item_type)
            try:
                items = app.datamanagerV2.find_items(item_type=item_type, item_state=ITEM_STATE.NEW)
                for item in items:
                    # Insert into staging, and remove from new
                    app.datamanagerV2.save_item(item, item_type=item_type, item_state=ITEM_STATE.STAGGING)
                    app.datamanagerV2.delete_item(item, item_type=item_type, item_state=ITEM_STATE.NEW)
                    response[item_type] += 1
            except DataException:  # This type of objects is not represented in our NEW database
                logger.debug("  ==> Type missing from DB : %s" % item_type)
    
    reset_configuration_stats()
    return response


def commit_all():
    syncuicommon.apply_staging_in_production()
    syncuicommon.compute_all_diff_staging_production()
    syncuicommon.reset_configuration_stats_cache()
    return {}


def terminate_syncui():
    logger.info('Start stopping UI process pid:[%s]' % os.getpid())
    transaction_protecter.drain_stop()
    logger.info('UI process is stop pid:[%s]' % os.getpid())
    os._exit(0)


def get_call_stats():
    return syncuicommon.get_call_stats()


def get_auth():
    response = {
        u'admin_level': u'',
        u'code'       : 200,
        u'msg'        : u'',
        u'user'       : None,
        u'enabled'    : False
    }
    login = app.request.GET.get(u'shinken_login', u'')
    password = app.request.GET.get(u'password', u'')
    requester = app.request.GET.get(u'requester', u'')
    
    try:
        password = base64.b64decode(password)
    except TypeError:
        response[u'code'] = 400
        response[u'msg'] = u'Parameter password is not in base64'
        return response
    
    if not login and not password:
        response[u'code'] = 400
        response[u'msg'] = u'Parameter login and password are mandatory'
        return response
    
    user = app.authenticate_user_by_modules(login, password, requester)
    if not user:
        response[u'code'] = 403
        response[u'msg'] = u'Forbidden user'
        return response
    
    response[u'user'] = user.get_raw_item()
    response[u'admin_level'] = user.get_admin_level()
    response[u'enabled'] = user.is_enabled()
    return response


def update_sync_keys():
    source_name = app.request.GET.get('source_name', '')
    old_pattern_name = app.request.GET.get('old_pattern', '').decode('utf8')
    new_pattern_name = app.request.GET.get('new_pattern', '').decode('utf8')
    old_pattern_sync_key = old_pattern_name.lower()
    new_pattern_sync_key = new_pattern_name.lower()
    logger.debug('[update_sync_keys] Will update sync-keys for source:[%s] [%s]->[%s]' % (source_name, old_pattern_sync_key, new_pattern_sync_key))
    for item_state in (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING):
        for item_type in ITEM_TYPE.ALL_TYPES:
            key_name = DEF_ITEMS[item_type]['key_name']
            if not ITEM_TYPE.has_work_area(item_type) and item_state == ITEM_STATE.WORKING_AREA:
                continue
            items = app.datamanagerV2.find_items(item_type, item_state, where={'sources': source_name})
            for item in items:
                raw_item = item.get_raw_item(flatten_links=False)
                raw_item[key_name] = raw_item[key_name].replace(old_pattern_name, new_pattern_name)
                raw_item['_SYNC_KEYS'] = [i.replace(old_pattern_sync_key, new_pattern_sync_key) for i in raw_item['_SYNC_KEYS']]
                app.datamanagerV2.save_item(raw_item, None, item_type, item_state, action=HISTORY_ACTION.AUTO_MODIFICATION)
    
    _data_provider_mongo = DataProviderMongo(app.mongodb_db)
    for item_type in ITEM_TYPE.ALL_TYPES:
        key_name = DEF_ITEMS[item_type]['key_name']
        items = _data_provider_mongo.find_items(item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=source_name)
        for item in items:
            item[key_name] = item[key_name].replace(old_pattern_name, new_pattern_name)
            item['_SYNC_KEYS'] = [i.replace(old_pattern_sync_key, new_pattern_sync_key) for i in item['_SYNC_KEYS']]
            make_item_hash(item)
            _data_provider_mongo.save_item(item, item_type=item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=source_name)
    
    conn = app.get_synchronizer_deamon_connection()
    conn.request("GET", "/set_merge_asked")


def call_memory_stats():
    password = app.request.GET.get('password', '')
    expanded = app.request.GET.get('expanded', False)
    if hashlib.sha256(password).hexdigest() == 'c2a450753e8f2ad51b3d87a65d7d652198e6b19ed84701f529b2d43222866f2f':
        return memory_stats.query_memory_stats(expanded)
    else:
        app.abort(403, 'incorrect password')


def call_ask_garbage():
    password = app.request.GET.get('password', '')
    if hashlib.sha256(password).hexdigest() == 'c2a450753e8f2ad51b3d87a65d7d652198e6b19ed84701f529b2d43222866f2f':
        return force_memory_trimming()
    else:
        app.abort(403, 'incorrect password')


# Note: internal calls MUST be protected against external access
pages = {
    import_done                   : {'routes': ['/internal/import_done'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    flatten_conf                  : {'routes': ['/internal/get_conf'], 'method': 'GET', 'wrappers': ['protect_internal_call']},
    get_conf                      : {'routes': ['/internal/get_existing_conf'], 'method': 'GET', 'wrappers': ['protect_internal_call']},
    get_whole_configuration       : {'routes': ['/internal/get_whole_configuration'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    apply_all_diffs               : {'routes': ['/internal/apply_all_diffs'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    validate_all_elements         : {'routes': ['/internal/validate_all_elements'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    force_trusted_source_behaviour: {'routes': ['/internal/force_trusted_source_behaviour'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    put_in_production             : {'routes': ['/internal/put_in_production'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    delete_in_production          : {'routes': ['/internal/delete_in_production'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    commit_all                    : {'routes': ['/internal/commit_all'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    get_inherited_item            : {'routes': ['/internal/item_by_id/<item_state>/<item_type>/<item_id>'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    get_call_stats                : {'routes': ['/internal/get_call_stats'], 'wrappers': ['protect_internal_call']},
    update_sync_keys              : {'routes': ['/internal/update_sync_keys'], 'wrappers': ['protect_internal_call', 'db_transaction']},
    terminate_syncui              : {'routes': ['/api/terminate'], 'wrappers': []},
    get_auth                      : {'routes': ['/get_auth'], 'method': 'GET', 'wrappers': ['json', 'protect_internal_call']},
    call_memory_stats             : {'routes': ['/memory_stats'], 'method': 'GET', 'wrappers': ['json']},
    call_ask_garbage              : {'routes': ['/ask_garbage'], 'method': 'GET', 'wrappers': ['json']},
}
