#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

import socket

from shinken.http_client import HTTPExceptions
from shinken.log import logger
from shinken.property import IntegerProp, StringProp, ListProp, EditableListProp, BoolProp
from shinken.satellitelink import SatelliteLink, SatelliteLinks

""" TODO: Add some comment about this class for the doc"""


class SynchronizerLink(SatelliteLink):
    id = 0
    my_type = 'synchronizer'
    properties = SatelliteLink.properties.copy()
    properties.update({
        'synchronizer_name': StringProp(),
        'host_name'        : StringProp(default=socket.gethostname()),
        'port'             : IntegerProp(default='7765'),
        'sources'          : EditableListProp(default=''),
        'taggers'          : EditableListProp(default=''),
        'show_refresh_and_clean_column': BoolProp(default='0'),
    })
    
    
    def get_name(self):
        return getattr(self, 'synchronizer_name', 'unknown')
    
    
    def get_config(self):
        return self.con.get_config()
    
    
    # Check is required when prop are set:
    # contacts OR contactgroups is need
    def is_correct(self):
        state = True
        cls = self.__class__
        
        for prop, entry in cls.properties.items():
            if not hasattr(self, prop) and entry.required:
                # This should raise an error afterwards?
                # Log the issue
                logger.warning("%s synchronizerlink is missing %s property" % (self.get_name(), prop))
                self.debug("%s synchronizerlink is missing %s property" % (self.get_name(), prop))
                state = False  # Bad boy...
        return state
    
    
    # Look for ourself as an arbiter. If we search for a specific arbiter name, go forit
    # If not look be our fqdn name, or if not, our hostname
    def is_me(self, lookup_name):
        logger.info("An synchronizer is launched with the hostname:%s from an synchronizerlink point of view of addr:%s" % (self.host_name, socket.getfqdn()))
        if lookup_name:
            return lookup_name == self.get_name()
        else:
            return self.host_name == socket.getfqdn() or self.host_name == socket.gethostname()
    
    
    def give_satellite_cfg(self):
        return {'port': self.port, 'address': self.address, 'name': self.synchronizer_name}
    
    
    def do_not_run(self):
        if self.con is None:
            self.create_connection()
        try:
            self.con.get('do_not_run')
            return True
        except HTTPExceptions, exp:
            self.con = None
            return False
    
    
    def get_satellite_list(self, daemon_type):
        if self.con is None:
            self.create_connection()
        try:
            r = self.con.get_satellite_list(daemon_type)
            return r
        except HTTPExceptions, exp:
            self.con = None
            return []
    
    
    def get_satellite_status(self, daemon_type, name):
        if self.con is None:
            self.create_connection()
        try:
            r = self.con.get_satellite_status(daemon_type, name)
            return r
        except HTTPExceptions, exp:
            self.con = None
            return {}
    
    
    def get_all_states(self):
        if self.con is None:
            self.create_connection()
        try:
            r = self.con.get_all_states()
            return r
        except HTTPExceptions, exp:
            self.con = None
            return None
    
    
    def get_objects_properties(self, table, *properties):
        if self.con is None:
            self.create_connection()
        try:
            r = self.con.get_objects_properties(table, *properties)
            return r
        except HTTPExceptions, exp:
            self.con = None
            return None


class SynchronizerLinks(SatelliteLinks):
    name_property = "synchronizer_name"
    inner_class = SynchronizerLink
    
    
    # We must have a realm property, so we find our realm
    def linkify(self, modules, sources, taggers):
        self.linkify_s_by_plug(modules)
        self.linkify_by_sources(sources)
        self.linkify_by_taggers(taggers)
    
    
    # Linkify with modules
    def linkify_by_sources(self, sources):
        for sync in self:
            new_sources = []
            for sname in sync.sources:
                sname = sname.strip()
                # don't tread void names
                if sname == '':
                    continue
                
                s = sources.find_by_name(sname)
                if s is not None:
                    new_sources.append(s)
                else:
                    err = "Error: the source %s is unknown for %s" % (sname, sync.get_name())
                    sync.configuration_errors.append(err)
            sync.sources = new_sources
    
    
    # Linkify with modules
    def linkify_by_taggers(self, taggers):
        for sync in self:
            new_taggers = []
            for sname in sync.taggers:
                sname = sname.strip()
                # don't tread void names
                if sname == '':
                    continue
                s = taggers.find_by_name(sname)
                if s is not None:
                    new_taggers.append(s)
                else:
                    err = "Error: the taggers %s is unknown for %s" % (sname, sync.get_name())
                    sync.configuration_errors.append(err)
            sync.taggers = new_taggers
