#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013:
#    Gabes Jean, naparuba@gmail.com
#
# This file is part of Shinken Enterprise, all rights reserved.

import httplib
import json
import os
import socket
import ssl
import threading
import time
import traceback
import urllib

import mapper
from shinken.basemodule import BaseModule, ModuleState
from shinken.brok import PersistantBrok
from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.webui.bottlewebui import run, route, request, response
from shinken.webui.cherrypybackend import CherryPyServerHTTP

if TYPE_CHECKING:
    from shinken.daemons.arbiterdaemon import Arbiter

_PATH_RECEIVED_ARCH_RETENTION = '/var/lib/shinken/architecture_export_received.json'
properties = {
    'daemons': ['arbiter', 'synchronizer'],
    'type'   : 'architecture_export',
    'phases' : ['configuration'],
}

logger = LoggerFactory.get_logger().get_sub_part(u'ARCHITECTURE-EXPORT')

MAP_GENERATION_TIMEOUT = 30


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info("Export Shinken architecture information to NagVis maps. %s" % plugin.get_name())
    
    # Catch errors
    path = plugin.path
    instance = ArchitectureExport(plugin, path)
    return instance


class ArchitectureExport(BaseModule):
    def __init__(self, mod_conf, path):
        BaseModule.__init__(self, mod_conf)
        self.path = path
        self.errors = None
        self.architecture_name = getattr(mod_conf, 'architecture_name', 'Shinken')
        self.send_my_architecture_to_recipients = getattr(mod_conf, 'send_my_architecture_to_recipients', [])
        if self.send_my_architecture_to_recipients:
            self.send_my_architecture_to_recipients = self.send_my_architecture_to_recipients.strip().split(',')
        
        self.map_base_url = getattr(mod_conf, 'map_base_url', "")
        
        self.thread_counter = 0
        self.threads_status = {}
        self.threads_status_lock = threading.RLock()
        
        self._broks_to_send = {}
        # http server management
        self.srv = None
        self.srv_lock = threading.RLock()
        self.thread = None
        self.serveropts = {}
        self.arbiter = None
        try:
            logger.debug("[%s] Configuration starting ..." % self.architecture_name)
            self.port = int(getattr(mod_conf, 'port', '7780'))
            self.host = getattr(mod_conf, 'host', '0.0.0.0')
            self.use_ssl = getattr(mod_conf, 'use_ssl', False)
            if isinstance(self.use_ssl, basestring):
                self.use_ssl = True if self.use_ssl == "1" else False
            self.ssl_key = getattr(mod_conf, 'ssl_key', "")
            self.ssl_cert = getattr(mod_conf, 'ssl_cert', "")
            self.listener_use_ssl = int(getattr(mod_conf, 'listener_use_ssl', 0))
            self.listener_login = getattr(mod_conf, 'listener_login', 'Shinken')
            self.listener_password = getattr(mod_conf, 'listener_password', 'OFU2SE4zOU1FMDdaQlJENFtljgwTcWn2hQ5ocksBWS0=')
            
            self.ssh_port = int(getattr(mod_conf, 'ssh_port', '22'))
            self.ssh_user = getattr(mod_conf, 'ssh_user', 'shinken')
            self.ssh_key_file = getattr(mod_conf, 'ssh_key_file', "/var/lib/shinken/.ssh/id_rsa.pub")
            self.ssh_timeout = int(getattr(mod_conf, 'ssh_timeout', '5'))
            
            self.map_realm_layout = getattr(mod_conf, 'map_realm_layout', 'sort_by_size')
            if self.map_realm_layout not in mapper.MAP_REALM_LAYOUTS.keys():
                self.errors = "The architecture-export Module could not initialize because the parameter 'map_realm_layout' has an illegal value : %s" % self.map_realm_layout
                logger.error("%s" % self.errors)
            
            if self.ssl_key and not self.use_ssl:
                self.ssl_key = ""
            if self.ssl_cert and not self.use_ssl:
                self.ssl_cert = ""
            logger.info("Configuration done, host: %s(%s)" % (self.host, self.port))
        except AttributeError:
            logger.error("The module is missing a property, check module declaration in architecture-export.cfg")
            raise
        except Exception, e:
            logger.error("Exception : %s" % str(e))
            raise
    
    
    # used by the arbiter to get objects from modules
    def get_objects(self):
        return {}
    
    
    def get_state(self):
        if self.errors:
            return {"status": ModuleState.CRITICAL, "output": self.errors}
        return {"status": ModuleState.OK, "output": "OK"}
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def hook_daemon_daemonized(self, arb):
        logger.info("[%s] Initialization of the Architecture Export module" % self.architecture_name)
        self.arbiter = arb
        if self.arbiter.me.spare:
            return
        
        self.lang = "en"
        for mod in arb.conf.modules:
            if mod.module_type == "webui":
                self.lang = getattr(mod, 'lang', 'en')
        
        self._start_http()
        # load previously presents architecture and send brok, the broker will already know that thoses maps exists
        try:
            self._broks_to_send = self._load_brok_arch()
            # Update broks info to account for arbiter address change
            self._broks_to_send = self._update_broks_arch_info(self._broks_to_send)
            self._add_arch_brok(self._broks_to_send)
        except Exception:
            pass
        
        # Send configuration after loading broks from retention to avoid conflicts when updating broks
        self.send_configuration(arb)
    
    
    def hook_tick(self, arb):
        to_del = []
        with self.threads_status_lock:
            for (thread_id, info) in self.threads_status.iteritems():
                status = info.get('status')
                thread = info.get('thread')
                server_uuid = info.get('server_uuid')
                architecture_name = info.get('architecture_name')
                
                if status == "DONE":
                    if thread:
                        thread.join()
                    
                    to_del.append(thread_id)
                    logger.info('[%s] Export of architecture successful.' % architecture_name)
                elif status == "RUNNING":
                    if (info.get('timeout_logged') is None) and (int(time.time()) - info['creation_time'] > MAP_GENERATION_TIMEOUT):
                        logger.warning('Generating maps for architecture %s is taking more than %s seconds.' % (architecture_name, MAP_GENERATION_TIMEOUT))
                        info['timeout_logged'] = True
                elif status == "ERROR":
                    if thread:
                        thread.join()
                    
                    logger.error('Generating maps for architecture %s failed.' % architecture_name)
                    to_del.append(thread_id)
                    self._broks_to_send.pop(server_uuid, None)
                    self._add_arch_brok(self._broks_to_send)
            
            for thread_id in to_del:
                del self.threads_status[thread_id]
    
    
    def quit(self):
        self.do_stop()
    
    
    def do_stop(self):
        self._save_brok_arch()
        self._stop_http()
    
    
    def _start_http(self):
        # We must protect against a user that spam the enable/disable button
        with self.srv_lock:
            # We already did start, skip it
            if self.thread is not None:
                if self.thread.is_alive():
                    return
                self.thread.join()
                self.thread = None
            self._init_http()
            self._start_thread()
    
    
    def _stop_http(self):
        # Already no more thread, we are great
        if self.thread is None:
            return
        # We must protect against a user that spam the enable/disable button
        with self.srv_lock:
            self.srv.stop()
            self.thread.join(1)
            self.thread = None
            logger.debug("[%s] Http listener stopped" % self.architecture_name)
    
    
    def _start_thread(self):
        self.thread = threading.Thread(None, target=self._http_start, name='architecture-export-http')
        self.thread.daemon = True
        self.thread.start()
    
    
    def _http_start(self):
        # Now block and run
        logger.info('[%s] Server starting' % self.architecture_name)
        self.srv.start()
    
    
    # return a json containing the previous architecture broks
    def _load_brok_arch(self):
        try:
            file_path = _PATH_RECEIVED_ARCH_RETENTION
            if not os.path.exists(os.path.dirname(file_path)):
                os.makedirs(os.path.dirname(file_path))
            if os.path.isfile(file_path):
                self._broks_to_send = json.load(open(file_path, 'r'))
                if self._broks_to_send is None:
                    self._broks_to_send = {}
        except Exception as e:
            self._broks_to_send = {}
            logger.error("Previous arch broks retention cannot be loaded from file [%s] and will be empty. \n%s" % (_PATH_RECEIVED_ARCH_RETENTION, e))
        
        return self._broks_to_send
    
    
    def _save_brok_arch(self):
        try:
            file_path = _PATH_RECEIVED_ARCH_RETENTION
            if not os.path.exists(os.path.dirname(file_path)):
                os.makedirs(os.path.dirname(file_path))
            with open(file_path, 'w') as fd:
                json.dump(self._broks_to_send, fd)
            logger.debug('save broks contents %s' % file_path)
        except:
            logger.error("Save broks error in file [%s] fail : [%s]" % (_PATH_RECEIVED_ARCH_RETENTION, traceback.format_exc()))
    
    
    def _add_arch_brok(self, _broks_to_send):
        # Avoid sending None broks to Broker that can cause errors
        if _broks_to_send is None:
            return
        
        # this is optional, remove the old architecture_export_map avoid spamming the brokers
        architecture_export_brok_type = 'architecture_export_map'
        for key, brok in self.arbiter.persistant_broks.items():
            if brok.type == architecture_export_brok_type:
                self.arbiter.persistant_broks.pop(key, None)
        brok = PersistantBrok(architecture_export_brok_type, _broks_to_send)
        self.arbiter.add(brok)
    
    
    def _update_broks_arch_info(self, _broks_to_send):
        for server_uuid, arch_info in _broks_to_send.iteritems():
            _broks_to_send[server_uuid] = self._get_brok_content(server_uuid, arch_info['name'])
        return _broks_to_send
    
    
    def _get_thread_id(self):
        self.thread_counter += 1
        return self.thread_counter
    
    
    # We have the configuration loaded here,
    # this is the time to send the map to my recipient,
    # if disable is True it's time to tell my recipient that my arch have been disabled
    def send_configuration(self, arbiter, disabled=False):
        # type: (Arbiter, bool) -> None
        server_uuid = arbiter.server_uuid.replace('\n', "")
        if len(arbiter.conf.synchronizers) >= 1:
            synchronizer = arbiter.conf.synchronizers[0]
        else:
            synchronizer = None
        
        broks_to_send = self._get_brok_content(server_uuid, self.architecture_name)
        mapper_instance = mapper.ArchitectureExportMapper(self.path,
                                                          self.architecture_name,
                                                          server_uuid,
                                                          synchronizer,
                                                          self.lang,
                                                          self.listener_use_ssl,
                                                          self.listener_login,
                                                          self.listener_password,
                                                          broks_to_send,
                                                          self.ssh_port,
                                                          self.ssh_user,
                                                          self.ssh_key_file,
                                                          self.ssh_timeout,
                                                          self.map_realm_layout)
        arch_realm = mapper_instance.get_architecture_realm(conf=arbiter.conf)
        
        for other_export_module in self.send_my_architecture_to_recipients:
            other_export_module = other_export_module.strip()
            original_address = other_export_module
            address_has_formatting_errors = False
            
            send_with_https = False
            if other_export_module.startswith('http://'):
                other_export_module = other_export_module.replace('http://', '')
            elif other_export_module.startswith('https://'):
                other_export_module = other_export_module.replace('https://', '')
                send_with_https = True
            else:
                logger.error('External export module address "%s" has an incorrect format (does not start with "http://" or "https://"). Skipping sending info to this address.' % original_address)
                address_has_formatting_errors = True
            
            if len(other_export_module.split(':')) != 2:
                logger.error('No port specified in in external export module address "%s". Skipping sending info to this address.' % original_address)
                address_has_formatting_errors = True
            
            if other_export_module[-1] == '/':
                other_export_module = other_export_module[:-1]
            
            if address_has_formatting_errors:
                continue
            
            nb_try = 0
            max_try = 3
            while nb_try < max_try:  # do the try for max_try times
                nb_try += 1
                try:
                    if send_with_https:
                        args = {}
                        # If we are in SSL mode, do not look at certificate too much
                        # NOTE: ssl.SSLContext is only available on last python 2.7 versions
                        if hasattr(ssl, 'SSLContext'):
                            ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1)
                            ssl_context.check_hostname = False
                            ssl_context.verify_mode = ssl.CERT_NONE
                            conn = httplib.HTTPSConnection(other_export_module, timeout=3, context=ssl_context)
                        else:
                            conn = httplib.HTTPSConnection(other_export_module, timeout=3)
                    else:
                        conn = httplib.HTTPConnection(other_export_module, timeout=3)
                    headers = {"Content-type": "application/json", "Accept": "text/plain"}
                    path = u"/v1/architecture/%s/%s" % (server_uuid, self.architecture_name,)
                    path = urllib.pathname2url(path.encode("utf-8"))
                    # change the HTTP method if we need to notify about a disable
                    http_method = "POST"
                    if disabled:
                        http_method = "DELETE"
                    conn.request(http_method, path, json.dumps(arch_realm), headers)
                    response = conn.getresponse()
                    res = response.read()
                    if response.status not in (httplib.CREATED, httplib.OK):
                        logger.error("An error occurred when sending Architecture [%s] to [%s]: %s - %s" % (self.architecture_name, other_export_module, response.status, res))
                    break
                except (httplib.HTTPException, socket.error, socket.timeout) as exp:
                    message = '[%s] cannot send Architecture to [%s]; "%s" occurred, still %d try' % (self.architecture_name, other_export_module, exp, max_try - nb_try)
                    if nb_try == 1:
                        logger.debug(message)
                    else:
                        logger.error(message)
                    time.sleep(nb_try)
    
    
    # generate a brok with the architecture_name and the url of the map
    def _get_brok_content(self, server_uuid, architecture_name):
        if not self.map_base_url:
            base_url = "http://%s" % self.arbiter.me.address
        else:
            base_url = self.map_base_url
            if base_url.endswith("/"):
                base_url = base_url[:-1]
            if not (base_url.startswith("http://") or base_url.startswith("https://")):
                base_url = "http://%s" % base_url
        
        return {
            'name'    : architecture_name,
            'tree_map': {
                'name': mapper.translate("realms_tree_link_name", self.lang),
                'url' : "%s/shinken-core-map/frontend/nagvis-js/index.php?mod=Map&show=shinken_global-%s" % (base_url, server_uuid),
            },
            'arch_map': {
                'name': mapper.translate("shinken_architecture_link_name", self.lang),
                'url' : "%s/shinken-core-map/frontend/nagvis-js/index.php?mod=Map&show=shinken_architecture-%s" % (base_url, server_uuid),
            }
        }
    
    
    # We initialize the HTTP part. It's a simple wsgi backend
    # with a select hack so we can still exit if someone ask it
    def _init_http(self):
        logger.info("[%s] Starting WS arbiter http socket" % self.architecture_name)
        
        
        # used when another Architecture Export module send us their configuration
        def receive_architecture(server_uuid, architecture_name):
            sender = request.environ.get('REMOTE_ADDR')
            logger.info("[%s] %s ask me to map the architecture" % (architecture_name, sender))
            response.content_type = 'application/json'
            
            if len(self.arbiter.conf.synchronizers) >= 1:
                synchronizer = self.arbiter.conf.synchronizers[0]
            else:
                synchronizer = None
            
            broks_to_send = self._get_brok_content(server_uuid, architecture_name)
            mapperInstance = mapper.ArchitectureExportMapper(self.path,
                                                             architecture_name,
                                                             server_uuid,
                                                             synchronizer,
                                                             self.lang,
                                                             self.listener_use_ssl,
                                                             self.listener_login,
                                                             self.listener_password,
                                                             broks_to_send,
                                                             self.ssh_port,
                                                             self.ssh_user,
                                                             self.ssh_key_file,
                                                             self.ssh_timeout,
                                                             self.map_realm_layout)
            try:
                thread_id = self._get_thread_id()
                self.threads_status[thread_id] = {
                    "status"           : "RUNNING",
                    "creation_time"    : int(time.time()),
                    "server_uuid"      : server_uuid,
                    "architecture_name": architecture_name
                }
                # calling the make_map (threaded) failed on permission denied
                mapperInstance.make_map(thread_id, self.threads_status, self.threads_status_lock, request.json)
                response.status = 201
                self._broks_to_send[server_uuid] = broks_to_send
                self._add_arch_brok(self._broks_to_send)
                self._save_brok_arch()
            except Exception as e:
                logger.error(e)
                response.status = 500
                self._broks_to_send.pop(server_uuid, None)
                self._add_arch_brok(self._broks_to_send)
            return 'done'
        
        
        # used when another Architecture Export module send us their configuration and notify us that we need to disable it
        def receive_architecture_to_disable(server_uuid, architecture_name):
            logger.info("[%s] Someone ask me to disable the architecture" % architecture_name)
            response.content_type = 'application/json'
            if len(self.arbiter.conf.synchronizers) >= 1:
                synchronizer = self.arbiter.conf.synchronizers[0]
            else:
                synchronizer = None
            mapper_instance = mapper.ArchitectureExportMapper(self.path,
                                                              architecture_name,
                                                              server_uuid,
                                                              synchronizer,
                                                              self.lang,
                                                              self.listener_use_ssl,
                                                              self.listener_login,
                                                              self.listener_password,
                                                              self._broks_to_send,
                                                              self.ssh_port,
                                                              self.ssh_user,
                                                              self.ssh_key_file,
                                                              self.ssh_timeout,
                                                              self.map_realm_layout)
            try:
                self._broks_to_send.pop(server_uuid, None)
                self._add_arch_brok(self._broks_to_send)
                # calling the make_map (threaded) failed on permission denied
                mapper_instance.remove_map(request.json)
                response.status = 201
            except Exception as e:
                logger.error(e)
                response.status = 500
            return 'done'
        
        
        # used when the addon system disable the module
        def module_disabled():
            logger.debug("[%s] Module disabled with the shinken-addons-disable command, remove my architecture to all my 'recipient'" % self.architecture_name)
            self.send_configuration(self.arbiter, disabled=True)
            response.status = 200
            return 'done'
        
        
        # Manually send architecture information
        def send_architecture_to_recipients():
            logger.debug("[%s] Sending architecture information to all recipients listed in the architecture-export module configuration" % self.architecture_name)
            self.send_configuration(self.arbiter)
            response.status = 200
            return 'done'
        
        
        try:
            # used when another Architecture Export module send us their configuration
            route('/v1/architecture/:server_uuid/:architecture_name', callback=receive_architecture, method=('POST', 'PUT', 'PATCH'))
            # used when another Architecture Export module send us their configuration and notify us that we need to disable it
            route('/v1/architecture/:server_uuid/:architecture_name', callback=receive_architecture_to_disable, method='DELETE')
            # used when the addon system disable the module
            route('/v1/architecture/disabled', callback=module_disabled, method='GET')
            # used when the addon system disable the module
            route('/v1/architecture/send', callback=send_architecture_to_recipients, method='POST')
            self.srv = run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=self.use_ssl, ssl_key=self.ssl_key, ssl_cert=self.ssl_cert, **self.serveropts)
        except Exception, e:
            logger.error("[%s] Exception : %s" % (self.architecture_name, str(e)))
            raise
        logger.info("[%s] Server loaded" % self.architecture_name)
