#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.

from shinkensolutions.lib_modules.configuration_reader import read_string_in_configuration, read_bool_in_configuration, read_int_in_configuration
from shinkensolutions.ssh_mongodb.sshtunnelmongomgr import mongo_by_ssh_mgr


class SLADatabaseConnection(object):
    
    def __init__(self, conf, _logger):
        self.logger = _logger
        self.uri = read_string_in_configuration(conf, 'mongo_uri', 'mongodb://localhost/?w=1&fsync=false')
        self.database = read_string_in_configuration(conf, 'mongo_database', 'shinken')
        self.use_ssh_tunnel = read_bool_in_configuration(conf, 'mongo_use_ssh_tunnel', False)
        self.use_ssh_retry_failure = read_int_in_configuration(conf, 'mongo_use_ssh_retry_failure', 1)
        self.mongo_timeout = read_int_in_configuration(conf, 'mongo_timeout', 5)
        self.ssh_user = read_string_in_configuration(conf, 'mongo_ssh_user', None)
        self.ssh_keyfile = read_string_in_configuration(conf, 'mongo_ssh_keyfile', None)
        
        self.logger.info('Load connection parameter to SLA base done.')
        self.logger.info('   mongo_uri                      : [%s]' % self.uri)
        self.logger.info('   mongo_database                 : [%s]' % self.database)
        self.logger.info('   mongo_use_ssh_tunnel           : [%s]' % self.use_ssh_tunnel)
        self.logger.info('   mongo_use_ssh_retry_failure    : [%s]' % self.use_ssh_retry_failure)
        self.logger.info('   mongo_ssh_user                 : [%s]' % self.ssh_user)
        self.logger.info('   mongo_ssh_keyfile              : [%s]' % self.ssh_keyfile)
        
        self.col_archive = None
        self.col_sla_info = None
        
        self._init_connection()
    
    
    def _init_connection(self):
        # Get a connection. If need, we will use a SSH tunnel
        con_result = mongo_by_ssh_mgr.get_connection(
            self.uri,
            fsync=False,
            use_ssh=self.use_ssh_tunnel,
            ssh_keyfile=self.ssh_keyfile,
            ssh_user=self.ssh_user,
            ssh_retry=self.use_ssh_retry_failure,
            requestor=self.logger.name,
            ssh_tunnel_timeout=self.mongo_timeout,
        )
        self.con = con_result.get_connection()
        self.db = getattr(self.con, self.database)
        self.col_archive = getattr(self.db, 'sla_archive')
        self.col_sla_info = getattr(self.db, 'sla_info')
        self.logger.info('Connection to sla base OK')
    
    
    def query_sla_archives(self, uuids, sla_date):
        _filter = {
            'uuid': {'$in': uuids},
            'year': sla_date.year,
            'yday': sla_date.yday,
        }
        
        self.logger.debug('Query sla archives with filter:[%s]' % _filter)
        found_archives = list(self.col_archive.find(_filter))
        self.logger.info('Query for %s items at date %s found %s sla archives' % (len(uuids), sla_date, len(found_archives)))
        return found_archives
    
    
    def query_sla_archive(self, uuid, sla_date):
        _filter = {
            'uuid': uuid,
            'year': sla_date.year,
            'yday': sla_date.yday,
        }
        
        self.logger.debug('Query sla archives with filter:[%s]' % _filter)
        found_archive = self.col_archive.find_one(_filter)
        return found_archive
