#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.

import time
from datetime import datetime

from shinken.log import PART_INITIALISATION
from shinken.thread_helper import Thread

DEFAULT_CHECK_INTERVAL = 5
DEFAULT_FIRST_MONITORING_DATE = datetime.strptime('2000-01-01', '%Y-%m-%d')


class SLAInfo(Thread):
    
    def __init__(self, sla_database_connection, logger):
        super(SLAInfo, self).__init__(logger)
        self.logger = logger
        self.sla_database_connection = sla_database_connection
        self._cache_date = None
        self._cache_sla_info_by_uuid = {}
        self._cache_sla_info_by_name = {}
    
    
    def init(self):
        time_start = time.time()
        all_sla_infos = self.sla_database_connection.col_sla_info.find({})
        
        tmp_cache = {}
        # handle double name with id !=
        for sla_info in all_sla_infos:
            if sla_info['_id'] == 'SLA_INFO':
                self._cache_date = sla_info.get('last_update', -1)
            else:
                name = '%s-%s' % (sla_info['host_name'], sla_info['service_description'])
                tmp = tmp_cache.get(name, [])
                tmp.append(sla_info)
                tmp_cache[name] = tmp
        
        invalide_sla_infos = []
        for tmp in tmp_cache.itervalues():
            if len(tmp) > 1:
                invalide_sla_infos.extend(tmp)
            else:
                sla_info = tmp[0]
                name = '%s-%s' % (sla_info['host_name'], sla_info['service_description'])
                sla_info_uuid = sla_info['_id']
                self._cache_sla_info_by_uuid[sla_info_uuid] = sla_info
                self._cache_sla_info_by_name[name] = sla_info
        
        self.logger.info(PART_INITIALISATION, 'Load %s elements info in cache done in %s' % (len(self._cache_sla_info_by_uuid), self.logger.format_chrono(time_start)))
        self.start_thread()
    
    
    def get_thread_name(self):
        return 'sla-info-thread'
    
    
    def loop_turn(self):
        sla_info = self.sla_database_connection.col_sla_info.find_one({'_id': 'SLA_INFO'})
        if sla_info and self._cache_date != sla_info.get('last_update', -1):
            self._reload_cache()
    
    
    def _reload_cache(self):
        time_start = time.time()
        self.logger.info('Elements info was updated we reload our cache')
        all_sla_infos = self.sla_database_connection.col_sla_info.find({})
        
        for sla_info in all_sla_infos:
            if sla_info['_id'] == 'SLA_INFO':
                self._cache_date = sla_info.get('last_update', -1)
            else:
                name = '%s-%s' % (sla_info['host_name'], sla_info['service_description'])
                sla_info_uuid = sla_info['_id']
                self._cache_sla_info_by_uuid[sla_info_uuid] = sla_info
                self._cache_sla_info_by_name[name] = sla_info
        
        self.logger.info(PART_INITIALISATION, 'Reload %s elements info in cache done in %s' % (len(self._cache_sla_info_by_uuid), self.logger.format_chrono(time_start)))
    
    
    def _get_cache_sla_info(self, item_uuid='', host_name='', service_description=''):
        if item_uuid:
            _where = {'_id': item_uuid}
            sla_info = self._cache_sla_info_by_uuid.get(item_uuid, None)
        else:
            service_description = '' if service_description is None else service_description
            _where = {'host_name': host_name, 'service_description': service_description}
            name = '%s-%s' % (host_name, service_description)
            sla_info = self._cache_sla_info_by_name.get(name, None)
        
        if sla_info:
            return sla_info
        
        sla_info = self.sla_database_connection.col_sla_info.find_one(_where)
        if sla_info:
            name = '%s-%s' % (sla_info['host_name'], sla_info['service_description'])
            self._cache_sla_info_by_uuid[sla_info['_id']] = sla_info
            self._cache_sla_info_by_name[name] = sla_info
            return sla_info
        return None
    
    
    def get_all_uuids(self):
        return [uuid[0:uuid.rfind('-')] if uuid.count('-') == 3 else uuid for uuid in self._cache_sla_info_by_uuid.iterkeys()]
    
    
    def get_uuid(self, hname, sdesc):
        info = self._get_cache_sla_info(host_name=hname, service_description=sdesc)
        if info:
            item_uuid = info['_id']
            if item_uuid.count('-') == 3:
                item_uuid = item_uuid[0:item_uuid.rfind('-')]
            return item_uuid
        else:
            return None
    
    
    def get_name(self, item_uuid):
        info = self._get_cache_sla_info(item_uuid=item_uuid)
        if info:
            return info.get('host_name', None), info.get('service_description', None)
        else:
            return None, None
