#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.


import time
from datetime import datetime, timedelta

from shinken.objects.timeperiod import Timeperiod
from .sla_database_connection import SLADatabaseConnection
from shinkensolutions.lib_modules.configuration_reader import read_string_in_configuration, read_int_in_configuration


class STATUS(object):
    OK = 0
    WARN = 1
    CRIT = 2
    UNKNOWN = 3
    MISSING_DATA = 4
    SHINKEN_INACTIVE = 5


LIST_STATUS = ('ok', 'warn', 'crit', 'unknown', 'missing', 'inactive')

CURRENT_ARCHIVE_VERSION = 2
STATUS_MISSING_DATA = 4

WARNING_COUNTS_AS_OK = 1
WARNING_COUNTS_AS_WARNING = 0

DOWNTIME_PERIOD_INCLUDE = 0
DOWNTIME_PERIOD_EXCLUDE = 1
DOWNTIME_PERIOD_OK = 2
DOWNTIME_PERIOD_CRITICAL = 3

UNKNOWN_PERIOD_INCLUDE = 0
UNKNOWN_PERIOD_EXCLUDE = 1
UNKNOWN_PERIOD_OK = 2

NO_DATA_PERIOD_INCLUDE = 0
NO_DATA_PERIOD_EXCLUDE = 1
NO_DATA_PERIOD_OK = 2

DOWNTIME_PERIOD_MAP = {
    'exclude' : DOWNTIME_PERIOD_EXCLUDE,
    'include' : DOWNTIME_PERIOD_INCLUDE,
    'ok'      : DOWNTIME_PERIOD_OK,
    'critical': DOWNTIME_PERIOD_CRITICAL,
}

UNKNOWN_PERIOD_MAP = {
    'exclude': UNKNOWN_PERIOD_EXCLUDE,
    'include': UNKNOWN_PERIOD_INCLUDE,
    'ok'     : UNKNOWN_PERIOD_OK,
}

NO_DATA_PERIOD_MAP = {
    'exclude': NO_DATA_PERIOD_EXCLUDE,
    'include': NO_DATA_PERIOD_INCLUDE,
    'ok'     : NO_DATA_PERIOD_OK,
}


def _get_next_period(period_start, timeperiod):
    # type: (float, Timeperiod) -> (int, int, bool)
    # When you ask a timeperiod at the last second of its range : "when is the next valid time", its return : "it's now brother !"
    # Here we have always the last second of a range, so we don't want "now" but the real next period.
    # So we need to add 1 second when we ask the is_valid or the next period, but we want to keep the date to give the good period start
    is_valid = timeperiod.is_time_valid(period_start + 1)
    
    if is_valid:
        # Another funny things, when you ask the next invalid period, the timeperiod keep the second we add to get the next period, so we remove it before return
        _period_end = timeperiod.get_next_invalid_time_from_t(period_start + 1, max_t=86400) - 1
    else:
        _period_end = timeperiod.get_next_valid_time_from_t(period_start + 1, max_t=86400)
    
    if _period_end is None:
        _period_end = period_start + 86400
    
    return int(period_start), int(_period_end), is_valid


class SlaReader(object):
    
    def __init__(self, _logger, conf):
        self.logger = _logger
        
        self.sla_database_connection = SLADatabaseConnection(conf, self.logger)
        
        self.warning_counts_as_ok = read_int_in_configuration(conf, 'warning_counts_as_ok', 0)
        self.unknown_period = UNKNOWN_PERIOD_MAP.get(read_string_in_configuration(conf, 'unknown_period', 'include'), UNKNOWN_PERIOD_INCLUDE)
        self.no_data_period = NO_DATA_PERIOD_MAP.get(read_string_in_configuration(conf, 'no_data_period', 'include'), NO_DATA_PERIOD_INCLUDE)
        self.downtime_period = DOWNTIME_PERIOD_MAP.get(read_string_in_configuration(conf, 'downtime_period', 'include'), DOWNTIME_PERIOD_INCLUDE)
        self.default_sla_thresholds = (99, 97)
        default_conf = getattr(conf, 'configuration_default_value', {})
        if default_conf:
            def_for_host = default_conf.get('host', {})
            self.default_sla_thresholds = (float(def_for_host.get('sla_warning_threshold', self.default_sla_thresholds[0])), float(def_for_host.get('sla_critical_threshold', self.default_sla_thresholds[1])))
        
        self.logger.info('Load read parameter to SLA base done.')
        self.logger.info('   warning_counts_as_ok   : [%s]' % self.warning_counts_as_ok)
        self.logger.info('   unknown_period         : [%s]' % self.unknown_period)
        self.logger.info('   no_data_period         : [%s]' % self.no_data_period)
        self.logger.info('   downtime_period        : [%s]' % self.downtime_period)
    
    
    @staticmethod
    def _compute_archive_day(sla_entry):
        ranges = sla_entry['ranges']
        if not ranges or len(ranges) == 0:
            # then simulate a void day, with an unknown state, the whole day, so tag this entry that we will 'lie' into it
            sla_entry['missing'] = True
            sla_entry['history_inactive'] = 1
            sla_entry['history_total'] = 1
            return
        
        total_period = 0
        for _range in ranges:
            start = _range['start']
            end = _range['end']
            status = _range['rc']
            period = end - start
            total_period += period
            
            if status == STATUS.OK:
                prefix = 'ok'
            elif status == STATUS.WARN:
                prefix = 'warn'
            elif status == STATUS.CRIT:
                prefix = 'crit'
            elif status == STATUS.UNKNOWN:
                prefix = 'unknown'
            elif status == STATUS.MISSING_DATA:
                prefix = 'missing'
            else:
                prefix = 'inactive'
            
            sla_entry['history_%s' % prefix] = sla_entry.get('history_%s' % prefix, 0) + period
            if _range['dt']:
                sla_entry['history_dt_%s' % prefix] = sla_entry.get('history_dt_%s' % prefix, 0) + period
        sla_entry['history_total'] = total_period
    
    
    def _compute_sla(self, archive_day):
        sla_format = '%s%s%s%s' % (self.warning_counts_as_ok, self.unknown_period, self.no_data_period, self.downtime_period)
        sla_ok = archive_day.get('history_ok', 0)
        # self.logger.debug('sla_ok : history_ok')
        if self.warning_counts_as_ok:
            sla_ok += archive_day.get('history_warn', 0)
            # self.logger.debug('sla_ok : history_warn')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_CRITICAL):
                sla_ok -= archive_day.get('history_dt_warn', 0)
                # self.logger.debug('sla_ok : - history_dt_warn')
        if self.downtime_period == DOWNTIME_PERIOD_OK:
            # We add downtime time but without the downtime OK part which is already in ok_sum
            sla_ok += archive_day.get('history_dt_crit', 0)
            # self.logger.debug('sla_ok : history_dt_crit')
            if self.warning_counts_as_ok == WARNING_COUNTS_AS_WARNING:
                sla_ok += archive_day.get('history_dt_warn', 0)
                # self.logger.debug('sla_ok : history_dt_warn')
            if not self.unknown_period == UNKNOWN_PERIOD_OK:
                sla_ok += archive_day.get('history_dt_unknown', 0)
                # self.logger.debug('sla_ok : history_dt_unknown')
            if not self.no_data_period == NO_DATA_PERIOD_OK:
                sla_ok += archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0)
                # self.logger.debug('sla_ok : history_dt_missing + history_dt_inactive')
        if self.downtime_period == DOWNTIME_PERIOD_CRITICAL:
            # We remove downtime OK part form ok_sum
            sla_ok -= archive_day.get('history_dt_ok', 0)
            # self.logger.debug('sla_ok : - history_dt_ok')
        if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
            # We remove downtime OK part form ok_sum
            sla_ok -= archive_day.get('history_dt_ok', 0)
            # self.logger.debug('sla_ok : - history_dt_ok')
        if self.unknown_period == UNKNOWN_PERIOD_OK:
            sla_ok += archive_day.get('history_unknown', 0)
            # self.logger.debug('sla_ok : history_unknown')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_CRITICAL):
                sla_ok -= archive_day.get('history_dt_unknown', 0)
                # self.logger.debug('sla_ok : - history_dt_unknown')
        if self.no_data_period == NO_DATA_PERIOD_OK:
            sla_ok += archive_day.get('history_missing', 0) + archive_day.get('history_inactive', 0)
            # self.logger.debug('sla_ok : history_missing + history_inactive')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_CRITICAL):
                sla_ok -= (archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0))
                # self.logger.debug('sla_ok : - history_dt_missing - history_dt_inactive')
        
        sla_crit = archive_day.get('history_crit', 0)
        # self.logger.debug('sla_crit : history_crit[%s]'%history_crit)
        if self.downtime_period == DOWNTIME_PERIOD_CRITICAL:
            sla_crit += \
                archive_day.get('history_dt_ok', 0) + \
                archive_day.get('history_dt_warn', 0) + \
                archive_day.get('history_dt_unknown', 0) + \
                archive_day.get('history_dt_missing', 0) + \
                archive_day.get('history_dt_inactive', 0)
        if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK):
            sla_crit -= archive_day.get('history_dt_crit', 0)
        
        sla_warn = 0
        if self.warning_counts_as_ok == WARNING_COUNTS_AS_WARNING:
            sla_warn = archive_day.get('history_warn', 0)
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
                sla_warn -= archive_day.get('history_dt_warn', 0)
        
        sla_unknown = 0
        if self.unknown_period == UNKNOWN_PERIOD_INCLUDE:
            sla_unknown = archive_day.get('history_unknown', 0)
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
                sla_unknown -= archive_day.get('history_dt_unknown', 0)
        elif self.unknown_period == UNKNOWN_PERIOD_EXCLUDE and self.downtime_period == DOWNTIME_PERIOD_INCLUDE:
            sla_unknown = archive_day.get('history_dt_unknown', 0)
        
        sla_missing = 0
        sla_inactive = 0
        if self.no_data_period == NO_DATA_PERIOD_INCLUDE:
            sla_missing = archive_day.get('history_missing', 0)
            sla_inactive = archive_day.get('history_inactive', 0)
            # self.logger.debug('sla_missing : history_missing')
            # self.logger.debug('sla_inactive : history_inactive')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
                sla_missing -= archive_day.get('history_dt_missing', 0)
                sla_inactive -= archive_day.get('history_dt_inactive', 0)
                # self.logger.debug('sla_missing : - history_dt_missing')a
                # self.logger.debug('sla_inactive : - history_dt_inactive')
        elif self.no_data_period == NO_DATA_PERIOD_EXCLUDE and self.downtime_period == DOWNTIME_PERIOD_INCLUDE:
            sla_missing = archive_day.get('history_dt_missing', 0)
            sla_inactive = archive_day.get('history_dt_inactive', 0)
            # self.logger.debug('sla_missing : history_dt_missing')
            # self.logger.debug('sla_inactive : history_dt_inactive')
        
        sla_total = archive_day.get('history_ok', 0) + archive_day.get('history_warn', 0) + archive_day.get('history_crit', 0)
        # self.logger.debug('sla_total : history_ok + history_warn + history_crit')
        if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
            sla_total -= archive_day.get('history_dt_ok', 0) + archive_day.get('history_dt_warn', 0) + archive_day.get('history_dt_crit', 0)
            # self.logger.debug('sla_total : - history_dt_ok - history_dt_warn - history_dt_crit')
        if self.unknown_period in (UNKNOWN_PERIOD_INCLUDE, UNKNOWN_PERIOD_OK):
            sla_total += archive_day.get('history_unknown', 0)
            # self.logger.debug('sla_total : + history_unknown')
            if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
                sla_total -= archive_day.get('history_dt_unknown', 0)
                # self.logger.debug('sla_total : - history_dt_unknown')
        elif self.downtime_period in (DOWNTIME_PERIOD_INCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
            sla_total += archive_day.get('history_dt_unknown', 0)
            # self.logger.debug('sla_total : + history_dt_unknown')
        if self.no_data_period in (NO_DATA_PERIOD_INCLUDE, NO_DATA_PERIOD_OK):
            sla_total += (archive_day.get('history_missing', 0) + archive_day.get('history_inactive', 0))
            # self.logger.debug('sla_total : history_missing + history_inactive')
            if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
                sla_total -= archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0)
                # self.logger.debug('sla_total : - history_dt_missing - history_dt_inactive')
        elif self.downtime_period in (DOWNTIME_PERIOD_INCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
            sla_total += archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0)
            # self.logger.debug('sla_total : + history_dt_missing + history_dt_inactive')
        # self.logger.debug('sla_ok:[%s] / sla_total:[%s] = [%d]' % (sla_ok, sla_total, sla_ok * 100 / (sla_total if sla_total else 1)))
        
        self._set_sla_value(archive_day, 'sla_ok', sla_ok)
        self._set_sla_value(archive_day, 'sla_crit', sla_crit)
        self._set_sla_value(archive_day, 'sla_warn', sla_warn)
        self._set_sla_value(archive_day, 'sla_unknown', sla_unknown)
        self._set_sla_value(archive_day, 'sla_missing', sla_missing)
        self._set_sla_value(archive_day, 'sla_inactive', sla_inactive)
        
        archive_day['sla_total'] = sla_total
        archive_day['sla_format'] = sla_format
    
    
    def _build_archive_for_missing_day(self, curr_year, curr_yday, item_uuid):
        _day = time.strptime('%d %d' % (curr_year, curr_yday), '%Y %j')
        start_of_day = int(time.mktime(_day))  # use UTC and not localtime calendar.timegm(_day)
        end_of_day = self._compute_end_of_day(start_of_day)
        
        empty_entry = {
            'uuid'            : item_uuid,
            'yday'            : curr_yday,
            'year'            : curr_year,
            'ranges'          : [{'rc': STATUS_MISSING_DATA, 'end': end_of_day, 'start': start_of_day, 'ack': False, 'dt': False, 'flg': False}],
            'version'         : CURRENT_ARCHIVE_VERSION,
            'history_inactive': end_of_day - start_of_day,
            'history_total'   : end_of_day - start_of_day,
        }
        
        self._compute_sla(empty_entry)
        for prefix in LIST_STATUS:
            _sum = empty_entry.get('sla_%s' % prefix, 0)
            if _sum:
                empty_entry['archive_%s' % prefix] = _sum
        empty_entry['archive_total'] = empty_entry['sla_total']
        empty_entry['archive_format'] = empty_entry['sla_format']
        
        return empty_entry
    
    
    @staticmethod
    def _set_sla_value(archive_day, sla_name, sla_value):
        if sla_value:
            archive_day[sla_name] = sla_value
        elif sla_name in archive_day:
            del archive_day[sla_name]
    
    
    @staticmethod
    def _compute_end_of_day(start_of_day):
        end_date = datetime.fromtimestamp(start_of_day) + timedelta(days=1)
        end_of_day = time.mktime(end_date.timetuple())
        return end_of_day
    
    
    @staticmethod
    def _truncate_sla_ranges_from_timeperiod(sla_entry, timeperiod):
        _date = datetime.strptime('%s %s 00:00:00' % (sla_entry['year'], sla_entry['yday']), '%Y %j %H:%M:%S')
        _date = time.mktime(_date.timetuple())
        _new_ranges = []
        
        # we get the first period
        _period_start, _period_end, _is_valid = _get_next_period(_date, timeperiod)
        
        for _range in reversed(sla_entry['ranges']):
            
            # The only case where the _range start is after the _period_end is a element that was not monitored before this date
            while _period_end < _range['start']:
                _period_start, _period_end, _is_valid = _get_next_period(_period_end, timeperiod)
                continue
            
            _must_loop = True
            while _must_loop:
                
                if _is_valid:
                    _truncated_range = _range.copy()
                    if _truncated_range['start'] < _period_start:
                        _truncated_range['start'] = _period_start
                    if _truncated_range['end'] > _period_end:
                        _truncated_range['end'] = _period_end
                        _period_start, _period_end, _is_valid = _get_next_period(_period_end, timeperiod)
                    else:
                        _must_loop = False
                    
                    _new_ranges.append(_truncated_range)
                    _truncated_range = None
                
                elif _range['end'] < _period_end:
                    _must_loop = False
                
                else:
                    _period_start, _period_end, _is_valid = _get_next_period(_period_end, timeperiod)
        
        sla_entry['ranges'] = _new_ranges
    
    
    def read_one_sla(self, uuid, sla_date):
        if uuid is None:
            return {
                'sla_total': 'N.A.',
                'sla_ok'   : 'N.A.',
            }
        
        sla_archive = self.sla_database_connection.query_sla_archive(uuid, sla_date)
        if not sla_archive:
            sla_archive = self._build_archive_for_missing_day(sla_date.year, sla_date.yday, '')
            self._compute_sla(sla_archive)
        
        sla_info = {
            'sla_total'   : sla_archive.get('sla_total', 0),
            'sla_ok'      : sla_archive.get('sla_ok', 0),
            'sla_crit'    : sla_archive.get('sla_crit', 0),
            'sla_warn'    : sla_archive.get('sla_warn', 0),
            'sla_unknown' : sla_archive.get('sla_unknown', 0),
            'sla_missing' : sla_archive.get('sla_missing', 0),
            'sla_inactive': sla_archive.get('sla_inactive', 0),
            'sla_thresholds': sla_archive.get('thresholds', self.default_sla_thresholds),
            'sla_date'    : time.strftime('%d_%m_%Y', time.strptime('%s-%s' % (sla_date.year, sla_date.yday), '%Y-%j'))
        }
        return sla_info
