#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

from ec_brok_handler import BrokHandlerModuleWorker
from ec_common import shared_data
from ec_database_connection import ECDatabaseConnection
from ec_writer_stats import ECWriterStats
from shinken.brokermodule import WorkerBasedBrokerModule
from shinken.log import PART_INITIALISATION
from shinkensolutions.date_helper import get_now
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration

ACCEPTED_BROK_TYPES = ('service_check_result', 'host_check_result', 'update_service_status', 'update_host_status', 'initial_service_status', 'initial_host_status', 'initial_broks_done')
MARGIN_SHINKEN_INACTIVE = 30  # in sec


class EventContainerBrokerModule(WorkerBasedBrokerModule):
    MODULE_WORKER_CLASS = BrokHandlerModuleWorker
    
    
    def __init__(self, configuration):
        global MARGIN_SHINKEN_INACTIVE
        WorkerBasedBrokerModule.__init__(self, configuration)
        self.database_connection = ECDatabaseConnection(configuration, self.logger)
        self.writer_stats = ECWriterStats(self, self.database_connection)
        MARGIN_SHINKEN_INACTIVE = read_int_in_configuration(configuration, 'time_before_shinken_inactive', MARGIN_SHINKEN_INACTIVE)
        self._init_completed = False
        self.day_keep_data = read_int_in_configuration(configuration, 'day_keep_data', 30)
    
    def init(self):
        self.database_connection.init(requester=self.get_name())
        last_start_time = self.database_connection.get_last_tick()
        if last_start_time:
            shinken_inactive_period = get_now() - last_start_time
            shared_data.set_last_start_time(last_start_time)
            
            if shinken_inactive_period > MARGIN_SHINKEN_INACTIVE:
                shared_data.set_shinken_inactive_period(shinken_inactive_period)
            else:
                shared_data.set_shinken_inactive_period(0)
        
        self.logger.info(PART_INITIALISATION, 'last_start_time : %s' % self.logger.format_time(last_start_time))
        
        WorkerBasedBrokerModule.init(self)
        self.writer_stats.start_thread()
        self._init_completed = True
    
    
    def hook_tick(self, daemon):
        self._update_active_range()
    
    
    def want_brok(self, brok):
        return brok.type in ACCEPTED_BROK_TYPES
    
    
    def _update_active_range(self):
        if self._init_completed:
            self.database_connection.update_tick_info(get_now())
    
    def get_day_keep_data(self):
        return self.day_keep_data
    
    def get_raw_stats(self, param=''):
        try:
            seconds_of_stats = int(param)
        except:
            seconds_of_stats = 60
        
        if not self._init_completed:
            raw_stats = {'is_ready': False}
        else:
            raw_stats = self.writer_stats.get_raw_stats(seconds_of_stats)
        return raw_stats
