#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

# This Class is an example of an Scheduler module
# Here for the configuration phase AND running one

import cPickle
import glob
import os
import re
import shutil
import time

from shinken.basemodule import BaseModule

# Hack to make 0.5 retention file loading in a 0.6 version
# because the commandCall class was moved
import shinken
from shinken.commandcall import CommandCall
from shinken.log import LoggerFactory

shinken.objects.command.CommandCall = CommandCall

properties = {
    'daemons' : ['scheduler'],
    'type'    : 'pickle_retention_file',
    'external': False,
}


def get_instance(plugin):
    """
    Called by the plugin manager to get a broker
    """
    path = plugin.path
    instance = Pickle_retention_scheduler(plugin, path)
    return instance


# NOTE: DO NOT EDIT THIS UNTIL YOU KNOW WHAT YOUR ARE DOING
# PATH_LEVEL_SEPARATOR = '_--_'
SUFFIX_FORMAT = '_--_realm--%s_--_scheduler--%s_--_id--%s.retention'


class Pickle_retention_scheduler(BaseModule):
    def __init__(self, modconf, path):
        BaseModule.__init__(self, modconf)
        self.path = path
        self.logger.debug("Get a pickle retention scheduler module for plugin %s" % modconf.get_name())

    
    
    # We want to remove all part that are not allowed as path
    def _protect_path_string(self, path_part):
        path_part = str(path_part).lower()
        path_part = re.sub('[^\w\-_]', '', path_part)
        return path_part
    
    
    def _get_file_suffix(self, scheduler):
        realm = self._protect_path_string(scheduler.sched_daemon.realm)
        daemon_id = self._protect_path_string(scheduler.sched_daemon.daemon_id)
        instance_name = self._protect_path_string(scheduler.instance_name)
        
        suffix = SUFFIX_FORMAT % (realm, instance_name, daemon_id)
        return suffix
    
    
    def _select_from_possible_files(self, possible_files):
        if len(possible_files) == 1:
            return possible_files[0]
        # more old first
        possible_files.sort(key=os.path.getmtime)
        return possible_files[-1]  # so most recent is the last
    
    
    def _find_retention_file_to_load(self, scheduler):
        realm = self._protect_path_string(scheduler.sched_daemon.realm)
        daemon_id = self._protect_path_string(scheduler.sched_daemon.daemon_id)
        instance_name = self._protect_path_string(scheduler.instance_name)
        
        # We are looking in the order:
        # 1= realm + name + id
        # 2= if the old file before 2.05.03 is present, take it and move it
        # 3= realm + other name + id
        # 4= realm + name + other id
        # 5= other realm + name + id
        # 6= other realm + other name + id
        # And if more tha 1 files, take the most recent one
        
        # 1
        my_file_suffix = SUFFIX_FORMAT % (realm, instance_name, daemon_id)
        normal_pth = self.path + my_file_suffix
        if os.path.exists(normal_pth):
            self.logger.debug('Find_retention_file_to_load: case 1, founded my file %s' % normal_pth)
            return normal_pth
        
        # 2
        if os.path.exists(self.path):
            shutil.move(self.path, normal_pth)
            self.logger.info('We did migrate the old migration file (%s) to the new path (%s)' % (self.path, normal_pth))
            return normal_pth
        
        # 3
        possible_files = glob.glob(self.path + SUFFIX_FORMAT % (realm, '*', daemon_id))
        if possible_files:
            pth = self._select_from_possible_files(possible_files)
            self.logger.debug('Find_retention_file_to_load: case 3 (realm + other name + id), founded %s in the files: %s' % (pth, ','.join(possible_files)))
            return pth
        
        # 4
        possible_files = glob.glob(self.path + SUFFIX_FORMAT % (realm, instance_name, '*'))
        if possible_files:
            pth = self._select_from_possible_files(possible_files)
            self.logger.debug('Find_retention_file_to_load: case 4 (realm + name + other id), founded %s in the files: %s' % (pth, ','.join(possible_files)))
            return pth
        
        # 5
        possible_files = glob.glob(self.path + SUFFIX_FORMAT % ('*', instance_name, daemon_id))
        if possible_files:
            pth = self._select_from_possible_files(possible_files)
            self.logger.debug('Find_retention_file_to_load: case 5 (other realm + name + id), founded %s in the files: %s' % (pth, ','.join(possible_files)))
            return pth
        
        # 6
        possible_files = glob.glob(self.path + SUFFIX_FORMAT % ('*', '*', daemon_id))
        if possible_files:
            pth = self._select_from_possible_files(possible_files)
            self.logger.debug('Find_retention_file_to_load: case 6 (other realm + other name + id), founded %s in the files: %s' % (pth, ','.join(possible_files)))
            return pth
        
        self.logger.debug('Find_retention_file_to_load: cannot find matching retention files.')
        return None
    
    
    # main function that is called in the retention creation pass
    def hook_save_retention(self, sched):
        self.logger.debug("Asking me to update the retention objects")
        
        suffix = self._get_file_suffix(sched)
        retention_path = self.path + suffix
        
        # Now the flat file method
        try:
            start = time.time()
            # Open a file near the path, with .tmp extension
            # so in case of a problem, we do not lose the old one
            f = open(retention_path + '.tmp', 'wb')
            # Just put hosts/services because checks and notifications
            # are already link into
            # all_data = {'hosts': sched.hosts, 'services': sched.services}
            
            # We create a all_data dict with list of dict of retention useful
            # data of our hosts and services
            all_data = sched.get_retention_data()
            
            cPickle.dump(all_data, f, protocol=cPickle.HIGHEST_PROTOCOL)
            f.flush()
            os.fsync(f.fileno())
            f.close()
            # Now move the .tmp file to the real path
            shutil.move(retention_path + '.tmp', retention_path)
            end = time.time()
        except IOError, exp:
            err = "Retention file creation failed: %s" % str(exp)
            raise Exception(err)
        counts = {}
        for (k, l) in all_data.iteritems():
            try:
                counts[k] = len(l)
            except Exception:
                counts[k] = 1
        self.logger.info(u'Updating retention_file %s with elements: %s' % (retention_path, u' -- '.join([u'%s [ %d ]' % (k, c) for (k, c) in counts.iteritems()])))
        self.logger.info(u'Retention file saved in %.03f seconds' % (end - start))
        return True
    
    
    # Should return if it succeed in the retention load or not
    def hook_load_retention(self, sched):
        retention_path = self._find_retention_file_to_load(sched)
        
        if not retention_path or not os.path.exists(retention_path):
            self.logger.info('Skipping reading retention data from missing file %s' % self.path)
            return True
        self.logger.info("Reading retention data from retention_file %s" % retention_path)
        try:
            f = open(retention_path, 'rb')
            all_data = cPickle.load(f)
            f.close()
        except (EOFError, ValueError, IOError), exp:
            err = 'Cannot load retention: %s' % str(exp)
            self.logger.error(err)
            raise Exception(err)
        except (IndexError, TypeError), exp:
            err = "The file %s is not compatible with this module version: %s" % (retention_path, exp)
            self.logger.error(err)
            raise Exception(err)
        
        # call the scheduler helper function for restoring values
        sched.restore_retention_data(all_data)
        
        self.logger.info("Retention objects loaded successfully.")
        
        return True
